/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.autoupdate;

import java.lang.ref.*;
import java.util.*;

//import junit.framework.*;
import org.netbeans.junit.*;

import java.util.List;
import java.awt.Image;
import java.awt.datatransfer.Transferable;
import java.beans.*;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.jar.*;
import java.util.jar.Manifest;
import java.util.regex.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

//import org.openide.ErrorManager;



public class NbmsTest extends AbstractTestHid {
    public NbmsTest(String name) {
        super (name);
    }
    
    public static void main(java.lang.String[] args) {
        if (args.length == 1) {
            junit.textui.TestRunner.run(new NbmsTest (args[0]));
        } else {
            junit.textui.TestRunner.run(new NbTestSuite(NbmsTest.class));
        }
        System.exit (0);
    }    
    
    
    
    public void testNBMFileIsGenerated () throws Exception {
        File f = generateBasicNbm ("3.8");

        ModuleUpdate mu = ModuleUpdate.getModuleUpdate (f);
        download (mu);
        installNBM (Downloader.getNBM (mu));
        
        
        Map userFiles = findFiles (userDir);
        assertNotNull ("Contains x.jar", userFiles.get ("modules/x.jar"));
        assertNotNull ("Contains y.jar", userFiles.get ("modules/y.jar"));
        assertNotNull ("Contains update tracking", userFiles.get ("update_tracking/org-openide.xml"));
        
        assertEquals ("Empty platform dir", 0, findFiles (platformDir).size ());
    }

    public void testNBMFileIsInstalledWithNonCanonicalPaths () throws Exception {
        File f = generateBasicNbm ("3.8");

        ModuleUpdate mu = ModuleUpdate.getModuleUpdate (f);
        download (mu);
        
        File file = Downloader.getNBM (mu);
        File noncanonical = new File (
            new File (
                new File (file.getParentFile (), ".."), 
                file.getParentFile ().getName ()
            ),
            file.getName ()
        );
        assertFalse ("Files are different", noncanonical.equals (file));
        assertEquals ("But point to the same file", noncanonical.getCanonicalFile (), file.getCanonicalFile ());
        
        installNBM (noncanonical);
        
        
        Map userFiles = findFiles (userDir);
        assertNotNull ("Contains x.jar", userFiles.get ("modules/x.jar"));
        assertNotNull ("Contains y.jar", userFiles.get ("modules/y.jar"));
        assertNotNull ("Contains update tracking", userFiles.get ("update_tracking/org-openide.xml"));
        
        assertEquals ("Empty platform dir", 0, findFiles (platformDir).size ());
    }
    
    public void testIfNoUpdateTrackingThenItCannotBeInstalledGlobally () throws Exception {
        File f = generateBasicNbm ("3.8");

        ModuleUpdate mu = ModuleUpdate.getModuleUpdate (f);
        
        assertNull ("Cannot be installed globally", mu.findInstallDirectory ());
        assertTrue ("We should not be allowed to install", Downloader.bannedWriteToInstall (mu));
    }

    public void testIfUpdateTrackingExistsButNoAutoUpdateFilePresentThenItCannotBeInstalledGlobally () throws Exception {
        File f = generateBasicNbm ("3.8");

        File update = new File (platformDir, "update_tracking");
        update.mkdirs ();
        File noAU = new File(platformDir, ".noautoupdate");
        noAU.createNewFile();

        ModuleUpdate mu = ModuleUpdate.getModuleUpdate (f);
        
        assertNull ("Cannot be installed globally", mu.findInstallDirectory ());
        assertTrue ("We should not be allowed to install", Downloader.bannedWriteToInstall (mu));
    }
    
    public void testIfUpdateTrackingExistsInstallIntoSharedDir () throws Exception {
        File f = generateBasicNbm ("3.8");
        
        File update = new File (platformDir, "update_tracking");
        update.mkdirs ();

        ModuleUpdate mu = ModuleUpdate.getModuleUpdate (f);
        
        assertEquals ("Install globally", platformDir, mu.findInstallDirectory ());
    }
    
    public void testUpdateTrackingExistsNbDirsAsWell () throws Exception {
        initClusters ();
        
        File f = generateBasicNbm ("3.8");
        
        File update = new File (platformDir, "update_tracking");
        update.mkdirs ();
        update = new File (clusterDir, "update_tracking");
        update.mkdirs ();
        update = new File (nextDir, "update_tracking");
        update.mkdirs ();

        ModuleUpdate mu = ModuleUpdate.getModuleUpdate (f);
        
        assertEquals (
            "If the NBM is not installed yet, then it goes to the first " +
            "netbeans.dir cluster", 
            clusterDir, mu.findInstallDirectory ()
        );
        
        download (mu);
        
        File downloadedNbm = Downloader.getNBM (mu);
        assertTrue ("Downloaded NBM " + downloadedNbm + " exists.", downloadedNbm.exists ());
        
        mu.setToInstallDir (true);
        assertTrue ("Moved to install cluster", Downloader.tryMove (mu));
        assertTrue (mu.findInstallDirectory ().toString ().endsWith ("clstr"));
        assertFalse ("Downloaded NBM " + downloadedNbm + " is moved to install dir and removed in user dir.", downloadedNbm.exists ());
        File movedNbm = Downloader.getMovedNBM (mu);
        installNBM (movedNbm);
        
        
        assertEquals ("Empty user dir", 0, findFiles (userDir).size ());
        assertEquals ("Empty platform dir", 0, findFiles (platformDir).size ());
        
        Map platFiles = findFiles (clusterDir);
        assertNotNull ("Contains x.jar", platFiles.get ("modules/x.jar"));
        assertNotNull ("Contains y.jar", platFiles.get ("modules/y.jar"));
        assertNotNull ("Contains update tracking", platFiles.get ("update_tracking/org-openide.xml"));
        
    }

    public void testCheckWeCannotInstallIntoUserDirIfWeContainSomethingInstalledInCore () throws Exception {
        doTestInstallToUserDirForPrefix ("core", false, false);
    }
    
    public void testCheckWeCannotInstallIntoUserDirIfWeContainSomethingInstalledInLib () throws Exception {
        doTestInstallToUserDirForPrefix ("lib", false, false);
    }
        
    public void testCheckWeCanInstallIntoUserDirIfWeContainSomethingInstalledInCore () throws Exception {
        doTestInstallToUserDirForPrefix ("core", true, true);
    }
    
    public void testCheckWeCannotInstallIntoUserDirIfWeWantToInstallInLib () throws Exception {
        doTestInstallToUserDirForPrefix ("lib", true, false);
    }
        
    private void doTestInstallToUserDirForPrefix (String prefix, boolean newModule, boolean can) throws Exception {
        File update = new File (platformDir, prefix);
        update.mkdirs ();        
                
        if (! newModule) {
            File f = new File (update, "dummy.jar");
            f.createNewFile ();
        }
        
        File f = generateBasicNbm (
            "3.9", // new version
            new String[] {
                "netbeans/" + prefix,
                "netbeans/" + prefix + "/dummy.jar"
            }
        ); 
        ModuleUpdate mu = ModuleUpdate.getModuleUpdate (f);
        download (mu);
        assertEquals (
            "No update_tracking, so we shouldn't install globally", 
            null, mu.findInstallDirectory ()
        );

        if (newModule && can) {
            assertFalse ("By default we need to install to user dir", mu.isToInstallDir ());
        } else {
            assertTrue ("By default we need to install to install dir", mu.isToInstallDir ());
        }
        try {
            mu.setToInstallDir (false);
        } catch (IllegalArgumentException iae) {
            // can ignore this, Autoupdate in GUI shouldn't set to false when forcedGlobal
        }
        assertEquals ("Can change toInstallDir?", can, ! mu.isToInstallDir ());
    }
    
    public void testInstallUpdate () throws Exception {
        
        File update = new File (platformDir, "update_tracking");
        update.mkdirs ();
        
        File f = generateBasicNbm ("3.8");
        ModuleUpdate mu = ModuleUpdate.getModuleUpdate (f);
        download (mu);
        
        assertEquals (
            "We do not have any clusters, we go into platformDir", 
            platformDir, mu.findInstallDirectory ()
        );
        
        mu.setToInstallDir (true);
        
        assertTrue ("Moved to install cluster", Downloader.tryMove (mu));
        File movedNbm = Downloader.getMovedNBM (mu);
        installNBM (movedNbm);
        
        Map platFiles = findFiles (platformDir);
        assertNotNull ("Contains x.jar", platFiles.get ("modules/x.jar"));
        assertNotNull ("Contains y.jar", platFiles.get ("modules/y.jar"));
        assertNotNull ("Contains update tracking", platFiles.get ("update_tracking/org-openide.xml"));
        
        assertEquals ("Empty user dir", 0, findFiles (userDir).size ());
        
        
        update = new File (clusterDir, "update_tracking");
        update.mkdirs ();
        update = new File (nextDir, "update_tracking");
        update.mkdirs ();
        initClusters ();
        
        f = generateBasicNbm (
            "3.9", // new version
            new String[] {
                "netbeans/modules/kuk.jar"
            }
        ); 
        mu = ModuleUpdate.getModuleUpdate (f);
        download (mu);
        mu.setToInstallDir (true);
        
        
        File id = mu.findInstallDirectory ();
        assertEquals (
            "If the NBM is installed it will offer override", platformDir, id
        );
        
        assertTrue ("Moved to platform cluster", Downloader.tryMove (mu));
        movedNbm = Downloader.getMovedNBM (mu);
        installNBM (movedNbm);
        
        platFiles = findFiles (platformDir);
        assertNull ("No x.jar anymore", platFiles.get ("modules/x.jar"));
        assertNull ("No y.jar anymore", platFiles.get ("modules/y.jar"));
        assertNotNull ("Contains update tracking", platFiles.get ("update_tracking/org-openide.xml"));
        assertNotNull ("kuk.jar is there", platFiles.get ("modules/kuk.jar"));
        
        assertEquals ("Empty userdir still", 0, findFiles (userDir).size ());
        
        org.netbeans.updater.UpdateTracking ut;
        ut = org.netbeans.updater.UpdateTracking.getTracking (platformDir, false);
        String cb = mu.getInfoCodenamebase ();
        assertTrue ("Installed", ut.isModuleInstalled (cb));
            
    }
    
    public void testInstallLaterXMLCreatedInSharedDirIssue47077 () throws Exception {
        initClusters ();
        
        File f = generateBasicNbm ("3.8");
        
        File update = new File (platformDir, "update_tracking");
        update.mkdirs ();
        update = new File (clusterDir, "update_tracking");
        update.mkdirs ();
        update = new File (nextDir, "update_tracking");
        update.mkdirs ();

        ModuleUpdate mu = ModuleUpdate.getModuleUpdate (f);
        download (mu);
        mu.setToInstallDir (true);
        assertTrue ("Moved to platform cluster", Downloader.tryMove (mu));
        File movedNbm = Downloader.getMovedNBM (mu);
        
        PreparedModules.getPrepared().addModule( mu );
        PreparedModules.getPrepared ().write ();
        
        assertEquals ("Empty user dir", 0, findFiles (userDir).size ());
        assertEquals ("Empty platform dir", 0, findFiles (platformDir).size ());
        
        Map platFiles = findFiles (clusterDir);
        
        assertNotNull ("Contains install_later.xml", platFiles.get ("update/download/install_later.xml"));
        
        if (platFiles.size () != 2) {
            fail ("Unexcepted files (else than nbm and install_later.xml): " + platFiles);
        }
    }

    public void testInstallLaterXMLCreatedIssue47077 () throws Exception {
        initClusters ();
        
        File f = generateBasicNbm ("3.8");
        
        File update = new File (platformDir, "update_tracking");
        update.mkdirs ();
        update = new File (clusterDir, "update_tracking");
        update.mkdirs ();
        update = new File (nextDir, "update_tracking");
        update.mkdirs ();

        ModuleUpdate mu = ModuleUpdate.getModuleUpdate (f);
        
        assertEquals (
            "If the NBM is not installed yet, then it goes to the first " +
            "netbeans.dir cluster", 
            clusterDir, mu.findInstallDirectory ()
        );
        
        download (mu);
        
        PreparedModules.getPrepared().addModule( mu );
        PreparedModules.getPrepared ().write ();
        
        assertEquals ("Empty cluster dir", 0, findFiles (clusterDir).size ());
        assertEquals ("Empty platform dir", 0, findFiles (platformDir).size ());
        
        Map platFiles = findFiles (userDir);
        
        assertNotNull ("Contains install_later.xml", platFiles.get ("update/download/install_later.xml"));
        
        if (platFiles.size () != 2) {
            fail ("Unexcepted files (else than nbm and install_later.xml): " + platFiles);
        }
    }

    //
    // Utilities
    //
    
    /** Generates sample basic NBM.
     * @param revision a number like 4.15
     */
    private File generateBasicNbm (String revision) throws Exception {
        return generateBasicNbm (revision, new String[] { 
            "netbeans/modules/x.jar", 
            "netbeans/modules/y.jar",
            "netbeans/platform5/core/openide.jar"
        });
    }
    
    /** Generates sample basic NBM.
     * @param revision a number like 4.15
     * @param list of files that should be in the NBM
     */
    private File generateBasicNbm (String revision, String[] fileList) throws Exception {
        String manifest =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
            "<!DOCTYPE module PUBLIC \"-//NetBeans//DTD Autoupdate Module Info 2.0//EN\" \"http://www.netbeans.org/dtds/autoupdate-info-2_0.dtd\">\n" +
            "<module codenamebase=\"org.openide\"\n" +
            "homepage=\"http://openide.netbeans.org/\"\n" +
            "distribution=\"http://www.netbeans.org/download/nbms/40/openide.nbm\"\n" +
            "license=\"standard-nbm-license.txt\"\n" +
            "downloadsize=\"0\"\n" +
            ">\n" +
            "<manifest OpenIDE-Module=\"org.openide/1\"\n" +
            "OpenIDE-Module-Display-Category=\"Infrastructure\"\n" +
            "OpenIDE-Module-Implementation-Version=\"031128\"\n" +
            "OpenIDE-Module-Long-Description=\"This pseudo-module represents the Open APIs (org.openide.*), which are used by all other NetBeans modules.\"\n" +
            "OpenIDE-Module-Name=\"Open APIs\"\n" +
            "OpenIDE-Module-Short-Description=\"The NetBeans Open APIs.\"\n" +
            "OpenIDE-Module-Specification-Version=\"" + revision + "\"\n" +
            "/>\n" +
            "<license name=\"standard-nbm-license.txt\"><![CDATA[This module is part of NetBeans and is open-source.\n" +
            "You can see http://www.netbeans.org/license.html for details.\n" +
            "\n" +
            "You may use the binary however you like. The source file license is:\n" +
            "\n" +
            "The Original Code is NetBeans. The Initial Developer of the Original\n" +
            "Code is Sun Microsystems, Inc. Portions Copyright 1997-2003 Sun\n" +
            "Microsystems, Inc. All Rights Reserved.\n" +
            "]]></license>\n" +
            "</module>\n";

        File f = generateNBM (fileList, manifest);
        
        return f;
    }
    
}
