/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.core;

import java.beans.PropertyVetoException;
import java.io.File;
import java.io.IOException;
import java.lang.ref.WeakReference;
import junit.framework.*;
import junit.textui.TestRunner;
import org.openide.filesystems.*;
import org.openide.nodes.Node;
import org.openide.loaders.DataObject;
import java.util.Enumeration;
import org.netbeans.junit.NbTestCase;
import org.openide.loaders.DataFolder;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.LocalFileSystem;
import org.openide.filesystems.Repository;
import org.openide.cookies.InstanceCookie;
import org.openide.loaders.InstanceDataObject;
import org.openide.modules.ModuleInfo;
import org.openide.util.Lookup;
import org.openide.util.Task;

/** Test Plain top manager. Must be run externally.
 * @author Jaroslav Tulach
 */
public class AutomountSupportTest extends TestCase {
    /** folder to store objects to */
    private DataFolder mount;
    /** task to wait for */
    private Task task;
    
    public AutomountSupportTest(String name) {
        super(name);
    }
    
    public static void main(String[] args) {
        TestRunner.run(new TestSuite(AutomountSupportTest.class));
        // In case it fails to shut itself down afterwards:
        System.exit(0);
    }
    
    /** If execution fails we wrap the exception with 
     * new log message.
     */
    protected void runTest () throws Throwable {
        ErrManager.messages.append ("Starting test ");
        ErrManager.messages.append (getName ());
        ErrManager.messages.append ('\n');
        
        try {
            super.runTest ();
        } catch (AssertionFailedError ex) {
            AssertionFailedError ne = new AssertionFailedError (ex.getMessage () + " Log:\n" + ErrManager.messages);
            ne.setStackTrace (ex.getStackTrace ());
            throw ne;
        }
    }
    
    /**/
    protected void setUp() throws Exception {
        System.setProperty ("org.openide.loaders.FolderList", "-5");
        Lookup.getDefault ().lookup (ModuleInfo.class);
        
        Object err = Lookup.getDefault ().lookup (ErrManager.class);
        if (err == null) {
            InstanceDataObject.create (
                DataFolder.findFolder (FileUtil.createFolder (
                    Repository.getDefault ().getDefaultFileSystem ().getRoot (), "Services"
                )), "LoggingErrManager", ErrManager.class
            );
        }
        err = Lookup.getDefault ().lookup (ErrManager.class);
        assertNotNull ("Error manager shall be in lookup", err);
        
        task = AutomountSupport.initialize ();
        task.waitFinished();
        
        FileObject fo = Repository.getDefault().getDefaultFileSystem().findResource("Mount"); // NOI18N        
        mount = DataFolder.findFolder (fo);
        
        DataObject[] arr = mount.getChildren ();
        for (int i = 0; i < arr.length; i++) {
            arr[i].delete ();
        }
        if (mount.getChildren ().length > 0) {
            fail ("Mount dir is not empty");
        }
        task.waitFinished();

        
        FileSystem[] arr2 = Repository.getDefault ().toArray ();
        for (int i = 0; i < arr2.length; i++) {
            if (!arr2[i].isDefault ()) {
                Repository.getDefault ().removeFileSystem (arr2[i]);
            }
        }
        task.waitFinished();
    }
    
    
    /** Checks whether a change in repository is propagated to mount dir.
     */
    public void testSavedAndDeleted () throws Exception {
        LocalFileSystem fs = new LocalFileSystem ();
        
        Repository.getDefault ().addFileSystem (fs);
        
        task.waitFinished ();
        
        DataObject obj = findInFolder (fs, mount);
        if (obj == null) {
            fail ("Local filesystem has not been written to disk");
        }
        
        obj.delete ();
        
        task.waitFinished ();
        
        int i = findInRepository (fs);
        if (i != -1) {
            fail ("Filesystem is still in repository. Index: " + i);
        }
    }
    
    
    
    /** Checks whether a change in mount dir is propagated to repository.
     */
    public void testCreateAndRemove () throws Exception {
        LocalFileSystem fs = new LocalFileSystem ();
        
        InstanceDataObject obj = InstanceDataObject.create (mount, null, fs, null);
        
        task.waitFinished ();

        int indx = findInRepository (fs);
        if (indx == -1) {
            fail ("FileSystem not found in repository");
        }

        Repository.getDefault ().removeFileSystem (fs);
        
        task.waitFinished ();

        if (obj.isValid ()) {
            fail ("Data object should be invalidated");
        }
        
        DataObject o = findInFolder (fs, mount);
        if (o != null) {
            fail ("There is a still object representing the filesystem: " + o);
        }
    }


    public void testCreateJustInMount () throws Exception {
        mount.getPrimaryFile().createFolder("sub1");        
        mount.getPrimaryFile().createFolder("sub2");
        mount.getPrimaryFile().createFolder("sub3");
        
        task.waitFinished ();
        
        FileSystem fs = new LocalFileSystem ();
        Repository.getDefault ().addFileSystem (fs);
        
        task.waitFinished ();
        
        int fsOccurenceCount = 0;
        Enumeration en = mount.children(true);
        while (en.hasMoreElements()) {
            DataObject dataObject = (DataObject) en.nextElement();
            InstanceCookie ic = (InstanceCookie)dataObject.getCookie(InstanceCookie.class);
            if (ic != null) {
                FileSystem fsInstance = (FileSystem)ic.instanceCreate();
                if (fsInstance == fs) {
                    // must be in mount folder
                    assertEquals(dataObject.getFolder(), mount);
                    assertEquals(++fsOccurenceCount, 1);
                }
            }
        }
    }
        
    public void testReorderRepository () throws Exception {
        LocalFileSystem lfs = new LocalFileSystem ();
        JarFileSystem jfs = new JarFileSystem ();
        MultiFileSystem mfs = new MultiFileSystem(new FileSystem [] {lfs, jfs});
        
        Repository rep = Repository.getDefault();
        
        rep.addFileSystem(lfs);        
        rep.addFileSystem(jfs);
        rep.addFileSystem(mfs);
        
        /** original order*/
        task.waitFinished ();
        assertOrder(rep.toArray());

        /** reorder*/        
        Repository.getDefault().reorder(new int[] { 0, 3, 2, 1});
        
        task.waitFinished ();
        assertOrder (new FileSystem[] {
                    Repository.getDefault().getDefaultFileSystem(),
                    mfs,
                    jfs,
                    lfs
                });
    }

    private void assertOrder (FileSystem[] fss) throws Exception {
        DataObject[] childs = mount.getChildren();
        boolean ommitedDefFs = false;
        boolean ok = true;
        for (int i = 0; i < fss.length; i++) {
            FileSystem fs = fss[i];
            if (fs.isDefault()) {
                ommitedDefFs = true; continue;
            }
            int idx = (ommitedDefFs && i > 0) ? (i - 1) : i; 
            InstanceCookie ic = (InstanceCookie)childs[idx].getCookie(InstanceCookie.class); 
            assertNotNull("Object " + childs[idx] + " does not have cookie", ic);
            FileSystem mountFs = (FileSystem)ic.instanceCreate();
            if (fs != mountFs) {
                ok = false;
            }
        }
        
        if (ok) {
            return;
        }
                
        //
        // report failure
        //
        StringBuffer sb = new StringBuffer ();
        sb.append ("Failure in order. Repository contains:");
        for (int i = 0; i < fss.length; i++) {
            sb.append ("\n  ");
            sb.append (i);
            sb.append ("th = ");
            sb.append (fss[i]);
        }
        sb.append ("\nOrder on disk is:");
        for (int i = 0; i < childs.length; i++) {
            sb.append ("\n  ");
            sb.append (i);
            sb.append ("th = ");
            sb.append (childs[i]);
            InstanceCookie ic = (InstanceCookie)childs[i].getCookie(InstanceCookie.class); 
            if (ic == null) {
                sb.append (" cookie is null");
                continue;
            }
            try {
                sb.append (" fs: ");
                sb.append (ic.instanceCreate());
            } catch (Exception ex) {
                sb.append (" exception: " + ex.getMessage ());
            }
        }
        fail (sb.toString ());
    }
    
    /** Tries to fool the automount by adding and removing the filesystem.
     */
    public void testFoolByRepetition () throws Exception {
        LocalFileSystem fs = new LocalFileSystem ();

        for (int i = 0; i < 100; i++) {
            Repository.getDefault ().addFileSystem(fs);
            
            if (findInRepository (fs) == -1) {
                fail ("Filesystem is not there after " + i + " tries");
            }
            
            Repository.getDefault ().removeFileSystem (fs);
            
            int indx = findInRepository (fs);
            if (indx != -1) {
                fail ("Filesystem is there after " + i + " tries with index " + indx);
            }
        }

        // wait
        task.waitFinished ();
        
        
        if (findInRepository (fs) != -1) {
            fail ("FileSystem remained there after the test");
        }
        
        if (findInFolder (fs, mount) != null) {
            fail ("There is a data object on the disk");
        }
    }
    
    /** Tries to fool the automount by adding and removing the filesystem.
     */
    public void testReaddingTheSame() throws Exception {
        LocalFileSystem fs1 = new LocalFileSystem ();

        Repository.getDefault ().addFileSystem(fs1);
        
        if (findInRepository (fs1) == -1) {
            fail ("Filesystem1 is not there.");
        }

        Repository.getDefault ().removeFileSystem (fs1);
        Repository.getDefault ().addFileSystem(fs1);

        DataObject d = null;
        
        // wait to finish
        task.waitFinished ();
        
        if ((d = findInFolder (fs1, mount)) == null) {
            fail ("A data object is not on the disk (1)");
        }
        d.delete();
        
        if (findInFolder (fs1, mount) != null) {
            fail ("A data object is still on the disk (2)");
        }
        
        // wait to finish
        task.waitFinished ();
        
        if (findInRepository (fs1) != -1) {
            fail ("FileSystem remained there after data object was deleted. (3)");
        }
    }
    
       public void testFileSystemMemoryLeakOnUnmount39380() {
            File dir = new File (System.getProperty("xtest.sketchpad"), "localfs");
            dir.mkdir();

            LocalFileSystem fs = new LocalFileSystem();
            try {
                Repository r = Repository.getDefault();
                fs.setRootDirectory(dir);
                r.addFileSystem(fs);

                WeakReference ref = new WeakReference (fs);
                r.removeFileSystem(fs);
                fs = null;
                NbTestCase.assertGC("filesystem can disappear", ref);
            }
            catch (IOException ioe) {
                fail ("Filesystem mounting failed - "+ioe.getLocalizedMessage());
            }
            catch (PropertyVetoException ioe) {
                fail ("Filesystem mounting failed - "+ioe.getLocalizedMessage());
            }
       }
       
    
    /** Find in repository.
     * @param fs filesystem to find out in repository
     * @return index of found fs or -1
     */
    private static int findInRepository (FileSystem fs) {
        FileSystem[] fsArr = Repository.getDefault ().toArray ();
        for (int i = 0; i < fsArr.length; i++) {
            if (fsArr[i] == fs) {
                return i;
            }
        }
        return -1;
    }
    
    /** Find in folder
     * @param fs filesystem to find in folder
     * @return the data object or null
     */
    private DataObject findInFolder (FileSystem fs, DataFolder mount) throws Exception {
        DataObject[] arr = mount.getChildren ();
        for (int i = 0; i < arr.length; i++) {
            InstanceCookie ic = (InstanceCookie)arr[i].getCookie (InstanceCookie.class);
            
            if (ic != null && ic.instanceCreate () == fs) {
                return arr[i];
            }
        }
        
        return null;
    }

    //
    // Logging support
    //
    public static final class ErrManager extends org.openide.ErrorManager {
        public static final StringBuffer messages = new StringBuffer ();
        
        private String prefix;
        
        public ErrManager () {
            this (null);
        }
        public ErrManager (String prefix) {
            this.prefix = prefix;
        }
        
        public Throwable annotate (Throwable t, int severity, String message, String localizedMessage, Throwable stackTrace, java.util.Date date) {
            return t;
        }
        
        public Throwable attachAnnotations (Throwable t, org.openide.ErrorManager.Annotation[] arr) {
            return t;
        }
        
        public org.openide.ErrorManager.Annotation[] findAnnotations (Throwable t) {
            return null;
        }
        
        public org.openide.ErrorManager getInstance (String name) {
            if (
                name.startsWith ("org.netbeans.core.AutomountSupport") ||
                name.startsWith ("org.openide.loaders.FolderList") ||
                name.startsWith ("org.openide.loaders.FolderInstance")
            ) {
                return new ErrManager ('[' + name + ']');
            } else {
                // either new non-logging or myself if I am non-logging
                return new ErrManager ();
            }
        }
        
        public void log (int severity, String s) {
            if (prefix != null) {
                messages.append (prefix);
                messages.append (s);
                messages.append ('\n');
            }
        }
        
        public void notify (int severity, Throwable t) {
        }
        
    } 
    
    
}
