/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.core;

import org.netbeans.junit.*;
import junit.textui.TestRunner;
import org.openide.util.io.NbMarshalledObject;

/** Tests HTTP Proxy settings.
 *
 * @author Jiri Rechtacek
 * @see http://www.netbeans.org/issues/show_bug.cgi?id=51641
 */
public class HttpSettingsTest extends NbTestCase {
    private IDESettings settings;
    private static String SYSTEM_PROXY_HOST = "system.cache.org";
    private static String SYSTEM_PROXY_PORT = "777";
    private static String USER_PROXY_HOST = "my.webcache";
    private static String USER_PROXY_PORT = "8080";

    private static String SILLY_USER_PROXY_HOST = "http://my.webcache";
    private static String SILLY_SYSTEM_PROXY_HOST = "http://system.cache.org";
    
    private static String NETBEANS_ORG = "*.netbeans.org";
    private static String OTHER_ORG = "*.other.org";
    
    public HttpSettingsTest (String name) {
        super (name);
    }
    
    public static void main(String[] args) {
        TestRunner.run (new NbTestSuite (HttpSettingsTest.class));
    }
    
    protected void setUp () throws Exception {
        super.setUp ();
        System.setProperty ("netbeans.system_http_proxy", SYSTEM_PROXY_HOST + ":" + SYSTEM_PROXY_PORT);
        System.setProperty ("netbeans.system_http_non_proxy_hosts", "*.other.org");
        System.setProperty ("http.nonProxyHosts", NETBEANS_ORG + ',' + NETBEANS_ORG);
        settings = (IDESettings)IDESettings.findObject(IDESettings.class, true);
        settings.initialize ();
        settings.setUserProxyHost (USER_PROXY_HOST);
        settings.setUserProxyPort (USER_PROXY_PORT);
    }
    
    public void testDirectConnection () {
        settings.setProxyType (IDESettings.DIRECT_CONNECTION);
        assertEquals ("Proxy type DIRECT_CONNECTION.", IDESettings.DIRECT_CONNECTION, settings.getProxyType ());
        assertEquals ("No Proxy Host set.", "", System.getProperty (IDESettings.KEY_PROXY_HOST));
        assertEquals ("No Proxy Port set.", "", System.getProperty (IDESettings.KEY_PROXY_PORT));
    }
    
    public void testAutoDetectProxy () {
        settings.setProxyType (IDESettings.AUTO_DETECT_PROXY);
        assertEquals ("Proxy type AUTO_DETECT_PROXY.", IDESettings.AUTO_DETECT_PROXY, settings.getProxyType ());
        assertEquals ("System Proxy Host: ", SYSTEM_PROXY_HOST, System.getProperty (IDESettings.KEY_PROXY_HOST));
        assertEquals ("System Proxy Port: ", SYSTEM_PROXY_PORT, System.getProperty (IDESettings.KEY_PROXY_PORT));
    }
    
    public void testManualSetProxy () {
        settings.setProxyType (IDESettings.MANUAL_SET_PROXY);
        assertEquals ("Proxy type MANUAL_SET_PROXY.", IDESettings.MANUAL_SET_PROXY, settings.getProxyType ());
        assertEquals ("Manual Set Proxy Host from IDESettings: ", USER_PROXY_HOST, settings.getProxyHost ());
        assertEquals ("Manual Set Proxy Port from IDESettings: ", USER_PROXY_PORT, settings.getProxyPort ());
        assertEquals ("Manual Set Proxy Host from System.getProperty(): ", USER_PROXY_HOST, System.getProperty (IDESettings.KEY_PROXY_HOST));
        assertEquals ("Manual Set Proxy Port from System.getProperty(): ", USER_PROXY_PORT, System.getProperty (IDESettings.KEY_PROXY_PORT));
    }
    
    public void testHttpSettingsSerialization () throws Exception {
        assertEquals ("Original user proxy host", USER_PROXY_HOST, settings.getProxyHost ());
        assertEquals ("Original user proxy port", USER_PROXY_PORT, settings.getProxyPort ());
        IDESettings deserializedSettings = (IDESettings) new NbMarshalledObject (settings).get ();
        assertEquals ("Original user proxy host returned from deserialized IDESettings", USER_PROXY_HOST, deserializedSettings.getProxyHost ());
        assertEquals ("Original user proxy port returned from deserialized IDESettings", USER_PROXY_PORT, deserializedSettings.getProxyPort ());
        deserializedSettings.setUserProxyHost ("new.cache");
        deserializedSettings.setUserProxyPort ("80");
        deserializedSettings.setUserNonProxyHosts ("*.mydomain.org");
        IDESettings againDeserializedSettings = (IDESettings) new NbMarshalledObject (deserializedSettings).get ();
        assertEquals ("New user proxy host returned from deserialized IDESettings after change", "new.cache", againDeserializedSettings.getProxyHost ());
        assertEquals ("New user proxy port returned from deserialized IDESettings after change", "80", againDeserializedSettings.getProxyPort ());
        assertEquals ("New user non proxy hosts returned from deserialized IDESettings after change", "*.mydomain.org", againDeserializedSettings.getNonProxyHosts ());
    }
    
    public void testIfTakeUpNonProxyFromProperty () {
        assertTrue (NETBEANS_ORG + " in one of Non-Proxy hosts.", settings.getNonProxyHosts ().indexOf (NETBEANS_ORG) != -1);
    }
    
    public void testNonProxy () {
        assertEquals ("The IDESettings takes as same value as System properties in initial.", System.getProperty ("http.nonProxyHosts"), settings.getNonProxyHosts ());
        
        // change value in IDESettings
        settings.setProxyType (IDESettings.MANUAL_SET_PROXY);
        settings.setUserNonProxyHosts ("myhost.mydomain.net");
        assertEquals ("IDESettings returns new value.", "myhost.mydomain.net", settings.getNonProxyHosts ());
        assertEquals ("System property http.nonProxyHosts was changed as well.", settings.getNonProxyHosts (), System.getProperty ("http.nonProxyHosts"));
        
        // switch proxy type to DIRECT_CONNECTION
        settings.setProxyType (IDESettings.DIRECT_CONNECTION);
        assertFalse ("IDESettings doesn't return new value if DIRECT_CONNECTION set.", "myhost.mydomain.net".equals (settings.getNonProxyHosts ()));
        assertEquals ("System property http.nonProxyHosts was changed as well.", settings.getNonProxyHosts (), System.getProperty ("http.nonProxyHosts"));
        
        // switch proxy type back to MANUAL_SET_PROXY
        settings.setProxyType (IDESettings.MANUAL_SET_PROXY);
        assertEquals ("IDESettings again returns new value.", "myhost.mydomain.net", settings.getNonProxyHosts ());
        assertEquals ("System property http.nonProxyHosts was changed as well.", settings.getNonProxyHosts (), System.getProperty ("http.nonProxyHosts"));
        
        // switch proxy type to AUTO_DETECT_PROXY
        settings.setProxyType (IDESettings.AUTO_DETECT_PROXY);
        assertFalse ("IDESettings doesn't return new value if AUTO_DETECT_PROXY set.", "myhost.mydomain.net".equals (settings.getNonProxyHosts ()));
        assertEquals ("System property http.nonProxyHosts was changed as well.", settings.getNonProxyHosts (), System.getProperty ("http.nonProxyHosts"));
                
        // switch proxy type back to MANUAL_SET_PROXY
        settings.setProxyType (IDESettings.MANUAL_SET_PROXY);
        assertEquals ("IDESettings again returns new value.", "myhost.mydomain.net", settings.getNonProxyHosts ());
        assertEquals ("System property http.nonProxyHosts was changed as well.", settings.getNonProxyHosts (), System.getProperty ("http.nonProxyHosts"));
    }
    
    public void testAvoidDuplicateNonProxySetting () {
        settings.setProxyType (IDESettings.MANUAL_SET_PROXY);
        assertTrue (NETBEANS_ORG + " is among Non-proxy hosts detect from OS.", settings.getNonProxyHosts ().indexOf (NETBEANS_ORG) != -1);
        assertFalse (NETBEANS_ORG + " is in Non-Proxy hosts only once.", settings.getNonProxyHosts ().indexOf (NETBEANS_ORG) < settings.getNonProxyHosts ().lastIndexOf (NETBEANS_ORG));
        assertEquals ("System property http.nonProxyHosts was changed as well.", settings.getNonProxyHosts (), System.getProperty ("http.nonProxyHosts"));
    }
    
    public void testReadNonProxySettingFromSystem () {
        settings.setProxyType (IDESettings.AUTO_DETECT_PROXY);
        assertTrue (OTHER_ORG + " is among Non-proxy hosts detect from OS.", settings.getNonProxyHosts ().indexOf (OTHER_ORG) != -1);
        assertEquals ("System property http.nonProxyHosts was changed as well.", settings.getNonProxyHosts (), System.getProperty ("http.nonProxyHosts"));
    }
    
}
