/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.db.api.explorer;

import java.awt.datatransfer.DataFlavor;
import org.netbeans.api.db.explorer.DatabaseConnection;
import org.netbeans.api.db.explorer.JDBCDriver;
import org.netbeans.modules.db.explorer.DbMetaDataTransferProvider;

/**
 * This class contains data flavors and classes for transferring database metadata
 * from the Database Explorer. The current version only supports transferring
 * database tables. Support for other database objects, such as columns or views
 * might be added in the future.
 *
 * @author Andrei Badea
 *
 * @since 1.1
 */
public final class DatabaseMetaDataTransfer {

    /**
     * The {@link DataFlavor} representing a database table.
     */
    public static DataFlavor TABLE_FLAVOR;
    
    /**
     * The {@link DataFlavor} representing a database view.
     */
    public static DataFlavor VIEW_FLAVOR;


    /**
     * The implementation of DbMetaDataTransferProvider, returned by
     * {@link #getProvider}.
     */
    private static DbMetaDataTransferProvider PROVIDER;

    static {
        try {
            TABLE_FLAVOR = new DataFlavor("application/x-java-netbeans-dbexplorer-table;class=org.netbeans.modules.db.api.explorer.DatabaseMetaDataTransfer$Table"); // NOI18N
            VIEW_FLAVOR = new DataFlavor("application/x-java-netbeans-dbexplorer-view;class=org.netbeans.modules.db.api.explorer.DatabaseMetaDataTransfer$View"); // NOI18N
        } catch (ClassNotFoundException e) {
            throw new AssertionError(e);
        }
    }

    /**
     * This method is referenced from the layer to register the provider
     * implementation.
     */
    private static synchronized DbMetaDataTransferProvider getProvider() {
        if (PROVIDER == null) {
            PROVIDER = new DbMetaDataTransferProviderImpl();
        }
        return PROVIDER;
    }

    /**
     * Represents a table during a drag and drop transfer.
     */
    public static final class Table {

        private final DatabaseConnection dbconn;
        private final JDBCDriver jdbcDriver;
        private final String tableName;

        private Table(DatabaseConnection dbconn, JDBCDriver jdbcDriver, String tableName) {
            this.dbconn = dbconn;
            this.jdbcDriver = jdbcDriver;
            this.tableName = tableName;
        }

        /**
         * Returns the {@link DatabaseConnection} this table comes from.
         */
        public DatabaseConnection getDatabaseConnection() {
            return dbconn;
        }

        /**
         * Returns the {@link JDBCDriver} which was used to connect to {@link #getDatabaseConnection}.
         */
        public JDBCDriver getJDBCDriver() {
            return jdbcDriver;
        }

        /**
         * Returns the name of this table.
         */
        public String getTableName() {
            return tableName;
        }

        public String toString() {
            return "Table[databaseConnection=" + dbconn + ",jdbcDriver=" + jdbcDriver + ",tableName=" + tableName + "]"; // NOI18N
        }
    }

    /**
     * The implementation of DbMetaDataTransferProvider, registered in the
     * default lookup and used by the Database Explorer.
     */
    private static final class DbMetaDataTransferProviderImpl implements DbMetaDataTransferProvider {

        public DataFlavor getTableDataFlavor() {
            return TABLE_FLAVOR;
        }

        public Object createTableData(DatabaseConnection dbconn, JDBCDriver jdbcDriver, String tableName) {
            return new Table(dbconn, jdbcDriver, tableName);
        }
     
        // View support
        public DataFlavor getViewDataFlavor() {
            return VIEW_FLAVOR;
        }
        
        public Object createViewData(DatabaseConnection dbconn, JDBCDriver jdbcDriver, String viewName) {
            return new View(dbconn, jdbcDriver, viewName);
        }
    }
    
    /**
     * Represents a view during a drag and drop transfer.
     */
    public static final class View {
        
        private final DatabaseConnection dbconn;
        private final JDBCDriver jdbcDriver;
        private final String viewName;
        
        private View(DatabaseConnection dbconn, JDBCDriver jdbcDriver, String viewName) {
            this.dbconn = dbconn;
            this.jdbcDriver = jdbcDriver;
            this.viewName = viewName;
        }
        
        /**
         * Returns the {@link DatabaseConnection} this table comes from.
         */
        public DatabaseConnection getDatabaseConnection() {
            return dbconn;
        }
        
        /**
         * Returns the {@link JDBCDriver} which was used to connect to {@link #getDatabaseConnection}.
         */
        public JDBCDriver getJDBCDriver() {
            return jdbcDriver;
        }
        
        /**
         * Returns the name of this view.
         */
        public String getViewName() {
            return viewName;
        }
        
        public String toString() {
            return "View[databaseConnection=" + dbconn + ",jdbcDriver=" + jdbcDriver + ",viewName=" + viewName + "]"; // NOI18N
        }
    }
}
