/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.debugger.jpda.ui.actions;

import com.sun.jdi.VMDisconnectedException;
import com.sun.jdi.request.EventRequestManager;
import com.sun.jdi.event.Event;
import com.sun.jdi.event.BreakpointEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import java.util.Collections;
import java.util.Set;
import org.netbeans.api.debugger.ActionsManager;
import org.netbeans.api.debugger.ActionsManagerListener;

import org.netbeans.api.debugger.DebuggerManager;
import org.netbeans.spi.debugger.ContextProvider;
import org.netbeans.api.debugger.Session;
import org.netbeans.api.debugger.jpda.JPDADebugger;
import org.netbeans.api.debugger.jpda.LineBreakpoint;
import org.netbeans.api.debugger.jpda.event.JPDABreakpointListener;
import org.netbeans.api.debugger.jpda.event.JPDABreakpointEvent;
import org.netbeans.api.debugger.jpda.JPDAStep;
import org.netbeans.modules.debugger.jpda.ui.EditorContextBridge;
import org.netbeans.spi.debugger.ActionsProviderSupport;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;

import org.netbeans.modules.debugger.jpda.ui.SourcePath;

import org.openide.util.NbBundle;

/**
 *
 * @author  Jan Jancura, Roman Ondruska
 */
public class RunIntoMethodActionProvider extends ActionsProviderSupport 
                                         implements PropertyChangeListener,
                                                    ActionsManagerListener {

    private JPDADebugger debugger;
    private Session session;
    private LineBreakpoint breakpoint;
    private ActionsManager lastActionsManager;
    private SourcePath sourcePath;
    
    public RunIntoMethodActionProvider(ContextProvider lookupProvider) {
        debugger = (JPDADebugger) lookupProvider.lookupFirst 
                (null, JPDADebugger.class);
        session = (Session) lookupProvider.lookupFirst 
                (null, Session.class);
        sourcePath = (SourcePath) lookupProvider.lookupFirst 
                (null, SourcePath.class);
        debugger.addPropertyChangeListener (debugger.PROP_STATE, this);
        EditorContextBridge.addPropertyChangeListener (this);
    }
    
    private void destroy () {
        debugger.removePropertyChangeListener (debugger.PROP_STATE, this);
        EditorContextBridge.removePropertyChangeListener (this);
    }
    
    private ActionsManager getActionsManager () {
        ActionsManager current = RunToCursorActionProvider.getCurrentActionsManager();
        if (current != lastActionsManager) {
            if (lastActionsManager != null) {
                lastActionsManager.removeActionsManagerListener(
                        ActionsManagerListener.PROP_ACTION_STATE_CHANGED, this);
            }
            current.addActionsManagerListener(
                    ActionsManagerListener.PROP_ACTION_STATE_CHANGED, this);
            lastActionsManager = current;
        }
        return current;
    }

    public void propertyChange (PropertyChangeEvent evt) {
        setEnabled (
            ActionsManager.ACTION_RUN_INTO_METHOD,
            getActionsManager().isEnabled(ActionsManager.ACTION_CONTINUE) &&
            (debugger.getState () == debugger.STATE_STOPPED) &&
            (EditorContextBridge.getCurrentLineNumber () >= 0) && 
            (EditorContextBridge.getCurrentURL ().endsWith (".java"))
        );
        if ( (debugger.getState () != debugger.STATE_RUNNING) &&
             (breakpoint != null)
        ) {
            DebuggerManager.getDebuggerManager ().removeBreakpoint (breakpoint);
            breakpoint = null;
        }
        if (debugger.getState () == debugger.STATE_DISCONNECTED) 
            destroy ();
    }
    
    public Set getActions () {
        return Collections.singleton (ActionsManager.ACTION_RUN_INTO_METHOD);
    }
     
    public void doAction (Object action) {
        if (breakpoint != null) {
            DebuggerManager.getDebuggerManager ().removeBreakpoint (breakpoint);
            breakpoint = null;
        }
        final String method = EditorContextBridge.getSelectedMethodName ();
        final int methodLine = EditorContextBridge.getCurrentLineNumber();
        
//        System.out.println("Currently in the method: " + EditorContextBridge.getCurrentMethodName() +
//                "on line" + methodLine);
         
        if (method.length () < 1) {
            NotifyDescriptor.Message descriptor = new NotifyDescriptor.Message (
                NbBundle.getMessage (
                RunIntoMethodActionProvider.class, 
                "MSG_Put_cursor_on_some_method_call"
                )
            );
            DialogDisplayer.getDefault ().notify (descriptor);
            return;
        }
        try {
            //Is debugger already at line with cursor?
            if (debugger.getCurrentThread().getCallStack()[0].getLineNumber(null) ==
                    EditorContextBridge.getCurrentLineNumber()) {
//                    System.out.println("The same line");
                    traceLineForMethod(method, methodLine);
                    session.getEngineForLanguage ("Java").getActionsManager ().doAction (
                        ActionsManager.ACTION_CONTINUE
                    );
                    return;
            }   
        } catch (com.sun.jdi.AbsentInformationException e) {
            e.printStackTrace();
        }
        
        breakpoint = LineBreakpoint.create (
            EditorContextBridge.getCurrentURL (),
            EditorContextBridge.getCurrentLineNumber ()
        );
        breakpoint.setHidden (true);
        breakpoint.setSuspend(LineBreakpoint.SUSPEND_NONE);
        breakpoint.addJPDABreakpointListener(new JPDABreakpointListener() {
            public void breakpointReached(JPDABreakpointEvent event) {
//                System.out.println("Breakpoint reached");
                traceLineForMethod(method, methodLine);
                DebuggerManager.getDebuggerManager().removeBreakpoint (breakpoint);
//                System.out.println("End of Breakpoint reached");
            }
        });
        DebuggerManager.getDebuggerManager ().addBreakpoint (breakpoint);
        session.getEngineForLanguage ("Java").getActionsManager ().doAction (
            ActionsManager.ACTION_CONTINUE
        );
        
    }
    
    private void traceLineForMethod(final String method, final int methodLine) {
        final int depth = debugger.getCurrentThread().getStackDepth();
        final JPDAStep step = debugger.createJPDAStep(JPDAStep.STEP_LINE, JPDAStep.STEP_INTO);
                step.addPropertyChangeListener(JPDAStep.PROP_STATE_EXEC, new PropertyChangeListener() {
                    public void propertyChange(PropertyChangeEvent evt) {
//                        System.out.println("Step fired.");
//                        System.out.println(debugger.getCurrentThread().getMethodName());
                        
                        if (
                                (
                                    debugger.getCurrentThread().getMethodName().
                                    equals(method) ||
                                    (debugger.getCurrentThread().getStackDepth() == depth && 
                                    debugger.getCurrentCallStackFrame().getLineNumber("java")
                                    != methodLine)
                                ) &&
                                sourcePath.sourceAvailable(debugger.getCurrentThread(),
                                    "java", false)
                            ) {
                            step.setHidden(false);
//                            System.out.println("Hidden -- false");
                        } else {
                            step.setHidden(true);
                            
                            if (step.getDepth() == JPDAStep.STEP_INTO)
                                step.setDepth(JPDAStep.STEP_OUT);
                            else step.setDepth(JPDAStep.STEP_INTO);
                            
                            step.addStep(debugger.getCurrentThread());
//                            System.out.println("Hidden -- true");
                        }
                    }
                });
                step.addStep(debugger.getCurrentThread());
    } 

    public void actionPerformed(Object action) {
        // Is never called
    }

    public void actionStateChanged(Object action, boolean enabled) {
        if (ActionsManager.ACTION_CONTINUE == action) {
            setEnabled (
                ActionsManager.ACTION_RUN_INTO_METHOD,
                enabled &&
                (debugger.getState () == debugger.STATE_STOPPED) &&
                (EditorContextBridge.getCurrentLineNumber () >= 0) && 
                (EditorContextBridge.getCurrentURL ().endsWith (".java"))
            );
        }
    }
}
