/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.mimelookup;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.netbeans.spi.editor.mimelookup.Class2LayerFolder;
import org.netbeans.spi.editor.mimelookup.InstanceProvider;
import org.netbeans.spi.editor.mimelookup.MimeLookupInitializer;
import org.openide.util.Lookup;
import org.openide.util.LookupEvent;
import org.openide.util.LookupListener;
import org.openide.util.WeakSet;
import org.openide.util.lookup.AbstractLookup;
import org.openide.util.lookup.InstanceContent;

/** MimeLookupInitializer that works over xml layer file system. It provides lookup
 *  of the desired looked-up class (or template of the class) in appropriate mime specific (or global)
 *  layer folder.
 *  <p>
 *  Subfolder where the looked-up class should be searched is retrieved from Class2LayerFolders 
 *  instances registrations, where the class is mapped to appropriated subfolder.
 *  
 *
 *  @author Martin Roskanin
 */
public class LayerMimeLookupInitializer implements MimeLookupInitializer{
    private static Map proxyChildResults = new HashMap();               
    private static Map mimePath2lazyLookup = new HashMap();               
    private static Map class2Provider = new HashMap();    
    /** Result of Lookup.getDefault().lookup(
                    new Lookup.Template(Class2LayerFolder.class))*/
    private static Lookup.Result class2LayerFolders;    
    /** Mapping of class to appropriate folder */
    private static Map class2folder = new HashMap();
    /** Listens on additions or removals of Class2LayerFolder.class instances */
    private static Class2LayerFoldersLookupListener class2LayerFoldersLookupListener;
    private static final String[] EMPTY_STRING_ARRAY = new String[0];
    
    private LayerMimeLookupInitializer parent;
    private String[] mimePath;
    
    /**
     * Creates a new instance of LayerMimeLookupInitializer 
     */
    public LayerMimeLookupInitializer() {
        // base or default LayerMimeLookupInitializer
        this.parent = null;
        this.mimePath = EMPTY_STRING_ARRAY;
    }
    

    public LayerMimeLookupInitializer(LayerMimeLookupInitializer parent, String mimeType){
        this.parent = parent;
        String[] parentMimePath = (parent != null)
            ? parent.mimePath
            : EMPTY_STRING_ARRAY;

        int parentMimePathLength = parentMimePath.length;
        mimePath = new String[parentMimePathLength + 1];
        System.arraycopy(parentMimePath, 0, mimePath, 0, parentMimePathLength);
        mimePath[parentMimePathLength] = mimeType;
        // init class2LayerFolders and start listening
        initClass2LayerFolders();
    }
    
    public org.openide.util.Lookup.Result child(String mimeType) {
        String parentPath = mimePath2String(mimePath);
        String path = (parentPath.length() == 0) ? parentPath + mimeType :
            parentPath + "/" + mimeType; //NOI18N
        synchronized (proxyChildResults){
            ProxyChildResult proxyChildResult = (ProxyChildResult)proxyChildResults.get(path);
            if (proxyChildResult == null){
                proxyChildResult = new ProxyChildResult(this, mimeType);
                proxyChildResults.put(path, proxyChildResult);
            }
            return proxyChildResult;
        }
    }

    
    public Lookup lookup() {
        synchronized (mimePath2lazyLookup){
            Lookup lookup = (Lookup)mimePath2lazyLookup.get(mimePath);
            if (lookup == null){
                lookup = new LazyLookup(mimePath);
                mimePath2lazyLookup.put (mimePath, lookup);
            }
            return lookup;
        }
    }
    

    private static synchronized void initClass2LayerFolders() {
        if (class2LayerFolders == null) {
            class2LayerFolders = Lookup.getDefault().lookup(
                    new Lookup.Template(Class2LayerFolder.class));
            Iterator it = class2LayerFolders.allInstances().iterator();
            synchronized (class2folder){
                while (it.hasNext()){
                    Object obj = it.next();
                    Class2LayerFolder c2lf = (Class2LayerFolder)obj;
                    class2folder.put(c2lf.getClazz(), c2lf);
                }
            }
            if (class2LayerFoldersLookupListener == null){
                class2LayerFoldersLookupListener = new Class2LayerFoldersLookupListener();
            }
            class2LayerFolders.addLookupListener(class2LayerFoldersLookupListener);
        }
    }
    
    private String mimePath2String(String[] mimePath){
        StringBuffer sb = new StringBuffer();
        for (int i=0; i<mimePath.length; i++){
            sb.append(mimePath[i]);
            if (i < mimePath.length-1){
                sb.append("/"); //NOI18N
            }
        }
        return sb.toString();
    }

    private class LazyLookup extends Lookup {
        
        private String mimePath[];
        private Map class2Lookup = new HashMap();
        private Map class2LookupObjects = new HashMap();
        
        private LazyLookup(String mimePath[]){
            this.mimePath = mimePath;
        }
        
        private void lookupChanged(){
            synchronized (mimePath2lazyLookup){
                Object obj = mimePath2lazyLookup.remove(mimePath);
                obj = null;
            }            
            
            ProxyChildResult proxyChildResult;
            synchronized (proxyChildResults){
                proxyChildResult = (ProxyChildResult)proxyChildResults.get(mimePath2String(mimePath));
            }
            if (proxyChildResult != null){
                proxyChildResult.resultChanged();
            }
        }
        
        public String toString(){
            StringBuffer sb  = new StringBuffer();
            sb.append("LazyLookup:"); //NOI18N
            sb.append(mimePath2String(mimePath));
            sb.append(" - class2Lookup:"); //NOI18N
            sb.append(class2Lookup);
            return sb.toString();
        }
        
        public Lookup.Result lookup(Lookup.Template template) {
            Lookup lookup = getLookup(template.getType());
            Lookup.Result res;
            if (mimePath != null && mimePath.length == 0){
                // #72873
                res = Lookup.EMPTY.lookup(template);
            } else {
                res = lookup.lookup(template);
            }
            return res;
        }

        public Object lookup(Class clazz) {
            Lookup lookup = getLookup(clazz);
            Object obj = lookup.lookup(clazz);
            return obj;
        }

        /** Lazy lookup creation
         */
        private Lookup getLookup(Class clazz){
            Lookup lookup;
            synchronized (class2Lookup){
                lookup = (Lookup)class2Lookup.get(clazz);
                if (lookup != null){
                    return lookup;
                }
            }
            
            Lookup lookup2 = createLookup(clazz);
            
            synchronized (class2Lookup){
                lookup = (Lookup)class2Lookup.get(clazz);
                if (lookup != null){
                    return lookup;
                } else {
                    class2Lookup.put(clazz, lookup2);
                    lookup = lookup2;
                }
            }
            
            return lookup;

        }
        
        private Lookup createLookup(Class clazz){
            InstanceContent ic = new InstanceContent();
            // we need to retrieve content objects from appropriate mimeFolder/subFolder
            // first, obtain subfolder from class2folder
            String subFolder = null;
            InstanceProvider instanceProvider = null;
            synchronized (class2folder){
                Object obj = class2folder.get(clazz);
                Class2LayerFolder c2lf = (Class2LayerFolder)obj;
                if (c2lf != null){
                    subFolder = c2lf.getLayerFolderName();
                    instanceProvider = c2lf.getInstanceProvider();
                }
            }
            
            LayerFolderObjectsProvider provider;
            synchronized (class2Provider){
                provider = (LayerFolderObjectsProvider)class2Provider.get(clazz);
                if (provider == null){
                    provider = new LayerFolderObjectsProvider(subFolder, clazz, instanceProvider);
                    class2Provider.put(clazz, provider);
                }
            }

            // retrieve objects and start listening on objects change.
            // If some objects will be added/removed, new lookup will be
            // created by LookupObjectsListener
            Lookup.Result lookupObjects = provider.folderObjects(mimePath);
            lookupObjects.addLookupListener(new LookupObjectsListener(clazz));
            class2LookupObjects.put(clazz, lookupObjects); 

            Iterator it = lookupObjects.allInstances().iterator();
            while (it.hasNext()){
                Object lookupObject = it.next();
                ic.add(lookupObject);
            }

            return new AbstractLookup(ic);
        }
        
        private class LookupObjectsListener implements LookupListener{
            private Class clazz;
            public LookupObjectsListener(Class clazz){
                this.clazz = clazz;
            }
            public void resultChanged(LookupEvent ev) {
                Lookup.Result result = ((Lookup.Result)ev.getSource());
                // refresh lookup content
                Collection newInstances = result.allInstances();
                // replace old lookup with new one
                boolean lookupObjectsChanged = false;
                synchronized (class2Lookup){
                    Lookup oldLookup = (Lookup)class2Lookup.get(clazz);
                    if (oldLookup != null){
                        Lookup.Result oldResult = oldLookup.lookup(new Template(clazz));
                        if (!newInstances.equals(oldResult.allInstances())){
                            lookupObjectsChanged = true;
                        }
                    }
                    if (lookupObjectsChanged){
                        class2Lookup.remove(clazz);
                        oldLookup = null;
                    }
                }
                if (lookupObjectsChanged){
                    Lookup.Result res = (Lookup.Result)class2LookupObjects.get(clazz);
                    if (res!=null) {
                        res.removeLookupListener(this);
                    }
                    result.removeLookupListener(this);
                    lookupChanged();
                }
            }
        }
    }
    
    
    private static class Class2LayerFoldersLookupListener implements LookupListener {
        public void resultChanged(LookupEvent ev) {
            Iterator it = ((Lookup.Result)ev.getSource()).allInstances().iterator();
            synchronized (class2folder){                
                class2folder.clear();                    
                while (it.hasNext()){
                    Class2LayerFolder c2lf = (Class2LayerFolder)it.next();
                    class2folder.put(c2lf.getClazz(), c2lf);
                }
            }
        }
    }

    /** 
     * Provides "on-demand" instances of child LayerMimeLookupInitializer.
     */
    private class ProxyChildResult extends Lookup.Result {

        private LayerMimeLookupInitializer parent;
        private String mimeType;
        private Collection instances = null;
        private final Set listeners = new WeakSet(10); // Set<LookupListener>


        
        public ProxyChildResult(LayerMimeLookupInitializer parent, String mimeType){
            this.parent = parent;
            this.mimeType = mimeType;
        }
        
        public void addLookupListener(LookupListener l) {
            synchronized (listeners) {
                listeners.add(l);
            }
        }

        public void removeLookupListener(LookupListener l) {
            synchronized (listeners) {
                listeners.remove(l);
            }
        }

        public void resultChanged() {
            LookupListener[] _listeners;
            synchronized (listeners) {
                if (listeners.isEmpty()) {
                    return;
                }
                _listeners = (LookupListener[])listeners.toArray(new LookupListener[listeners.size()]);
            }
            LookupEvent ev = new LookupEvent(this);
            if (_listeners == null){
                return;
            }
            for (int i = 0; i < _listeners.length; i++) {
                LookupListener lst = _listeners[i];
                if (lst != null){
                    lst.resultChanged(ev);
                }
            }
        }
        
        public synchronized Collection allInstances() {
            if (instances == null) {
                ArrayList ret = new ArrayList();
                ret.add(new LayerMimeLookupInitializer(parent, mimeType));
                instances = Collections.unmodifiableCollection(ret);
            }
            return instances;
        }
        
    }
    
}
