/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.mimelookup;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.modules.ModuleInstall;
import org.openide.util.TopologicalSortException;


/**
 *
 * @author Martin Roskanin, Miloslav Metelka
 */
public class MimeLookupUtils {

    /** Flag for testing the sorting support by debugging messages. */
    private static final boolean debugSort
        = Boolean.getBoolean("netbeans.debug.editor.mimelookup.utils.sort.debug"); // NOI18N

    private static List hidden = new ArrayList();
    
    private MimeLookupUtils(){
        
    }
    
    /** Merge together the base and mime folders.
     * @param dataFoldersList List of folders2merge.
     * @return list of data objects resulted from merging.
     */
    public static List mergeObjects(List dataFoldersList) {
        Map name2dob = new HashMap();
        Map edges = new HashMap();
        hidden.clear();
        for (int i =0; i<dataFoldersList.size(); i++){
            DataFolder folder = (DataFolder) dataFoldersList.get(i);
            addDataObjects(name2dob, folder.getChildren());
        }

        for (int i =0; i<dataFoldersList.size(); i++){
            DataFolder folder = (DataFolder) dataFoldersList.get(i);
            addEdges(edges, name2dob, folder);
        }
        
        try {
            return org.openide.util.Utilities.topologicalSort(name2dob.values(), edges);
        } catch (TopologicalSortException ex) {
            List corrected = ex.partialSort();
            ErrorManager em = ErrorManager.getDefault();
            em.log (ErrorManager.WARNING, "Note: MimeLookup objects cannot " +
                    "be consistently sorted due to ordering conflicts."); // NOI18N
            em.notify (ErrorManager.INFORMATIONAL, ex);
            em.log (ErrorManager.WARNING, "Using partial sort: " + corrected); // NOI18N
            return corrected;
        }
    }
    
    /** Append array of dataobjects to the existing list of dataobjects. Also
     * append the names of added dataobjects so that the both dobs and names
     * list are in the same order.
     * @param dobs valid existing list of dataobjects
     * @param names valid existing list of names of the dataobjects from dobs list.
     * @param addDobs dataobjects to be added.
     */
    private static void addDataObjects(Map name2dob, DataObject[] addDobs) {
        int addDobsLength = addDobs.length;
        for (int i = 0; i < addDobsLength; i++) {
            DataObject dob = addDobs[i];
            String dobName = dob.getPrimaryFile().getNameExt();
            if ((dobName != null && dobName.endsWith("_hidden"))) { //NOI18N
                String originalName = dobName.substring(0, dobName.indexOf("_hidden"));//NOI18N
                hidden.add(originalName);
            } else if (isHidden(dob.getPrimaryFile())) {
                hidden.add(dobName);
            } else {
                if (!hidden.contains(dobName)){
                    if (dob instanceof DataFolder){
                       name2dob.put(dob.getPrimaryFile().getPath(), dob);    
                    } else {
                        name2dob.put(dobName, dob);
                    }
                }
            }
        }
    }
    
    private static boolean isHidden(FileObject fo){
        for (Enumeration e = fo.getAttributes();
            e.hasMoreElements();
        ) {
            String name = (String)e.nextElement();
            if ("hidden".equals(name)) { //NOI18N
                Object value = fo.getAttribute(name);
                if ((value instanceof Boolean) && ((Boolean) value).booleanValue()){
                    return true;
                }
            }
        }
        return false;
    }

    /** Append the pairs - first dob name then second dob name to the list
     * of sort pairs for the dataobjects from the given folder.
     */
    private static void addEdges(Map edges, Map name2dob, DataFolder folder) {
        FileObject primaryFile = folder.getPrimaryFile();
        String path = primaryFile.getPath();
        for (Enumeration e = primaryFile.getAttributes();
            e.hasMoreElements();
        ) {
            String name = (String)e.nextElement();
            int slashIndex = name.indexOf("/"); // NOI18N
            if (slashIndex != -1) { //NOI18N
                Object value = primaryFile.getAttribute(name);
                if ((value instanceof Boolean) && ((Boolean) value).booleanValue()){
                    String name1 = name.substring(0, slashIndex);
                    String name2 = name.substring(slashIndex + 1);
                    if (debugSort) {
                        System.err.println("SORT-PAIR: [" + name1 + ", " + name2 + "]"); // NOI18N
                    }
                    
                    DataObject dob = (DataObject)name2dob.get(name1);
                    if (dob==null){
                        String pathName = path + "/" + name1; //NOI18N
                        dob = (DataObject)name2dob.get(pathName);
                    }
                    DataObject target = (DataObject)name2dob.get(name2);
                    if (target==null){
                        String pathName = path +  "/" + name1; //NOI18N
                        target = (DataObject)name2dob.get(pathName);
                    }
                    if (dob != null && target != null) {
                        Collection targetVertices = (Collection)edges.get(dob);
                        if (targetVertices == null) { // none target vertices yet
                            // Use just singleton list to save space
                            targetVertices = Collections.singletonList(target);
                            edges.put(dob, targetVertices);

                        } else if (targetVertices.size() == 1) { // singleton list
                            targetVertices = new HashSet(targetVertices);
                            targetVertices.add(target);
                            edges.put(dob, targetVertices);

                        } else {
                            targetVertices.add(target);
                        }
                    }
                }
            }   
        }
    }
    
    
}
