/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.mimelookup;

import java.io.File;
import java.io.IOException;
import java.lang.ref.Reference;
import java.lang.reflect.Field;
import java.util.Map;
import org.netbeans.api.editor.mimelookup.MimeLookup;
import org.netbeans.junit.NbTestCase;
import org.openide.util.Lookup;
import org.openide.util.Lookup.Result;
import org.openide.util.Lookup.Template;
import org.openide.util.LookupEvent;
import org.openide.util.LookupListener;
import org.openide.util.WeakListeners;

/** 
 * Testing weak reference maps of MimeLookup
 * 
 * @author Martin Roskanin
 */
public class MimeLookupWeakRefTest extends NbTestCase {

    final int resultChangedCount[] = new int[1];
    
    public MimeLookupWeakRefTest(String testName) {
        super(testName);
    }
    
    protected void setUp() throws Exception {
        String fsstruct [] = new String [] {
            "Editors/text/xml/java-lang-StringBuffer.instance", //NOI18N
            "Editors/text/x-java/text/xml/java-lang-String.instance", //NOI18N                    
            "Editors/text/x-java/java-lang-Exception.instance", //NOI18N  
        };

        EditorTestLookup.setLookup(fsstruct, getWorkDir(), new Object[] {},
                   getClass().getClassLoader());
        
    }

    private void createFile(String file) throws IOException{
        EditorTestLookup.createFile(new File(getWorkDir(), file)); //NOI18N        
    }
    
    public void testWeakListenersAddingToResult() throws IOException{
        String rootMime = "text/x-java";
        resultChangedCount[0] = 0;
        MimeLookup lookup = MimeLookup.getMimeLookup(rootMime);
        Lookup.Result result = lookup.lookup(new Lookup.Template(Object.class));
        LookupListener lookupListener = new LookupListener(){
            public void resultChanged(LookupEvent ev) {
                resultChangedCount[0]++;
            }
        };
        result.allInstances();
        LookupListener weakLookupListener = (LookupListener) WeakListeners.create(
                LookupListener.class, lookupListener, result);
        result.addLookupListener(weakLookupListener);
        //simulate module installation, new file will be added
        createFile("Editors/text/x-java/" + //NOI18N        
                "java-lang-StringBuffer.instance"); //NOI18N        
        
        checkResultChange(1);        
        lookupListener = null;
        gc();
        resultChangedCount[0] = 0;
        
        createFile("Editors/text/x-java/" + //NOI18N        
                "java-lang-String.instance"); //NOI18N        
        checkResultChange(0);        
    }
    
    private void checkResultChange(final int count) throws IOException{
        // wait for firing event
        MimeLookupTestUtils.waitMaxMilisForValue(2000, new MimeLookupTestUtils.ValueResolver(){
            public Object getValue(){
                return Boolean.FALSE;
            }
        }, Boolean.TRUE);
        assertTrue(("resultChangedCount is:"+resultChangedCount[0]+" instead of "+count), resultChangedCount[0] == count);
    }
    
    /**
     * Testing weak caching of MimeLookup.mime2lookup map
     * after simple MimeLookup.getMimeLookup call
     */
    public void testMimeLookupRootWeakMap(){
        String mime = "text/xml";
        Lookup lookup = MimeLookup.getMimeLookup(mime); //NOI18N
        checkWeakRef(MimeLookup.class, null, "mime2lookup", mime, true);
        // reference lost, after gc, there should be no instance in the mime2lookup map
        lookup = null;
        gc();
        checkWeakRef(MimeLookup.class, null, "mime2lookup", mime, false);
    }

    /**
     * Testing weak caching of MimeLookup.mime2lookup map and MimeLookup.mime2childLookup map
     * after simple MimeLookup.getMimeLookup.childLookup call
     */
    public void testMimeLookupChildWeakMap(){
        String rootMime = "text/x-java";
        String childMime = "text/xml";
        MimeLookup rootLookup = MimeLookup.getMimeLookup(rootMime);
        Lookup childLookup = rootLookup.childLookup(childMime); //NOI18N
        checkWeakRef(rootLookup.getClass(), rootLookup, "mime2childLookup", childMime, true);
        checkWeakRef(rootLookup.getClass(), null, "mime2lookup", rootMime, true);            
            
        // reference lost, after gc, there should be no instance in the mime2childLookup map
        childLookup = null;
        gc();
        checkWeakRef(rootLookup.getClass(), rootLookup, "mime2childLookup", childMime, false);
            
        // reference on rootLookup lost.
        rootLookup = null;
        gc();
        checkWeakRef(MimeLookup.class, null, "mime2lookup", rootMime, false);            
    }

    /**
     * Testing weak caching of MimeLookup.mime2lookup map and MimeLookup.mime2childLookup map
     * after Result listening on MimeLookup.getMimeLookup and MimeLookup.getMimeLookup.childLookup
     */
    public void testMimeLookupWeakMapAfterResultListening(){
        String rootMime = "text/x-java";
        String childMime = "text/xml";
        MimeLookup rootLookup = MimeLookup.getMimeLookup(rootMime);
        Lookup childLookup = rootLookup.childLookup(childMime); //NOI18N
        
        Exception e = (Exception) rootLookup.lookup(Exception.class);
        assertNotNull(e);
        String s = (String) childLookup.lookup(String.class);
        assertNotNull(s);

        checkWeakRef(rootLookup.getClass(), rootLookup, "mime2childLookup", childMime, true);
        checkWeakRef(MimeLookup.class, null, "mime2lookup", rootMime, true);
        
        // reference lost, after gc, there should be no instance in the mime2childLookup map
        childLookup = null;
        gc();
        checkWeakRef(rootLookup.getClass(), rootLookup, "mime2childLookup", childMime, false);
            
        // reference on rootLookup lost.
        rootLookup = null;
        gc();
        checkWeakRef(MimeLookup.class, null, "mime2lookup", rootMime, false);            
        
        rootLookup = MimeLookup.getMimeLookup(rootMime);
        childLookup = rootLookup.childLookup(childMime); //NOI18N
        
        e = (Exception) rootLookup.lookup(Exception.class);
        assertNotNull(e);
        s = (String) childLookup.lookup(String.class);
        assertNotNull(s);

        Result rootResult = rootLookup.lookup(new Template(Exception.class));
        assertTrue(rootResult.allInstances().size() == 1);
        
        Result childResult = childLookup.lookup(new Template(String.class));
        assertTrue(childResult.allInstances().size() == 1);
        
        LookupListener rootListener = new LookupListener(){
            public void resultChanged(LookupEvent ev){}
        };
        
        LookupListener childListener = new LookupListener(){
            public void resultChanged(LookupEvent ev){}
        };
        
        rootResult.addLookupListener(rootListener);
        childResult.addLookupListener(childListener);
        
        // reference lost, after gc, there should be no instance in the mime2childLookup map
        childLookup = null;
        gc();
        checkWeakRef(rootLookup.getClass(), rootLookup, "mime2childLookup", childMime, true);

        // reference on rootLookup lost.
        rootLookup = null;
        gc();
        checkWeakRef(MimeLookup.class, null, "mime2lookup", rootMime, true);            
        
        rootResult.addLookupListener(rootListener);
        childResult.addLookupListener(childListener);
        rootListener = null;
        childListener = null;
        rootResult = null;
        childResult = null;
        gc();
        checkWeakRef(MimeLookup.class, null, "mime2lookup", rootMime, false);
    }
    
    
    private void checkWeakRef(Class cls, 
            Object clazzInstance, 
            String mapName, 
            String refKey, 
            boolean refExists){
        try {
            Field fld = cls.getDeclaredField(mapName); //NOI18N
            fld.setAccessible(true);
            Map map = (Map) fld.get(clazzInstance);
            Reference ref = (Reference) map.get(refKey); //NOI18N
            assertNotNull("Reference should not be null.", ref); //NOI18N
            if (refExists){
                assertNotNull("Referenced object should not be null.", ref.get()); //NOI18N
            } else {
                assertNull("Referenced object should be null.", ref.get()); //NOI18N
            }
        } catch (Exception e) {
            System.err.println(e);
        }
    }

    private void gc(){
        System.gc();
        System.gc();
        System.gc();
        System.gc();
        System.gc();
    }
    
}
