/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.i18n;


import java.awt.event.ActionEvent;
import java.io.IOException;
import java.util.Arrays;
import javax.swing.JPanel;

import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.util.UserCancelException;
import org.openide.ErrorManager;
import org.netbeans.api.project.Project;


/**
 * Customizer for <code>ResourceHolder</code> object which is encapsulated by <code>I18nString</code> instance.
 * It's used inside <code>I18nPanel</code>.
 *
 * @author  Peter Zavadsky
 */
public class ResourcePanel extends JPanel {

    /** Name for resource property. */
    public static final String PROP_RESOURCE = "property_resource";
    
    /** <code>I18nString</code> which resource is cutomized. */
    private I18nString i18nString;
    
    /** the Project this customizer is invoked in. It is used to
     * identify the sources to browse. */
    private Project project;
     
    /** the file for that resource should be selected **/
    private FileObject file;



    /** Creates new form ResourceHolderCustomizer.
     * @param project the Project this customizer is invoked in
     */
    public ResourcePanel(Project project, FileObject file) {
        this.project = project;
        this.file = file;
        initComponents();
        initAccessibility();        
        enableComponents();
    }

    /** Sets the file for that resource should be selected **/
    public void setFile(FileObject fo) {
        this.file = fo;
    }

    public FileObject getFile() { return file;}
       
    /** Setter for <code>i18nString</code> property. */
    public void setI18nString(I18nString i18nString) {        
        this.i18nString = i18nString;
        
        updateValues();
    }

    public void setProject(Project project) {
        if (this.project != project) {
            this.project = project;
            resourceText.setText(""); // NOI18N
        }      
        enableComponents();
    }

    public Project getProject( ) {
        return this.project;
    }

    /** Helper method. Reflects values changes to UI. */
    private void updateValues() {
        // Set resource texfield.
        resourceText.setText(getResourceName(i18nString.getSupport().getResourceHolder().getResource()));
    }

    /** Helper method. Changes resource. */
    private void changeResource(DataObject resource) {
        if(resource == null)
            throw new IllegalArgumentException();

        DataObject oldValue = i18nString.getSupport().getResourceHolder().getResource();
        
        if(oldValue != null && oldValue.equals(resource))
            return;
        
        i18nString.getSupport().getResourceHolder().setResource(resource);
        String newResourceValue = i18nString.getSupport().getResourceHolder()
                                  .getValueForKey(i18nString.getKey());
        if (newResourceValue != null) {
            i18nString.setValue(newResourceValue);
        }
        updateValues();
        
        firePropertyChange(PROP_RESOURCE, oldValue, resource);
        
        I18nUtil.getOptions().setLastResource2(resource);
    }

    /** Handler delegate. */
    private void resourceTextActionPerformedDelegate(ActionEvent evt) {
        DataObject resource = findResource(resourceText.getText());
        if(resource != null)
            changeResource(resource);
        else
            resourceText.setText(getResourceName(i18nString.getSupport().getResourceHolder().getResource()));
    }

    /** Helper method. Finds <code>DataObject</code> (resource bundle) for specified name. */
    private DataObject findResource(String bundleName) {
        try {
            // It has to provide package name separated by slashes (not by dots).
            bundleName = bundleName.replace('.', '/');
            
            String resourceName = bundleName + ".properties"; // NOI18N // PENDING

            FileObject fileObject = Util.getResource(file, resourceName);
            
            if(fileObject == null)
                return null;
            
            DataObject dataObject = DataObject.find(fileObject);

            // Check for validity of found data object for this resource holder.
            if(isResourceClass(dataObject.getClass()))
                return dataObject;
            else
                return null;
        } catch(IOException e) {
            return null;
        }
    }

    /** Helper method. Checks if specified class is valid resource class. */
    private boolean isResourceClass(Class clazz) {
        return Arrays.asList(i18nString.getSupport().getResourceHolder().getResourceClasses()).contains(clazz);
    }
    
    /** Helper method. Gets name of specified <code>DataObject</code> (resource bundle) . */
    private String getResourceName(DataObject resource) {
        if(resource == null) {
            return ""; // NOI18N
        }
        else {
            String name = Util.getResourceName(file, resource.getPrimaryFile(), '.', false );
            if (name != null) return name; else return "";
        }
    }

    private void initAccessibility() {
        this.getAccessibleContext().setAccessibleDescription(I18nUtil.getBundle().getString("ACS_ResourcePanel"));                        
        newButton.getAccessibleContext().setAccessibleDescription(I18nUtil.getBundle().getString("ACS_CTL_NewButton"));
        browseButton.getAccessibleContext().setAccessibleDescription(I18nUtil.getBundle().getString("ACS_CTL_BrowseButton"));
        resourceText.getAccessibleContext().setAccessibleDescription(I18nUtil.getBundle().getString("ACS_ResourceText"));
    }
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents() {//GEN-BEGIN:initComponents
        java.awt.GridBagConstraints gridBagConstraints;

        resourceLabel = new javax.swing.JLabel();
        resourceText = new javax.swing.JTextField();
        jPanel1 = new javax.swing.JPanel();
        browseButton = new javax.swing.JButton();
        newButton = new javax.swing.JButton();

        setLayout(new java.awt.GridBagLayout());

        resourceLabel.setText(I18nUtil.getBundle().getString("LBL_BundleName"));
        resourceLabel.setLabelFor(resourceText);
        // setBorder(new TitledBorder(I18nUtil.getBundle().getString("LBL_ResourceTitleBorder")));
        resourceLabel.setDisplayedMnemonic((I18nUtil.getBundle().getString("LBL_BundleName_Mnem")).charAt(0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(12, 12, 0, 0);
        add(resourceLabel, gridBagConstraints);

        resourceText.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                resourceTextActionPerformed(evt);
            }
        });
        resourceText.addFocusListener(new java.awt.event.FocusAdapter() {
            public void focusLost(java.awt.event.FocusEvent evt) {
                resourceTextFocusLost(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(12, 12, 0, 11);
        add(resourceText, gridBagConstraints);

        jPanel1.setLayout(new java.awt.GridLayout(1, 2, 5, 0));

        browseButton.setMnemonic((I18nUtil.getBundle().getString("CTL_BrowseButton_Mnem")).charAt(0));
        browseButton.setText(I18nUtil.getBundle().getString("CTL_BrowseButton"));
        browseButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                browseButtonActionPerformed(evt);
            }
        });

        jPanel1.add(browseButton);

        newButton.setMnemonic((I18nUtil.getBundle().getString("CTL_NewButton_Mnem")).charAt(0));
        newButton.setText(I18nUtil.getBundle().getString("CTL_NewButton"));
        newButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                newButtonActionPerformed(evt);
            }
        });

        jPanel1.add(newButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(11, 0, 11, 11);
        add(jPanel1, gridBagConstraints);

    }//GEN-END:initComponents

    private void newButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_newButtonActionPerformed
        try {
            // PENDING at the moment create from first class.
            // It should be possible for user to choose which kind of class.
            DataObject template = i18nString.getSupport().getResourceHolder().getTemplate(i18nString.getSupport().getResourceHolder().getResourceClasses()[0]);
          
            DataObject resource = SelectorUtils.instantiateTemplate(project, this.file, template);
            
            if(resource != null) {
                changeResource(resource);
            }
            
        } catch (UserCancelException uce) {
            // ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, uce); // nobody is interested in the message            
        } catch (IOException ioe) {
            ErrorManager.getDefault().notify(ioe);
        }
    }//GEN-LAST:event_newButtonActionPerformed

    private void browseButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_browseButtonActionPerformed
        // Selects source data objects which could be i18n-ized.

      DataObject resource = SelectorUtils.selectBundle(this.project, file);
      if (resource != null) {
	changeResource(resource);
      }
    }//GEN-LAST:event_browseButtonActionPerformed

    private void resourceTextFocusLost(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_resourceTextFocusLost
        resourceTextActionPerformedDelegate(new ActionEvent(evt.getSource(), evt.getID(), null));
    }//GEN-LAST:event_resourceTextFocusLost

    private void resourceTextActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_resourceTextActionPerformed
        resourceTextActionPerformedDelegate(evt);
    }//GEN-LAST:event_resourceTextActionPerformed

    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton browseButton;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JButton newButton;
    private javax.swing.JLabel resourceLabel;
    private javax.swing.JTextField resourceText;
    // End of variables declaration//GEN-END:variables


    /**
     * Set new resource holder.
     */
    public void setResource(DataObject resource) {
        if (isResourceClass(resource.getClass())) {
            changeResource(resource);
        }
    }        
    
    private void enableComponents() {
        boolean enabled = this.project != null;
        this.setEnabled(enabled);
        browseButton.setEnabled(enabled);
        newButton.setEnabled(enabled);
        resourceText.setEnabled(enabled);        
        resourceLabel.setEnabled(enabled); 
    }
}
