/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.metadata;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.queries.SourceForBinaryQuery;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileSystem;
import org.openide.filesystems.Repository;
import org.openide.util.Utilities;
import org.openide.util.WeakListeners;

/**
 * Wrapper for a ClassPath whose {@link contains(FileObject)} method
 * will also search any source roots for the classpath's binary roots.
 * Implemented as a cache because the trivial approach (retrieving
 * the source roots whenever <code>contains()</code> is called)
 * is much slower.
 *
 * @author Andrei Badea
 */
public class ClassPathSourceCache {

    public final static String PROP_ROOTS = ClassPath.PROP_ROOTS;

    // the invalidResults and invalidSourceRoots counters are used
    // to detect that a change in the class path entries or the
    // SFBQ results occured while the respective caches are being
    // computed (in this case the caches cannot be set, because they
    // are obsolete).
    //
    // This is needed in order to prevent the following scenario:
    //
    // 1) Start with sourceRoots == null.
    // 2) Thread A enters getSourceRoots() and computes newSourceRoots.
    // 3) Thread B preempts A and causes the roots of a SFBQ.Result
    //    to change. The old roots is what newSourceRoots was computed from in
    //    step 2, so thread A now holds an obsolete set of roots.
    // 4) Thread A enters the second synchronized block in getSourceRoots()
    //    and sets the cache to the obsolete roots. The cache is never
    //    set to the new, correct set of roots.

    /**
     * The classpath the source roots are retrieved from.
     * Held weakly in order to avoid leaks.
     */
    private final WeakReference<ClassPath> classPathRef;
    private final Listener listener = new Listener();

    private final PropertyChangeSupport propChangeSupport = new PropertyChangeSupport(this);

    /**
     * The SFBQ results cache built from the classpath entries.
     * Not private because used in tests.
     */
    List<SourceForBinaryQuery.Result> results;

    /**
     * Incremented each time the classpath entries change.
     */
    private long invalidResults;

    /**
     * The source roots cache built from the SBFQ results.
     * Not private because used in tests.
     */
    Set<FileObject> sourceRoots;

    /**
     * Incremented each time a change occurs in any SFBQ result
     * in {@link #results}.
     */
    private long invalidSourceRoots;

    /**
     * If true, then the whole class has been deinitialized
     * because the classpath was GCd.
     */
    boolean deinitialized;

    private boolean resolveSources;

    public static ClassPathSourceCache newInstance(ClassPath classPath, boolean resolveSources) {
        ClassPathSourceCache created = new ClassPathSourceCache(classPath, resolveSources);
        created.initialize();
        return created;
    }

    private ClassPathSourceCache(ClassPath classPath, boolean resolveSources) {
        assert classPath != null;
        classPathRef = new CleanupReference<ClassPath>(classPath);
        this.resolveSources = resolveSources;
    }

    /**
     * Starts listening on the classpath. This should not be done in
     * the constructor, because it would expose <code>this</code> before
     * the constructor has finished. This is the only reason for the
     * static factory method + private constructor instead of a plain
     * public constructor.
     */
    private void initialize() {
        ClassPath classPath  = classPathRef.get();
        if (classPath != null) {
            classPath.addPropertyChangeListener(WeakListeners.propertyChange(listener, classPath));
        }
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propChangeSupport.addPropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propChangeSupport.removePropertyChangeListener(listener);
    }

    /**
     * Returns true if the roots returned by {@link #getRoots()} contain
     * the specified <code>FileObject</code>.
     *
     * @param  f the file to search for.
     * @return true if <code>f</code> could be find on any source root;
     *         false otherwise.
     */
    public boolean contains(FileObject f) {
        if (findOwnerRoot(getSourceRootSet(), f) != null) {
            return true;
        }
        return false;
    }

    public FileObject[] getRoots() {
        Set<FileObject> sourceRoots = getSourceRootSet();
        return sourceRoots.toArray(new FileObject[sourceRoots.size()]);
    }

    private Set<FileObject> getSourceRootSet() {
        long invalidSourceRootsCopy;

        synchronized (this) {
            if (sourceRoots != null) {
                return sourceRoots;
            }

            invalidSourceRootsCopy = invalidSourceRoots;
        }

        final Set<FileObject> newSourceRoots = new HashSet<FileObject>();
        // add the original classpath roots
        // (in an atomic action as a workaround for issue 85995)
        try {
            Repository.getDefault().getDefaultFileSystem().runAtomicAction(new FileSystem.AtomicAction() {
                public void run() throws IOException {
                    ClassPath classPath = classPathRef.get();
                    if (classPath != null) {
                        for (FileObject root : classPath.getRoots()) {
                            newSourceRoots.add(root);
                        }
                    }
                }
            });
        } catch (IOException e) {
            // should never occur, as ClassPath.getRoots() does not throw an IOException
            ErrorManager.getDefault().notify(e);
        }

        if (resolveSources) {
            // add the source roots computed from original classpath's binary roots
            for (SourceForBinaryQuery.Result result : getResults()) {
                for (FileObject sourceRoot : result.getRoots()) {
                    newSourceRoots.add(sourceRoot);
                }
            }
        }

        synchronized (this) {
            if (invalidSourceRoots == invalidSourceRootsCopy) {
                if (sourceRoots == null) {
                    sourceRoots = newSourceRoots;
                }
                return sourceRoots;
            } else {
                return newSourceRoots;
            }
        }
    }

    private List<SourceForBinaryQuery.Result> getResults() {
        long invalidResultsCopy;

        synchronized (this) {
            if (results != null) {
                return results;
            }

            // just to be sure: sourceRoots must be also null at this moment
            // (the source roots cannot be valid since we don't even know
            // the source for binary query results -- which the source roots
            // are retrieved from)
            assert sourceRoots == null;

            invalidResultsCopy = invalidResults;
        }

        List<SourceForBinaryQuery.Result> newResults = null;
        ClassPath classPath = classPathRef.get();
        if (classPath == null) {
            newResults = Collections.emptyList();
        } else {
            newResults = new ArrayList<SourceForBinaryQuery.Result>();
            List<ClassPath.Entry> entries = classPath.entries();
            for (ClassPath.Entry entry : entries) {
                newResults.add(SourceForBinaryQuery.findSourceRoots(entry.getURL()));
            }
        }

        synchronized (this) {
            if (invalidResults == invalidResultsCopy) {
                if (results == null) {
                    for (SourceForBinaryQuery.Result result : newResults) {
                        result.addChangeListener(listener);
                    }
                    results = newResults;
                }
                return results;
            } else {
                return newResults;
            }
        }
    }

    private static FileObject findOwnerRoot(Set<FileObject> roots, FileObject resource) {
        for (FileObject f = resource; f != null; f = f.getParent()) {
            if (roots.contains(f)) {
                return f;
            }
        }
        return null;
    }

    /**
     * Listens on the classpath and on all SFBQ results and invalidates
     * the results and sourceRoots caches.
     */
    private final class Listener implements PropertyChangeListener, ChangeListener {

        public void propertyChange(PropertyChangeEvent evt) {
            String propertyName = evt.getPropertyName();
            if (ClassPath.PROP_ENTRIES.equals(propertyName)) {
                if (resolveSources) {
                    entriesChanged();
                }
            } else if (ClassPath.PROP_ROOTS.equals(propertyName)) {
                rootsChanged();
            }
        }

        public void stateChanged(ChangeEvent evt) {
            rootsChanged();
        }

        private void entriesChanged() {
            synchronized (ClassPathSourceCache.this) {
                if (!deinitialized) {
                    if (results != null) {
                        for (SourceForBinaryQuery.Result result : results) {
                            result.removeChangeListener(this);
                        }
                    }
                    results = null;
                    invalidResults++;
                    sourceRoots = null;
                    invalidSourceRoots++;
                    propChangeSupport.firePropertyChange(PROP_ROOTS, null, null);
                }
            }
        }

        private void rootsChanged() {
            synchronized (ClassPathSourceCache.this) {
                if (!deinitialized) {
                    sourceRoots = null;
                    invalidSourceRoots++;
                    propChangeSupport.firePropertyChange(PROP_ROOTS, null, null);
                }
            }
        }
    }

    /**
     * Weak reference holding the cache's classpath and cleans up when
     * the classpath is GCd.
     */
    private final class CleanupReference<T> extends WeakReference<T> implements Runnable {

        public CleanupReference(T referent) {
            super(referent, Utilities.activeReferenceQueue());
        }

        public void run() {
            synchronized (ClassPathSourceCache.this) {
                if (results != null) {
                    for (SourceForBinaryQuery.Result result : results) {
                        result.removeChangeListener(listener);
                    }
                }
                results = Collections.emptyList();
                invalidResults++;
                sourceRoots = Collections.emptySet();
                invalidSourceRoots++;

                deinitialized = true;
            }
        }
    }
}
