/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.j2ee.metadata;

import static java.beans.Introspector.*;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.jmi.javamodel.Parameter;

import java.lang.reflect.Modifier;
import java.util.List;
import java.util.Arrays;
import java.util.Collections;

/**
 * A lot of code in this class looks like java.beans.Introspector.class.
 * Unfortunately, we can not use that class because Java beans convention is to
 * have accessors methods public where as an entity bean accessors can be
 * protected as well.
 *
 * @author Sanjeeb.Sahoo@Sun.COM
 */
public class JMIClassIntrospector {
    private static final String GET_PREFIX = "get"; // NOI18N

    private static final String SET_PREFIX = "set"; // NOI18N

    private static final int GET_PREFIX_SIZE = 3;

    private static final int SET_PREFIX_SIZE = 3;

    private static final String IS_PREFIX = "is"; // NOI18N

    private static final int IS_PREFIX_SIZE = 2; // NOI18N

    public static boolean isGetter(Method method) {
        String name = method.getName();
        if (name.length() <= GET_PREFIX_SIZE &&
                !name.startsWith(IS_PREFIX)) {
            // Optimization. Don't bother with invalid propertyNames.
            return false;
        }
        final String resultTypeName = method.getType().getName();
        final List parameters = method.getParameters();
        if (parameters.isEmpty() &&
                !resultTypeName.equals(Constants.VOID) &&
                !Modifier.isStatic(method.getModifiers())) {
            if (name.startsWith(GET_PREFIX)) { // Simple Getter
                // check if the first char after get is uppercase
                return Character.isUpperCase(name.charAt(GET_PREFIX_SIZE));
            } else {
                if (name.startsWith(IS_PREFIX) &&
                        (resultTypeName.equals(Constants.PRIMITIVE_BOOLEAN) ||
                        resultTypeName.equals(Constants.WRAPPER_BOOLEAN))) {
                    // check if the first char after get is uppercase
                    return Character.isUpperCase(name.charAt(IS_PREFIX_SIZE));
                }
            }
        }
        return false;
    }

    public static boolean isSetter(Method method) {
        String name = method.getName();
        if (name.startsWith(SET_PREFIX)) {
            final String resultTypeName = method.getType().getName();
            final List parameters = method.getParameters();
            if (parameters.size() == 1 &&
                    resultTypeName.equals(Constants.VOID) &&
                    !Modifier.isStatic(method.getModifiers())) {
                return Character.isUpperCase(name.charAt(SET_PREFIX_SIZE));
            }
        }
        return false;
    }

    /**
     * calcualtes property name from a given getter method. It does not
     * validate if input argument. So if the input argument is not a valid
     * getter, then it can give incorrect result.
     * @param getter a method object which must be a valid getter method.
     * @return property name.
     * @see #isGetter(org.netbeans.jmi.javamodel.Method)
     */
    public static String getPropertyNameFromGetter(Method getter) {
        assert(isGetter(getter));
        String methodName = getter.getName();
        if(methodName.startsWith(IS_PREFIX)) {
            return decapitalize(methodName.substring(IS_PREFIX_SIZE));
        } else {
            return decapitalize(methodName.substring(GET_PREFIX_SIZE));
        }
    }

    /**
     * calcualtes property name from a given setter method. It does not
     * validate if input argument. So if the input argument is not a valid
     * setter, then it can give incorrect result.
     * @param setter a method object which must be a valid setter method.
     * @return property name.
     * @see #isSetter(org.netbeans.jmi.javamodel.Method)
     */
    public static String getPropertyNameFromSetter(Method setter) {
        assert(isSetter(setter));
        String methodName = setter.getName();
        return decapitalize(methodName.substring(SET_PREFIX_SIZE));
    }

    /**
     * This method does not validate the input.
     * @param getter method which must be a valid getter method.
     * @return the matching setter method. null, if none found.
     */
    public static Method getSetter(Method getter) {
        String setterName;
        final String getterName = getter.getName();
        if(getterName.startsWith(IS_PREFIX)) {
            setterName = SET_PREFIX +
                    getterName.substring(IS_PREFIX_SIZE);
        } else {
            setterName = SET_PREFIX +
                    getterName.substring(GET_PREFIX_SIZE);
        }

        Method setter = getter.getDeclaringClass().getMethod(
                setterName,
                Arrays.asList(getter.getType()),
                true);

        if (setter!=null) {
            // check to see if it meets other requirements like
            // correct return type, nonstatic.
            if(setter.getType().getName().equals(Constants.VOID) &&
                    !Modifier.isStatic(setter.getModifiers())) {
                return setter;
            }
        }
        return null;
    }

    /**
     * This method does not validate the input.
     * @param setter method which must be a valid setter method.
     * @return the matching getter method. null, if none found.
     */
    public static Method getGetter(Method setter) {
        assert(isSetter(setter));

        String getterName = GET_PREFIX +
                    setter.getName().substring(SET_PREFIX_SIZE);

        Method getter = setter.getDeclaringClass().getMethod(
                getterName,
                Collections.emptyList(),
                true);
        final Type setterParamType = Parameter.class.cast(
                setter.getParameters().get(0)).getType();
        if (getter == null &&
                setterParamType.getName().equals(Constants.PRIMITIVE_BOOLEAN)) {
            // this is a boolean property, so let's see if there is an isProperty
            getterName = IS_PREFIX +
                    setter.getName().substring(SET_PREFIX_SIZE);
            getter = setter.getDeclaringClass().getMethod(
                    getterName,
                    Collections.emptyList(),
                    true);
        }
        if (getter!=null) {
            // check to see if it meets other requirements like
            // correct return type, nonstatic.
            if(getter.getType().equals(setterParamType) &&
                    !Modifier.isStatic(getter.getModifiers())) {
                return getter;
            }
        }

        return null;
    }

    /**
     * Returns a String which capitalizes the first letter of the string.
     */
    public static String capitalize(String name) {
        return name.substring(0, 1).toUpperCase() + name.substring(1);
    }

    public static Method getGetter(Field field) {
        JavaClass javaClass = (JavaClass) field.getDeclaringClass();
        //TODO handle boolean
        return javaClass.getMethod(GET_PREFIX + capitalize(field.getName()), Collections.EMPTY_LIST, false);
    }
}
