/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.j2ee.metadata;

import org.netbeans.spi.java.classpath.ClassPathImplementation;

import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Collections;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import java.lang.ref.WeakReference;
import java.util.concurrent.CopyOnWriteArraySet;
import org.netbeans.spi.java.classpath.PathResourceImplementation;
import org.openide.util.Utilities;
import org.openide.util.WeakListeners;

/** ProxyClassPathImplementation provides read only proxy for ClassPathImplementations.
 *  The order of the resources is given by the order of its delegates.
 *  The proxy is designed to be used as a union of class paths.
 *  E.g. to be able to easily iterate or listen on all design resources = sources + compile resources
 */
public class WeakProxyClassPathImplementation implements ClassPathImplementation {
    
    final private CopyOnWriteArraySet<WeakReference<ClassPathImplementation>> classPathRefs = new CopyOnWriteArraySet<WeakReference<ClassPathImplementation>>();
    final private PropertyChangeListener classPathsListener = new DelegatesListener ();

    private List<PathResourceImplementation> resourcesCache;
    private ArrayList<PropertyChangeListener> listeners;

    public WeakProxyClassPathImplementation (ClassPathImplementation[] classPaths) {
        if (classPaths == null) {
            throw new IllegalArgumentException ();
        }
        for (ClassPathImplementation classPath : classPaths) {
            if (classPaths == null) {
                continue;
            }
            classPath.addPropertyChangeListener (WeakListeners.propertyChange(classPathsListener, classPath));
            classPathRefs.add(new CleanupReference<ClassPathImplementation>(classPath));
        }
    }

    public List <PathResourceImplementation> getResources() {
        synchronized (this) {
            if (this.resourcesCache != null) {
                return this.resourcesCache;
            }
        }
        List<PathResourceImplementation> result = new ArrayList<PathResourceImplementation>(classPathRefs.size() * 10);
        for (WeakReference<ClassPathImplementation> classPathRef : classPathRefs) {
            ClassPathImplementation classPath = classPathRef.get();
            if (classPath == null) {
                continue;
            }
            List<PathResourceImplementation> subPath = classPath.getResources();
            assert subPath != null : "ClassPathImplementation.getResources() returned null. ClassPathImplementation.class: " 
                + classPath.getClass().toString() + " ClassPathImplementation: " + classPath.toString();
            result.addAll(subPath);
        }
        synchronized (this) {
            if (this.resourcesCache == null) {
                resourcesCache = Collections.unmodifiableList(result);
            }
            return this.resourcesCache;
        }
    }

    public synchronized void addPropertyChangeListener(PropertyChangeListener listener) {
        if (this.listeners == null)
            this.listeners = new ArrayList<PropertyChangeListener>();
        this.listeners.add (listener);
    }

    public synchronized void removePropertyChangeListener(PropertyChangeListener listener) {
        if (this.listeners == null)
            return;
        this.listeners.remove (listener);
    }
    
    public String toString () {
        StringBuffer builder = new StringBuffer("[");   //NOI18N
        for (WeakReference<ClassPathImplementation> classPathRef : classPathRefs) {
            ClassPathImplementation classPath = classPathRef.get();
            if (classPath != null) {
                builder.append (classPath.toString());
                builder.append(", ");   //NOI18N
            }
        }
        builder.append ("]");   //NOI18N
        return builder.toString ();
    }
    
    private void change(String propertyName) {
        Iterator it = null;
        synchronized (this) {
            WeakProxyClassPathImplementation.this.resourcesCache = null;    //Clean the cache
            if (WeakProxyClassPathImplementation.this.listeners == null) {
                return;
            }
            it = ((ArrayList)WeakProxyClassPathImplementation.this.listeners.clone()).iterator();
        }
        PropertyChangeEvent event = new PropertyChangeEvent (this, propertyName, null, null);
        while (it.hasNext()) {
            ((PropertyChangeListener)it.next()).propertyChange (event);
        }
    }

    private class DelegatesListener implements PropertyChangeListener {

        public void propertyChange(PropertyChangeEvent evt) {
            change(evt.getPropertyName());
        }
    }
    
    private class CleanupReference<T> extends WeakReference<T> implements Runnable {
        
        public CleanupReference(T referent) {
            super(referent, Utilities.activeReferenceQueue());
        }
        
        public void run() {
            classPathRefs.remove(this);
            change(ClassPathImplementation.PROP_RESOURCES);
        }
    }

}
