/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.metadata;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.lang.ref.WeakReference;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.swing.event.ChangeListener;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.queries.SourceForBinaryQuery;
import org.netbeans.modules.j2ee.persistence.wizard.fromdb.ChangeSupport;
import org.netbeans.spi.java.classpath.ClassPathFactory;
import org.netbeans.spi.java.classpath.ClassPathImplementation;
import org.netbeans.spi.java.classpath.PathResourceImplementation;
import org.netbeans.spi.java.classpath.support.ClassPathSupport;
import org.netbeans.spi.java.queries.SourceForBinaryQueryImplementation;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.filesystems.URLMapper;

/**
 *
 * @author Andrei Badea
 */
public class ClassPathSourceCacheTest extends TestBase {

    private FileObject root1FO;
    private FileObject root2FO;
    private URL root1URL;
    private URL root2URL;

    private URL binary1URL;
    private FileObject src1;

    private URL binary2URL;
    private FileObject src2;

    private SourceForBinaryQueryImpl s4bq1;
    private SourceForBinaryQueryImpl s4bq2;

    public ClassPathSourceCacheTest(String testName) {
        super(testName);
    }

    public void setUp() throws Exception {
        clearWorkDir();
        FileObject workDir = FileUtil.toFileObject(getWorkDir());

        root1FO = workDir.createFolder("root1");
        root2FO = workDir.createFolder("root2");

        root1URL = URLMapper.findURL(root1FO, URLMapper.INTERNAL);
        root2URL = URLMapper.findURL(root2FO, URLMapper.INTERNAL);

        binary1URL = FileUtil.getArchiveRoot(URLMapper.findURL(workDir.createData("binary1.jar"), URLMapper.INTERNAL));
        src1 = workDir.createFolder("src1");

        binary2URL = FileUtil.getArchiveRoot(URLMapper.findURL(workDir.createData("binary2.jar"), URLMapper.INTERNAL));
        src2 = workDir.createFolder("src2");


        s4bq1 = new SourceForBinaryQueryImpl(binary1URL, new FileObject[] { src1 });
        s4bq2 = new SourceForBinaryQueryImpl(binary2URL, new FileObject[] { src2 });
        setLookups(new Object[] { s4bq1, s4bq2 });





    }


    public void testClassPathSourceCache() throws Exception {










        ClassPathImpl classPathImpl = new ClassPathImpl(new URL[] { root1URL, root2URL, binary1URL });
        ClassPath classPath = ClassPathFactory.createClassPath(classPathImpl);
        ClassPathSourceCache cache = ClassPathSourceCache.newInstance(classPath, true);
        PCL pcl = new PCL();
        cache.addPropertyChangeListener(pcl);

        assertFalse(pcl.testChangeAndReset());
        assertTrue(cache.contains(src1));
        assertFalse(cache.contains(src2));
        assertRoots(new FileObject[] { root1FO, root2FO, src1 }, cache.getRoots());

        // adding a binary root with sources to the classpath

        classPathImpl.changeResources(new URL[] { root1URL, root2URL, binary1URL, binary2URL });

        assertTrue(pcl.testChangeAndReset());
        assertTrue(cache.contains(src1));
        assertTrue(cache.contains(src2));
        assertRoots(new FileObject[] { root1FO, root2FO, src1, src2 }, cache.getRoots());

        // removing a source root from a SFBQ result

        s4bq1.getResult().changeRoots(new FileObject[0]);

        assertTrue(pcl.testChangeAndReset());
        assertFalse(cache.contains(src1));
        assertTrue(cache.contains(src2));
        assertRoots(new FileObject[] { root1FO, root2FO, src2 }, cache.getRoots());
        // let's also test contains()
        assertTrue(cache.contains(root1FO));
        assertTrue(cache.contains(root2FO));
        assertFalse(cache.contains(src1));
        assertTrue(cache.contains(src2));

        // deleting root2FO should remove it from the result of getRoots()

        root2FO.delete();

        assertTrue(pcl.testChangeAndReset());
        assertFalse(cache.contains(src1));
        assertTrue(cache.contains(src2));
        assertRoots(new FileObject[] { root1FO, src2 }, cache.getRoots());
        assertTrue(cache.contains(root1FO));
        assertFalse(cache.contains(root2FO));
        assertFalse(cache.contains(src1));
        assertTrue(cache.contains(src2));

        // adding some files underneath src2 to test that contains() works correctly

        FileObject foo = src2.createFolder("foo");
        FileObject bar = foo.createData("bar");

        assertTrue(cache.contains(foo));
        assertTrue(cache.contains(bar));

        // should be able to GC the classpath

        WeakReference<ClassPath> classPathRef = new WeakReference<ClassPath>(classPath);
        classPathImpl = null; // as it holds classPath in its listener list
        classPath = null;

        assertGC("Should be able to GC classPath", classPathRef);

        synchronized (cache) {
            assertTrue("The cache should be deinitialized", cache.deinitialized);
            assertEquals("The cache source roots should have deinitialized", 0, cache.sourceRoots.size());
            assertEquals("The cache SFBQ results should have deinitialized", 0, cache.results.size());
        }

        assertEquals("Should have unregistered any SFBQ result listeners", 0, s4bq1.getResult().getListenerCount());
        assertEquals("Should have unregistered any SFBQ result listeners", 0, s4bq2.getResult().getListenerCount());
    }

    public void testNoResolveSources() throws Exception {
        ClassPathImpl classPathImpl = new ClassPathImpl(new URL[] { root1URL, root2URL, binary1URL });
        ClassPath classPath = ClassPathFactory.createClassPath(classPathImpl);
        ClassPathSourceCache cache = ClassPathSourceCache.newInstance(classPath, false);

        assertRoots(new FileObject[] { root1FO, root2FO }, cache.getRoots());

        // adding a binary root with sources to the classpath, but we are not resolving the sources
        // thus the roots should not contain them

        classPathImpl.changeResources(new URL[] { root1URL, root2URL, binary1URL, binary2URL });

        assertRoots(new FileObject[] { root1FO, root2FO }, cache.getRoots());
    }

    private static void assertRoots(FileObject[] expected, FileObject[] actual) {
        Set<FileObject> expectedSet = new HashSet<FileObject>(Arrays.asList(expected));
        Set<FileObject> actualSet = new HashSet<FileObject>(Arrays.asList(actual));
        assertEquals("The expected array contains duplicates", expected.length, expectedSet.size());
        assertEquals("The actual array contains duplicates", actual.length, actualSet.size());
        assertEquals(expectedSet, actualSet);
    }

    private static final class SourceForBinaryQueryImpl implements SourceForBinaryQueryImplementation {

        private final URL url;
        private final FileObject[] roots;

        private ResultImpl result;

        public SourceForBinaryQueryImpl(URL url, FileObject[] roots) {
            this.url = url;
            this.roots = roots;
        }

        public SourceForBinaryQuery.Result findSourceRoots(URL binaryRoot) {
            if (url.equals(binaryRoot)) {
                synchronized (this) {
                    if (result == null) {
                        result = new ResultImpl(roots);
                    }
                    return result;
                }
            }
            return null;
        }

        public synchronized ResultImpl getResult() {
            return result;
        }
    }

    private static final class ResultImpl implements SourceForBinaryQuery.Result {

        private final ChangeSupport changeSupport = new ChangeSupport(this);

        private FileObject[] roots;

        public ResultImpl(FileObject[] roots) {
            this.roots = roots;
        }

        public synchronized void changeRoots(FileObject[] roots) {
            this.roots = roots;
            changeSupport.fireChange();
        }

        public synchronized FileObject[] getRoots() {
            return roots;
        }

        public void addChangeListener(ChangeListener listener) {
            changeSupport.addChangeListener(listener);
        }

        public void removeChangeListener(ChangeListener listener) {
            changeSupport.removeChangeListener(listener);
        }

        public int getListenerCount() {
            return changeSupport.getListenerCount();
        }
    }

    private static final class ClassPathImpl implements ClassPathImplementation {

        private final PropertyChangeSupport propChangeSupport = new PropertyChangeSupport(this);
        private List<PathResourceImplementation> resources;

        public ClassPathImpl(URL[] urls) {
            computeResources(urls);
        }

        public synchronized void changeResources(URL[] urls) {
            computeResources(urls);
            propChangeSupport.firePropertyChange(ClassPathImplementation.PROP_RESOURCES, null, null);
        }

        private synchronized void computeResources(URL[] urls) {
            resources = new ArrayList<PathResourceImplementation>();
            for (URL url : urls) {
                resources.add(ClassPathSupport.createResource(url));
            }
        }

        public synchronized List getResources() {
            return resources;
        }

        public void addPropertyChangeListener(PropertyChangeListener listener) {
            propChangeSupport.addPropertyChangeListener(listener);
        }

        public void removePropertyChangeListener(PropertyChangeListener listener) {
            propChangeSupport.removePropertyChangeListener(listener);
        }
    }

    private static final class PCL implements PropertyChangeListener {

        private int changeCount;

        public void propertyChange(PropertyChangeEvent event) {
            if (ClassPathSourceCache.PROP_ROOTS.equals(event.getPropertyName())) {
                changeCount++;
            }
        }

        private boolean testChangeAndReset() {
            boolean result = changeCount > 0;
            changeCount = 0;
            return result;
        }
    }
}
