/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.refactoring;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AttributeValue;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.ParameterizedType;
import org.netbeans.jmi.javamodel.StringLiteral;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.modules.j2ee.common.JMIUtils;
import org.netbeans.modules.javacore.api.JavaModel;

/**
 * This class resolves associations between entities. Entities might be associated
 * via annotations such as OneToOne/OneToMany/ManyToMany. Note that currently this
 * class can only find annotated associations.
 *
 * @author Erno Mononen
 */
public class EntityAssociationResolver {
    
    /**
     * The entity whose associations are resolved.
     */
    private JavaClass entity;
    /**
     * List of all possible entities to whom <code>entity</code> might be associated,
     * for example all entities in a project.
     */
    private List<JavaClass> allEntities;
    
    // supported annotations
    private static final List<String> ANNOTATIONS = Arrays.asList(
            new String[]{"OneToOne", "OneToMany", "ManyToOne", "ManyToMany",});
    
    private static final String MAPPED_BY = "mappedBy";
    private static final String TARGET_ENTITY = "targetEntity";
    
    
    /**
     * Creates a new instance of EntityAssociationResolver
     * @param entity the entity whose associations are resolved. Must be an annotated
     * entity and must not be null.
     * @param allEntities list of all possible entities to whom <code>entity</code> might be associated,
     * for example all entities in a project. Must not be null.
     */
    public EntityAssociationResolver(JavaClass entity, List<JavaClass> allEntities) {
        assert entity != null;
        assert allEntities != null;
        this.entity = entity;
        this.allEntities = allEntities;
    }
    
    /**
     * Gets all annotation references that refer to given <code>property</code> via <tt>mappedBy</tt>
     * element.
     * @param property the property of our entity (field or method) whose references are to be looked
     * for.
     * @return associations that refer to given <code>property</code> via 'mappedBy' element.
     */
    public List<EntityAnnotationReference> getMappedByReferences(Feature property){
        
        if (!(property instanceof Method || property instanceof Field)){
            return Collections.emptyList();
        }
        
        JavaClass reference = findReferencesOfType(resolveType(property));
        if (reference == null){
            return Collections.emptyList();
        }
        return getReferences(reference, MAPPED_BY, Utility.getPropertyName(property.getName()));
    }
    
    
    private List<Annotation> getFeatureAnnotations(JavaClass javaClass){
        List<Annotation> result = new ArrayList<Annotation>();
        for (Object elem : javaClass.getFeatures()) {
            Feature feature = (Feature) elem;
            for (Object elem2 : feature.getAnnotations()) {
                Annotation annotation = (Annotation) elem2;
                result.add(annotation);
            }
        }
        return result;
    }
    
    /**
     * Gets the name of f the given feature's type, i.e. return type's name for
     * method and for field its type's name. For a method that returns a generic collection or for
     * a field whose type is a generic collection this method will 
     * return the type of its elements.
     * @param feature the feature whose type's name is to be resolved.
     * @return name of the given feature's type.
     */
    private String resolveType(Feature feature){
        Type returnType = null;
        if (feature instanceof Method){
            Method method = (Method) feature;
            returnType = method.getType();
        } else if (feature instanceof Field){
            Field field = (Field) feature;
            returnType = field.getType();
        }
        if (returnType instanceof ParameterizedType){
            ParameterizedType parametrizedType = (ParameterizedType) returnType;
            List list = parametrizedType.getParameters();
            if (!list.isEmpty()){
                returnType = (Type) list.get(0);
            }
        }
        return returnType != null ? returnType.getName() : null;
    }
    
    /**
     * Gets references from our entity to given javaClass.
     * TODO: document me.
     * @param javaClass the
     * @param annotationElementName the name of the annotation whose references we're
     *  looking for.
     * @param propertyName the name of the property.
     * @return list of associations.
     */
    private List<EntityAnnotationReference> getReferences(JavaClass javaClass,
            String annotationElementName, String propertyName){
        
        List<EntityAnnotationReference> result = new ArrayList<EntityAnnotationReference>();
        
        for (Object elem : javaClass.getFeatures()) {
            Feature feature = (Feature) elem;
            
            if (!entity.getName().equals(resolveType(feature))){
                continue;
            }
            
            for (Object elem2 : feature.getAnnotations()){
                Annotation annotation = (Annotation) elem2;
                
                if (ANNOTATIONS.contains(annotation.getTypeName().getName())){
                    
                    for (Object elem3 : annotation.getAttributeValues()) {
                        AttributeValue attributeValue = (AttributeValue) elem3;
                        
                        if (attributeValue.getName().equals(annotationElementName)
                        && propertyName.equals(getStringValue(attributeValue))){
                            
                            result.add(new EntityAnnotationReference(entity, javaClass, feature, annotation, attributeValue));
                        }
                    }
                }
            }
        }
        return result;
    }
    
    private String getStringValue(AttributeValue attributeValue){
        return ((StringLiteral) attributeValue.getValue()).getValue();
    }
    
    private JavaClass findReferencesOfType(String clazz){
        for (JavaClass each : allEntities) {
            if (each.getName().equals(clazz)){
                return each;
            }
        }
        return null;
    }
    
}
