/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.refactoring;

import java.io.IOException;
import java.util.*;
import javax.jmi.reflect.RefObject;

import org.netbeans.api.java.classpath.GlobalPathRegistry;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ui.OpenProjects;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.modules.j2ee.dd.api.ejb.*;
import org.netbeans.modules.j2ee.deployment.devmodules.spi.J2eeAppProvider;
import org.netbeans.modules.j2ee.deployment.devmodules.spi.J2eeModuleProvider;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.web.api.webmodule.WebModule;
import org.netbeans.modules.web.spi.webmodule.WebModuleImplementation;
import org.netbeans.modules.websvc.api.webservices.WebServicesSupport;
import org.netbeans.spi.project.SubprojectProvider;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;

/**
 *
 * @author mg116726
 */
public class Utility {
    
    private static final ErrorManager err =
            ErrorManager.getDefault().getInstance("org.netbeans.modules.j2ee.refactoring");   // NOI18N
    
    public static final String ENTERPRISE_BEAN = "javax.ejb.EnterpriseBean"; //NOI18N
    public static final String SESSION_BEAN = "javax.ejb.SessionBean"; //NOI18N
    public static final String ENTITY_BEAN = "javax.ejb.EntityBean"; //NOI18N
    public static final String MESSAGE_DRIVEN_BEAN = "javax.ejb.MessageDrivenBean"; //NOI18N
    public static final String EJB_LOCAL_HOME = "javax.ejb.EJBLocalHome"; //NOI18N
    public static final String EJB_HOME = "javax.ejb.EJBHome"; //NOI18N
    public static final String EJB_LOCAL_OBJECT = "javax.ejb.EJBLocalObject"; //NOI18N
    public static final String EJB_OBJECT = "javax.ejb.EJBObject"; //NOI18N
    
    public static final String MESSAGE_LISTENER = "javax.jms.MessageListener"; //NOI18N
    
    public static final String EJB_CREATE = "ejbCreate";            //NOI18N
    public static final String EJB_ACTIVATE = "ejbActivate";        //NOI18N
    public static final String EJB_PASSIVATE = "ejbPassivate";      //NOI18N
    public static final String EJB_REMOVE = "ejbRemove";            //NOI18N
    public static final String EJB_LOAD = "ejbLoad";      //NOI18N
    public static final String EJB_STORE = "ejbStore";      //NOI18N
    public static final String SET_SESSION_CONTEXT = "setSessionContext";      //NOI18N
    public static final String SET_ENTITY_CONTEXT = "setEntityContext";      //NOI18N
    public static final String SET_MSG_CONTEXT = "setMessageDrivenContext";      //NOI18N
    public static final String UNSET_ENTITY_CONTEXT = "unsetEntityContext";      //NOI18N
    public static final String ON_MESSAGE = "onMessage";      //NOI18N
    
    public static final String[] NO_PARAMS = new String[] {};
    public static final String[] SET_SESSION_CONTEXT_PARAMTYPENAMES = new String[] {"javax.ejb.SessionContext"};
    public static final String[] SET_ENTITY_CONTEXT_PARAMTYPENAMES = new String[] {"javax.ejb.EntityContext"};
    public static final String[] SET_MSG_CONTEXT_PARAMTYPENAMES = new String[] {"javax.ejb.MessageDrivenContext"};
    public static final String[] ON_MESSAGE_PARAMTYPENAMES = new String[] {"javax.jms.Message"};
    public static final String PREFIX_SET = "set"; //NOI18N
    public static final String PREFIX_GET = "get"; //NOI18N
    public static final String PREFIX_FIND = "find"; //NOI18N
    public static final String PREFIX_EJBSELECT = "ejbSelect"; //NOI18N
    public static final String PREFIX_EJBPOSTCREATE = "ejbPostCreate";  //NOI18N
    public static final String PREFIX_EJBCREATE = "ejbCreate"; //NOI18N
    public static final String PREFIX_CREATE = "create"; //NOI18N
    public static final String PREFIX_EJBHOME = "ejbHome"; //NOI18N
    public static final String PREFIX_EJBFIND = "ejbFind"; //NOI18N
    private static final String PREFIX_POSTCREATE = "postCreate";
    
    /** Creates a new instance of Utility */
    public Utility() {
    }
    
    public static List getTypedParams(List params) {
        if (params != null) {
            List typedParams = new ArrayList(params.size());
            for (int i=0; i < params.size(); i++) {
                Element param = (Element) params.get(i);
                Type type = param instanceof Parameter ? ((Parameter) param).getType() : (Type) param;
                typedParams.add(type);
            }
            return typedParams;
        }
        return Collections.EMPTY_LIST;
    }
    
    public static EnterpriseBeans getEnterpriseBeansFromDD(FileObject fo) {
        err.log(">Utility.getEnterpriseBeansFromDD (" + fo + ")" );
        org.netbeans.modules.j2ee.api.ejbjar.EjbJar emod = org.netbeans.modules.j2ee.api.ejbjar.EjbJar.getEjbJar(fo);
        if (emod != null) {
            FileObject ejbJarFO = emod.getDeploymentDescriptor();
            EjbJar ejbJarDD = null;
            try {
                ejbJarDD = org.netbeans.modules.j2ee.dd.api.ejb.DDProvider.getDefault().getMergedDDRoot(emod.getMetadataUnit());
            } catch (IOException ioe) {
                //ignore
            }
            if ((ejbJarDD != null) && (ejbJarDD.getStatus()!=EjbJar.STATE_INVALID_UNPARSABLE)) {
                return ejbJarDD.getEnterpriseBeans();
            }
        }
        return null;
    }
    
    public static JavaClass getImplClassForHome(EnterpriseBeans eBeans, Class beanType, String homeClassName,
            boolean local) {
        if (eBeans != null) {
            Ejb[] ejbs = eBeans.getEjbs();
            if (ejbs != null) {
                for (int i = 0; i < ejbs.length; i++) {
                    Ejb ejb = ejbs[i];
                    if (ejb != null && beanType.isAssignableFrom(ejb.getClass())) {
                        EntityAndSession bean = (EntityAndSession) ejb;
                        if (homeClassName.equals(getHomeInterface(bean, local))) {
                            JavaClass javaClass = resolveRealClass(bean.getEjbClass());
                            if (javaClass != null) {
                                return javaClass;
                            }
                        }
                    }
                }
            }
        }
        return null;
    }
    
    public static Method getMethodInImplClassForHome(FileObject fo, Class beanType, String className,
            String methodName, List parameters, boolean local) {
        if (methodName == null || methodName.length() == 0) {
            return null;
        }
        JavaClass ejbImplClass = getImplClassForHome(getEnterpriseBeansFromDD(fo), beanType, className, local);
        if (ejbImplClass != null) {
            return ejbImplClass.getMethod(methodName, getTypedParams(parameters), false);
        } else {
            return null;
        }
    }
    
    private static JavaClass getOtherHomeClassForHome(EnterpriseBeans eBeans, Class beanType, String homeClassName,
            boolean local) {
        if (eBeans != null) {
            Ejb[] ejbs = eBeans.getEjbs();
            if (ejbs != null) {
                for (int i = 0; i < ejbs.length; i++) {
                    Ejb ejb = ejbs[i];
                    if (ejb != null && beanType.isAssignableFrom(ejb.getClass())) {
                        EntityAndSession bean = (EntityAndSession) ejbs[i];
                        if (homeClassName.equals(getHomeInterface(bean, !local))) {
                            JavaClass javaClass = resolveRealClass(getHomeInterface(bean, local));
                            if (isSubTypeOf(javaClass, local ? EJB_LOCAL_HOME : EJB_HOME)) {
                                return javaClass;
                            }
                        }
                    }
                }
            }
        }
        return null;
    }
    
    public static Method getMethodInOtherHomeClassForHome(FileObject fo, Class beanType, String className,
            String methodName, List parameters, boolean local) {
        JavaClass ejbHomeClass = getOtherHomeClassForHome(getEnterpriseBeansFromDD(fo), beanType, className, local);
        if (ejbHomeClass != null) {
            return ejbHomeClass.getMethod(methodName, getTypedParams(parameters), false);
        }
        return null;
    }
    
    private static JavaClass getHomeClassForImpl(EnterpriseBeans eBeans, Class beanType, String implClassName,
            boolean local) {
        Ejb bean = EjbInterfaceType.BEAN_IMPL.getBean(eBeans, beanType, implClassName);
        if (bean != null) {
            EjbInterfaceType ejbInterfaceType = local ? EjbInterfaceType.LOCAL : EjbInterfaceType.LOCAL_HOME;
            return ejbInterfaceType.resolveClass(bean);
        } else {
            return null;
        }
    }
    
    public static Method getMethodInHomeClassForImpl(FileObject fo, Class beanType, String className,
            String methodName, List parameters, boolean local) {
        if (methodName == null || methodName.length() == 0) {
            return null;
        }
        JavaClass ejbHomeClass = getHomeClassForImpl(getEnterpriseBeansFromDD(fo), beanType, className, local);
        if (ejbHomeClass != null) {
            return ejbHomeClass.getMethod(methodName, getTypedParams(parameters), false);
        }
        return null;
    }
    
    private static String getHomeInterface(EntityAndSession bean, boolean local) {
        return local ? bean.getLocalHome() : bean.getHome();
    }
    
    public static String addPrefix(String methodName, String prefix) {
        return prefix + methodName.substring(0,1).toUpperCase() + methodName.substring(1);
    }
    
    public static String stripNonEmptyPrefix(String methodName, String prefix) {
        methodName = stripPrefix(methodName, prefix);
        if (methodName != null && methodName.length() > 1 && isFirstUpperCase(methodName)) {
            return methodName.substring(0, 1).toLowerCase() + methodName.substring(1);
        }
        return null;
    }
    
    
    public static String stripPrefix(String methodName, String prefix) {
        if (methodName.startsWith(prefix)) {
            return methodName.substring(prefix.length());
        }
        return null;
    }
    
    public static String replacePrefix(String methodName, String oldPrefix, String newPrefix) {
        if (newPrefix == null) {
            return stripNonEmptyPrefix(methodName, oldPrefix);
        } else if (methodName.startsWith(oldPrefix) && methodName.length() > oldPrefix.length()) {
            return newPrefix + methodName.substring(oldPrefix.length());
        } else {
            return null;
        }
    }
    
    /** finds finder method in implementation class that coresponds to method in home interface (local or remote) based on valu of 'local' */
    public static Method getImplFinderMethodForHomeFinderMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getImplFinderMethodForHomeFinderMethod (" + method + ", " + fo + ", " + className + ", " + local + ")" );
        return getMethodInImplClassForHome(fo, EntityAndSession.class, className, local, method, PREFIX_FIND, PREFIX_EJBFIND);
    }
    
    /** finds create method in implementation class that coresponds to method in home interface (local or remote) based on value of 'local' */
    public static Method getImplCreateMethodForHomeCreateMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getImplCreateMethodForHomeCreateMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        String oldPrefix = PREFIX_CREATE;
        String newPrefix = PREFIX_EJBCREATE;
        Class beanType = EntityAndSession.class;
        return getMethodInImplClassForHome(fo, beanType, className, local, method, oldPrefix, newPrefix);
        
    }
    
    /** finds create method in implementation class that coresponds to method in home interface (local or remote) based on value of 'local' */
    public static Method getImplPostCreateMethodForHomeCreateMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getImplPostCreateMethodForHomeCreateMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        return getMethodInImplClassForHome(fo, EntityAndSession.class, className, local, method, PREFIX_EJBPOSTCREATE, PREFIX_CREATE);
    }
    
    public static Method getMethod(Method origMethod, String origPrefix, String prefix) {
        ClassDefinition jc = origMethod.getDeclaringClass(); // get affected class
        err.log("classdefinition classDefinition: " + jc);
        if ((jc instanceof JavaClass)) {
            String methodName = replacePrefix(origMethod.getName(), origPrefix, prefix);
            if (methodName == null || methodName.length() == 0) {
                return null;
            }
            return jc.getMethod(methodName, getTypedParams(origMethod.getParameters()), false);
        } else {
            return null;
        }
    }
    
    public static Method getMethodInHomeClassForImpl(FileObject fo, Class beanType, String className, boolean local,
            Method origMethod, String origPrefix, String prefix) {
        String methodName = replacePrefix(origMethod.getName(), origPrefix, prefix);
        return getMethodInHomeClassForImpl(fo, beanType, className, methodName, origMethod.getParameters(),
                local);
    }
    
    public static Method getMethodInImplClassForHome(FileObject fo, Class beanType, String className, boolean local,
            Method origMethod, String origPrefix, String prefix) {
        String methodName = replacePrefix(origMethod.getName(), origPrefix, prefix);
        return getMethodInImplClassForHome(fo, beanType, className, methodName,
                origMethod.getParameters(), local);
    }
    
    /** finds home method in implementation class that coresponds to method in home interface (local or remote) based on value of 'local' */
    public static Method getImplHomeMethodForHomeHomeMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getImplHomeMethodForHomeHomeMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        String implMethodName = addPrefix(method.getName(), PREFIX_EJBHOME);
        return getMethodInImplClassForHome(fo, EntityAndSession.class, className, implMethodName,
                method.getParameters(), local);
    }
    
    /** finds home method in home/localhome interface that coresponds to method in localhome/home interface - 'local' if true, finds local, otherwise remote */
    public static Method getHomeHomeMethodForHomeHomeMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getHomeHomeMethodForHomeHomeMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        String methodName = method.getName();
        return getMethodInOtherHomeClassForHome(fo, EntityAndSession.class, className, methodName,
                method.getParameters(), local);
    }
    
    /** finds finder method in home/local home iface that coresponds to method in localhome/home interface 'local' - if true, finds local, otherwise remote*/
    public static Method getHomeFinderMethodForHomeFinderMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getHomeFinderMethodForHomeFinderMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        String methodName = method.getName();
        if (!methodName.startsWith(PREFIX_FIND)) {
            return null;
        }
        return getMethodInOtherHomeClassForHome(fo, EntityAndSession.class, className, methodName,
                method.getParameters(), local);
    }
    
    /** finds create method in home/local home iface that coresponds to method in localhome/home interface 'local' - if true, finds local, otherwise remote*/
    public static Method getHomeCreateMethodForHomeCreateMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getHomeCreateMethodForHomeCreateMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        String methodName = method.getName();
        if (!methodName.startsWith(PREFIX_CREATE)) {
            return null;
        }
        return getMethodInOtherHomeClassForHome(fo, EntityAndSession.class, className, methodName,
                method.getParameters(), local);
    }
    
    /** finds finder method in home interface that coresponds to method in implementation class for EntityBeans*/
    public static Method getHomeFinderMethodForImplFinderMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getHomeFinderMethodForImplFinderMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        return getMethodInHomeClassForImpl(fo, Entity.class, className, local, method, PREFIX_FIND, PREFIX_EJBFIND);
    }
    
    /** finds home method in home interface that coresponds to method in implementation class for EntityBeans*/
    public static Method getHomeHomeMethodForEntityImplHomeMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getHomeHomeMethodForEntityImplHomeMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        String methodName = stripNonEmptyPrefix(method.getName(), PREFIX_EJBHOME);
        return getMethodInHomeClassForImpl(fo, Entity.class, className, methodName, method.getParameters(), local);
    }
    
    /** finds home method in home interface that coresponds to method in implementation class for SessionBeans*/
    public static Method getHomeHomeMethodForSessionImplHomeMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getHomeHomeMethodForSessionImplHomeMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        String methodName = stripNonEmptyPrefix(method.getName(), PREFIX_EJBHOME);
        return getMethodInHomeClassForImpl(fo, Session.class, className, methodName, method.getParameters(), local);
    }
    
    /** finds create method in home interface that coresponds to method in implementation class for SessionBeans*/
    public static Method getHomeCreateMethodForSessionImplCreateMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getHomeCreateMethodForSessionImplCreateMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        return getMethodInHomeClassForImpl(fo, Session.class, className, local, method, PREFIX_EJBCREATE, PREFIX_CREATE);
    }
    
    /**
     * finds create method in home interface that coresponds to method in implementation class for EntityBeans
     */
    public static Method getHomeCreateMethodForEntityImplCreateMethod(Method method, FileObject fo, String className,
            boolean local) {
        err.log(">Utility.getHomeCreateMethodForEntityImplCreateMethod (" + method + ", " + fo + ", " + className +
                ", " + local + ")");
        String oldPrefix = PREFIX_EJBCREATE;
        String newPrefix = PREFIX_CREATE;
        Class beanType = Entity.class;
        return getMethodInHomeClassForImpl(fo, beanType, className, local, method, oldPrefix, newPrefix);
    }
    
    /** finds create method in home interface that coresponds to postCreate method in implementation class for EntityBeans*/
    public static Method getHomeCreateMethodForEntityImplPostCreateMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getHomeCreateMethodForEntityImplPostCreateMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        return getMethodInHomeClassForImpl(fo, Entity.class, className, local, method, PREFIX_EJBPOSTCREATE, PREFIX_CREATE);
    }
    
    /** finds create method in home interface that coresponds to method in implementation class for EntityBeans*/
    public static Method getHomePostCreateMethodForEntityImplCreateMethod(Method method, FileObject fo, String className, boolean local) {
        err.log(">Utility.getHomePostCreateMethodForEntityImplCreateMethod (" + method + ", " + fo + ", " + className + ", " + local + ")");
        return getMethodInHomeClassForImpl(fo, Entity.class, className, local, method, PREFIX_EJBCREATE, PREFIX_POSTCREATE);
    }
    
    /** finds create method in home interface that coresponds to method in implementation class for EntityBeans*/
    public static Method getImplPostCreateMethodForEntityImplCreateMethod(Method method) {
        return getMethod(method, PREFIX_EJBCREATE, PREFIX_EJBPOSTCREATE);
    }
    
    /** finds create method in home interface that coresponds to method in implementation class for EntityBeans*/
    public static Method getImplCreateMethodForEntityImplPostCreateMethod(Method method) {
        String oldPrefix = PREFIX_EJBPOSTCREATE;
        String newPrefix = PREFIX_EJBCREATE;
        return getMethod(method, oldPrefix, newPrefix);
    }
    
    /** checks if a methods signature matches one of javax.ejb.SessionBean interface methods */
    public static boolean isSignatureFromSessionBeanInterface(Method method) {
        return (equalMethods(method, EJB_PASSIVATE, NO_PARAMS)) ||
                (equalMethods(method, EJB_ACTIVATE, NO_PARAMS)) ||
                (equalMethods(method, EJB_REMOVE, NO_PARAMS)) ||
                (equalMethods(method, SET_SESSION_CONTEXT, SET_SESSION_CONTEXT_PARAMTYPENAMES));
    }
    
    /** checks if a methods signature matches one of javax.ejb.EntityBean interface methods */
    public static boolean isSignatureFromEntityBeanInterface(Method method) {
        return (equalMethods(method, EJB_PASSIVATE, NO_PARAMS)) ||
                (equalMethods(method, EJB_ACTIVATE, NO_PARAMS)) ||
                (equalMethods(method, EJB_REMOVE, NO_PARAMS)) ||
                (equalMethods(method, EJB_LOAD, NO_PARAMS)) ||
                (equalMethods(method, EJB_STORE, NO_PARAMS)) ||
                (equalMethods(method, UNSET_ENTITY_CONTEXT, NO_PARAMS)) ||
                (equalMethods(method, SET_ENTITY_CONTEXT, SET_ENTITY_CONTEXT_PARAMTYPENAMES));
    }
    
    /** checks if a methods signature matches one of javax.ejb.MessageDrivenBean interface methods */
    public static boolean isSignatureFromMessageDrivenBeanInterface(Method method) {
        return (equalMethods(method, EJB_CREATE, NO_PARAMS)) ||
                (equalMethods(method, EJB_REMOVE, NO_PARAMS)) ||
                (equalMethods(method, SET_MSG_CONTEXT, SET_MSG_CONTEXT_PARAMTYPENAMES));
    }
    
    /** checks if a methods signature matches one of javax.ejb.MessageDrivenBean interface methods */
    public static boolean isSignatureFromMessageListenerInterface(Method method) {
        return equalMethods(method, Utility.ON_MESSAGE, Utility.ON_MESSAGE_PARAMTYPENAMES);
    }
    
    /** Returns collection of entity beans from dd or null if something happened */
    public static EjbJar getEjbJar(org.netbeans.modules.j2ee.api.ejbjar.EjbJar emodule) {
        if (emodule != null) {
            FileObject ejbJarFO = emodule.getDeploymentDescriptor();
            try {
                return org.netbeans.modules.j2ee.dd.api.ejb.DDProvider.getDefault().getMergedDDRoot(emodule.getMetadataUnit());
            } catch (IOException ioe) {
                //ignore
            }
        }
        return null;
    }
    
    /** Returns collection of entity beans from dd or null if something happened */
    public static Entity[] getEntityBeans(org.netbeans.modules.j2ee.api.ejbjar.EjbJar emodule) {
        EjbJar ejbJarDD = getEjbJar(emodule);
        if (ejbJarDD != null) {
            EnterpriseBeans eBeans = ejbJarDD.getEnterpriseBeans();
            if (eBeans != null) {
                return eBeans.getEntity();
            }
        }
        return null;
    }
    
    /** Returns relationships from dd or null if something happened */
    public static EjbRelation[] getRelationships(org.netbeans.modules.j2ee.api.ejbjar.EjbJar emodule) {
        EjbJar ejbJarDD = getEjbJar(emodule);
        if (ejbJarDD != null) {
            Relationships rels = ejbJarDD.getSingleRelationships();
            if (rels != null) {
                return rels.getEjbRelation();
            }
        }
        return null;
    }
    
    /** Returns collection of entity beans from dd or null if something happened */
    public static AssemblyDescriptor getAssemblyDescriptor(org.netbeans.modules.j2ee.api.ejbjar.EjbJar emodule) {
        EjbJar ejbJarDD = getEjbJar(emodule);
        if (ejbJarDD != null) {
            return ejbJarDD.getSingleAssemblyDescriptor();
        }
        return null;
    }
    
    public static String getEjbNameForClass(org.netbeans.modules.j2ee.dd.api.ejb.EjbJar ejbJarDD, ClassDefinition jc) {
        EnterpriseBeans eBeans = ejbJarDD.getEnterpriseBeans();
        if ((eBeans != null)) {
            Ejb[] ejbs = eBeans.getEjbs();
            for (int e=0; e<ejbs.length; e++) {
                Ejb ejb = ejbs[e];
                String ejbClass = ejb.getEjbClass();
                err.log("ejbclass: " + ejbClass);
                if (ejbClass.equals(jc.getName())) {
                    return ejb.getEjbName();
                }
            }
        }
        return null;
    }
    
    /** Returns collection of entity beans from dd or null if something happened */
    public static Session[] getSessionBeans(org.netbeans.modules.j2ee.api.ejbjar.EjbJar emodule) {
        EjbJar ejbJarDD = getEjbJar(emodule);
        if (ejbJarDD != null) {
            EnterpriseBeans eBeans = ejbJarDD.getEnterpriseBeans();
            if (eBeans != null) {
                return eBeans.getSession();
            }
        }
        return null;
    }
    
    /** Checks if signature of a 'method' matches 'name' and a set of paramNames */
    public static boolean equalMethods(Method method, String name, String[] paramNames) {
        if (!name.equals(method.getName())) {
            return false;
        }
        if (method.getParameters().size() != paramNames.length) {
            return false;
        }
        Iterator it = method.getParameters().iterator();
        for (int i=0; i < paramNames.length; i++) {
            Parameter param = (Parameter)it.next();
            if (!param.getType().getName().equals(paramNames[i])) {
                return false;
            }
        }
        return true;
    }
    
    public static boolean equalMethods(Method method1, Method method2) {
        if (method1 == null || method2 == null) {
            return method1 == method2;
        }
        if (!method1.getName().equals(method2.getName())) {
            return false;
        }
        List parameters1 = method1.getParameters();
        List parameters2 = method2.getParameters();
        int n = parameters1.size();
        if (n != parameters2.size()) {
            return false;
        }
        for (int i = 0; i < n; i++) {
            Parameter param1 = (Parameter) parameters1.get(i);
            Parameter param2 = (Parameter) parameters2.get(i);
            if (!param1.getType().getName().equals(param2.getType().getName())) {
                return false;
            }
        }
        return true;
    }
    
    public static void addProblemsToEnd(Problem[] where, Problem what) {
        where[0] = addProblemsToEnd(where[0], what);
    }
    
    public static Problem addProblemsToEnd(Problem where, Problem what) {
        if (where == null) {
            return what;
        } else{
            if(what != null) {
                Problem tail = where;
                while (tail.getNext() != null) {
                    tail = tail.getNext();
                }
                tail.setNext(what);
            }
            return where;
        }
    }
    
    /** Finds all ejb projects that depend on a project which is owner of the element */
    public static Collection/*<EjbJar>*/ getRelevantEjbModules(Element element) {
        FileObject fo = JavaModel.getFileObject(element.getResource());
        return Utility.getRelevantEjbModules(fo);
    }
    
    public static Collection/*<EjbJar>*/ getRelevantEjbModules(RefObject refObject) {
        if (refObject instanceof Element){
            return getRelevantEjbModules((Element)refObject);
        }
        return Collections.emptyList();
    }
    
    /** Finds all ejb projects that depend on a project which is owner of FileObject 'fo' */
    public static Collection/*<EjbJar>*/ getRelevantEjbModules(FileObject fo) {
        
        Project affectedProject = FileOwnerQuery.getOwner(fo);
        Collection ejbmodules = new ArrayList();
        Collection projects = new ArrayList();
        
        if (affectedProject != null) {
            // first check if the project which directly contains fo is relevant
            org.netbeans.modules.j2ee.api.ejbjar.EjbJar emod =
                    org.netbeans.modules.j2ee.api.ejbjar.EjbJar.getEjbJar(affectedProject.getProjectDirectory());
            if (emod != null) {
                projects.add(affectedProject);
            } else {
                return Collections.EMPTY_SET;
            }
            for (Project project : OpenProjects.getDefault().getOpenProjects()){
                Object isJ2eeApp = project.getLookup().lookup(J2eeAppProvider.class);
                if (isJ2eeApp != null) {
                    J2eeAppProvider j2eeApp = (J2eeAppProvider) isJ2eeApp;
                    J2eeModuleProvider[] j2eeModules = j2eeApp.getChildModuleProviders();
                    
                    if (j2eeModules != null) {
                        J2eeModuleProvider affectedPrjProvider =
                                (J2eeModuleProvider) affectedProject.getLookup().lookup(J2eeModuleProvider.class);

                        if (affectedPrjProvider != null) {
                            if (Arrays.asList(j2eeModules).contains(affectedPrjProvider)) {
                                for (int k = 0; k < j2eeModules.length; k++) {
                                    FileObject[] sourceRoots = j2eeModules[k].getSourceRoots();
                                    if (sourceRoots != null && sourceRoots.length > 0){
                                        FileObject srcRoot = sourceRoots[0];
                                        Project p = FileOwnerQuery.getOwner(srcRoot);
                                        if ((p != null) && (!projects.contains(p))) {
                                            projects.add(p);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                Object obj = project.getLookup().lookup(SubprojectProvider.class);
                if ((obj != null) && (obj instanceof SubprojectProvider)) {
                    Set subprojects = ((SubprojectProvider) obj).getSubprojects();
                    if (subprojects.contains(affectedProject)) {
                        org.netbeans.modules.j2ee.api.ejbjar.EjbJar em = org.netbeans.modules.j2ee.api.ejbjar.EjbJar
                                .getEjbJar(project.getProjectDirectory());
                        if (em != null) {
                            if (!projects.contains(project)) { // include each project only once
                                projects.add(project);
                            }
                        }
                    }
                }
            }
        }
        
        for (int j=0; j < projects.size(); j++) {
            Project prj = (Project)((ArrayList)projects).get(j);
            org.netbeans.modules.j2ee.api.ejbjar.EjbJar ejb =
                    org.netbeans.modules.j2ee.api.ejbjar.EjbJar.getEjbJar(prj.getProjectDirectory());
            if (ejb != null) {
                ejbmodules.add(ejb);
            }
        }
        
        err.log("Affected ejb modules: " + ejbmodules);
        return ejbmodules;
    }
    
    /** Finds all WS projects affected by the change of FileObject 'fo' */
    public static Collection/*WebServicesSupport*/ getRelevantWSModules(FileObject fo) {
        
        Project affectedProject = FileOwnerQuery.getOwner(fo);
        Collection wsmodules = new ArrayList();
        Collection projects = new ArrayList();
        
        if (affectedProject != null) {
            // first check if the project which directly contains fo is relevant
            WebServicesSupport wsmod = WebServicesSupport.getWebServicesSupport(affectedProject.getProjectDirectory());
            if (wsmod != null) {
                projects.add(affectedProject);
            } else {
                return Collections.EMPTY_SET;
            }
            
            for (Project project : OpenProjects.getDefault().getOpenProjects()){
                
                Object isJ2eeApp = project.getLookup().lookup(J2eeAppProvider.class);
                if (isJ2eeApp != null) {
                    J2eeAppProvider j2eeApp = (J2eeAppProvider)isJ2eeApp;
                    J2eeModuleProvider[] j2eeModules = j2eeApp.getChildModuleProviders();
                
                    if (j2eeModules != null) {
                        J2eeModuleProvider affectedPrjProvider =
                                (J2eeModuleProvider)affectedProject.getLookup().lookup(J2eeModuleProvider.class);
                        if (affectedPrjProvider != null) {
                            if (Arrays.asList(j2eeModules).contains(affectedPrjProvider)) {
                                for (int i=0; i<j2eeModules.length; i++) {
                                    FileObject[] sourceRoots = j2eeModules[i].getSourceRoots();
                                    if (sourceRoots != null && sourceRoots.length > 0){
                                        FileObject srcRoot = sourceRoots[0];
                                        Project p = FileOwnerQuery.getOwner(srcRoot);
                                        if ((p != null) && (!projects.contains(p))) {
                                            projects.add(p);
                                        }
                                    }
                                }
                            }
                        }
                    }
                    Object obj = project.getLookup().lookup(SubprojectProvider.class);
                    if ((obj != null) && (obj instanceof SubprojectProvider)) {
                        Set subprojects = ((SubprojectProvider)obj).getSubprojects();
                        if (subprojects.contains(affectedProject)) {
                            WebServicesSupport ws =
                                    WebServicesSupport.getWebServicesSupport(project.getProjectDirectory());
                            if (ws != null) {
                                if (!projects.contains(project)) { // include each project only once
                                    projects.add(project);
                                }
                            }
                        }
                    }
                }
            }
        }
        
        for (int j=0; j < projects.size(); j++) {
            Project prj = (Project)((ArrayList)projects).get(j);
            WebServicesSupport websvc = WebServicesSupport.getWebServicesSupport(prj.getProjectDirectory());
            wsmodules.add(websvc);
        }
        
        err.log("Affected ws modules: " + wsmodules);
        return wsmodules;
    }
    
    /** Finds all web projects that depend on a project which is owner of FileObject 'fo' */
    public static Collection/*WebModule*/ getRelevantWebModules(Element e) {
        Resource r = e.getResource();
        if (r != null) {
            FileObject f = JavaModel.getFileObject(r);
            if (f != null) {
                Collection/*WebModule*/ c = getRelevantWebModules(f);
                return c;
            }
            
        }
        return Collections.EMPTY_LIST;
    }
    
    /** Finds all web projects that depend on a project which is owner of FileObject 'fo' */
    public static Collection/*WebModule*/ getRelevantWebModules(FileObject fo) {
        
        Collection webmodules = new ArrayList();
        Collection projects = new ArrayList();
        
        if ( fo == null ) {
            ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, "Passed null to getRelevantWebModules()");
            return webmodules;
        }
        
        Project affectedProject = FileOwnerQuery.getOwner(fo);
        if (affectedProject != null) {
            // first check if the project which directly contains fo is relevant
            WebModule wmod = WebModule.getWebModule(affectedProject.getProjectDirectory());
            if (wmod != null) {
                projects.add(affectedProject);
            } else {
                return Collections.EMPTY_SET;
            }
            Set globalPath = GlobalPathRegistry.getDefault().getSourceRoots();
            Iterator iter = globalPath.iterator();
            while (iter.hasNext()) {
                FileObject sourceRoot = (FileObject)iter.next();
                Project project = FileOwnerQuery.getOwner(sourceRoot);
                if (project != null) {
                    err.log("found this project: " + project);
                    Object isJ2eeApp = project.getLookup().lookup(J2eeAppProvider.class);
                    if (isJ2eeApp != null) {
                        J2eeAppProvider j2eeApp = (J2eeAppProvider)isJ2eeApp;
                        err.log("j2eeapp: " + j2eeApp);
                        J2eeModuleProvider[] j2eeModules = j2eeApp.getChildModuleProviders();
                        err.log("j2ee modules: " + j2eeModules);
                        if (j2eeModules != null) {
                            J2eeModuleProvider affectedPrjProvider =
                                    (J2eeModuleProvider)affectedProject.getLookup().lookup(J2eeModuleProvider.class);
                            if (affectedPrjProvider != null) {
                                if (Arrays.asList(j2eeModules).contains(affectedPrjProvider)) {
                                    for (int i=0; i<j2eeModules.length; i++) {
                                        if (j2eeModules[i] instanceof WebModuleImplementation) {
                                            FileObject docBase =
                                                    ((WebModuleImplementation)j2eeModules[i]).getDocumentBase();
                                            Project p = FileOwnerQuery.getOwner(docBase);
                                            if ((p != null) && (!projects.contains(p))) {
                                                projects.add(p);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                    Object obj = project.getLookup().lookup(SubprojectProvider.class);
                    if ((obj != null) && (obj instanceof SubprojectProvider)) {
                        Set subprojects = ((SubprojectProvider)obj).getSubprojects();
                        if (subprojects.contains(affectedProject)) {
                            WebModule wm = WebModule.getWebModule(project.getProjectDirectory());
                            if (wm != null) {
                                if (!projects.contains(project)) { // include each project only once
                                    projects.add(project);
                                }
                            }
                        }
                    }
                }
            }
        }
        
        for (int j=0; j < projects.size(); j++) {
            Project prj = (Project)((ArrayList)projects).get(j);
            WebModule web = WebModule.getWebModule(prj.getProjectDirectory());
            webmodules.add(web);
        }
        
//        err.log("Affected web modules: " + webmodules);
        return webmodules;
    }
    
    public static boolean isCmpField(JavaClass jc, String fieldName) {
        if ((jc != null) && (fieldName != null)) {
            Collection emodules = getRelevantEjbModules(jc);
            Iterator e = emodules.iterator();
            while (e.hasNext()) {
                org.netbeans.modules.j2ee.api.ejbjar.EjbJar ejbJar =
                        (org.netbeans.modules.j2ee.api.ejbjar.EjbJar)e.next();
                Entity[] eBeans = getEntityBeans(ejbJar);
                if (eBeans != null) {
                    for (int i=0; i < eBeans.length; i++) {
                        Entity ent = eBeans[i];
                        String ejbClass = ent.getEjbClass();
                        if (jc.getName().equals(ejbClass)) {
                            CmpField[] cmpFields = ent.getCmpField();
                            if (cmpFields != null) {
                                for (int c=0; c<cmpFields.length; c++) {
                                    CmpField cmpField = cmpFields[c];
                                    // TODO: guessing of field name passed to this method should be enhanced
                                    // (e.g. when first letter of field is upper-case), just equals shoud be used
                                    if (fieldName.equalsIgnoreCase(cmpField.getFieldName())) {
                                        return true;
                                    }
                                }
                            }
                            return false;
                        }
                    }
                }
            }
        }
        return false;
    }
    
    public static boolean isCmrField(JavaClass jc, String fieldName) {
        if ((jc != null) && (fieldName != null)) {
            Collection emodules = Utility.getRelevantEjbModules(jc);
            Iterator e = emodules.iterator();
            while (e.hasNext()) {
                org.netbeans.modules.j2ee.api.ejbjar.EjbJar ejbJar =
                        (org.netbeans.modules.j2ee.api.ejbjar.EjbJar)e.next();
                EjbRelation[] ejbRelations = getRelationships(ejbJar);
                if (ejbRelations != null) {
                    String ejbName = getEntityEjbNameForClass(jc);
                    for (int i=0; i < ejbRelations.length; i++) {
                        EjbRelation relation = ejbRelations[i];
                        if (relation != null) {
                            EjbRelationshipRole role = relation.getEjbRelationshipRole();
                            if (role != null) {
                                RelationshipRoleSource relSource = role.getRelationshipRoleSource();
                                String roleEjbName = relSource.getEjbName();
                                if (roleEjbName.equals(ejbName)) {
                                    CmrField cmrField = role.getCmrField();
                                    if (cmrField != null) {
                                        String cmrFieldName = cmrField.getCmrFieldName();
                                        // TODO: guessing of field name passed to this method should be enhanced
                                        // (e.g. when first letter of field is upper-case), just equals shoud be used
                                        if (cmrFieldName.equalsIgnoreCase(fieldName)) {
                                            return true;
                                        }
                                    }
                                }
                            }
                            EjbRelationshipRole role2 = relation.getEjbRelationshipRole2();
                            if (role2 != null) {
                                RelationshipRoleSource relSource = role2.getRelationshipRoleSource();
                                String roleEjbName = relSource.getEjbName();
                                if (roleEjbName.equals(ejbName)) {
                                    CmrField cmrField = role2.getCmrField();
                                    if (cmrField != null) {
                                        String cmrFieldName = cmrField.getCmrFieldName();
                                        // TODO: guessing of field name passed to this method should be enhanced
                                        // (e.g. when first letter of field is upper-case), just equals shoud be used
                                        if (cmrFieldName.equalsIgnoreCase(fieldName)) {
                                            return true;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        return false;
    }
    
    /* works for entity beans only to be quicker */
    public static String getEntityEjbNameForClass(JavaClass jc) {
        if (jc == null) {
            return null;
        }
        Resource res = jc.getResource();
        if (res != null) {
            FileObject fo = JavaModel.getFileObject(res);
            if (fo != null) {
                Project prj = FileOwnerQuery.getOwner(fo);
                if (prj != null) {
                    org.netbeans.modules.j2ee.api.ejbjar.EjbJar emod =
                            org.netbeans.modules.j2ee.api.ejbjar.EjbJar.getEjbJar(prj.getProjectDirectory());
                    Entity[] entityBeans = getEntityBeans(emod);
                    if (entityBeans != null) {
                        for (int e=0; e<entityBeans.length; e++) {
                            Entity entity = entityBeans[e];
                            String ejbClass = entity.getEjbClass();
                            err.log("ejbclass: " + ejbClass);
                            if (ejbClass.equals(jc.getName())) {
                                return entity.getEjbName();
                            }
                        }
                    }
                }
            }
        }
        
        return null;
    }
    
    /**
     * Creates full class name from package name and simple class name
     * @param pkg package name
     * @param simpleName simple class name
     * @return full class name
     */
    public static String getClassName(String pkg, final String simpleName) {
        return (pkg == null || pkg.length() == 0 ? "" : pkg + ".") + simpleName;
    }
    
    public static Ejb getEjb(EjbJar ejbJar, ClassDefinition classDefinition) {
        if (classDefinition == null) {
            return null;
        }
        EnterpriseBeans enterpriseBeans = ejbJar.getEnterpriseBeans();
        if (enterpriseBeans == null) {
            return null;
        }
        String className = classDefinition.getName();
        Ejb[] ejbs = enterpriseBeans.getEjbs();
        for (int i = 0; i < ejbs.length; i++) {
            Ejb ejb = ejbs[i];
            if (className.equals(ejb.getEjbClass())) {
                return ejb;
            }
            if (ejb instanceof EntityAndSession) {
                EntityAndSession entityAndSession = ((EntityAndSession) ejb);
                if (isSubTypeOf(entityAndSession.getLocal(), classDefinition) ||
                        isSubTypeOf(entityAndSession.getLocalHome(), classDefinition) ||
                        isSubTypeOf(entityAndSession.getRemote(), classDefinition) ||
                        isSubTypeOf(entityAndSession.getHome(), classDefinition)) {
                    return ejb;
                }
            }
        }
        return null;
    }
    
    private static boolean isSubTypeOf(String thisClassName, ClassDefinition classDefinition) {
        return thisClassName != null && resolveClass(thisClassName).isSubTypeOf(classDefinition);
    }
    
    public static String getFieldName(Method method) {
        final String methodName = method.getName();
        if (methodName.length() > 3 && (methodName.startsWith(PREFIX_SET) || methodName.startsWith(PREFIX_GET))) {
            final char c = methodName.charAt(3);
            if (Character.isUpperCase(c)) {
                return Character.toLowerCase(c) + methodName.substring(4);
            }
        }
        return null;
    }
    
    private static String getGetterName(String fieldName) {
        return PREFIX_GET + firstToUpperCase(fieldName);
    }
    
    
    private static String getSetterName(String fieldName) {
        return PREFIX_SET + firstToUpperCase(fieldName);
    }
    
    public static Method getGetterMethod(JavaClass javaClass, String fieldName) {
        if (javaClass == null) {
            return null;
        }
        return javaClass.getMethod(getGetterName(fieldName), Collections.EMPTY_LIST, true);
    }
    
    private static Method getSetterMethod(JavaClass javaClass, String fieldName, Type type) {
        if (javaClass == null) {
            return null;
        }
        return javaClass.getMethod(getSetterName(fieldName), Arrays.asList(new Type[]{type}), true);
    }
    
    public static Collection getAccessMethods(Entity entity, String fieldName) {
        Collection accessMethods = new LinkedList();
        JavaClass javaClass = resolveClass(entity.getEjbClass());
        Method method = getGetterMethod(javaClass, fieldName);
        if (method != null) {
            accessMethods.add(method);
            Type type = method.getType();
            addToCollection(accessMethods, getSetterMethod(javaClass, fieldName, type));
            
            addAccessMethods(accessMethods, resolveClass(entity.getLocal()), fieldName, type);
            addAccessMethods(accessMethods, resolveClass(entity.getRemote()), fieldName, type);
        }
        return accessMethods;
    }
    
    private static void addAccessMethods(Collection accessMethods, JavaClass javaClass, String fieldName, Type type) {
        if (javaClass != null) {
            addToCollection(accessMethods, getGetterMethod(javaClass, fieldName));
            addToCollection(accessMethods, getSetterMethod(javaClass, fieldName, type));
        }
    }
    
    public static JavaClass resolveClass(String className) {
        return (JavaClass) resolveType(className);
    }
    
    private static Type resolveType(String typeName) {
        Type type = JavaModel.getDefaultExtent().getType().resolve(typeName);
        if (type instanceof UnresolvedClass) {
            Type basicType = JavaModel.getDefaultExtent().getType().resolve("java.lang." + typeName);  // NOI18N;
            if (!(basicType instanceof UnresolvedClass)) {
                return basicType;
            }
        }
        return type;
    }
    
    
    
    public static void addToCollection(Collection collection, Object object) {
        if (object != null) {
            collection.add(object);
        }
    }
    
    public static org.netbeans.modules.j2ee.api.ejbjar.EjbJar getApiEjbJar(Element element) {
        FileObject fileObject = JavaModel.getFileObject(element.getResource());
        if (fileObject != null) {
            return org.netbeans.modules.j2ee.api.ejbjar.EjbJar.getEjbJar(fileObject);
        }
        return null;
    }
    
    public static FileObject getEjbJarFileObject(Element element) {
        FileObject fileObject = JavaModel.getFileObject(element.getResource());
        if (fileObject != null) {
            Project project = FileOwnerQuery.getOwner(fileObject);
            if (project != null) {
                final org.netbeans.modules.j2ee.api.ejbjar.EjbJar ejbJar =
                        org.netbeans.modules.j2ee.api.ejbjar.EjbJar.getEjbJar(project.getProjectDirectory());
                if (ejbJar != null) {
                    return ejbJar.getDeploymentDescriptor();
                }
            }
        }
        return null;
    }
    
    public static EjbJar getEjbJar(FileObject fo) {
        if (fo != null) {
            try {
                return DDProvider.getDefault().getMergedDDRoot(org.netbeans.modules.j2ee.api.ejbjar.EjbJar.getEjbJar(fo).getMetadataUnit());
            } catch (IOException ioe) {
                //ignore
            }
        }
        return null;
    }
    
    public static Method findMethod(ClassDefinition classDefinition, Method method,
            boolean includeSuperTypes) {
        return getMethod(classDefinition, method.getName(), method.getParameters(), includeSuperTypes);
    }
    
    public static Method getMethod(ClassDefinition classDefinition, String methodName, List parameters,
            boolean includeSuperTypes) {
        if (classDefinition == null || methodName == null) {
            return null;
        }
        return classDefinition.getMethod(methodName, getTypedParams(parameters), includeSuperTypes);
    }
    
    private static List getQueryMethodParams(Query query) {
        String[] methodParam = query.getQueryMethod().getMethodParams().getMethodParam();
        List params = new LinkedList();
        for (int i = 0; i < methodParam.length; i++) {
            params.add(resolveType(methodParam[i]));
        }
        return params;
    }
    
    private static void addMethodToCollection(String className, Collection queryMethods, String methodName,
            List params) {
        JavaClass javaClass = resolveClass(className);
        addToCollection(queryMethods, getMethod(javaClass, methodName, params, false));
    }
    
    public static Collection getQueryMethods(Entity entity, Query query) {
        Collection queryMethods = new LinkedList();
        final List params = getQueryMethodParams(query);
        final String methodName = query.getQueryMethod().getMethodName();
        if (methodName.startsWith(PREFIX_EJBSELECT)) {
            addMethodToCollection(entity.getEjbClass(), queryMethods, methodName, params);
        } else if (methodName.startsWith(PREFIX_FIND)) {
            addMethodToCollection(entity.getLocalHome(), queryMethods, methodName, params);
            addMethodToCollection(entity.getHome(), queryMethods, methodName, params);
        }
        return queryMethods;
    }
    
    public static JavaClass resolveRealClass(String className) {
        JavaClass javaClass = resolveClass(className);
        return javaClass instanceof UnresolvedClass ? null : javaClass;
    }
    
    public static boolean isSubTypeOf(ClassDefinition classDefinition, String className) {
        if (classDefinition == null) {
            return false;
        }
        JavaClass javaClass = resolveClass(className);
        if (javaClass instanceof UnresolvedClass) {
            for (Iterator it = classDefinition.getInterfaces().iterator(); it.hasNext();) {
                if (className.equals(((JavaClass) it.next()).getName())) {
                    return true;
                }
            }
            return false;
        } else {
            return classDefinition.isSubTypeOf(javaClass);
        }
    }
    
    public static boolean isFirstUpperCase(String s) {
        return s.length() != 0 && Character.isUpperCase(s.charAt(0));
    }
    
    public static boolean isFirstLowerCase(String s) {
        return s.length() != 0 && Character.isLowerCase(s.charAt(0));
    }
    
    public static String firstToUpperCase(String s) {
        return s.length() == 0 ? s : Character.toUpperCase(s.charAt(0)) + s.substring(1);
    }
    
    public static String firstToLowerCase(String s) {
        return s.length() == 0 ? s : Character.toLowerCase(s.charAt(0)) + s.substring(1);
    }
    
    /**
     * Constructs new name for given class.
     * @param originalFullyQualifiedName old fully qualified name of the class.
     * @param newName new unqualified name of the class.
     * @return new fully qualified name of the class.
     */
    public static String renameClass(String originalFullyQualifiedName, String newName){
        if (isEmpty(originalFullyQualifiedName) || isEmpty(newName)){
            throw new IllegalArgumentException("Old and new name of the class must be given.");
        }
        int lastDot = originalFullyQualifiedName.lastIndexOf('.');
        return (lastDot <= 0) ? newName : originalFullyQualifiedName.substring(0, lastDot + 1) + newName;
    }
    
    /**
     * @param qualifiedName fully qualified name of a class
     * @return unqualified name
     */
    public static String unqualify(String qualifiedName){
        int lastDot = qualifiedName.lastIndexOf(".");
        return (lastDot < 0 ) ? qualifiedName : qualifiedName.substring(lastDot + 1);
        
    }
    
    /**
     * @return true if given str is null or empty.
     */
    public static boolean isEmpty(String str){
        return str == null || "".equals(str.trim());
    }
    
    /**
     * @return property name resolved from given <code>getter</code>, i.e.
     * if given arg was <code>getProperty</code>, this method will return
     * <code>property</code>. Given getter must follow JavaBeans naming
     * converntions for this method to work correctly.
     */
    public static String getPropertyName(String getter){
        if (!getter.startsWith("get")){
            return getter;
        }
        String sub = getter.substring(3);
        char c = sub.charAt(0);
        if (!Character.isUpperCase(c)){
            return getter;
        }
        return Character.toLowerCase(c) + sub.substring(1);
    }
    
    /**
     * @return true if given <code>project</code> is an EJB 3.0 project, false
     * otherwise.
     */
    public static boolean isEjb30(Project project){
        if (project == null){
            return false;
        }
        try {
            org.netbeans.modules.j2ee.api.ejbjar.EjbJar[] modules = org.netbeans.modules.j2ee.api.ejbjar.EjbJar.getEjbJars(project);
            if (modules != null && modules.length > 0){
                return org.netbeans.modules.j2ee.dd.api.ejb.EjbJar.VERSION_3_0.equals(DDProvider.getDefault().getMergedDDRoot(modules[0].getMetadataUnit()).getVersion().toString());
            }
        } catch (IOException e) {
            ErrorManager.getDefault().notify(e);
        }
        return false;
    }
    
    /**
     * @return true if given <code>refObject</code> is an instance of
     * Element and its project is a EJB 3.0 project, false otherwise.
     */
    public static boolean isEjb30(RefObject refObject){
        if (!(refObject instanceof Element)){
            return false;
        }
        Element element = (Element) refObject;
        FileObject fo = JavaModel.getFileObject(element.getResource());
        return isEjb30(FileOwnerQuery.getOwner(fo));
    }
    
    /**
     * @param ejbModules collection of <code>EjbJar</code>s.
     * @return true if any of the given modules has an ejb-jar.xml file.
     */
    public static boolean hasEjbJar(Collection ejbModules){
        for (Object elem : ejbModules) {
            org.netbeans.modules.j2ee.api.ejbjar.EjbJar em = (org.netbeans.modules.j2ee.api.ejbjar.EjbJar)elem;
            FileObject ejbJarFO = em.getDeploymentDescriptor();
            if (ejbJarFO != null){
                return true;
            }
        }
        return false;
    }
    
    /**
     * @param refObject
     * @return true if any of the relevant ajb modules of the given refObject
     * has an ejb-jar.xml file.
     */
    public static boolean hasEjbJar(RefObject refObject){
        return hasEjbJar(getRelevantEjbModules(refObject));
    }
}
