/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.refactoring.jaxwssupport;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AttributeValue;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.modules.j2ee.refactoring.rename.AbstractRenameRefactoringElement;
import org.netbeans.modules.j2ee.refactoring.rename.JaxWsXmlRenameRefactoring;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.ExternalChange;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.websvc.api.jaxws.project.config.Endpoint;
import org.netbeans.modules.websvc.api.jaxws.project.config.Endpoints;
import org.netbeans.modules.websvc.api.jaxws.project.config.EndpointsProvider;
import org.netbeans.modules.websvc.api.jaxws.project.config.JaxWsModel;
import org.netbeans.modules.websvc.api.jaxws.project.config.JaxWsModelProvider;
import org.netbeans.modules.websvc.api.jaxws.project.config.Service;
import org.netbeans.modules.websvc.jaxws.api.JAXWSSupport;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.util.NbBundle;

/**
 * Base class for jax-ws.xml and sun-jaxws.xml refactorings.
 *
 * @author Erno Mononen
 */
public abstract class JaxWsXmlRefactoringSupport {
    
    /**
     * Name of an annotation that represents web service.
     */
    protected static final String WS_ANNOTATION = "WebService";
    /**
     * Name of an annotation element that represents wsdl location.
     */
    protected static final String WSDL_LOCATION_ELEMENT = "wsdlLocation";
    
    /**
     * Creates a new instance of JaxWsRefactoringSupport
     */
    protected JaxWsXmlRefactoringSupport() {
    }
    
    /**
     * @return the model representing jax-ws.xml from the project of given
     * <code>javaClass</code> or null if was not found.
     */
    protected JaxWsModel getModel(JavaClass javaClass){
        JaxWsModelProvider provider = JaxWsModelProvider.getDefault();
        
        FileObject fo = JavaModel.getFileObject(javaClass.getResource());
        Project project = FileOwnerQuery.getOwner(fo);
        
        FileObject jaxWsXml = project.getProjectDirectory().getFileObject("nbproject/jax-ws.xml");
        // no jax-ws.xml file in the project
        if (jaxWsXml == null){
            return null;
        }
        
        JaxWsModel model = null;
        try {
            model = provider.getJaxWsModel(jaxWsXml);
        } catch (IOException ex){
            ErrorManager.getDefault().notify(ex);
        }
        
        return model;
    }
    
    /**
     * @return true if the given javaClass represents a web service from wsdl.
     */
    protected boolean isWebSvcFromWsdl(JavaClass javaClass){
        for (Object elem : javaClass.getAnnotations()) {
            Annotation ann = (Annotation) elem;
            if (ann.getTypeName() == null){
                continue;
            }
            if (ann.getTypeName().getName().equals(WS_ANNOTATION)){
                for (Object elem2 : ann.getAttributeValues()) {
                    AttributeValue value = (AttributeValue) elem2;
                    if (value.getName().equals(WSDL_LOCATION_ELEMENT)){
                        return true;
                    }
                }
            }
        }
        return false;
    }
    
    /**
     * @return Endpoints whose implementation class is equal to given <code>javaClass</code>, 
     * never null.
     */
    protected List<Endpoint> getEndpoints(JavaClass javaClass){
        Endpoints endpoints = getEndpointsModel(javaClass);
        
        if (endpoints == null){
            return Collections.emptyList();
        }
        
        List<Endpoint> result = new ArrayList<Endpoint>();
        for (Endpoint each : endpoints.getEndpoints()) {
            if (javaClass.getName().equals(each.getImplementation())){
                result.add(each);
            }
        }
        return result;
    }
    
    /**
     * @return the model representing <code>sun-jaxws.xml</code> in the 
     * project of given <code>javaClass</code> or null.
     */
    protected Endpoints getEndpointsModel(JavaClass javaClass){
        FileObject sunjaxwsFile = getSunJaxWsXmlFile(javaClass);
        if (sunjaxwsFile == null){
            return null;
        }

        Endpoints endpoints = null;
        try {
            endpoints = EndpointsProvider.getDefault().getEndpoints(sunjaxwsFile);
        } catch (IOException ex) {
            ErrorManager.getDefault().notify(ex);
        }
        
        return endpoints;
    }
    
    /**
     * @return the file object representing <code>sun-jaxws.xml</code> in the 
     * project of given <code>javaClass</code> or null.
     */
    protected FileObject getSunJaxWsXmlFile(JavaClass javaClass){
        FileObject javaClassFO = JavaModel.getFileObject(javaClass.getResource());
        JAXWSSupport jaxwssupport = JAXWSSupport.getJAXWSSupport(javaClassFO);
        FileObject folder = jaxwssupport.getDeploymentDescriptorFolder();
        return folder.getFileObject("sun-jaxws.xml");
    }
}
