/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.refactoring.safedelete;

import java.text.MessageFormat;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.modules.j2ee.persistence.dd.persistence.model_1_0.PersistenceUnit;
import org.netbeans.modules.j2ee.persistence.provider.ProviderUtil;
import org.netbeans.modules.j2ee.persistence.unit.PUDataObject;
import org.netbeans.modules.j2ee.refactoring.DefaultPositionBoundsResolver;
import org.netbeans.modules.j2ee.refactoring.J2EERefactoring;
import org.netbeans.modules.j2ee.refactoring.PersistenceRefactoringUtil;
import org.netbeans.modules.j2ee.refactoring.Utility;
import org.netbeans.modules.javacore.internalapi.ExternalChange;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.api.SafeDeleteRefactoring;
import org.netbeans.modules.refactoring.spi.RefactoringElementImplementation;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;

/**
 * Safe delelete refactoring for entity classes listed
 * in persistence.xml.
 *
 * @author Erno Mononen
 */
public class PersistenceXmlSafeDeleteRefactoring implements J2EERefactoring {
    
    private Element element;
    private SafeDeleteRefactoring safeDeleteRefactoring;
    
    /**
     * Creates a new instance of PersistenceXmlSafeDeleteRefactoring
     */
    public PersistenceXmlSafeDeleteRefactoring(Element element, SafeDeleteRefactoring safeDeleteRefactoring) {
        this.element = element;
        this.safeDeleteRefactoring = safeDeleteRefactoring;
    }
    
    public Problem preCheck() {
        return null;
    }
    
    public Problem checkParameters() {
        return null;
    }
    
    public Problem fastCheckParameters() {
        return null;
    }
    
    public Problem prepare(RefactoringElementsBag refactoringElementsBag) {
        
        Problem problem = null;
        
        for (Object elem : element.getResource().getClassifiers()) {
            JavaClass jClass = (JavaClass) elem;
            
            for (FileObject ddFile : PersistenceRefactoringUtil.getPersistence(jClass)) {
                PUDataObject dataObject = ProviderUtil.getPUDataObject(ddFile);
                
                if (!ProviderUtil.isValid(dataObject)){
                    Problem newProblem =
                            new Problem(false, NbBundle.getMessage(PersistenceXmlSafeDeleteRefactoring.class, "TXT_PersistenceXmlInvalidProblem", ddFile.getPath()));
                    problem = Utility.addProblemsToEnd(problem, newProblem);
                    continue;
                }
                
                
                PersistenceUnit[] persistenceUnits = ProviderUtil.getPersistenceUnits(dataObject);
                
                for (int i = 0; i < persistenceUnits.length; i++) {
                    PersistenceUnit unit = persistenceUnits[i];
                    
                    for (int j = 0; j < unit.getClass2().length; j++) {
                        
                        if (unit.getClass2()[j].equals(jClass.getName())){
                            
                            RefactoringElementImplementation refactoringElem =
                                    new PersistenceXmlSafeDeleteRefactoringElement(unit, jClass.getName(), dataObject, ddFile);
                            
                            refactoringElementsBag.add(safeDeleteRefactoring, refactoringElem);
                        }
                        
                    }
                }
                
            }
        }
        return problem;
    }
    
    /**
     * Safely delete element for persistence.xml
     */
    private static class PersistenceXmlSafeDeleteRefactoringElement extends AbstractSafeDeleteRefactoringElement implements ExternalChange {
        
        private PersistenceUnit persistenceUnit;
        private PUDataObject puDataObject;
        private String clazz;
        
        /** Creates a new instance of PersistenceXmlSafeDeleteRefactoringElement */
        public PersistenceXmlSafeDeleteRefactoringElement(PersistenceUnit persistenceUnit, String clazz, PUDataObject puDataObject, FileObject parentFile) {
            this.parentFile = parentFile;
            this.clazz = clazz;
            this.persistenceUnit = persistenceUnit;
            this.puDataObject = puDataObject;
        }
        
        /**
         * Returns text describing the refactoring formatted for display (using HTML tags).
         * @return Formatted text.
         */
        public String getDisplayText() {
            Object[] args = new Object [] {parentFile.getNameExt(), clazz};
            return MessageFormat.format(NbBundle.getMessage(PersistenceXmlSafeDeleteRefactoringElement.class, "TXT_PersistenceXmlSafeDeleteClass"), clazz);
        }
        
        public void performExternalChange() {
            ProviderUtil.removeManagedClass(persistenceUnit, clazz, puDataObject);
        }
        
        public void undoExternalChange() {
            ProviderUtil.addManagedClass(persistenceUnit, clazz, puDataObject);
        }
        
        /** Performs the change represented by this refactoring element.
         */
        public void performChange() {
            JavaMetamodel.getManager().registerExtChange(this);
        }
        
        public PositionBounds getPosition() {
            try {
                return new DefaultPositionBoundsResolver(DataObject.find(parentFile), clazz).getPositionBounds();
            } catch (DataObjectNotFoundException ex) {
                ErrorManager.getDefault().notify(ex);
            }
            return null;
        }
        
        
    }
    
    
}
