/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.test.j2ee.refactoring;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.lang.reflect.Modifier;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.StringTokenizer;
import javax.jmi.reflect.InvalidObjectException;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.TypeClass;
import org.netbeans.junit.diff.LineDiff;
import org.netbeans.junit.ide.ProjectSupport;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.api.RefactoringElement;
import org.netbeans.modules.refactoring.api.RefactoringSession;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;


/** LogTestCase
 * @author Jan Becicka
 */
public class RefactoringTestCase extends LogTestCase {
    
    protected static TypeClass typeProxy;
    protected static JavaClass jc;
    
    public RefactoringTestCase(java.lang.String testName) {
        super(testName);
    }
    
    protected void setUp() {
        super.setUp();
        setJavaClass("java.lang.String");
    }
    
    protected void setJavaClass(String name) {
        Utility.beginTrans(true);
        try {
            jc = Utility.findClass(name);
            typeProxy = ((JavaModelPackage) jc.refOutermostPackage()).getType();
        } finally {
            Utility.endTrans();
        }
    }
    
    /**
     * Stores problems into log file.
     */
    public void logProblems(Problem problem) {
        Problem p=problem;
        if (p != null) {
            ArrayList list=new ArrayList();
            while (p != null) {
                if (p.isFatal()) {
                    list.add("Problem fatal: "+p.getMessage());
                } else {
                    list.add("Problem: "+p.getMessage());
                }
                p=p.getNext();
            }
            Collections.sort(list);
            for (int i=0;i < list.size();i++) {
                log(list.get(i));
            }
        }
    }
    
    /**
     * Stores problems into ref file. Problems should be sorted.
     * @return true if problem is not null and one of them is fatal
     */
    public boolean refProblems(Problem problem) {
        Problem p=problem;
        boolean ret=false;
        if (p != null) {
            ArrayList list=new ArrayList();
            while (p != null) {
                if (p.isFatal()) {
                    ret=true;
                    list.add("Problem fatal: "+p.getMessage());
                } else {
                    list.add("Problem: "+p.getMessage());
                }
                p=p.getNext();
            }
            Collections.sort(list);
            for (int i=0;i < list.size();i++) {
                ref(list.get(i));
            }
        }
        return ret;
    }
    
    /**
     * Makes diffs of files in map (before refactoring) with refactored files.
     * @param files map of resource to temporary file
     * @param session refactoring session
     * @param name original name
     * @param renamed new name
     */
    protected void compareResources(HashMap files, RefactoringSession session, String name, String renamed) throws Exception {
        Collection result = session.getRefactoringElements();
        ref("Found "+String.valueOf(result.size())+" occurrence(s).");
        Object[] ress=files.keySet().toArray();
        Arrays.sort(ress);
        for (int i=0;i < ress.length;i++) {
            String key = (String) ress[i];
            if (key.equals("empty")) {
                continue;
            }
            Res res=(Res)(files.get(key));
            File oldfile=res.backup;
            if (oldfile == null || !oldfile.exists()) { //resource has new name
                log("compareResources: backuped old file is null - changed resource ");
                continue;
            }
            File actualfile;
            boolean b = false;
            if (key.endsWith(".xml")) {
                actualfile = new File(key);
                b = true;  
            } else if (res.resource != null) {
                try {
                    actualfile = new File(classPathWorkDir, res.resource.getName());
                } catch (InvalidObjectException ioe) {
                    actualfile = new File(classPathWorkDir, "xyz.abc");
                }
            } else {
                actualfile = new File(classPathWorkDir, key);
            }
            if (!actualfile.exists()) { //moved
                if (key.indexOf(name.replace('.','/')) == 0) { //package name
                    String rsnm=key; // a/b/c/A.java
                    String nm=name.replace('.','/');  // a/b/c
                    rsnm=renamed.replace('.','/')+rsnm.substring(nm.length()); // b/b/c/A.java
                    actualfile=new File(classPathWorkDir, rsnm);
                } else if (key.indexOf("-config.xml") > 0) { //need special handling for ws-config.xml file
                    // absolute/path/to/ws/config.xml
                    String rsnm=renamed.replace('.','/').substring(0, renamed.lastIndexOf('.'))
                        + key.substring(key.lastIndexOf(File.separatorChar));
                    actualfile=new File(classPathWorkDir, rsnm);
                }
            }
            LineDiff ldiff=new LineDiff();
            File diff=File.createTempFile("xtest", "refactoring");
            ldiff.diff(actualfile, oldfile, diff);
            if (b) {
                ref("\n"+key.substring(
                        classPathWorkDir.getParentFile().getParentFile()
                        .getAbsolutePath().length() + 1).replace('\\', '/') +"\n");
            } else {
                ref("\n"+key +"\n");
            }
            Collections.sort(res.texts);
            for (int j=0;j < res.texts.size();j++) {
                String msg = (String) res.texts.get(j);
                String substr = "/qa-functional/data/projects/";
                if (msg.indexOf(substr) < 0) {
                    //common case
                    ref("      " + msg);
                } else {
                    //cannot use absolute paths in goldenfiles nor logs
                    StringTokenizer tok = new StringTokenizer(msg, " ");
                    StringBuffer sb = new StringBuffer();
                    while (tok.hasMoreTokens()) {
                        String t = tok.nextToken();
                        if (t.indexOf(substr) > 0) {
                            boolean isWindows = File.pathSeparator.equals(";");
                            int index = classPathWorkDir.getAbsolutePath().length();
                            if (!isWindows) {
                                index--;
                            }
                            sb.append(t.substring(index).replace('/', '.'));
                            sb.append(" ");
                        } else {
                            sb.append(t);
                            sb.append(" ");
                        }
                    }
                    ref("      " + sb.toString().trim());
                }
            }
            ref("\nFile diff:\n");
            ref(diff);
            diff.delete();
            oldfile.delete();
            Utility.copyFile(actualfile, new File(getWorkDir(), actualfile.getName()));
        }
        ref("");
        Res empty=(Res)(files.get("empty"));
        if (empty != null) {
            for (int i=0;i < empty.texts.size();i++) {
                ref(empty.texts.get(i));
            }
        }
    }
    
    /**
     * makes map of files mapped to resources names of usages, files are backuped into tmp files
     */
    protected HashMap getResources(RefactoringSession session) throws Exception {
        Collection result = session.getRefactoringElements();
        ArrayList list=new ArrayList();
        ArrayList xmlFiles = new ArrayList();
        HashMap files=new HashMap();
        for (Iterator it=result.iterator();it.hasNext();) {
            Object o=it.next();
            //we're interested in RefactoringElements
            if (o instanceof RefactoringElement) {
                RefactoringElement wue=(RefactoringElement) o;
                Element el = wue.getJavaElement();
                if (el != null) {
                    //we have java file
                    Resource resource = el.getResource();
                    if (resource != null && resource.isValid()) {
                        Res res;
                        if (!list.contains(resource)) {
                            list.add(resource);
                            res=new Res();
                            files.put(resource.getName(), res);
                        } else {
                            res=(Res)(files.get(resource.getName()));
                        }
                        res.resource = resource;
                        res.texts.add(getDisplayText(wue));
                    }
                } else {
                    //we have DD (or other non-java file)
                    FileObject fo = wue.getParentFile();
                    File f = FileUtil.toFile(fo);
                    Res res;
                    if (!xmlFiles.contains(f.getAbsolutePath())) {
                        xmlFiles.add(f.getAbsolutePath());
                        res=new Res();
                        files.put(f.getAbsolutePath(), res);
                    } else {
                        res=(Res)(files.get(f.getAbsolutePath()));
                    }
                    res.texts.add(getDisplayText(wue));
                }
            } // if (o instanceof RefactoringElement)
        }
        files.put("empty", new Res());
        Collections.sort(list, new Comparator() {
            public int compare(Object o1, Object o2) {
                return ((Resource)o1).getName().compareTo(((Resource)o2).getName());
            }
        });
        Collections.sort(xmlFiles);
        File folder=getWorkDir();
        for (int i=0;i < list.size();i++) {
            Resource r=(Resource)(list.get(i));
            Res res=(Res)(files.get(r.getName()));
            String name=r.getName().replace('/', '_')+".test";
            name=name.replace('\\', '_');
            File fl=new File(folder, name);
            if (!fl.getParentFile().exists()) {
                fl.getParentFile().mkdirs();
            }
            PrintStream ps=new PrintStream(new FileOutputStream(fl));
            ps.print(r.getSourceText());
            ps.close();
            res.backup=fl;
        }
        for (int i=0;i < xmlFiles.size();i++) {
            String s = (String) xmlFiles.get(i);
            File f = new File(s);
            Res res=(Res)(files.get(s));
            String name=s.substring(s.lastIndexOf(File.separatorChar) + 1).replace('/', '_')+".test";
            name=name.replace('\\', '_');
            File fl=new File(folder, name);
            if (!fl.getParentFile().exists()) {
                fl.getParentFile().mkdirs();
            }
            try {
                copyFile(f, fl);
            } catch (Exception e) {
                e.printStackTrace(System.err);
            }
            res.backup=fl;
        }
        return files;
    }
    
    protected void refUsages(RefactoringSession session) {
        Collection result = session.getRefactoringElements();
        ArrayList list;
        HashMap map=new HashMap();
        for (Iterator it=result.iterator();it.hasNext();) {
            Object o=it.next();
            if (o instanceof RefactoringElement) {
                RefactoringElement wue=(RefactoringElement) o;
                Element el = wue.getJavaElement();
                String s = null;
                if (el != null && el.getResource() != null) {
                    s=el.getResource().getName().replace(File.separatorChar,'/');
                } else {
                    FileObject fo = wue.getParentFile();
                    if (fo != null) {
                        String tmp = FileUtil.toFile(fo).getAbsolutePath();
                        int i = classPathWorkDir.getParentFile().getParentFile()
                        .getAbsolutePath().length();
                        s = tmp.substring(i + 1).replace(File.separatorChar, '/');
                    }
                }
                if (s != null) {
                    list=(ArrayList)(map.get(s));
                    if (list == null) {
                        list=new ArrayList();
                        map.put(s, list);
                    }
                    list.add(getDisplayText(wue));
                } else {
                    log("refUsages without resource");
                    log(getDisplayText(wue));
                    map.put(getDisplayText(wue), "");
                }
            }
        }
        ref("Found "+String.valueOf(result.size())+" occurrence(s).");
        Object[] keys=map.keySet().toArray();
        Arrays.sort(keys);
        for (int i=0;i < keys.length;i++) {
            ref("");
            if (map.get(keys[i]) instanceof ArrayList) {
                ref(keys[i]);
                list=(ArrayList)(map.get(keys[i]));
                Collections.sort(list);
                for (int j=0;j < list.size();j++) {
                    ref("      "+list.get(j));
                }
            } else {
                ref(keys[i]);
            }
        }
        ref("");
    }
    
    protected String getDisplayText(RefactoringElement elm) {
        String app="";
        if (elm.getStatus() == RefactoringElement.WARNING) {
            app=" [ warning! ]";
        } else if (elm.getStatus() == RefactoringElement.GUARDED) {
            app=" [ error: code is in guarded block ]";
        }
        return elm.getDisplayText()+app;
    }
    
    protected String getModifier(int modifier) {
        String ret="";
        if (Modifier.isPublic(modifier)) {
            ret+="public";
        } else if (Modifier.isProtected(modifier)) {
            ret+="protected";
        } else if (Modifier.isPrivate(modifier)) {
            ret+="private";
        }
        if (Modifier.isAbstract(modifier)) {
            ret+=" abstract";
        } else if (Modifier.isFinal(modifier)) {
            ret+=" final";
        } else if (Modifier.isInterface(modifier)) {
            ret+=" interface";
        } else if (Modifier.isNative(modifier)) {
            ret+=" native";
        } else if (Modifier.isStatic(modifier)) {
            ret+=" static";
        } else if (Modifier.isSynchronized(modifier)) {
            ret+=" synchronized";
        } else if (Modifier.isTransient(modifier)) {
            ret+=" transient";
        } else if (Modifier.isVolatile(modifier)) {
            ret+=" volatile";
        }
        return ret.trim();
    }
    
    public void prepareProject() {
        File projectDir = new File(getDataDir(), "projects/EJBModule");
        ProjectSupport.openProject(projectDir);
        classPathWorkDir= new File(projectDir, "src/java");
    }
    
    
    private static void copyFile(File src, File dest) {
        try {
            dest.createNewFile();
            FileChannel srcChannel = new FileInputStream(src).getChannel();
            FileChannel dstChannel = new FileOutputStream(dest).getChannel();
            dstChannel.transferFrom(srcChannel, 0, srcChannel.size());
            srcChannel.close();
            dstChannel.close();
        } catch (IOException e) {
            //ignore
            e.printStackTrace(System.out);
        }
    }
    
    
    
    class Res {
        public File backup;
        public ArrayList texts;
        public Resource resource = null;
        
        public Res() {
            texts=new ArrayList();
        }
        
        public String toString() {
            return "\nRes[backup: " + backup + ", texts: " + texts + "]\n\n";
        }
    }
}

