/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.test.j2ee.wizard;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import junit.framework.Test;
import junit.framework.TestSuite;
import org.netbeans.jellytools.EditorWindowOperator;
import org.netbeans.jellytools.JellyTestCase;
import org.netbeans.jellytools.NewProjectNameLocationStepOperator;
import org.netbeans.jellytools.NewProjectWizardOperator;
import org.netbeans.jellytools.ProjectsTabOperator;
import org.netbeans.jellytools.nodes.Node;
import org.netbeans.jemmy.EventTool;
import org.netbeans.jemmy.operators.JComboBoxOperator;
import org.netbeans.junit.NbTestCase;
import org.netbeans.junit.NbTestSuite;
import org.netbeans.test.j2ee.lib.Reporter;
import org.netbeans.test.j2ee.lib.RequiredFiles;
import org.netbeans.test.j2ee.lib.J2eeProjectSupport;
import org.netbeans.test.j2ee.lib.Utils;

/**
 * Test New File wizards in J2EE area. These tests are
 * part of J2EE Functional test suite. Each test checks
 * if all files were created (deployment descriptors,
 * directories for sources etc.) and if project node
 * is expanded after finishing New Project wizard.
 *
 * @author jungi
 * @see <a href="http://qa.netbeans.org/modules/j2ee/promo-f/testspec/j2ee-wizards-testspec.html">J2EE Wizards Test Specification</a>
 */
public class NewProjectWizardsTest extends JellyTestCase {
    
    private static final int EJB = 0;
    private static final int WEB = 1;
    private static final int WEB_JAKARTA = 2;
    private static final int J2EE_DEFAULT = 3;
    private static final int APP_CLIENT_DEFAULT = 4;
    
    private String projectLocation;
    private String projectName;
    private String version;
    private Reporter reporter;

    public NewProjectWizardsTest(String testName) {
        this(testName, "1.4");
    }
    
    public NewProjectWizardsTest(String testName, String version) {
        super(testName);
        this.version = version;
    }
    
    public void setUp() throws Exception {
        reporter = Reporter.getReporter((NbTestCase)this);
    }
    
    public void tearDown() throws Exception {
        reporter.close();
    }
    
    public static void main(java.lang.String[] args) {
        junit.textui.TestRunner.run(suite());
    }
    
    public static Test suite() {
        TestSuite suite = new NbTestSuite(NewProjectWizardsTest.class);
        return suite;
    }

    /**
     * Create EJB Module which name contains spaces
     * and default project settings.
     */
    public void testDefaultNewEJBModWizard() throws Exception {
        Utils.openOutputTab();
        projectLocation = getWorkDir().getAbsolutePath();
        projectName = "def EJB Mod" + version;
        NewProjectWizardOperator wiz
                = WizardUtils.createNewProject("Enterprise", "EJB Module");
        NewProjectNameLocationStepOperator op
                = WizardUtils.setProjectNameLocation(projectName,
                projectLocation);
        WizardUtils.setJ2eeSpecVersion(op, WizardUtils.MODULE_EJB, version);
        wiz.finish();
        checkProjectStructure(EJB);
        checkProjectNodes();
    }
    
    /**
     * Create EJB Module with default project settings.
     */
    public void testNewEJBModWizard() throws Exception {
        projectLocation = getWorkDir().getAbsolutePath();
        projectName = "BadModule" + version;
        NewProjectWizardOperator wiz
                = WizardUtils.createNewProject("Enterprise", "EJB Module");
        NewProjectNameLocationStepOperator op
                = WizardUtils.setProjectNameLocation(projectName,
                projectLocation);
        WizardUtils.setJ2eeSpecVersion(op, WizardUtils.MODULE_EJB, version);
        wiz.finish();
        checkProjectStructure(EJB);
        checkProjectNodes();
    }
    
    
    /**
     * Create Enterprise Application Client project with default project
     * settings.
     */
    public void testDefaultAppClientWizard() throws Exception {
        projectLocation = getWorkDir().getAbsolutePath();
        projectName = "App client" + version;
        NewProjectWizardOperator wiz
                = WizardUtils.createNewProject("Enterprise", "Enterprise Application Client");
        NewProjectNameLocationStepOperator op
                = WizardUtils.setProjectNameLocation(projectName,
                projectLocation);
        WizardUtils.setJ2eeSpecVersion(op, WizardUtils.MODULE_CAR, version);
        wiz.finish();
        checkProjectStructure(APP_CLIENT_DEFAULT);
        checkProjectNodes();
    }
    
    /**
     * Create Web Application which name contains spaces
     * and default project settings.
     */
    public void testDefaultNewWebModWizard() throws Exception {
        projectLocation = getWorkDir().getAbsolutePath();
        projectName = "def Web app" + version;
        NewProjectWizardOperator wiz
                = WizardUtils.createNewProject("Web", "Web Application");
        NewProjectNameLocationStepOperator op
                = WizardUtils.setProjectNameLocation(projectName,
                projectLocation);
        WizardUtils.setJ2eeSpecVersion(op, WizardUtils.MODULE_WAR, version);
        wiz.finish();
        Thread.sleep(1000);
        checkProjectStructure(WEB);
        checkProjectNodes();
    }
    
    /**
     * Create Web Application with Jakarta source structure.
     */
    public void testDefaultNewJakartaWebModWizard() throws Exception {
        projectLocation = getWorkDir().getAbsolutePath();
        projectName = "Jakarta Web App" + version;
        NewProjectWizardOperator wiz
                = WizardUtils.createNewProject("Web", "Web Application");
        NewProjectNameLocationStepOperator nlsop
                = WizardUtils.setProjectNameLocation(projectName,
                projectLocation);
        JComboBoxOperator cbop = new JComboBoxOperator(nlsop, "BluePrints");
        cbop.setSelectedIndex(1); //jakarta source structure
        WizardUtils.setJ2eeSpecVersion(nlsop, WizardUtils.MODULE_WAR, version);
        wiz.finish();
        Thread.sleep(1000);
        checkProjectStructure(WEB_JAKARTA);
        checkProjectNodes();
    }
    
    /**
     * Create Enterprise Application project with default project
     * settings (ejb and web module are as well ).
     */
    public void testDefaultNewJ2eeAppWizard() throws Exception {
        projectLocation = getWorkDir().getAbsolutePath();
        projectName = "def EAR app" + version;
        NewProjectWizardOperator wiz
                = WizardUtils.createNewProject("Enterprise", "Enterprise Application");
        NewProjectNameLocationStepOperator op
                = WizardUtils.setProjectNameLocation(projectName,
                projectLocation);
        WizardUtils.setJ2eeSpecVersion(op, WizardUtils.MODULE_EAR, version);
        wiz.finish();
        checkProjectStructure(J2EE_DEFAULT);
        Node root = checkProjectNodes();
        Node modules = new Node(root, "Java EE Modules");
        modules.expand();
        String[] s = modules.getChildren();
        assertEquals("Expected: \"def_EAR_app" + version + "-ejb.jar\", was: \"" + s[1]
                + "\"", "def_EAR_app" + version + "-ejb.jar", s[1]);
        assertEquals("Expected: \"def_EAR_app" + version + "-war.war\", was: \"" + s[0]
                + "\"", "def_EAR_app" + version + "-war.war", s[0]);
    }

    private void checkProjectStructure(int prjType) {
        RequiredFiles r = null;
        switch (prjType) {
            case EJB:
                r = readRF("structures/ejbProject.str");
                break;
            case WEB:
                r = readRF("structures/webProject.str");
                break;
            case WEB_JAKARTA:
                r = readRF("structures/webProjectJakarta.str");
                break;
            case J2EE_DEFAULT:
                r = readRF("structures/defEAR.str");
                break;
            case APP_CLIENT_DEFAULT:
               r = readRF("structures/carProject.str");
               break;
            default:
                throw new IllegalArgumentException();
        }
        try {
            Thread.currentThread().sleep(5000);
        } catch (InterruptedException ex) {
            //do nothing
        }
        Set/*<String>*/ l = J2eeProjectSupport.getFileSet(projectLocation + File.separatorChar + projectName);
        Set/*<String>*/ rf = r.getRequiredFiles();
        reporter.ref("Project: " + projectLocation);
        reporter.ref("Expected: " + rf);
        reporter.ref("Real: " + l);
        if ((EJB == prjType) && ("5".equals(version))) {
            Set result = getDifference(l, rf);
            if (!result.remove("src" + File.separator + "conf" + File.separator + "ejb-jar.xml")) {
                fail("Files: " + result + " are missing in project at: " + projectLocation);
            }
        } else {
            assertTrue("Files: " + getDifference(l, rf) + " are missing in project at: " + projectLocation , l.containsAll(rf));
        }
        rf = r.getRequiredFiles();
        reporter.ref("Project: " + projectLocation);
        reporter.ref("Expected: " + rf);
        reporter.ref("Real: " + l);
        Set s = getDifference(rf, l);
        assertTrue("Files: " + s + " are new in project: " + projectLocation , s.isEmpty());
    }
    
    public void closeProjects() {
        ProjectsTabOperator pto = new ProjectsTabOperator();
        pto.getProjectRootNode("def EJB Mod").performPopupAction("Close Project");
        pto.getProjectRootNode("BadModule").performPopupAction("Close Project");
        pto.getProjectRootNode("def EAR app").performPopupAction("Close Project");
        pto.getProjectRootNode("def Web app").performPopupAction("Close Project");
//        pto.getProjectRootNode("def App client").performPopupAction("Close Project");
        //pto.getProjectRootNode("Jakarta Web App").performPopupAction("Close Project");
        new EventTool().waitNoEvent(2500);
    }
    
    private RequiredFiles readRF(String fileName) {
        RequiredFiles rf = null;
        try {
            rf = new RequiredFiles(new File(getDataDir(), fileName));
        } catch (IOException ioe) {
            ioe.printStackTrace(reporter.getLogStream());
        }
        assertNotNull(rf);
        return rf;
    }
    
    private Set getDifference(Set s1, Set s2) {
        Set result = new HashSet();
        s2.removeAll(s1);
        for (Iterator i = s2.iterator(); i.hasNext();) {
            String s = (String) i.next();
            if (s.indexOf(".LCK") < 0) {
                result.add(s);
            } else {
                reporter.log("Additional file: " + s);
            }
        }
        return result;
    }
    
    private Node checkProjectNodes() {
        Node node = new ProjectsTabOperator().getProjectRootNode(projectName);
        try {
            Thread.sleep(1500);
        } catch (InterruptedException ie) {
            //do nothing
        }
        assertTrue("Project " + projectName + " is not expanded", node.isExpanded());
        return node;
    }
}
