/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.common.ui.nodes.elements;

import java.beans.FeatureDescriptor;
import java.beans.PropertyEditor;
import java.lang.ref.Reference;
import java.lang.ref.SoftReference;
import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;
import java.util.List;
import javax.jmi.reflect.JmiException;
import org.netbeans.jmi.javamodel.CallableFeature;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.MultipartId;
import org.netbeans.jmi.javamodel.Parameter;
import org.netbeans.modules.j2ee.common.ui.nodes.editors.IdentifierArrayEditor;
import org.netbeans.modules.j2ee.common.ui.nodes.editors.MethodParameterArrayEditor;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.openide.ErrorManager;
import org.openide.nodes.Node;
import org.openide.nodes.PropertySupport;
import org.openide.util.NbBundle;


/**
 * This is light copy of org.netbeans.modules.java.ui.nodes.elements.ElementNode
 *
 * @see org.netbeans.modules.java.ui.nodes.elements.ElementNode
 * @author Martin Adamek
 */
public class ElementNode {
    
    /** 
     * Name of {@link ConstructorElement#getParameters parameters} property  for {@link MethodElement methods} 
     * and {@link ConstructorElement constructors}.
     */
    public static final String PROP_PARAMETERS = "parameters"; // NOI18N

    /** 
     * Name of {@link ConstructorElement#getExceptions exceptions} property for {@link ConstructorElement constructors} 
     * and {@link MethodElement methods}.
     */
    public static final String PROP_EXCEPTIONS = "exceptions"; // NOI18N

    static String getString(String key) {
        return NbBundle.getMessage(ElementNode.class, key);
    }

    /** @see #getModel */
    public static void setModel(org.netbeans.jmi.javamodel.Element el, FeatureDescriptor fd) {
        JavaModelPackage model = JavaMetamodel.getManager().getJavaExtent(el);
        fd.setValue("JavaModelPackage", model); // NOI18N
    }

    /** extracts model from descriptor or provides default one if custom not exists*/
    public static JavaModelPackage getModel(FeatureDescriptor fd) {
        JavaModelPackage model = (JavaModelPackage) fd.getValue("JavaModelPackage"); // NOI18N
        if (model == null) {
            model = JavaMetamodel.getManager().getDefaultExtent();
            ErrorManager.getDefault().notify(
                    ErrorManager.INFORMATIONAL,
                    new IllegalStateException("missing JavaModelPackage")); // NOI18N
        }
        return model;
    }

    /** creates a node property for constructor or method parameters.
     * @param element element owning parameters
     * @param canW <code>false</code> to force property to be read-only
     * @return the property
     */
    public static Node.Property createParametersProperty(CallableFeature element, boolean canW) {
        Node.Property prop = new ParametersProperty(element, canW);
        setModel(element, prop);
        return prop;
    }

    /** creates a node property for constructor or method exceptions.
     * @param element element owning exceptions
     * @param canW <code>false</code> to force property to be read-only
     * @return the property
     */
    public static Node.Property createExceptionsProperty(CallableFeature element, boolean canW) {
        Node.Property prop = new ExceptionsProperty(element, canW);
        setModel(element, prop);
        return prop;
    }

    private final static class ParametersProperty extends ElementProp {
        
        private final CallableFeature element;
        
        private ParametersProperty(CallableFeature element, boolean canW) {
            super(PROP_PARAMETERS, Parameter[].class, canW);
            this.element = element;
        }

        public Object getValue() {
            List l = element.getParameters();
            return (Parameter[]) l.toArray(new Parameter[l.size()]);
        }

        public PropertyEditor createPropertyEditor() {
            return new MethodParameterArrayEditor(canWrite());
        }

        public void setValue(final Object val) throws IllegalArgumentException,
                       IllegalAccessException, InvocationTargetException {
            
            super.setValue(val);
            if (!(val instanceof Parameter[]))
                throw new IllegalArgumentException();
            
            boolean fail = true;
            try {
                JavaMetamodel.getDefaultRepository().beginTrans(true);
                try {
                    List l = element.getParameters();
                    l.clear();
                    l.addAll(Arrays.asList((Parameter[]) val));
                    fail = false;
                } finally {
                    JavaMetamodel.getDefaultRepository().endTrans(fail);
                }
            } catch (JmiException e) {
                IllegalArgumentException iae = new IllegalArgumentException();
                iae.initCause(e);
                throw iae;
            }
        }
        
    }

    private final static class ExceptionsProperty extends ElementProp {
        
        private final CallableFeature element;
        
        public ExceptionsProperty(CallableFeature element, boolean canW) {
            super(PROP_EXCEPTIONS, MultipartId[].class, canW);
            this.element = element;
        }

        protected PropertyEditor createPropertyEditor() {
            return new IdentifierArrayEditor();
        }

        public Object getValue () {
            Object ret = null;
            try {
                JavaMetamodel.getDefaultRepository().beginTrans(false);
                try {
                    List l = element.getExceptionNames();
                    ret = l.toArray(new MultipartId[l.size()]);
                } finally {
                    JavaMetamodel.getDefaultRepository().endTrans();
                }
            } catch (JmiException e) {
                ErrorManager.getDefault().notify(e);
            }
            return ret;
        }

        public void setValue(final Object val) throws IllegalArgumentException,
                IllegalAccessException, InvocationTargetException {
            super.setValue(val);
            if (!(val instanceof MultipartId[]))
                throw new IllegalArgumentException();
                       
            boolean fail = true;
            try {
                JavaMetamodel.getDefaultRepository().beginTrans(true);
                try {
                    List exs = element.getExceptionNames();
                    exs.clear();
                    exs.addAll(Arrays.asList((MultipartId[]) val));
                    fail = false;
                } finally {
                    JavaMetamodel.getDefaultRepository().endTrans(fail);
                }
            } catch (JmiException e) {
                IllegalArgumentException iae = new IllegalArgumentException();
                iae.initCause(e);
                throw iae;
            }
        }
    }

    // ================== Property support for element nodes =================

    /** Property support for element nodes properties.
    */
    static abstract class ElementProp extends PropertySupport {
        
        /** caches a reference to the property editor */
        private Reference editor = null;
        
        /** Constructs a new ElementProp - support for properties of
        * element hierarchy nodes.
        *
        * @param name The name of the property
        * @param type The class type of the property
        * @param canW The canWrite flag of the property
        */
        public ElementProp(String name, java.lang.Class type, boolean canW) {
            super(name, type,
                  getString("PROP_" + name), // NOI18N
                  getString("HINT_" + name), // NOI18N
                  true, canW);
        }

        /** Setter for the value. This implementation only tests
        * if the setting is possible.
        *
        * @param val the value of the property
        * @exception java.lang.IllegalAccessException when this ElementProp was constructed
        *            like read-only.
        */
        public void setValue (Object val) throws IllegalArgumentException,
                IllegalAccessException, InvocationTargetException {
            if (!canWrite())
                throw new IllegalAccessException(getString("MSG_Cannot_Write")); // NOI18N
        }
            
        public final PropertyEditor getPropertyEditor() {
            PropertyEditor pe; 
            if (editor == null || (pe = (PropertyEditor) editor.get()) == null) {
                pe = createPropertyEditor();
                editor = new SoftReference(pe);
            }
            return pe;
        }
        
        /**
         * override just to provide own property editor.
         * @return own property editor
         */ 
        protected PropertyEditor createPropertyEditor() {
            return super.getPropertyEditor();
        }

    }
    
}
