/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.j2ee.verification;

import org.netbeans.jmi.javamodel.*;

/**
 * A rules engine knows about rules. It applies them on appropriate nodes. It is
 * implemented as a visitor. Typically an instance of this class is passed to a
 * {@link TreeWalker} along with a {@link JavaClass}. As the TreeWalker
 * traverses all the nodes in tree, it calls appropriate visit method on the
 * RulesEngine. In visit methods, RulesEngine applies rules specific to a node
 * type. Typical use case looks like this:
 * <p/>
 * <p> <blockquote><pre>
 *     RulesEngine aRulesEngine;
 *     JavaClass javaClass;
 *     new TreeWalker(aRulesEngine)).visitClass(javaClass);
 * </pre></blockquote>
 *
 * @author Sanjeeb.Sahoo@Sun.COM
 */
public class RulesEngine extends EmptyVisitor {

    /**
     * This is a callback interface.
     */
    public static interface ErrorHandler {

        /**
         * This method is called every time rules engine comes across a
         * problem.
         *
         * @param problem representing violation of a rule.
         * @param rule    that was violated.
         */
        void onError(Problem problem, Rule rule);
    }

    private final RulesRepository rulesRepository;

    private ErrorHandler handler;

    private ProblemFinderContext ctx;

    public RulesEngine(
            ErrorHandler handler,
            ProblemFinderContext ctx,
            RulesRepository rulesRepository) {
        this.handler = handler;
        this.ctx = ctx;
        this.rulesRepository = rulesRepository;
    }

    public void setHandler(ErrorHandler handler) {
        this.handler = handler;
    }

    public void setCtx(ProblemFinderContext ctx) {
        this.ctx = ctx;
    }

    /**
     * Applies all rules applicable to this node. It only applies rules to this
     * node and does not apply rules to the children of this node.
     *
     * @param clazz node which will be tested for any rule violation.
     * @see #applyRule(Rule, Element, ProblemFinderContext)}
     */
    @Override public void visitClass(JavaClass clazz) {
        for (Rule rule : rulesRepository.getClazzRules()) {
            applyRule(rule, clazz, ctx);
        }
    }

    /**
     * Applies all rules applicable to this node. It only applies rules to this
     * node and does not apply rules to the children of this node.
     *
     * @param constr node which will be tested for any rule violation.
     * @see #applyRule(Rule, Element, ProblemFinderContext)}
     */
    @Override public void visitConstructor(Constructor constr) {
        for (Rule rule : rulesRepository.getConstructorRules()) {
            applyRule(rule, constr, ctx);
        }
    }

    /**
     * Applies all rules applicable to this node. It only applies rules to this
     * node and does not apply rules to the children of this node.
     *
     * @param method node which will be tested for any rule violation.
     * @see #applyRule(Rule, Element, ProblemFinderContext)}
     */
    @Override public void visitMethod(Method method) {
        for (Rule rule : rulesRepository.getMethodRules()) {
            applyRule(rule, method, ctx);
        }
    }

    /**
     * Applies all rules applicable to this node. It only applies rules to this
     * node and does not apply rules to the children of this node.
     *
     * @param field node which will be tested for any rule violation.
     * @see #applyRule(Rule, Element, ProblemFinderContext)}
     */
    @Override public void visitField(Field field) {
        for (Rule rule : rulesRepository.getFieldRules()) {
            applyRule(rule, field, ctx);
        }
    }

    @Override public void visitAnnotation(Annotation annotation) {
        for (Rule rule : rulesRepository.getAnnotationRules()) {
            applyRule(rule, annotation, ctx);
        }
    }

    /**
     * Applies a single rule on a given element (or a node). If the node
     * violates the rule, then it creates an instance of {@link Problem} and
     * calls {@link ErrorHandler#onError(Problem, Rule)}
     *
     * @param rule    Rule that is being applied
     * @param element node to which this rule will be applied.
     * @param context carries any additional info that may be used by while
     *                applying the rule
     */
    private void applyRule(
            Rule rule,
            Element element,
            ProblemFinderContext context) {
        if (Boolean.getBoolean("DEBUG_RULES_ENGINE")) {
            String elementDetails = (element instanceof NamedElement ?
                    NamedElement.class.cast(element).getName() :
                    element.toString());
            System.err.println("RulesEngine.applyRule: " +
                    "rule = [" + rule.getId() + "] " +
                    "element = [" + elementDetails + "]");
        }
        rule.reset();
        Problem problem = rule.apply(element, context);
        if (problem != null) {
            if (Boolean.getBoolean("DEBUG_RULES_ENGINE")) {
                System.err.println("RulesEngine.applyRule: " +
                        "rule = ["+rule.getId()+ "] VIOLATED. " +
                        "Error message: " + problem.getMessage());
            }
            handler.onError(problem, rule);
        }
    }

}
