/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.j2ee.verification;

import org.netbeans.jmi.javamodel.*;
import org.openide.util.NbBundle;

import java.util.List;

/**
 * This class supplies the traversal strategy over a javamodel.
 * This is itself a {@link Visitor}! Other visitors make use of it.
 * How this works?
 * An instance of an actual visitor is passed to this class. We call the
 * visitor as {@link #piggyBackedVisitor}. When appropriate visit method is
 * called on TreeWalker, TreeWalker traverses down (i.e. descends) all the nodes
 * in tree. As it traverses the nodes, it calls appropriate visit method on the
 * piggyBackedVisitor. Code snippet below shows how to use this class
 * to traverse an object graph:
 *
 * <p> <blockquote><pre>
 *     Visitor aVisitor;
 *     JavaClass javaClass;
 *     new TreeWalker(someVisitor)).visitClass(javaClass);
 * </pre></blockquote>
 *
 * @author Sanjeeb.Sahoo@Sun.COM
 */
public class TreeWalker implements Visitor {

    // TODO: Allow support for node which reprsents an extent

    /**
     *  the visitor which supplies actual code for visit method
     */
    private final Visitor piggyBackedVisitor;

    /**
     * Create a new TreeWalker.
     *
     * @param piggyBackedVisitor whose visit method will be called
     * @throws IllegalArgumentException if piggyBackedVisitor is itself a
     * TreeWalker
     */
    public TreeWalker(Visitor piggyBackedVisitor) {
        if(piggyBackedVisitor instanceof TreeWalker) {
            throw new IllegalArgumentException(
                    NbBundle.getMessage(this.getClass(),
                            "MSG_TreeWalkerIllegalArgument", // NOI18N
                            piggyBackedVisitor));
        }
        this.piggyBackedVisitor = piggyBackedVisitor;
    }

    public void visitClass(JavaClass javaClass) {
//        System.err.println("Visiting " + javaClass.getName());
        piggyBackedVisitor.visitClass(javaClass);

        // traverse annotations at class level
        for(Annotation annotation : (List<Annotation>)javaClass.getAnnotations()) {
            this.visitAnnotation(annotation);
        }

        // now traverse all children
        Element classElements[] =
                (Element[]) javaClass.getContents().toArray(new Element[]{});

        for (Element e : classElements) {
            if (e instanceof JavaClass) {
                // See http://www.netbeans.org/issues/show_bug.cgi?id=74127
                // So, we are not recursing here.
                // Rules for inner classes are by JEEVerificationAnnotationProvider
                // where it iterates over all the classes defined in a resource.
                // this.visitClass(JavaClass.class.cast(e)); // recursion
            } else if (e instanceof Constructor) {
                this.visitConstructor(Constructor.class.cast(e));
            } else if (e instanceof Method) {
                this.visitMethod(Method.class.cast(e));
            } else if (e instanceof Field) {
                this.visitField(Field.class.cast(e));
            } else if (e instanceof Annotation) {
                this.visitAnnotation(Annotation.class.cast(e));
            } else {
                // we don't care for the moment
            }
        }
    }

    public void visitConstructor(Constructor constr) {
        piggyBackedVisitor.visitConstructor(constr);
        // for the moment, we don't care about traversing parameters,
        // inner classes etc. defined inside a constructor.

        // traverse annotations attached to this element
        for(Annotation annotation : (List<Annotation>)constr.getAnnotations()) {
            this.visitAnnotation(annotation);
        }
    }

    public void visitMethod(Method method) {
        piggyBackedVisitor.visitMethod(method);

        // traverse annotations attached to this element
        for(Annotation annotation : (List<Annotation>)method.getAnnotations()) {
            this.visitAnnotation(annotation);
        }

        // for the moment, we don't care about traversing parameters,
        // inner classes etc. defined inside a method.
    }

    public void visitField(Field field) {
        piggyBackedVisitor.visitField(field);

        // traverse annotations attached to this element
        for(Annotation annotation : (List<Annotation>)field.getAnnotations()) {
            this.visitAnnotation(annotation);
        }
    }

    public void visitAnnotation(Annotation annotation) {
        piggyBackedVisitor.visitAnnotation(annotation);
    }

}
