/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.j2ee.verification.persistence;

import static org.netbeans.modules.j2ee.metadata.Constants.*;
import static org.netbeans.modules.j2ee.verification.persistence.BeanAccessType.*;
import org.netbeans.modules.j2ee.persistence.dd.PersistenceUtils;
import static org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIAnnotations.*;
import static org.netbeans.modules.j2ee.verification.ProblemFindingUtils.*;
import org.netbeans.jmi.javamodel.*;
import static org.netbeans.modules.j2ee.verification.JEEVerificationAnnotationProvider.*;
import org.netbeans.modules.j2ee.persistence.dd.orm.model_1_0.Entity;
import org.netbeans.modules.j2ee.persistence.dd.orm.model_1_0.EntityMappings;
import org.netbeans.modules.j2ee.persistence.dd.orm.model_1_0.IdClass;
import org.netbeans.modules.j2ee.persistence.dd.orm.model_1_0.MappedSuperclass;

import java.util.*;
import org.netbeans.modules.javacore.api.JavaModel;
import org.openide.filesystems.FileObject;
/**
 * @author Sanjeeb.Sahoo@Sun.COM
 */
public class PersistenceAPIHelper {
    
    //TODO: Move helper method about getter/setter to Introspector
    
    private static final Set fieldOrPropertyJSR220Annotations = new TreeSet(Arrays.asList(new String[]{
        "javax.persistence.JoinColumn", "javax.persistence.JoinColumns", //NOI18N
        "javax.persistence.AttributeOverride", "javax.persistence.AttributeOverrides", //NOI18N
        PersistenceAPIAnnotations.EMBEDDED_ID,
        "javax.persistence.Transient", //NOI18N
        "javax.persistence.Version",  //NOI18N
        "javax.persistence.Basic", //NOI18N
        "javax.persistence.Lob", //NOI18N
        "javax.persistence.ManyToOne", //NOI18N
        "javax.persistence.OneToOne", //NOI18N
        "javax.persistence.OneToMany", //NOI18N
        "javax.persistence.JoinTable", //NOI18N
        "javax.persistence.ManyToMany", //NOI18N
        "javax.persistence.MapKey", //NOI18N
        "javax.persistence.OrderBy", //NOI18N
        "javax.persistence.PrimaryKeyJoinColumn", //NOI18N
        "javax.persistence.Embedded", //NOI18N
        "javax.persistence.SequenceGenerator", //NOI18N
        "javax.persistence.TableGenerator", //NOI18N
        "javax.persistence.Column", //NOI18N
        PersistenceAPIAnnotations.ID,
        PersistenceAPIAnnotations.TEMPORAL,
        PersistenceAPIAnnotations.ENUMERATED,
        "javax.persistence.GeneratedValue" //NOI18N
    }));

    public static boolean isJSR220FieldOrPropertyAccessAnnotation(Annotation a){
        AnnotationType type = a.getType();
        
        if (type == null){
            return false; // Unresolved annotation class
        }
        
        return fieldOrPropertyJSR220Annotations.contains(type.getName());
    }

    public static BeanAccessType findBeanAccessType(JavaClass javaClass){
        BeanAccessType accessType = UNDEFINED;
        if(javaClass.getName().equals(OBJECT)) {
            return accessType; // optimization: java.lang.Object can't be an entity
        }
        Element classElements[] = (Element[])javaClass.getContents().toArray(new Element[]{});
        
        for (int i = 0; i < classElements.length; i++) {
            if (classElements[i] instanceof Method || classElements[i] instanceof Field){
                AnnotableElement element = (AnnotableElement)classElements[i];

                Annotation a = findFirstAnnotationFromGivenSet(element, fieldOrPropertyJSR220Annotations);
                
                if (a != null){
                    boolean fieldAccess = element instanceof Field;
                    
                    if (accessType == UNDEFINED){ // first annotated element
                        accessType = fieldAccess ? FIELD : PROPERTY;
                        
                    } else{ // accessType is already determined, checking consistency
                        if (accessType == FIELD && !fieldAccess
                                || accessType == PROPERTY && fieldAccess){
                            // inconsistency detected
                            accessType = INCONSISTENT;
                            break; // no point to search any further
                        }
                    }
                }
            }
        }
        
        tmpDbg(javaClass.getName() + " has " + accessType + " access type.");
        
        return accessType;
    }
    
    /**
     * This method returns the entity or the mapped superclass
     * that is the root of the entity hierarchy.
     * @param javaClass is the javaClass whose hierarchy will be examined.
     * @return the persistence root class. returns null if there is no
     * entity or mapped super class in the hierarchy.
     */
    public static JavaClass findPersistenceRootClass(JavaClass javaClass) {
        JavaClass persistenceRoot = null;
        for(JavaClass nextClass = javaClass;
        nextClass != null;
        nextClass = nextClass.getSuperClass()) {
            if(isEntity(nextClass) || isMappedSuperclass(nextClass)) {
                persistenceRoot = nextClass;
            }
        }
        tmpDbg("Persistence root for " + javaClass.getName() + " is " +
                persistenceRoot.getName());
        return persistenceRoot;
    }

    /**
     * This method returns the entity superclass
     * that is the root of the entity hierarchy.
     * @param javaClass is the javaClass whose hierarchy will be examined.
     * @return the entity root class. returns null if there is no
     * entity super class in the hierarchy.
     */
    public static JavaClass findEntityRootClass(JavaClass javaClass) {
        JavaClass entityRoot = null;
        for(JavaClass nextClass = javaClass;
        nextClass != null;
        nextClass = nextClass.getSuperClass()) {
            if(isEntity(nextClass)) {
                entityRoot = nextClass;
            }
        }
        tmpDbg("Entity root for " + javaClass.getName() + " is " +
                (entityRoot == null ? null : entityRoot.getName()));
        return entityRoot;
    }

    /**
     * Utility method that returns the first field or method in this class
     * that is annotated as @Id or @EmbeddedId.
     * @param javaClass
     * @return
     */
    private static ClassMember findFirstIdTypeMember(JavaClass javaClass) {
        Set idAnnotations = new TreeSet(Arrays.asList(ID, EMBEDDED_ID));
        for(ClassMember cm : (List<ClassMember>)javaClass.getContents()) {
            if(cm instanceof Field || cm instanceof Method) {
                if(findFirstAnnotationFromGivenSet(cm, idAnnotations) != null) {
                    return cm;
                }
            }
        }
        return null;
    }

    /**
     * This method returns the access type for the entity class hierarchy.
     * The access type of hierarchy is decided based on the access type of
     * Id/EMbeddedId field or property only.
     *
     * @param javaClass JavaClass whose hierarchy will be examined.
     * @return the persistence root class' access type. It returns
     * {@link UNDEFINED} if there is no persistence class in hierarchy.
     */
    public static BeanAccessType findAccessTypeOfHierarchy(
            JavaClass javaClass) {
        BeanAccessType beanAccessTypeOfHierarchy = UNDEFINED;
        if(javaClass.getName().equals(OBJECT)) { // optimization when there is no hierarchy
            final ClassMember firstIdTypeMember = findFirstIdTypeMember(javaClass);
            if(firstIdTypeMember==null) {
                beanAccessTypeOfHierarchy = UNDEFINED; 
            } else {
                beanAccessTypeOfHierarchy =
                        firstIdTypeMember instanceof Field ? FIELD : PROPERTY;
            }
        } else {
            // since entity subclass can not have ID, let's start with entity root if it exists.
            JavaClass startingPoint = findEntityRootClass(javaClass);
            if (startingPoint == null) {
                startingPoint = javaClass;
            }
            for (JavaClass nextClass = startingPoint;
                 !nextClass.getName().equals(OBJECT);
                 nextClass = nextClass.getSuperClass()) {
                final ClassMember firstIdTypeMemberInSuperClass =
                        findFirstIdTypeMember(nextClass);
                if(firstIdTypeMemberInSuperClass!=null) {
                    beanAccessTypeOfHierarchy =
                            firstIdTypeMemberInSuperClass instanceof Field ?
                            FIELD : PROPERTY;
                }
            }
        }
        tmpDbg("Access type for entity hierarchy for " + javaClass.getName() +
                " is " + beanAccessTypeOfHierarchy);
        return beanAccessTypeOfHierarchy;
    }

    /**
     * Utility method to find access type of the given IdClass.
     * Access type of an IdClass is determined by the access type of
     * the entity that uses the class. So we go over all the entities
     * that use this class as IdClass and then arrive at the access type
     * of this class. When an IdClass is used in different entities having
     * conflicting access type, this method returns
     * {@link BeanAccessType.INCONSISTENT}
     * @param javaClass JavaClass representing an IdClass.
     * @return access type for this IdClass.
     */
    public static BeanAccessType findIdClassAccessType(JavaClass javaClass) {
        EntityMappings mappings = PersistenceUtils.getEntityMappings(JavaModel.getFileObject(javaClass.getResource()));
        BeanAccessType result = UNDEFINED;
        if (mappings != null) {
            final String idClassName = javaClass.getName();
            for (Entity entity : mappings.getEntity()) {
                final IdClass idClassOfEntity = entity.getIdClass();
                if (idClassOfEntity != null &&
                        idClassName.equals(idClassOfEntity.getClass2())) {
                    // found an entity that uses this class as IdClass
                    // now check what is the access type of the entity

                    String entityAccessType = entity.getAccess();
                    if ("FIELD".equals(entityAccessType)) { // NOI18N
                        if (result == UNDEFINED) {
                            result = FIELD;
                            continue;
                        } else if (result == PROPERTY) {
                            result = INCONSISTENT;
                            break; // no need to search further
                        }
                    } else if ("PROPERTY".equals(entityAccessType)) { // NOI18N
                        if (result == UNDEFINED) {
                            result = PROPERTY;
                            continue;
                        } else if (result == FIELD) {
                            result = INCONSISTENT;
                            break; // no need to search further
                        }
                    }
                }
            }
        }
        tmpDbg("Id Class " + javaClass.getName() + " has " + result + " access type.");
        return result;
    }


    /**
     * Utility method to find out if a class is an id class.
     * This is costly operation as it has to iterate over all the entities
     * to find out if this class is used as id class in any of them.
     *
     * @param javaClass
     * @return
     */
    public static boolean isIdClass(JavaClass javaClass) {
        FileObject fo = JavaModel.getFileObject(javaClass.getResource());
        
        if (fo == null){
            return false; // the class has just been deleted, nevermind
        }
        
        EntityMappings mappings = PersistenceUtils.getEntityMappings(fo);
        if (mappings == null){
            return false;
        }

        final String name = javaClass.getName();
        for(Entity e : mappings.getEntity()) {
            final IdClass idClass = e.getIdClass();
            if (idClass != null &&
                    name.equals(idClass.getClass2())) {
                return true;
            }
        }
        // now look in MappedSuperclass...
        for(MappedSuperclass e : mappings.getMappedSuperclass()) {
            final IdClass idClass = e.getIdClass();
            if (idClass != null &&
                    name.equals(idClass.getClass2())) {
                return true;
            }
        }

        return false;
    }

    public static boolean isEmbeddable(JavaClass javaClass) {
        return findAnnotation(javaClass, EMBEDDABLE) != null;
    }
    
    public static boolean isMappedSuperclass(JavaClass javaClass) {
        return findAnnotation(javaClass, MAPPED_SUPERCLASS) != null;
    }
    
    public static boolean isEntity(JavaClass javaClass) {
        return findAnnotation(javaClass, ENTITY) != null;
        // B'cos of a bug in ORM model, we are not using it at this point.
//        EntityMappings mappings = PersistenceUtils.getEntityMappings(JavaModel.getFileObject(javaClass.getResource()));
//
//        if (mappings == null){
//            return false;
//        }
//
//        Entity entity = PersistenceUtils.getEntity(javaClass, mappings);
//        return entity != null;
    }

    public static String getAccessorName(String fieldName){
        String fieldNameStartingFromUpperCase = Character.toUpperCase(fieldName.charAt(0)) + fieldName.substring(1);
        return "get" + fieldNameStartingFromUpperCase; //NOI18N
    }
    
    public static String getMutatorName(String fieldName){
        String fieldNameStartingFromUpperCase = Character.toUpperCase(fieldName.charAt(0)) + fieldName.substring(1);
        return "set" + fieldNameStartingFromUpperCase; //NOI18N
    }
    
    public static String getFieldNameFromAccessorName(String accessorName){
        String pref = "get"; //NOI18N
        int prefLen = pref.length();
        
        if (!accessorName.startsWith(pref) || accessorName.length() <= prefLen){
            return null;
        }
        
        return Character.toLowerCase(accessorName.charAt(prefLen)) + accessorName.substring(prefLen + 1);
    }
    
    public static String getFieldNameFromMutatorName(String accessorName){
        String pref = "set"; //NOI18N
        int prefLen = pref.length();
        
        if (!accessorName.startsWith(pref) || accessorName.length() <= prefLen){
            return null;
        }
        
        return Character.toLowerCase(accessorName.charAt(prefLen)) + accessorName.substring(prefLen + 1);
    }
    
    public static Method findFieldMutator(JavaClass javaClass, Field field) {
        List args = Collections.singletonList(field.getType());
        return javaClass.getMethod(getMutatorName(field.getName()), args, false);
    }
    
    public static Method findFieldAccessor(JavaClass javaClass, Field field){
        Method r = javaClass.getMethod(getAccessorName(field.getName()), Collections.EMPTY_LIST, false);
        /* TODO: handle boolean fields ('is' and 'get' accessor prefixes allowed)
        if (r == null && field.getType().getName().equals("boolean")){ //NOI18N
            String getterName = getAccessorName
            r =
        }
         */
        return r;
    }
    
    public static Entity getEntity(ClassMember m){
        
        if (m != null){
            EntityMappings mappings = getEntityMappings(m);
            
            if (mappings != null){
                // this cast is safe as getEntityMappings() didn't return null
                JavaClass javaClass = (JavaClass) m.getDeclaringClass();
                return PersistenceUtils.getEntity(javaClass, mappings);
            }
        }
        return null;
    }
    
    public static EntityMappings getEntityMappings(ClassMember m){
        if (m != null){
            ClassDefinition classDef = m.getDeclaringClass();
            
            if (classDef instanceof JavaClass){ // ignore anonymous classes
                FileObject fo = JavaModel.getFileObject(((JavaClass)classDef).getResource());
                
                return fo == null ? null : PersistenceUtils.getEntityMappings(fo);
            }
        }
        return null;
    }

    private static Collection<String> integralTypes = new TreeSet<String>(Arrays.asList(
            WRAPPER_BYTE, WRAPPER_CHAR, WRAPPER_SHORT, WRAPPER_INT, 
            WRAPPER_LONG, PRIMITIVE_BYTE, PRIMITIVE_CHAR, PRIMITIVE_SHORT, 
            PRIMITIVE_INT, PRIMITIVE_LONG)); // NOI18N

    /**
     * Utility method to decide if a given type is integral type or not?
     * @param type Type which will be evaluated. Can be null.
     * @return returns true if the input type is an integral type. If input type
     * is null (this happens when type is not yet resolved), this method returns
     * false.
     */
    public static boolean isIntegralType(Type type) {
        if(type!=null) {
            return integralTypes.contains(type.getName());
        }
        return false;
    }

    private static Collection<String> fixedBasicTypes = new TreeSet<String>(Arrays.asList(
            WRAPPER_BYTE, WRAPPER_CHAR, WRAPPER_SHORT, WRAPPER_INT, 
            WRAPPER_LONG, PRIMITIVE_BYTE, PRIMITIVE_CHAR, PRIMITIVE_SHORT, 
            PRIMITIVE_INT, PRIMITIVE_LONG,
            PRIMITIVE_FLOAT, PRIMITIVE_DOUBLE, WRAPPER_FLOAT, WRAPPER_DOUBLE,
            DATE, CALENDAR, SQL_DATE, SQL_TIME, SQL_TIMESTAMP,
            "byte[]", "java.lang.Byte[]", "char[]", "java.lang.Character[]" // NOI18N
            )); // NOI18N

    /**
     * Utility method to decide if a given type is one of the allowable
     * types to be used in conjuntion with Basic annotation?
     * This only lists fixed types, not the dynamic ones like enum or Serializable.
     * @param type Type which will be evaluated. Can be null.
     * @return returns true if the input type is a fixed basic type. If input type
     * is null (this happens when type is not yet resolved), this method returns
     * false.
     */
    public static boolean isFixedBasicType(Type type) {
        if(type!=null) {
            return fixedBasicTypes.contains(type.getName());
        }
        return false;
    }

    /**
     * Utility method to find out if any member is annotated as Id or
     * EmbeddedId in this class? It does not check any of the inheritted
     * members.
     *
     * @param javaClass JavaClass whose members will be inspected.
     * @return returns true if atleast one member is annotated as Id or EmbeddedId
     */
    public static boolean isAnyMemberAnnotatedAsIdOrEmbeddedId(JavaClass javaClass) {
        boolean result = false;
        for (Element classElement : (List<Element>)javaClass.getContents()) {
            if (classElement instanceof Method ||
                    classElement instanceof Field){
                AnnotableElement element =
                        AnnotableElement.class.cast(classElement);
                Annotation annId = findAnnotation(element, ID);
                Annotation annEmbdId = findAnnotation(element, EMBEDDED_ID);

                if (annId != null || annEmbdId != null){
                    result = true;
                    break;
                }
            }
        }
        return result;
    }

    /**
     * Utility method to find out all the members(i.e. fields and or methods)
     * that are annotated as @Id.
     *
     * @param javaClass
     * @return
     */
    public static AnnotableElement[] getSimpleIdFields(JavaClass javaClass) {
        ArrayList<AnnotableElement> result = new ArrayList<AnnotableElement>();
        for (Element classElement : (List<Element>)javaClass.getContents()) {
            if (classElement instanceof Method ||
                    classElement instanceof Field){
                AnnotableElement element =
                        AnnotableElement.class.cast(classElement);
                if(findAnnotation(element, ID)!=null) {
                    result.add(element);
                }
            }
        }
        return result.toArray(new AnnotableElement[result.size()]);
    }

    public static AnnotableElement getEmbeddedId(JavaClass javaClass) {
        for (Element classElement : (List<Element>)javaClass.getContents()) {
            if (classElement instanceof Method ||
                    classElement instanceof Field){
                AnnotableElement element =
                        AnnotableElement.class.cast(classElement);
                if(findAnnotation(element, ID)!=null) {
                    return element;
                }
            }
        }
        return null;
    }

    /**
     * @param javaClass
     * @return the JavaClass corresponding to the IdClass specified
     * in the value attribute in @IdClass. return null, if class
     * can not be resolved or value is not specified.
     */
    public static JavaClass getIdClass(JavaClass javaClass) {
        Annotation annotation = findAnnotation(javaClass, ID_CLASS);
        if(annotation!=null) {
            for(AttributeValue attr :
                    (List<AttributeValue>)annotation.getAttributeValues()) {
                if (attr.getName() == null || // since value is a single method of @IdClass, this is allowed
                        PersistenceAPIAnnotations.VALUE.equals(attr.getName())) {
                    tmpDbg("value=" + attr.getValue());
                    final ClassExpression ce;
                    try {
                        ce = ClassExpression.class.cast(attr.getValue());
                    }catch(ClassCastException cce) {
                        return null; // user has not specified a class literal
                    }
                    return JavaClass.class.cast(ce.getClassName().getType());
                }
            }
        }
        return null;
    }

    /**
     * Set of annotations that can be applied at class level provided
     * the class is annotated as Entity
     */
    public static final Set entityClassAnnotations = new TreeSet(Arrays.asList(
            ID_CLASS, SEQUENCE_GENERATOR, TABLE_GENERATOR, 
            INHERITANCE,  DISCRIMINATOR_COLUMN, DISCRIMINATOR_VALUE,
            TABLE, SECONDARY_TABLE, SECONDARY_TABLES,
            PK_JOIN_COLUMN, PK_JOIN_COLUMNS, 
            ATTRIBUTE_OVERRIDE, ATTRIBUTE_OVERRIDES,
            ASSOCIATION_OVERRIDE, ASSOCIATION_OVERRIDES));

}
