/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.verification.persistence.hints;

import java.awt.Dialog;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import org.netbeans.jmi.javamodel.AnnotableElement;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AttributeValue;
import org.netbeans.jmi.javamodel.ClassMember;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.Parameter;
import org.netbeans.jmi.javamodel.ParameterizedType;
import org.netbeans.jmi.javamodel.StringLiteral;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.jmi.javamodel.TypedElement;
import org.netbeans.modules.editor.hints.spi.ChangeInfo;
import org.netbeans.modules.j2ee.common.JMIGenerationUtil;
import org.netbeans.modules.j2ee.verification.JEEVerificationContextInfo;
import org.netbeans.modules.j2ee.verification.JEEVerificationHint;
import org.netbeans.modules.j2ee.verification.ProblemFindingUtils;
import org.netbeans.modules.j2ee.verification.persistence.BeanAccessType;
import org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIHelper;
import org.netbeans.modules.javacore.api.JavaModel;

import static org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIAnnotations.ONE_TO_ONE;
import static org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIAnnotations.ONE_TO_MANY;
import static org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIAnnotations.MANY_TO_ONE;
import static org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIAnnotations.MANY_TO_MANY;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.openide.DialogDescriptor;
import org.openide.DialogDisplayer;
import org.openide.util.NbBundle;


/**
 *
 * @author Tomasz.Slota@Sun.COM
 */
public abstract class CreateRelationshipAbstractHint extends JEEVerificationHint {
    
    private static final String MAPPED_BY = "mappedBy"; //NOI18N
    private String annotationClass;
    private String complimentaryAnnotationClassName;
    private String relationName;
    private static Set<String> relationAnnotations = new TreeSet<String>(Arrays.asList(ONE_TO_ONE, ONE_TO_MANY, MANY_TO_MANY, MANY_TO_ONE));
    
    
    /** Creates a new instance of CreateOneToOneAnnotationHint */
    public CreateRelationshipAbstractHint(JEEVerificationContextInfo problemContext,
            String annotationClass, String complimentaryAnnotationClassName) {
        
        super(problemContext);
        this.annotationClass = annotationClass;
        this.complimentaryAnnotationClassName = complimentaryAnnotationClassName;
        
        int dotPos = annotationClass.lastIndexOf('.');
        relationName = dotPos > -1 ? annotationClass.substring(dotPos+1) : annotationClass;
    }
    
    @Override public ChangeInfo implement() {
        boolean owningSide = isOwningSideByDefault();
        JavaClass javaClass = getProblemContext().getJavaClass();
        String localFieldName = null;
        String mappedBy = null;
        Element availableElements[] = null;
        JavaClass targetClass = null;
        
        JavaModel.getJavaRepository().beginTrans(false);
        try {
            JavaModel.setClassPath(javaClass.getResource());
            targetClass = getTargetEntityClass();
            
            // get corresponding field
            localFieldName = getFieldNameFromElement(getProblemContext().getElement());
            mappedBy = getExistingFieldInRelation(javaClass, targetClass, localFieldName);
            
            if (mappedBy == null){
                Element[] compatibleElements = getCompatibleFieldsOrProperties(targetClass, javaClass);
                availableElements = filterOutElementsInRelation(compatibleElements);
            }
            
        } finally {
            JavaModel.getJavaRepository().endTrans(false);
        }
        
        if (mappedBy == null){
            // couldn't get the corresponding field automatically, display dialog
            
            CreateRelationshipPanel pnlPickOrCreateField = new CreateRelationshipPanel();
            pnlPickOrCreateField.setEntityClassNames(javaClass.getSimpleName(), targetClass.getSimpleName());
            pnlPickOrCreateField.setAvailableSelection(getAvailableRelationTypeSelection());
            pnlPickOrCreateField.setAvailableFields(wrapFields(availableElements));
            Set<String> existingFieldNames = getExistingFieldNames(targetClass);
            String defaultFieldName = genDefaultFieldName(javaClass, existingFieldNames);
            
            pnlPickOrCreateField.setDefaultFieldName(defaultFieldName);
            pnlPickOrCreateField.setExistingFieldNames(existingFieldNames);
            
            DialogDescriptor ddesc = new DialogDescriptor(pnlPickOrCreateField,
                    NbBundle.getMessage(CreateOneToOneAnnotationHint.class, "LBL_CreateRelationDlgTitle", relationName, targetClass.getSimpleName()));
            
            pnlPickOrCreateField.setDlgDescriptor(ddesc);
            Dialog dlg = DialogDisplayer.getDefault().createDialog(ddesc);
            dlg.setLocationRelativeTo(null);
            dlg.setVisible(true);
            
            if (ddesc.getValue() == DialogDescriptor.OK_OPTION){
                String fieldName = null;
                
                if (pnlPickOrCreateField.wasCreateNewFieldSelected()){
                    // create new
                    fieldName = pnlPickOrCreateField.getNewIdName();
                }else{
                    // pick existing
                    fieldName = pnlPickOrCreateField.getSelectedField().toString();
                }
                
                owningSide = pnlPickOrCreateField.owningSide();
                
                // do not create field for self-reflective relationships - see #77435
                if (!(javaClass.getName().equals(targetClass.getName()) && localFieldName.equals(fieldName))){
                    createFieldOrPropertyAtTargetClass(owningSide, targetClass, fieldName);
                }
                
                mappedBy = fieldName;
            }
        }
        
        if (mappedBy != null){
            JavaModel.getJavaRepository().beginTrans(true);
            try {
                Annotation ann = JMIGenerationUtil.createAnnotation(javaClass, annotationClass, Collections.EMPTY_LIST);
                
                if (!owningSide){
                    addMappedByAttribute(ann, mappedBy);
                }
                
                Element e = getProblemContext().getElement();
                ((AnnotableElement)e).getAnnotations().add(ann);
            } finally {
                JavaModel.getJavaRepository().endTrans(false);
            }
        }
        
        return null;
    }

    protected JavaClass getTargetEntityClass() {
        JavaClass targetClass;
        
        TypedElement e = (TypedElement) getProblemContext().getElement();
        targetClass = (JavaClass)e.getType();
        
        while (targetClass instanceof ParameterizedType) {
            targetClass = ((ParameterizedType)targetClass).getDefinition();
        }
        return targetClass;
    }
    
    protected String genDefaultFieldName(final JavaClass javaClass, Set<String> existingNames) {
        String defaultFieldNameBase = javaClass.getSimpleName();
        
        char initial = Character.toLowerCase(defaultFieldNameBase.charAt(0));
        defaultFieldNameBase = initial + defaultFieldNameBase.substring(1);
        
        if (isMultiValuedAtTargetEntity()){
            defaultFieldNameBase += "s"; //NOI18N
        }
        
        String defaultFieldName = null;
        int suffix = 0;
        
        do{
            defaultFieldName = defaultFieldNameBase + (suffix == 0 ? "": suffix); //NOI18N
            suffix ++;
        }
        while (existingNames.contains(defaultFieldName));
        
        return defaultFieldName;
    }
    
    private ElementWrapper[] wrapFields(Element fields[]){
        List<ElementWrapper> r = new ArrayList<ElementWrapper>();
        
        for (Element element : fields){
            r.add(ElementWrapper.create(element));
        }
        
        return r.toArray(new ElementWrapper[0]);
    }
    
    private String getExistingFieldInRelation(JavaClass localClass, JavaClass targetClass, String localFieldName){
        for (Element e : getCompatibleFieldsOrProperties(targetClass, localClass)){
            for (Annotation a : getRelationAnnotations((AnnotableElement) e)){
                for (AttributeValue av : (List<AttributeValue>)a.getAttributeValues()){
                    if (MAPPED_BY.equals(av.getName()) &&
                            localFieldName.equals(((StringLiteral)av.getValue()).getValue())){
                        
                        return getFieldNameFromElement(e);
                    }
                }
            }
        }
        
        return null;
    }
    
    private Element[] getCompatibleFieldsOrProperties(JavaClass javaClass, JavaClass type){
        List<Element> r = new ArrayList<Element>();
        
        Class elementClass = PersistenceAPIHelper.findAccessTypeOfHierarchy(javaClass) == BeanAccessType.FIELD ?
            Field.class : Method.class;
        
        for (Feature f : (List<Feature>) javaClass.getFeatures()){
            if (elementClass.isInstance(f)){
                Element element = (Element)f;
                
                if (typeMatches((TypedElement)element, type)){
                    r.add(element);
                }
            }
        }
        
        return r.toArray(new Element[0]);
    }
    
    protected boolean typeMatches(TypedElement element, JavaClass type){
        if (element.getType() == null){
            return false;
        }
        
        Type typeToCheck = null;
        
        if (isMultiValuedAtTargetEntity()){
            if (element.getType() instanceof ParameterizedType){
                ParameterizedType pt = (ParameterizedType)element.getType();
                
                if (pt.getParameters().size() == 1){
                    typeToCheck = (Type) pt.getParameters().get(0);
                }
            }
        } else{
            typeToCheck = element.getType();
        }
        
        if (typeToCheck != null && typeToCheck.getName().equals(type.getName())){
            return true;
        }
        
        return false;
    }
    
    
    private static String getFieldNameFromElement(Element e){
        if (e instanceof NamedElement){
            String elementName = ((NamedElement)e).getName();
            
            if (e instanceof Field){
                return elementName;
            } else if (e instanceof Method){
                return PersistenceAPIHelper.getFieldNameFromAccessorName(elementName);
            }
        }
        
        return null;
    }
    
    private Annotation[] getRelationAnnotations(AnnotableElement ae){
        ArrayList<Annotation> r = new ArrayList<Annotation>();
        
        for (Annotation a : (List<Annotation>)ae.getAnnotations()){
            if (a.getType() != null && relationAnnotations.contains(a.getType().getName())){
                r.add(a);
            }
        }
        
        return r.toArray(new Annotation[0]);
    }
    
    private void createFieldOrPropertyAtTargetClass(boolean owningSide, JavaClass targetClass, String fieldName) {
        JavaModel.getJavaRepository().beginTrans(true);
        try {
            JavaModel.setClassPath(targetClass.getResource());
            Annotation annotation = null;
            
            // create the field
            Field field = targetClass.getField(fieldName, false);
            String fieldType = getFieldToBeCreatedTypeName(targetClass);
            
            if (field == null){
                field = JMIGenerationUtil.createField(targetClass, fieldName, Modifier.PRIVATE, fieldType);
                targetClass.getFeatures().add(0,field);
            }
            
            if (PersistenceAPIHelper.findBeanAccessType(targetClass) == BeanAccessType.PROPERTY){
                // create getter
                Method getter = createMethodIfNeeded(targetClass, PersistenceAPIHelper.getAccessorName(fieldName), fieldType, Collections.EMPTY_LIST); //NOI18N
                getter.setBodyText("return " + fieldName + ";");
                annotation = createAnnotationAtTargetField(getter, targetClass);
                
                
                // create setter
                List setterArg = Collections.singletonList(field.getType());
                String getterName = PersistenceAPIHelper.getMutatorName(fieldName);
                
                if (targetClass.getMethod(getterName, setterArg, false) == null){
                    Method setter = createMethodIfNeeded(targetClass, getterName, "void", setterArg); //NOI18N
                    Parameter idParameter = JMIGenerationUtil.createParameter(targetClass, fieldName, fieldType);
                    setter.getParameters().add(idParameter);
                    setter.setBodyText("this." + fieldName + " = " + fieldName + ";"); //NOI18N
                }
            } else{ // assuming field access
                annotation = createAnnotationAtTargetField(field, targetClass);
            }
            
            if (annotation != null && owningSide){
                addMappedByAttribute(annotation, getFieldNameFromElement(getProblemContext().getElement()));
            }
        } finally {
            JavaModel.getJavaRepository().endTrans(false);
        }
    }

    private Annotation createAnnotationAtTargetField(AnnotableElement element, JavaClass targetClass) {
        Annotation annotation = null;
        
        for (Annotation existingAnn : (List<Annotation>)element.getAnnotations()){
            if (complimentaryAnnotationClassName.equals(existingAnn.getType().getName())){
                annotation = existingAnn;
                break;
            }
        }
        
        if (annotation == null){
            annotation = JMIGenerationUtil.createAnnotation(targetClass, complimentaryAnnotationClassName, Collections.EMPTY_LIST);
            element.getAnnotations().add(annotation);
        }
        
        return annotation;
    }

    protected String getFieldToBeCreatedTypeName(JavaClass targetClass) {
        String fieldType = getProblemContext().getJavaClass().getName();
        JavaClass fieldTypeClass = (JavaClass) JavaModel.getDefaultExtent().getType().resolve(fieldType);
        String simpleTypeName = JavaModelUtil.resolveImportsForType(targetClass, fieldTypeClass).getName();
        
        if (isMultiValuedAtTargetEntity()){
            JavaClass listTypeClass = (JavaClass) JavaModel.getDefaultExtent().getType().resolve("java.util.List"); //NOI18N
            JavaModelUtil.resolveImportsForType(targetClass, listTypeClass).getName();
            fieldType = "List<" + simpleTypeName + ">"; //NOI18N
        }
        
        return fieldType;
    }
    
    private static Method createMethodIfNeeded(JavaClass parentClass, String methodName, String returnType, List args){
        Method method = parentClass.getMethod(methodName, args, false);
        
        if (method == null){
            method = JMIGenerationUtil.createMethod(parentClass, methodName, Modifier.PUBLIC, returnType);
            parentClass.getFeatures().add(method);
        }
        
        return method;
    }
    
    
    private void addMappedByAttribute(Annotation ann, String value){
        AttributeValue av = JMIGenerationUtil.createAttributeValue(getProblemContext().getJavaClass(),
                MAPPED_BY, value);
        
        ann.getAttributeValues().add(av);
    }
    
    @Override public String getText(){
        return NbBundle.getMessage(CreateRelationshipAbstractHint.class, "LBL_CreateRelationHint", relationName);
    }
    
    @Override public int getType(){
        return ERROR;
    }
    
    protected boolean isMultiValuedAtTargetEntity(){
        return false;
    }

    protected Element[] filterOutElementsInRelation(Element[] compatibleElements) {
        ArrayList<Element> r = new ArrayList<Element>();
        
        for (Element e:compatibleElements){
            if (ProblemFindingUtils.findFirstAnnotationFromGivenSet((AnnotableElement)e, relationAnnotations) == null){
                r.add(e);
            }
        }
        
        return r.toArray(new Element[0]);
    }

    private Set<String> getExistingFieldNames(JavaClass javaClass) {
        Set<String> r = new TreeSet<String>();
        
        for (ClassMember m : (List<ClassMember>)javaClass.getContents()){
            if (m instanceof Field){
                r.add(m.getName());
            }
        }
        
        return r;
    }

    protected boolean isOwningSideByDefault() {
        return true;
    }

    protected CreateRelationshipPanel.AvailableSelection getAvailableRelationTypeSelection() {
        return CreateRelationshipPanel.AvailableSelection.BOTH;
    }
    
    private static abstract class ElementWrapper{
        private Element element;
        
        ElementWrapper(Element element){
            this.element = element;
        }
        
        Element getElement(){
            return element;
        }
        
        static ElementWrapper create(Element e){
            if (e instanceof Field){
                return new FieldWrapper((Field)e);
            } else if (e instanceof Method){
                return new GetterWrapper((Method)e);
            }
            
            return null;
        }
        
        private static class FieldWrapper extends ElementWrapper{
            FieldWrapper(Field field) {
                super(field);
            }
            
            @Override public String toString(){
                return ((Field)getElement()).getName();
            }
        }
        
        private static class GetterWrapper extends ElementWrapper{
            GetterWrapper(Method getter) {
                super(getter);
            }
            
            @Override public String toString(){
                return getFieldNameFromElement(getElement());
            }
        }
    }
}

