/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.j2ee.verification.persistence.rules.clazz;

import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.j2ee.verification.AbstractRule;
import org.netbeans.modules.j2ee.verification.Problem;
import org.netbeans.modules.j2ee.verification.ProblemFinderContext;
import org.netbeans.modules.j2ee.verification.persistence.BeanAccessType;
import org.netbeans.modules.j2ee.metadata.JMIClassIntrospector;
import org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIHelper;
import org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIProblemFinder;
import org.netbeans.modules.j2ee.verification.persistence.predicates.clazz.AnnotatedAsIdClassPred;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Collections;

/**
 * all properties found in IdClass must be also defined in the classthat uses
 * that class as IdClass.
 *
 * @author Sanjeeb.Sahoo@Sun.COM
 */
public class AllPropertiesOfIdClassDefinedInEntity extends AbstractRule {

    private List<String> missingAccessorNames = new ArrayList<String>();

    public AllPropertiesOfIdClassDefinedInEntity() {
        super(null, Arrays.asList(new AnnotatedAsIdClassPred()));
    }

    public String getId() {
        return PersistenceAPIProblemFinder.RuleNames.AllPropertiesOfIdClassDefinedInEntity;
    }

    protected String getMessageKey() {
        return "MSG_MissingIdProperty"; // NOI18N
    }

    @Override protected boolean isApplicable(
            Element subject, ProblemFinderContext ctx) {
        return (ctx.isIsEntityClass() || ctx.isMappedSuperclass()) &&
                ctx.getBeanAccessType() == BeanAccessType.PROPERTY &&
                super.isApplicable(subject, ctx);
    }

    @Override public Problem apply(Element subject, ProblemFinderContext ctx) {
        if (isApplicable(subject, ctx)) {
            JavaClass javaClass = ctx.getMainJavaClass();
            JavaClass idClass = PersistenceAPIHelper.getIdClass(javaClass);
            if(idClass==null) return null; // unresolved class
            // See, we don't search in super class of idClass
            for (ClassMember cm : (List<ClassMember>) idClass.getContents()) {
                if (cm instanceof Method) {
                    Method idMethod = Method.class.cast(cm);
                    if (JMIClassIntrospector.isGetter(idMethod)) {
                        final Method entityMethod = javaClass.getMethod(
                                idMethod.getName(),
                                Collections.emptyList(),
                                false); // no inheritance of property related methods b'cos it has to be annotated in this class.
                        if (entityMethod == null) {
                            missingAccessorNames.add(idMethod.getName()+"()"); // NOI18N
                        }
                    } else if(JMIClassIntrospector.isSetter(idMethod)) {
                        final Parameter param = Parameter.class.cast(
                                idMethod.getParameters().get(0));
                        final Method entityMethod = javaClass.getMethod(
                                idMethod.getName(),
                                Arrays.asList(param.getType()),
                                false); // no inheritance of property related methods b'cos it has to be annotated in this class.
                        if (entityMethod == null) {
                            final String methodName = idMethod.getName()+
                                    "(" + param.getType().getName()+ ")"; // NOI18N
                            missingAccessorNames.add(methodName);
                        }
                    }
                }
            }
            if (!missingAccessorNames.isEmpty()) {
                return createProblem(ctx, subject);
            }
        }
        return null;
    }

    @Override protected Object[] getMessageParameters() {
        StringBuilder sb = new StringBuilder();
        for (String f : missingAccessorNames) {
            sb.append(f).append(" "); // NOI18N
        }
        Object[] result = {sb.toString()};
        return result;
    }

    @Override public void reset() {
        missingAccessorNames.clear();
    }
}
