/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.deployment.plugins.api;

import java.io.File;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import javax.enterprise.deploy.model.DDBean;
import javax.enterprise.deploy.model.DeployableObject;
import javax.enterprise.deploy.spi.DeploymentConfiguration;
import javax.enterprise.deploy.spi.exceptions.ConfigurationException;
import javax.enterprise.deploy.spi.exceptions.OperationUnsupportedException;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.modules.j2ee.deployment.common.api.Datasource;
import org.netbeans.modules.j2ee.deployment.common.api.DatasourceAlreadyExistsException;
import org.netbeans.modules.j2ee.deployment.common.api.OriginalCMPMapping;
import org.netbeans.modules.j2ee.deployment.config.ConfigSupportImpl;
import org.netbeans.modules.j2ee.deployment.devmodules.spi.J2eeModuleProvider;
import org.netbeans.modules.j2ee.deployment.impl.Server;
import org.netbeans.modules.j2ee.deployment.impl.ServerRegistry;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.Lookup;


/**
 * <p>
 * ConfigurationSupport provides support for server specific configuration. It
 * supplements the JSR-88 DeploymentConfiguration class of life cycle management 
 * and some additional module change notifications.
 * </p>
 *
 * <h3>
 * The life cycle of the DeploymentConfiguration object.
 * </h3>
 *
 * <p>
 * When a new project is created or an existing one is opened, the j2eeserver
 * will create a DeploymentConfiguration object for the project target server by
 * calling DeploymentManager.createConfiguration(). After the configuration is 
 * created, the j2eeserver will initialize it by calling ConfigurationSupport.initConfiguration().
 * In this moment, plugin should make sure that the server specific deployment 
 * descriptors exist and create them, if they do not. From now on the server specific
 * deployment descriptors for the target server should be ready to respond to the 
 * module changes.
 * </p>
 *
 * <p>
 * In case that the project target server is changed, the old server is notified
 * by calling ConfigurationSupport.disposeConfiguration() that the old
 * DeploymentConfiguration will be disposed, which means that the j2eeserver will
 * release the reference to it and the configuration will be no longer receiving 
 * notifications of the module changes. The server specific deployment descriptors for
 * the old server will thus stop responding to the module changes.
 * </p>
 *
 * <p>
 * The life cycle for the new target server is the same as if an existing project 
 * is opened, see above.
 * </p>
 *
 * <p>
 * In case that the target server plugin needs to have access to the initilized 
 * deployment configuration before the j2eeserver created and initilized it by 
 * itself, the plugin can request the j2eeserver to create and initilize it by 
 * calling ConfigurationSupport.requestCreateConfiguration(). This may happen
 * for example when the IDE is started and if the server specific deployment 
 * descriptor was left open in the editor before the IDE was shutdown the last time.
 * After calling ConfigurationSupport.requestCreateConfiguration() the life cycle 
 * is the same as if the j2eeserver created and initialized the deployment 
 * configuration during project opening by itself.
 * </p>
 *
 * <p>
 * In case that other than target server plugin needs to have access to the initilized 
 * deployment configuration, the plugin can request the j2eeserver to create and 
 * initilize the configuration by calling ConfigurationSupport.requestCreateConfiguration(). 
 * The j2eeserver will create the configuration by calling DeploymentManager.createConfiguration() 
 * and will initilize it by calling ConfigurationSupport.initConfiguration(), however, 
 * with the keepUpdated=false parameter. This will tell the plugin that its server
 * specific deployment descriptors should not respond to the module changes. In 
 * fact, the deployment configuration will not be receiving module change events 
 * in this case. Since the j2eeserver will not even keep a reference to this 
 * configuration, the ConfigurationSupport.disposeConfiguration() will never be 
 * called for this configuration, there is also no reason for it.
 * </p>
 *
 *
 * @author sherold
 *
 * @since 1.8
 */
public abstract class ConfigurationSupport {
    
    /**
     *  Setting the CMP and CMR mapping info to the given configuration for the EJB by the given name.
     *  This call will be called after CMP wizard created all CMP beans (all new ejb events fired).
     *
     *  @param config   The configuration to push the CMP mapping to
     *  @param mappings All the mapping info needed to be pushed in one batch.
     */
    public abstract void setMappingInfo(DeploymentConfiguration config, 
                                        OriginalCMPMapping[] mappings);
    
    /**
     * Ensure resource is defined for the specified DD bean (for example entity bean).
     *
     * @param config    deployment configuration
     * @param bean      DD bean in question
     */
    public abstract void ensureResourceDefined(DeploymentConfiguration config, 
                                               DDBean bean);
    
    /**
     * Ensure resource is defined for the specified DD bean (for example entity bean).
     *
     * @param config    deployment configuration
     * @param bean      DD bean in question
     * @param jndiName  the JNDI name of the resource where the bean is stored
     */
    public void ensureResourceDefined(DeploymentConfiguration config, 
                                               DDBean bean, String jndiName) {
    }
    
    /**
     * Return the context root (context path) defined for the module specified 
     * by the deployable object.
     *
     * @param config  deployment configuration
     * @param deplObj deployable object that specifies the module
     *
     * @return context root
     *
     * @throws ConfigurationException reports errors in getting the web context
     *         root
     */
    public abstract String getWebContextRoot(DeploymentConfiguration config, 
                                             DeployableObject deplObj) 
                                    throws ConfigurationException;

    /**
     * Set the web context root (context path) for the module specified by the 
     * deployable object.
     *
     * @param config      deployment configuration
     * @param deplObj     deployable object that specifies the module
     * @param contextRoot context root to be set
     *
     * @throws ConfigurationException reports errors in setting the web context
     *         root
     */
    public abstract void setWebContextRoot(DeploymentConfiguration config, 
                                           DeployableObject deplObj,
                                           String contextRoot) 
                                   throws ConfigurationException;
    
    /**
     * Initialize the specified deployment configuration. If keepUpdated is true 
     * ensure the server specific deployment descriptors are created if they do 
     * not exist and that they are ready to respond to the module changes. If
     * keepUpdated is false, the module changes should not be reflected in the 
     * server specific deployment descriptors.
     * 
     * @param config        the deployment configuration that should be initialized.
     * @param files         server specific deployment configuration files.
     * @param resourceDir   directory containing definition for enterprise resources 
     *                      needed for the module execution or null if not supported.
     * @param ensureCreated if keepUpdated is true ensure the server specific 
     *                      deployment descriptors are created if they do not 
     *                      exist and that they are ready to respond to the module 
     *                      changes. If keepUpdated is false, the module changes 
     *                      should not be reflected in the server specific deployment 
     *                      descriptors.
     *
     * @throws ConfigurationException reports errors in deployment configuration
     *         initialization.
     */
    public abstract void initConfiguration(DeploymentConfiguration config, 
                                           File[] files, 
                                           File resourceDir, 
                                           boolean keepUpdated) 
                                    throws ConfigurationException;
    
    /**
     * The specified deployment configuration instance is about to be disposed,
     * which means that the j2eeserver will release the reference to it and the 
     * configuration will be no longer receiving notifications of the module changes.
     *
     * @param config deployment configuration which is about to be disposed.
     */
    public abstract void disposeConfiguration(DeploymentConfiguration config);
    
    /**
     * <p>
     * Request the j2eeserver to create and initialize the deployment configuration
     * for the specified server and module the given server specific deployment 
     * descriptor file belongs to. If the deployment configuration has already 
     * been created nothing will happen, otherwise a new instance will be created 
     * by calling DeploymentManager.createConfiguration() and then initialized by 
     * calling ConfigurationSupport.initConfiguration().
     * </p>
     *
     * <p>
     * The j2eeserver creates and initializes the deployment configuration for the
     * server that is set as a target server for the given module. In some cases,
     * however, the data object representing the server specific deployment descriptor
     * may require access to the initialized deployment configuration even before
     * the configuration is created and initialized by the j2eeserver itself. This
     * may happen for instance when the server specific deployment descriptor is
     * being reopened during the IDE startup if it was left open before the IDE 
     * was shut down the last time.
     * </p>
     *
     * @param fo        server specific deployment descriptor file object
     * @param serverID  server (plugin) ID
     */
    public static final void requestCreateConfiguration(FileObject fo, 
                                                        String serverID) 
                                            throws ConfigurationException {
        Project project = FileOwnerQuery.getOwner(fo);
        if (project == null) {
            throw new ConfigurationException("File " + FileUtil.toFile(fo).getAbsolutePath() + " is not part of any project"); // NOI18N
        }
        Lookup l = project.getLookup();
        J2eeModuleProvider moduleProvider = (J2eeModuleProvider)l.lookup(J2eeModuleProvider.class);
        if (moduleProvider == null) {
            throw new ConfigurationException("Project " + project + " does not provide J2eeModuleProvider in its lookup"); // NOI18N
        }
        Server server = ServerRegistry.getInstance().getServer(serverID);
        ConfigSupportImpl conf = (ConfigSupportImpl)moduleProvider.getConfigSupport();
        conf.createDeploymentConfiguration(server);
    }
    
    /**
     * Update the enterprise resource directory for the specified deployment
     * configuration.
     *
     * @param config      deployment configuration
     * @param resourceDir new enterprise resource directory.
     * @since 1.12
     */
    public abstract void updateResourceDir(DeploymentConfiguration config, File resourceDir);
    
    /**
     * Gets the data sources saved in the module.
     *
     * @param config deployment configuration
     * @return a set of data sources. The default implementation returns an empty set.
     *
     * @since 1.15
     */
    public Set<Datasource> getDatasources(DeploymentConfiguration config) {
        return Collections.<Datasource>emptySet();
    }
    
    /**
     * Tests whether data source creation is supported.
     *
     * @return true if data source creation is supported. The default implementation returns false.
     *
     * @since 1.15
     */
    public boolean isDatasourceCreationSupported() {
        return false;
    }
            
    /**
     * Creates and saves data source in the module if it does not exist yet in the module.
     * Data source is considered to be existing when JNDI name of the found data source and the one
     * just created equal.
     *
     * @param config deployment configuration
     * @param jndiName name of data source
     * @param url database URL
     * @param username database user
     * @param password user's password
     * @param driver fully qualified name of the database driver class
     * @return created data source
     * @exception OperationUnsupportedException if operation is not supported
     * @exception ConfigurationException if data source configuration file is not parseable
     * @exception DatasourceAlreadyExistsException if conflicting data source is found
     *
     * @since 1.15
     */
    public Datasource createDatasource(DeploymentConfiguration config, String jndiName, String  url, String username, String password, String driver) 
    throws OperationUnsupportedException, ConfigurationException, DatasourceAlreadyExistsException
    {
        throw new OperationUnsupportedException(""); // NOI18N
    }
}
