/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.editor.ext.java;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.StringTokenizer;
import org.netbeans.editor.LocaleSupport;
import org.netbeans.editor.ext.CompletionJavaDoc;
import org.netbeans.editor.ext.ExtEditorUI;

/**
 *  Support for javadoc in code completion.
 *  Contains also static utilities methods for preparing javadoc HTML content
 *
 *  @author  Martin Roskanin
 *  @since   05/2004
 */
public abstract class JavaCompletionJavaDoc extends CompletionJavaDoc{
    
    private static ArrayList omittedTags = new ArrayList();
    static{
        omittedTags.add("@serial"); //NOI18N
        omittedTags.add("@serialField"); //NOI18N
        omittedTags.add("@serialData"); //NOI18N
        omittedTags.add("@author"); //NOI18N
        omittedTags.add("@version"); //NOI18N
        omittedTags.add("@beaninfo"); //NOI18N
    }
    
    
    
    /** Creates a new instance of JavaCompletionJavaDoc */
    public JavaCompletionJavaDoc(ExtEditorUI extEditorUI) {
        super(extEditorUI);
    }
    
    protected boolean isNotFullyQualifiedInnerClass(String inner, String pkgName){
        if (inner.indexOf(".") != inner.lastIndexOf(".")){ //NOI18N
            // fully qualified
            return false;
        }
        return (getFinder().getExactClass( (pkgName.length()>0) ? (pkgName+"."+inner) : inner ) != null); //NOI18N
    }
    
    protected boolean isNotFullyQualifiedInnerClass (String inner, JCClass cls){
        return isNotFullyQualifiedInnerClass(inner, cls.getPackageName());
    }

    protected String findProperClass(String name, String pkgName){

        String ret = null;

        JCClass retClass;
        if (pkgName!=null && pkgName.length()>0){
            retClass = getFinder().getExactClass(pkgName+"."+name); //NOI18N
        }else{
            retClass = getFinder().getExactClass(name); 
        }

        if (retClass != null){
            return retClass.getFullName();
        }
        
        List classes = getFinder().findClasses(null, name, true);
        if (classes.size()>0){
            ret = ((JCClass)classes.get(0)).getFullName();
        }
        
        return ret;
    }
    
    protected String findProperClass(String name, JCClass cls){
        if (cls == null) return null;
        return findProperClass(name, cls.getPackageName());
    }
    
    private int getLastParenthesisOccurence(String prm){
        int level = 0;
        for(int i = 0;  i<prm.length(); i++){
            char ch = prm.charAt(i);
            if (ch == ' ' && level == 0) return i; //NOI18N
            if (ch == '(') level++; //NOI18N
            if (ch == ')'){ //NOI18N
                if (level == 1){
                    return i;
                }else{
                    level--;
                }
            }
        }
        return -1;
    }

    protected String createAnchor(String prm, String clsFQN, String clsPkgName, boolean linkedObjectExists){
        
        StringBuffer ret= new StringBuffer(prm.length() + 50); //NOI18N
        String label = ""; //NOI18N
        int lastParenthesisOccurence = getLastParenthesisOccurence(prm);
        
        int indexOfSpaceAfterLastParenthesis = prm.indexOf(" ",lastParenthesisOccurence); // NOI18N

        if ( indexOfSpaceAfterLastParenthesis > 0){
            label = prm.substring(indexOfSpaceAfterLastParenthesis+1);
        }

        if (prm.length() > 0 && prm.charAt(0) == '#'){ //NOI18N
            if (linkedObjectExists){
                ret.append("<a href='").append(clsFQN).append(((label.length()>0) ? prm.substring(0,indexOfSpaceAfterLastParenthesis):prm)).append( //NOI18N
                "'>").append(((label.length() > 0)?label:prm.substring(1))).append("</a>"); //NOI18N
            }else{
                ret.append(((label.length() > 0)?label:prm.substring(1)));
            }
                
        }else{
            if (linkedObjectExists){
                String href = ((label.length()>0)?prm.substring(0,indexOfSpaceAfterLastParenthesis):prm);
                if (href.indexOf(".")<0 || isNotFullyQualifiedInnerClass(href, clsPkgName)){ //NOI18N
                    // stick a package before
                    String refCls = (href.indexOf("#") > 0) ? href.substring(0, href.indexOf("#")) : href; //NOI18N
                    String refMember = (href.indexOf("#") > 0) ? href.substring(href.indexOf("#")+1) : ""; //NOI18N
                    String fullClassName = findProperClass(refCls, clsPkgName);
                    if (fullClassName != null){
                        href = (refMember.length()>0) ? fullClassName+"#"+refMember : fullClassName; //NOI18N
                    }else{
                        return "<a href='"+href+"'>"+ //NOI18N
                        ((label.length() > 0)?label:prm.replace('#','.'))+"</a>"; //NOI18N
                    }
                }
                ret.append("<a href='").append(href).append("'>").append( //NOI18N
                ((label.length() > 0)?label:prm.replace('#','.'))).append("</a>"); //NOI18N
            }else{
                ret.append(((label.length() > 0)?label:prm.replace('#','.')));
            }
        }
        return ret.toString();
        
    }

    protected String createAnchor(String prm, String clsFQN, String clsPkgName){
        boolean linkedObjectExists = (parseLink(prm, clsFQN, clsPkgName) != null );
        return createAnchor(prm, clsFQN, clsPkgName, linkedObjectExists);
    }

    
    /** Creates a link from given javadoc @see or @link parameter and root class
     *  @param prm link parameter such as <code>java.awt.Component#addHierarchyListener</code>
     *  @param cls root class to which link a link parameter without class specification like <code>#addHierarchyListener</code>
     */
    public String createAnchor(String prm, JCClass cls){
        return createAnchor(prm, cls.getFullName(), cls.getPackageName());
    }
    
    protected String createLinks(String content, String classFQN, String pkgName){
        StringBuffer ret = new StringBuffer(content.length());
        StringTokenizer strTok = new StringTokenizer(content ,"{"); //NOI18N
        String next=""; //NOI18N
        boolean linkAppended = false;
        boolean useOriginalContent = true;
        while (strTok.hasMoreTokens()){
            next = strTok.nextToken("{"); //NOI18N
            if (linkAppended && next.length() > 0 && next.charAt(0) == '}') next = next.substring(1); //NOI18N
            linkAppended = false;
            if (strTok.hasMoreTokens()) {
                String link = strTok.nextToken("}"); //NOI18N
                int indexOfLink = link.indexOf("@link"); // NOI18N
                if (indexOfLink > 0 && link.length()>6 && Character.isWhitespace(link.charAt(link.indexOf("@link")+5))) { //NOI18N
                    useOriginalContent = false;
                    int linkPos = indexOfLink+6; //NOI18N
                    if (next.length() > 0 && next.charAt(0) == '}') next = next.substring(1); //NOI18N
                    String prm  =link.substring(linkPos).trim();
                    boolean linkedObjectExists = (parseLink(prm, classFQN, pkgName) != null );                    
                    link = createAnchor(prm, classFQN, pkgName, linkedObjectExists);
                    linkAppended = true;
                }
                ret.append(next);
                ret.append(link);
            } else {
                ret.append(next);                
            }
        }
        return useOriginalContent ? content : ret.toString();
    }

    /** Processes given javadoc content and replaces @link parameters to anchors 
     *  @param content raw javadoc content
     *  @param cls root class
     */
    public String createLinks(String content, JCClass cls){
        return createLinks(content, cls.getFullName(), cls.getPackageName());
    }

    /** Prepares raw javadoc content 
     *  @param cls root class
     *  @param member String representation of field or method
     *  @param content raw javadoc content text
     */
    public String prepareJavaDocContent(JCClass cls, String member, String content){
        return prepareJavaDocContent(cls, member, content, null);
    }

     /** Prepares raw javadoc content with given javadoc parameters
     *  @param rootClsFQN root class FQN
     *  @param rootClsPkgName rott class package name
     *  @param member String representation of field or method
     *  @param content raw javadoc content text
     *  @param tags javadoc tag items array
     */
    public String prepareJavaDocContent(String rootClsFQN, String rootClsPkgName, String member, String content, CompletionJavaDoc.JavaDocTagItem tags[]){
        if (content != null){
            content = createLinks(content, rootClsFQN, rootClsPkgName);
        }else{
            content = ""; //NOI18N
        }
        StringBuffer sb = new StringBuffer(512);
        
        //sb.insert(0,"<font size='4'><code><b>"+cls.toString()+"</b></code></font>"); //NOI18N
        sb.append("<font size='+0'><b>").append(createAnchor(rootClsFQN, rootClsFQN, rootClsPkgName)).append("</b></font>"); //NOI18N
        if (member.length()>0) sb.append("<pre>").append(member).append("</pre>"); //NOI18N
        sb.append("<blockquote>").append(content).append("</blockquote>"); //NOI18N
        if (tags!=null){
            Arrays.sort(tags);
            String curParam = ""; //NOI18N
            int curItem = 0;
            for (int i=0; i<tags.length; i++){
                if (!curParam.equals(tags[i].getName())){
                    if (isOmittedJavaDocTag(tags[i].getName())) {
                        continue;
                    }
                    

                    if(!curParam.equals("")) sb.append("</blockquote>"); //NOI18N
                    sb.append("<b>").append(LocaleSupport.getString(BUNDLE_PREFIX+tags[i].getName(),tags[i].getName())).append("</b><blockquote>"); //NOI18N
                    curParam = tags[i].getName();
                    curItem = 0;
                }
                
                if (curItem>0)
                   sb.append("<br>"); //NOI18N
                

                if (curParam.equals("@param")){ //NOI18N
                    String prm = tags[i].getText();
                    int spacePos = prm.indexOf(" "); //NOI18N
                    if (spacePos>0){
                        String anchoredText = createLinks(prm.substring(spacePos), rootClsFQN, rootClsPkgName);
                        sb.append("<code>").append(prm.substring(0,spacePos)).append("</code> - ").append(anchoredText); //NOI18N
                    }else{
                        sb.append(prm);
                    }
                }else if (curParam.equals("@see")){ //NOI18N
                    String txt = tags[i].getText();
                    char first = txt.length() > 0 ? txt.charAt(0): '\0';
                    if (first == '"'){ //NOI18N
                        if (txt.length() > 1 && txt.charAt(txt.length()-1) == '"') {
                            txt = txt.substring(1,txt.length()-1); //NOI18N
                        }
                        else {
                            txt = txt.substring(1,txt.length()); //NOI18N
                        }
                        sb.append(txt);
                    }else if (first == '<'){ //NOI18N
                        sb.append(txt);
                    }else{
                        sb.append(createAnchor(txt, rootClsFQN, rootClsPkgName));
                    }
                }else{
                    sb.append(createLinks(tags[i].getText(), rootClsFQN, rootClsPkgName));
                }
                
                curItem++;
            }
            if(!curParam.equals("")) sb.append("</blockquote>"); //NOI18N
        }
        
        return sb.toString();
    }
    
     /** Prepares raw javadoc content with given javadoc parameters
     *  @param cls root class
     *  @param member String representation of field or method
     *  @param content raw javadoc content text
     *  @param tags javadoc tag items array
     */
    public String prepareJavaDocContent(JCClass cls, String member, String content, CompletionJavaDoc.JavaDocTagItem tags[]){
        return prepareJavaDocContent(cls.getFullName(), cls.getPackageName(), member, content, tags);
    }

    
    /** Checks whether a tag should be omitted in javadoc */
    public boolean isOmittedJavaDocTag(String tag){
        return omittedTags.contains(tag);
    }

    
    /** Parses given link such as <code>java.awt.Component#addHierarchyListener</code>
     *  and returns parsed Object
     *  @return Object of JCClass, JCMethod, JCConstructor or JCField 
     */
    public Object parseLink(String link, JCClass cls){
        if (cls == null) return parseLink(link, null, null);
        return parseLink(link, cls.getFullName(), cls.getPackageName());
    }

    /** Parses given link such as <code>java.awt.Component#addHierarchyListener</code>
     *  and returns parsed Object
     *  @return Object of JCClass, JCMethod, JCConstructor or JCField 
     */
    public Object parseLink(String link, String clsFQN, String pkgName){
        return null;
    }
    
    /** Parses given link such as <code>java.awt.Component#addHierarchyListener</code>
     *  and returns parsed Object
     *  @return Object of JCClass, JCMethod, JCConstructor or JCField 
     */
    /*
    public Object parseLink(String link, Object obj){
       if (obj instanceof JCClass || obj == null){
            return parseLink(link, (JCClass)obj);
        }
        return null;
    }
     */
    
    protected abstract JCFinder getFinder();
    
    
}
