/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.java;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.event.KeyEvent;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import javax.swing.Action;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import javax.swing.text.Position;
import org.netbeans.api.editor.completion.Completion;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.Utilities;
import org.netbeans.editor.ext.java.JavaCompletion;
import org.netbeans.editor.ext.java.JavaSyntaxSupport;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.jmi.javamodel.ElementPartKindEnum;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.Parameter;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.spi.editor.completion.CompletionDocumentation;
import org.netbeans.spi.editor.completion.CompletionItem;
import org.netbeans.spi.editor.completion.CompletionResultSet;
import org.netbeans.spi.editor.completion.CompletionProvider;
import org.netbeans.spi.editor.completion.CompletionTask;
import org.netbeans.spi.editor.completion.support.AsyncCompletionQuery;
import org.netbeans.spi.editor.completion.support.AsyncCompletionTask;
import org.openide.ErrorManager;
import org.openide.util.NbBundle;

/**
 *
 * @author Jan Lahoda
 */
public class ElementCreatingCompletionProvider implements CompletionProvider {
    
    /**
     * Creates a new instance of ElementCreatingCompletionProvider 
     */
    public ElementCreatingCompletionProvider() {
    }

    public int getAutoQueryTypes(JTextComponent component, String typedText) {
        return 0;
    }

    public CompletionTask createTask(int queryType, JTextComponent component) {
        if (queryType == COMPLETION_QUERY_TYPE)
            return new AsyncCompletionTask(new Query(), component);
        return null;
    }
    
    private static class Query extends AsyncCompletionQuery {
        private JTextComponent component;

        public Query () {}
        
        protected void prepareQuery(JTextComponent component) {
            this.component = component;
        }
        
        protected void query(CompletionResultSet resultSet, Document doc, int caretOffset) {
            if (!((JavaSyntaxSupport)Utilities.getSyntaxSupport(component)).isCompletionDisabled(caretOffset)) {
                List/*<CompletionItem>*/ results = queryImpl(doc, caretOffset);
                assert (results != null);
                resultSet.addAllItems(results);
            }
            resultSet.finish();
        }
        
    }
    
    public static List/*<CompletionItem>*/ queryImpl(Document document, int offset) {
        if (!(document instanceof BaseDocument))
            return Collections.EMPTY_LIST;
        
        BaseDocument doc = (BaseDocument) document;
        JMIUtils utils = JMIUtils.get(doc);

        utils.beginTrans(false);        
        try {
            NbJavaJMISyntaxSupport ssup = (NbJavaJMISyntaxSupport)doc.getSyntaxSupport();
            Feature f = ssup.getFeatureAtPos(offset, true);
            if (f instanceof ClassDefinition) {
                //check that the offset is in the body:
                //first of all, check it is not in the header:
                int headerEnd = JavaMetamodel.getManager().getElementPartPosition(f, ElementPartKindEnum.HEADER, -1).getEnd().getOffset();
                if (offset <= headerEnd)
                    return Collections.EMPTY_LIST;
                
                //second, check that the offset is after the opening '{':
                //XXX : ideally, the JMI should provide this information
                int nonWhite = Utilities.getFirstNonWhiteFwd(doc, headerEnd + 1, offset);
                
                if (nonWhite == -1)
                    return Collections.EMPTY_LIST;
                
                //check that offset is not after the ending '}':
                
                if (offset >= JavaMetamodel.getManager().getElementPosition(f).getEnd().getOffset())
                    return Collections.EMPTY_LIST;
                
                String prefix = getPrefix(doc, offset);
                
                ClassDefinition clazz = (ClassDefinition) f;
                
                List results = new ArrayList();
                List abstractMethods = new ArrayList();
                List commonMethods = new ArrayList();
                
                //1. Check for overridable methods:
                List methods = utils.findMethods(clazz, "", false, false, null, false, true, null, false, true); // NOI18N
                
                for (Iterator i = methods.iterator(); i.hasNext(); ) {
                    Method method = (Method) i.next();
                    
                    if (isOverridable(clazz, method) && !isOverriden(clazz, method)) {
                        OverrideMethodResultItem item = new OverrideMethodResultItem(clazz, method, doc, prefix);
                        
                        if (Modifier.isAbstract(method.getModifiers()))
                            abstractMethods.add(item);
                        else
                            commonMethods.add(item);
                    }
                }
                
                results.addAll(abstractMethods);
                results.addAll(commonMethods);
                
                //2. find all fields:
                List/*<Field>*/ fields = new ArrayList();
                Collection els = clazz.getFeatures();
                Object[] features = els.toArray();
                
                for (int cntr = 0; cntr < features.length; cntr++) {
                    if (features[cntr] instanceof Field) {
                        fields.add(features[cntr]);
                    }
                }
                
                //3. check for missing setters/getters.                
                for (Iterator i = fields.iterator(); i.hasNext(); ) {
                    Field field = (Field) i.next();
                    
                    int getterSetter = GeneratorUtils.checkForGetterSetter(field);
                    
                    if ((getterSetter & GeneratorUtils.GETTER) == 0) {
                        results.add(new SetterGetterResultItem(true, field, doc, offset, prefix));
                    }
                    if ((getterSetter & GeneratorUtils.SETTER) == 0) {
                        results.add(new SetterGetterResultItem(false, field, doc, offset, prefix));
                    }
                }
                return filterResult(results, prefix);
            }
        } catch (BadLocationException e) {
            ErrorManager.getDefault().notify(e);
        } finally {
            utils.endTrans(false);
        }
        
        return Collections.EMPTY_LIST;
    }
    
    private static String getPrefix(BaseDocument doc, int caretOffset) {
        String prefix = null;
        
        try {
            int[] identifierSpan = Utilities.getIdentifierBlock(doc, caretOffset);
            
            if (identifierSpan != null) {
                int start = identifierSpan[0];
                int end   = identifierSpan[1];
                
                if (end > caretOffset) 
                    end = caretOffset;
                
                prefix = doc.getText(start, end - start);
                
                //it seems that prefix may be "{", so removing all unneeded characters:
                int index = 0;
                
                while (index < prefix.length() && !Character.isJavaIdentifierStart(prefix.charAt(index)))
                    index++;
                
                if (index >= prefix.length())
                    return "";
                
                prefix = prefix.substring(index);
                
                index = 1;
                
                while (index < prefix.length() && Character.isJavaIdentifierPart(prefix.charAt(index)))
                    index++;
                
                return prefix.substring(0, index);
            }
        } catch (BadLocationException e) {
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, e);
        }
        
        if (prefix == null)
            prefix = "";
        
        return prefix;
    }
    
    private static abstract class CreateElementResultItem implements CompletionItem {
        
        public CreateElementResultItem() {
        }
        
        // CompletionItem implementation
        
        public static final String COMPLETION_SUBSTITUTE_TEXT= "completion-substitute-text"; // NOI18N
        
        static int substituteOffset = -1;
        static String toAdd;

        public void defaultAction(JTextComponent component) {
            int substOffset = substituteOffset;
            if (substOffset == -1)
                substOffset = component.getCaret().getDot();
            Completion.get().hideAll();
            create(substOffset);
        }
        
        public void processKeyEvent(KeyEvent evt) {
        }

        public CharSequence getSortText() {
            return toString();
        }

        public boolean instantSubstitution(JTextComponent c) {
            return false;
        }
        
        public CompletionTask createDocumentationTask() {
            return new AsyncCompletionTask(new DocQuery(this));
        }
        
        public CompletionTask createToolTipTask() {
            return null;
        }
        
        public abstract String getItemText();
        
        public abstract void create(int offset);
        
    }
    
    private static class SetterGetterResultItem extends CreateElementResultItem {
        
        private static NbSetterGetterPaintComponent setterGetterComponent = null;
        
        private boolean isGetter = false;
        private Field   field    = null;
        private String  methodName = null;
        
        private String fieldName;
        private String typeName;
        private Color  typeColor;
        private boolean deprecated;
        private int    modifiers;
        
        private String text;
        
        private Document document;
        private int      offset;
        private String   prefix;
        
        /**Has to be called from an MDR transaction
         */
        public SetterGetterResultItem(boolean isGetter, Field field, Document doc, int offset, String prefix) {
            this.isGetter = isGetter;
            this.field = field;
            this.methodName = GeneratorUtils.getGetterSetterName(field, isGetter);
            
            this.deprecated = field.isDeprecated();
            this.fieldName = field.getName();
            this.typeName = JMIUtils.getTypeName(field.getType(), false, false);
            this.typeColor = NbJMIResultItem.getTypeColor(field.getType());
            this.modifiers = modifiers;
            this.text = MessageFormat.format(NbBundle.getBundle(ElementCreatingCompletionProvider.class).getString("Create_getter_setter_for_field"), new Object[] {isGetter ? new Integer(0) : new Integer(1), methodName, field.getName()});
            
            this.document = doc;
            this.offset = offset;
            this.prefix = prefix;
        }
        
        public String getItemText() {
            return methodName;
        }
        
        public String toString() {
            return text;
        }
        
        public int getSortPriority() {
            return 300;
        }
        
        public CharSequence getInsertPrefix() {
            return getItemText();
        }
        
        public void create(int offset) {
            BaseDocument bdoc = (BaseDocument) document;
            Position position = null;
            
            bdoc.atomicLock();
            
            try {
                position = bdoc.createPosition(offset - prefix.length());
                document.remove(position.getOffset(), prefix.length());
                
                if (Utilities.isRowWhite(bdoc, position.getOffset())) {
                    int lineStart = Utilities.getRowStart(bdoc, position.getOffset());
                    int lineEnd   = Utilities.getRowEnd(bdoc, position.getOffset());
                    
                    document.remove(lineStart, lineEnd - lineStart);
                }
            } catch (BadLocationException e) {
                ErrorManager.getDefault().notify(e);
            } finally {
                bdoc.atomicUnlock();
            }
            
            JMIUtils utils = JMIUtils.get(bdoc);
            boolean fail = true;
            utils.beginTrans(true);
            try {
                GeneratorUtils.createGetterSetter(field, isGetter, position.getOffset());
                fail = false;
            } finally {
                utils.endTrans(fail);
            }
        }
        
        public int getPreferredWidth(Graphics g, Font defaultFont) {
            Component renderComponent = getPaintComponent();
            return renderComponent.getPreferredSize().width;
        }
        
        public void render(Graphics g, Font defaultFont, Color defaultColor,
        Color backgroundColor, int width, int height, boolean selected) {
            Component renderComponent = getPaintComponent();
            renderComponent.setFont(defaultFont);
            renderComponent.setForeground(defaultColor);
            renderComponent.setBackground(backgroundColor);
            renderComponent.setBounds(0, 0, width, height);
            ((NbJMIPaintComponent)renderComponent).paintComponent(g);
        }
        
        private Component getPaintComponent() {
            if (setterGetterComponent == null) {
                setterGetterComponent = new NbSetterGetterPaintComponent();
            }
            
            setterGetterComponent.setMethodName(methodName);
            setterGetterComponent.setFieldName(fieldName);
            setterGetterComponent.setTypeName(typeName);
            setterGetterComponent.setTypeColor(typeColor);
            setterGetterComponent.setDeprecated(deprecated);
            setterGetterComponent.setModifiers(modifiers);
            setterGetterComponent.setGetter(isGetter);
            
            return setterGetterComponent;
        }
        
    }
    
    private static Color METHOD_COLOR = Color.red.darker().darker();
    private static Color FIELD_COLOR = Color.blue.darker();
        
    private static class NbSetterGetterPaintComponent extends NbJMIPaintComponent.NbFieldPaintComponent {
        
        private String methodName;
        private boolean getter;
        
        public NbSetterGetterPaintComponent() {
            super(false);
        }
        
        protected void draw(Graphics g) {
            drawIcon(g, getIcon());
            
            //PENDING: resolve how to show this and make lookups into the Bundle
            drawString(g, NbBundle.getMessage(ElementCreatingCompletionProvider.class, getter ? "LBL_Create_Getter" : "LBL_Create_Setter") + " "); //NOI18N
            drawString(g, methodName, METHOD_COLOR);
            drawString(g, " " + NbBundle.getMessage(ElementCreatingCompletionProvider.class, "LBL_For_Field") + " "); // NOI18N
            
            boolean strike = isDeprecated();
            
            if ((getModifiers() & JavaCompletion.LOCAL_MEMBER_BIT) != 0){
                // it is local field, draw as bold
                drawString(g, getFieldName(), FIELD_COLOR, getDrawFont().deriveFont(Font.BOLD), strike);
            }else{
                drawString(g, getFieldName(), FIELD_COLOR , null, strike);
            }
            drawTypeName(g, getTypeName(), getTypeColor());
        }

        public void setMethodName(String methodName) {
            this.methodName = methodName;
        }

        public void setGetter(boolean getter) {
            this.getter = getter;
        }
        
    }

    private static class OverrideMethodResultItem extends CreateElementResultItem {
        
        private static NbOverrideMethodPaintComponent component;
        
        private Method method;
        private ClassDefinition where;
        private Document document;

        private List params = new ArrayList();
        private List excs = new ArrayList();
        private int modifiers;
        private String cfName, typeName;
        private Color typeColor;
        private boolean deprecated;
        
        private String methodName;
        private String text;
        
        private String prefix;
        
        private boolean isImplement;
        
        /**Should be always called inside an MDR transaction.
         */
        public OverrideMethodResultItem(ClassDefinition clazz, Method method, Document document, String prefix) {
            this.method = method;
            this.document = document;
            this.where = clazz;
            int modifiers = method.getModifiers();
            List params = new ArrayList();
            
            for (Iterator it = method.getParameters().iterator(); it.hasNext();) {
                Parameter prm = (Parameter) it.next();
                Type type = prm.getType();
                params.add(new NbJMIResultItem.ParamStr(type.getName(), JMIUtils.getTypeName(type, false, false), prm.getName(), prm.isVarArg(), NbJMIResultItem.getTypeColor(type)));
            }
            
            List excs = new ArrayList();
            
            for (Iterator it = method.getExceptions().iterator(); it.hasNext();) {
                JavaClass ex = (JavaClass) it.next();
                excs.add(new NbJMIResultItem.ExcStr(ex.getSimpleName(), NbJMIResultItem.getTypeColor(ex)));
            }
            
            Type methodType = method.getType();
            cfName = method.getName();
            this.modifiers = modifiers;
            typeName = JMIUtils.getTypeName(methodType, false, false);
            typeColor = NbJMIResultItem.getTypeColor(methodType);
            this.params = params;
            this.excs = excs;
            deprecated = method.isDeprecated();
            
            this.methodName = Modifier.toString(modifiers) + " " + method.getName() + getParameters(method.getParameters()) + ":" + methodType.getName(); // NOI18N
            this.text = MessageFormat.format(NbBundle.getBundle(ElementCreatingCompletionProvider.class).getString("Override_method"), new Object[] {getMethodName()});
            
            this.prefix = prefix;
            
            this.isImplement = Modifier.isAbstract(modifiers);
        }
        
        private String getParameters(List parameters) {
            StringBuffer result = new StringBuffer();
            
            result.append("("); // NOI18N      
            for (Iterator i = parameters.iterator(); i.hasNext(); ) {
                Parameter p = (Parameter) i.next();
                
                result.append(p.getType().getName());
                result.append(" "); // NOI18N
                result.append(p.getName());
                
                if (i.hasNext()) {
                    result.append(", "); // NOI18N
                }
            }
            result.append(")"); // NOI18N
            return result.toString();
        }
        
        private String getMethodName() {
            //TODO: type parameters:
            return methodName;
        }
        
        public String getItemText() {
            return cfName;
        }
        public String toString() {
            return text;
        }
        
        public void create(int offset) {
            BaseDocument bdoc = (BaseDocument) document;
            Position position = null;
            
            bdoc.atomicLock();
            
            try {
                position = bdoc.createPosition(offset - prefix.length());
                document.remove(position.getOffset(), prefix.length());
                
                if (Utilities.isRowWhite(bdoc, position.getOffset())) {
                    int lineStart = Utilities.getRowStart(bdoc, position.getOffset());
                    int lineEnd   = Utilities.getRowEnd(bdoc, position.getOffset());
                    
                    document.remove(lineStart, lineEnd - lineStart);
                }
            } catch (BadLocationException e) {
                ErrorManager.getDefault().notify(e);
            } finally {
                bdoc.atomicUnlock();
            }
            
            JMIUtils utils = JMIUtils.get(bdoc);
            boolean fail = true;
            utils.beginTrans(true);
            try {
                GeneratorUtils.overrideMethod(where, method, position.getOffset());
                fail = false;
            } finally {
                utils.endTrans(fail);
            }
        }

        public Component getPaintComponent(boolean isSelected) {
            if (component == null) {
                component = new NbOverrideMethodPaintComponent();
            }
            
            component.setSelected(isSelected);
            component.setFeatureName(cfName);
            component.setModifiers(modifiers);
            component.setTypeName(typeName);
            component.setTypeColor(typeColor);
            component.setParams(params);
            component.setExceptions(excs);
            component.setDeprecated(deprecated);

            return component;
        }

        public int getPreferredWidth(Graphics g, Font defaultFont) {
            Component renderComponent = getPaintComponent(false);
            return renderComponent.getPreferredSize().width;
        }
        
        public void render(Graphics g, Font defaultFont, Color defaultColor,
        Color backgroundColor, int width, int height, boolean selected) {
            Component renderComponent = getPaintComponent(selected);
            renderComponent.setFont(defaultFont);
            renderComponent.setForeground(defaultColor);
            renderComponent.setBackground(backgroundColor);
            renderComponent.setBounds(0, 0, width, height);
            ((NbJMIPaintComponent)renderComponent).paintComponent(g);
        }
        
        public int getSortPriority() {
            return isImplement ? 0 : 500;
        }
        
        public CharSequence getInsertPrefix() {
            return getItemText();
        }
        
    }
    
    private static class NbOverrideMethodPaintComponent extends NbJMIPaintComponent.NbMethodPaintComponent {
        public NbOverrideMethodPaintComponent () {}
        
        protected void draw(java.awt.Graphics g) {
            boolean strike = isDeprecated();
            drawIcon(g, getIcon());

            if ((getCFModifiers() & JavaCompletion.LOCAL_MEMBER_BIT) != 0){
                drawString(g, getCFName(), METHOD_COLOR , getDrawFont().deriveFont(Font.BOLD), strike);
            }else{
                drawString(g, getCFName(), METHOD_COLOR, null, strike);
            }
            drawParameterList(g, getParamList());
            if ((getCFModifiers() & Modifier.ABSTRACT) != 0)
                drawString(g, NbBundle.getBundle(ElementCreatingCompletionProvider.class).getString("Implement_CC_suffix"));
            else
                drawString(g, NbBundle.getBundle(ElementCreatingCompletionProvider.class).getString("Override_CC_suffix"));
            drawTypeName(g, getTypeName(), getTypeColor());
        }
        
    }
    
    /**
     * Determines if the given method is overriden in the hierarchy.
     *
     * @param  method  method for which overriding methods are looked for
     * @return true, if given method is overriden in subclasses
     */
    private static boolean isOverriden(ClassDefinition root, Method m) {
        List params = m.getParameters();
        String name = m.getName();
        List paramTypes = new ArrayList();
        for (Iterator iter = params.iterator(); iter.hasNext(); ) {
            paramTypes.add(((Parameter)iter.next()).getType());
        }
        
        return root.getMethod(name, paramTypes, false) != null;
    }
    
    //Copied from JMIInheritanceSupport:
    /** Determines if the given method can be overriden in the root class. */
    private static boolean isOverridable(ClassDefinition root, Method m) {
        if (!isAccessibleMethod(root, m)) {
            return false;
        }
        
        int mods = m.getModifiers();
        return !(Modifier.isFinal(mods) || Modifier.isStatic(mods));
    }

    /** Determines, if the given method is accessible from the root class. */
    private static boolean isAccessibleMethod(ClassDefinition root, Method m) {
        ClassDefinition cls = m.getDeclaringClass();
        int modifs = cls instanceof JavaClass ? ((JavaClass)cls).getModifiers() : 0;
        
        if ((modifs & Modifier.PRIVATE) != 0 ||
            ((modifs & (Modifier.PUBLIC | Modifier.PROTECTED | Modifier.PRIVATE)) == 0 &&
             !isSamePackage(root, cls))) {
            return false;
        }
        if ((cls instanceof JavaClass) && ((JavaClass)cls).isInterface())
            return true;
        modifs = m.getModifiers();
        if ((modifs & Modifier.PRIVATE) != 0) {
            return false;
        }
        if ((modifs & (Modifier.PUBLIC | Modifier.PROTECTED)) != 0)
            return true;
        return isSamePackage(root, cls);
    }
    
    /** Deteremines if the class is in the same package as the root class */
    private static boolean isSamePackage(ClassDefinition root, ClassDefinition cd) {
        String p1 = root.getResource().getPackageName();
        String p2 = cd.getResource().getPackageName();
        return p1 == null ? p2 == null : p1.equals(p2);
    }
    
    private static List filterResult(List list, String prefix) {
        if (prefix.length() == 0)
            return list;
                    
        List result = new ArrayList();
        
        for (Iterator i = list.iterator(); i.hasNext(); ) {
            Object item = i.next();
            
            if (item instanceof CreateElementResultItem) {
                CreateElementResultItem completionItem = (CreateElementResultItem) item;
                
                if (completionItem.getItemText().startsWith(prefix))
                    result.add(completionItem);
            } else {
                result.add(item);
            }
        }
        
        return result;
    }
    
    static class DocQuery extends AsyncCompletionQuery {
        
        Object item;
        
        DocQuery(Object item) {
            this.item = item;
        }

        protected void query(CompletionResultSet resultSet, Document doc, int caretOffset) {
            DocItem dItem = null;
            
            if (item instanceof OverrideMethodResultItem) {
                OverrideMethodResultItem override = (OverrideMethodResultItem) item;
                
                String pattern = null;
                
                if (override.isImplement)
                    pattern = NbBundle.getBundle(ElementCreatingCompletionProvider.class).getString("JDOC_IMPLEMENT");
                else
                    pattern = NbBundle.getBundle(ElementCreatingCompletionProvider.class).getString("JDOC_OVERRIDE");
                
                dItem = new DocItem(MessageFormat.format(pattern, new Object[] {override.methodName}));
            }
            
            if (item instanceof SetterGetterResultItem) {
                SetterGetterResultItem setter = (SetterGetterResultItem) item;
                
                String pattern = NbBundle.getBundle(ElementCreatingCompletionProvider.class).getString("JDOC_CREATE_GETTER_SETTER");
                
                dItem = new DocItem(MessageFormat.format(pattern, new Object[] {
                    setter.isGetter ? new Integer(0) : new Integer(1),
                            setter.methodName,
                            setter.fieldName
                }));
            }
            
            if (dItem != null) {
                resultSet.setDocumentation(dItem);
            }
            resultSet.finish();
        }
        
        private static class DocItem implements CompletionDocumentation {
            
            private String text;
            
            public DocItem(String text) {
                this.text = text;
            }
            
            public CompletionDocumentation resolveLink(String link) {
                return null;
            }
            
            public String getText() {
                return text;
            }
            
            public URL getURL() {
                return null;
            }
            
            public Action getGotoSourceAction() {
                return null;
            }
            
        }
        
    }
}
