/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.java;

import java.util.ArrayList;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.SyntaxSupport;
import org.netbeans.editor.TokenContextPath;
import org.netbeans.editor.TokenID;
import org.netbeans.editor.TokenProcessor;
import org.netbeans.editor.Utilities;
import org.netbeans.editor.ext.Completion;
import org.netbeans.editor.ext.CompletionQuery;
import org.netbeans.editor.ext.ExtSyntaxSupport;
import org.netbeans.editor.ext.ExtUtilities;
import org.netbeans.editor.ext.java.JCClass;
import org.netbeans.editor.ext.java.JCType;
import org.netbeans.editor.ext.java.JCConstructor;
import org.netbeans.editor.ext.java.JCMethod;
import org.netbeans.editor.ext.java.JCField;
import org.netbeans.editor.ext.java.JCParameter;
import org.netbeans.editor.ext.java.JCFinder;
import org.netbeans.editor.ext.java.JCResultItem;
import org.netbeans.editor.ext.java.JavaTokenContext;
import org.netbeans.modules.editor.NbEditorUtilities;
import org.netbeans.spi.java.classpath.support.ClassPathSupport;
import org.openide.src.ClassElement;
import org.openide.src.ConstructorElement;
import org.openide.src.MethodElement;
import org.openide.src.FieldElement;
import org.openide.src.Type;
import org.openide.src.MethodParameter;
import org.openide.filesystems.FileObject;

/**
* Extended Java Completion support
*
* @author Miloslav Metelka
*/
public class JCExtension {

    public static boolean equals(JCClass cls1, ClassElement cls2) {
        return cls1.getFullName().equals(cls2.getName().getFullName());
    }

    public static boolean equals(JCType typ1, Type typ2) {
        // Get array depth of the type
        int arrayDepth = 0;
        while(typ2.isArray()) {
            arrayDepth++;
            typ2 = typ2.getElementType();
        }

        String fn = typ2.isPrimitive() ? typ2.getFullString()
                    : typ2.getClassName().getFullName();
        return (typ1.getArrayDepth() == arrayDepth)
               && typ1.getClazz().getFullName().equals(fn);
    }

    public static boolean equals(JCField fld1, FieldElement fld2) {
        return fld1.getName().equals(fld2.getName().getFullName())
               && equals(fld1.getType(), fld2.getType());
    }

    public static boolean equals(JCConstructor ctr1, ConstructorElement ctr2) {
        JCParameter[] parms1 = ctr1.getParameters();
        MethodParameter[] parms2 = ctr2.getParameters();
        if (parms2 == null || parms1.length != parms2.length) {
            return false;
        }
        for (int i = parms1.length - 1; i >= 0; i--) {
            if (!equals(parms1[i].getType(), parms2[i].getType())) {
                return false;
            }
        }
        return true;
    }

    public static boolean equals(JCMethod mtd1, MethodElement mtd2) {
        return mtd1.getName().equals(mtd2.getName().getFullName())
               && equals((JCConstructor)mtd1, (ConstructorElement)mtd2);
    }

    public static FieldElement findFieldElement(JCField fld, ClassElement ce) {
        FieldElement[] fes = ce.getFields();
        if (fes != null) {
            for (int i = 0; i < fes.length; i++) {
                if (equals(fld, fes[i])) {
                    return fes[i];
                }
            }
        }
        return null;
    }

    public static ConstructorElement findConstructorElement(JCConstructor ctr, ClassElement ce) {
        ConstructorElement[] ces = ce.getConstructors();
        if (ces != null) {
            for (int i = 0; i < ces.length; i++) {
                if (equals(ctr, ces[i])) {
                    return ces[i];
                }
            }
        }
        return null;
    }

    public static MethodElement findMethodElement(JCMethod mtd, ClassElement ce) {
        MethodElement[] mes = ce.getMethods();
        if (mes != null) {
            for (int i = 0; i < mes.length; i++) {
                if (equals(mtd, mes[i])) {
                    return mes[i];
                }
            }
        }
        return null;
    }

    /**
     * Returns one classpath which is merge of SOURCE, COMPILE and BOOT
     * classpaths of the given file object.
     */
    public static ClassPath getFullClassPath(FileObject context) {
        ArrayList l = new ArrayList();
        ClassPath cp = ClassPath.getClassPath(context, ClassPath.SOURCE);
        if (cp != null) {
            l.add(cp);
        }
        cp = ClassPath.getClassPath(context, ClassPath.COMPILE);
        if (cp != null) {
            l.add(cp);
        }
        cp = ClassPath.getClassPath(context, ClassPath.BOOT);
        if (cp != null) {
            l.add(cp);
        }
        return ClassPathSupport.createProxyClassPath((ClassPath[])l.toArray(new ClassPath[l.size()]));
    }
    

    /** Finds an inner class with the same name as the resultClassItem */
    static JCClass findResultInnerClass(JCFinder finder, JCClass topClass, String innerClsName, Document doc){
        if (topClass == null || innerClsName == null) return null;

        ClassElement ce = ClassElement.forName(topClass.getFullName(),
                NbEditorUtilities.getDataObject (doc).getPrimaryFile());
        if (ce == null) return null;
        ClassElement innerClasses[] = ce.getClasses();
        
        if (innerClsName != null){
            for (int i=0; i<innerClasses.length; i++){
                if (!(innerClsName.equals(innerClasses[i].getName().getName()) || innerClsName.equals(innerClasses[i].getName().getFullName()))) continue;
                JCClass innerCls = finder.getExactClass(innerClasses[i].getName().getFullName());
                if (innerCls!=null){
                    return innerCls;
                }
            }
        }
        return null;
    }


    static int findEndOfMethod(JTextComponent textComp, int startPos){
        return findEndOfMethod((BaseDocument)textComp.getDocument(), startPos);
    }

    static int findEndOfMethod(BaseDocument doc, int startPos){
        ExtSyntaxSupport ssup = (ExtSyntaxSupport)doc.getSyntaxSupport();
        ClosingParenProcessor tp = new ClosingParenProcessor();
        try {
            ssup.tokenizeText(tp, startPos, doc.getLength(), false);
        } catch (BadLocationException ex) {
        }
        return tp.getClosingParenOffset();
    }
    
    private static final class ClosingParenProcessor implements TokenProcessor {
        
        private int bufferStartOffset;
        private int level = 0;
        private int closingParenOffset = -1;
        
        public int eot(int offset) {
            return 0;
        }
        
        public void nextBuffer(char[] buffer, int offset, int len, int startPos, int preScan, boolean lastBuffer) {
            bufferStartOffset = startPos - offset;
        }
        
        public boolean token(TokenID tokenID, TokenContextPath tokenContextPath, int tokenBufferOffset, int tokenLength) {
            int offset = bufferStartOffset + tokenBufferOffset;
            switch (tokenID.getNumericID()) {
                case JavaTokenContext.LPAREN_ID:
                    level++;
                    break;
                case JavaTokenContext.RPAREN_ID:
                    if (level == 0) {
                        closingParenOffset = offset + 1;
                        return false;
                    }
                    level--;
                    break;
                case JavaTokenContext.SEMICOLON_ID:
                    return false;
            }
            return true;
        }
        
        public int getClosingParenOffset() {
            return closingParenOffset;
        }
        
    }
    
    private static Object findImportedItem(NbJavaSyntaxSupport nbJavaSup, CompletionQuery.Result result){
        return findImportedItem(nbJavaSup, result, null);
    }
    
    private static Object findImportedItem(NbJavaSyntaxSupport nbJavaSup, CompletionQuery.Result result, String wordAtCaret){
        nbJavaSup.refreshJavaImport();
        String sourcePkg = nbJavaSup.getPackage();
        for (int x=0; x<result.getData().size(); x++){
            Object itemObj = result.getData().get(x);
            if (itemObj instanceof JCResultItem){
                itemObj = ((JCResultItem)itemObj).getAssociatedObject();
            }
            
            if (itemObj instanceof JCClass) {
                JCClass cls = (JCClass)itemObj;
                String itemPkg = cls.getPackageName();
                if(nbJavaSup.isImported(cls) || sourcePkg.equals(itemPkg) ){
                    if (wordAtCaret!=null && !wordAtCaret.equals(cls.getName())) continue;
                    Object itmx = result.getData().get(x);
                    if (itmx instanceof JCResultItem){
                        itmx = ((JCResultItem)itmx).getAssociatedObject();
                    }
                    
                    return itmx;
                }
            }
        }
        return null;
    }
    
    /** Completion query result can contain more ambiguous items. This method will scan all innerclasses, whether 
     *  the item is not declared in inner class */
    private static Object findResultInnerItem(NbJavaSyntaxSupport nbJavaSup, CompletionQuery.Result result, Document doc){
        JCClass topClass = nbJavaSup.getTopClass();
        
        if (topClass!=null){
            ClassElement ce = ClassElement.forName(topClass.getFullName(),
                    NbEditorUtilities.getDataObject (doc).getPrimaryFile());
            if (ce!=null){
                ClassElement innerClasses[] = ce.getClasses();
                for (int i = 0; i<result.getData().size();  i++){
                    
                    Object itmx = result.getData().get(i);
                    if (itmx instanceof JCResultItem){
                        itmx = ((JCResultItem)itmx).getAssociatedObject();
                    }
                    
                    
                    if (itmx instanceof JCClass){
                        return findResultInnerClass(nbJavaSup.getFinder(), topClass, ((JCClass)(itmx)).getFullName(),doc);
                    }

                    if (itmx instanceof JCMethod){
                        JCMethod mtd = (JCMethod)itmx;
                        JCClass clz = mtd.getClazz();
                        if (clz == null) continue;
                        JCClass innerClass = findResultInnerClass(nbJavaSup.getFinder(), topClass, clz.getFullName(),doc);
                        if (innerClass == null) continue;
                        JCMethod methods[] = innerClass.getMethods();
                        for (int j=0; j<methods.length; j++){
                            if (methods[j].equals(mtd)){
                                return methods[j];
                            }
                        }
                    }
                    
                    if (itmx instanceof JCConstructor){
                        JCConstructor constructor = (JCConstructor)itmx;
                        JCClass clz = constructor.getClazz();
                        if (clz == null) continue;
                        JCClass innerClass = findResultInnerClass(nbJavaSup.getFinder(), topClass, clz.getFullName(),doc);
                        if (innerClass == null) continue;
                        JCConstructor constructors[] = innerClass.getConstructors();
                        for (int j=0; j<constructors.length; j++){
                            if (constructors[j].equals(constructor)){
                                return constructors[j];
                            }
                        }
                        return innerClass;
                    }
                    
                    if (itmx instanceof JCField){
                        JCField fld = (JCField)itmx;
                        JCClass clz = fld.getClazz();
                        if (clz == null) continue;
                        JCClass innerClass = findResultInnerClass(nbJavaSup.getFinder(), topClass, clz.getFullName(),doc);
                        if (innerClass == null) continue;
                        JCField fields[] = innerClass.getFields();
                        for (int j=0; j<fields.length; j++){
                            if (fields[j].equals(fld)){
                                return fields[j];
                            }
                        }
                    }
                    
                }
            }
        }
        return null;
    }
    
    public static Object findItemAtCaretPos(JTextComponent target){
        Object item = null;
        Completion completion = ExtUtilities.getCompletion(target);
        SyntaxSupport sup = Utilities.getSyntaxSupport(target);
        NbJavaSyntaxSupport nbJavaSup = (NbJavaSyntaxSupport)sup.get(NbJavaSyntaxSupport.class);

        if (completion != null) {
            if (completion.isPaneVisible()) { // completion pane visible
                item = completion.getSelectedValue();
                if (item instanceof JCResultItem){
                    item = ((JCResultItem)item).getAssociatedObject();
                }

                if (item != null) {
                    return item;
                }
            } else { // pane not visible
                try {
                    int dotPos = target.getCaret().getDot();
                    BaseDocument doc = (BaseDocument)target.getDocument();
                    int[] idFunBlk = NbEditorUtilities.getIdentifierAndMethodBlock(doc, dotPos);

                    if ( (idFunBlk == null)
                    || ((dotPos>0) && (doc.getChars(dotPos-1, 1)[0] == '(') && (dotPos == idFunBlk[0]))){
                        idFunBlk = new int[] { dotPos, dotPos };
                    }

                    for (int ind = idFunBlk.length - 1; ind >= 1; ind--) {
                        CompletionQuery.Result result = completion.getQuery().query(
                                                            target, idFunBlk[ind], sup);
                        if (result != null && result.getData().size() > 0) {
                            Object itm = result.getData().get(0);
                            if (itm instanceof JCResultItem){
                                itm = ((JCResultItem)itm).getAssociatedObject();
                            }

                            if ( ((itm instanceof JCConstructor) || (itm instanceof JCMethod) )&& (result.getData().size() > 1) ){
                                // It is overloaded method, lets check for the right one
                                int endOfMethod = findEndOfMethod(target, idFunBlk[ind]);

                                if (endOfMethod>-1){
                                    CompletionQuery.Result resultx = completion.getQuery().query(target, endOfMethod, sup);
                                    if (resultx != null && resultx.getData().size() > 0) {
                                        Object itmx = resultx.getData().get(0);
                                        if (itmx instanceof JCResultItem){
                                            itmx = ((JCResultItem)itmx).getAssociatedObject();
                                        }

                                        return itmx;
                                    }
                                }

                            } 

                            // check whether the item is not declared in the inner class
                            Object innerClassItem = findResultInnerItem(nbJavaSup, result, doc);
                            if (innerClassItem !=null){
                                return innerClassItem;
                            }

                            Object importedItem = findImportedItem(nbJavaSup, result);
                            if (importedItem != null) return importedItem;
                            Object itmx = result.getData().get(0);
                            if (itmx instanceof JCResultItem){
                                return ((JCResultItem)itmx).getAssociatedObject();
                            }

                        }
                    }
                } catch (BadLocationException e) {
                }
            }
        }

        // Complete the messages
        return null;

    }
    

}
