/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.editor.java;

import java.lang.reflect.Modifier;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;

import org.netbeans.editor.SyntaxSupport;
import org.netbeans.editor.Utilities;
import org.netbeans.editor.ext.Completion;
import org.netbeans.editor.ext.CompletionJavaDoc;
import org.netbeans.editor.ext.ExtEditorUI;
import org.netbeans.editor.ext.JavaDocPane;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.editor.NbEditorUtilities;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.URLMapper;
import org.openide.loaders.DataObject;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;
import org.openide.util.RequestProcessor;

/**
 *  Support for javadoc in code completion working with java JMI interfaces
 *
 *  @author Martin Roskanin, Dusan Balek
 *  @since 04/2004
 */
class NbJMICompletionJavaDoc extends NbCompletionJavaDoc {

    private boolean goToSourceEnabled = false;    
    private String lastBase="";    
    private JavaDocPane pane;
    private JMIUtils jmiUtils;
    private NbJavaJMISyntaxSupport jmiSup;
    private Comparator contentComparator;
    private Map membersCache = new HashMap();
    
    private static String ARRAY_LENGTH_FIELD_JAVADOC =
        NbBundle.getBundle(NbJMICompletionJavaDoc.class).getString("array_length_field_javadoc"); //NOI18N

    private static String CLASS_CONSTANT_JAVADOC = 
        NbBundle.getBundle(NbJMICompletionJavaDoc.class).getString("class_constant_javadoc"); //NOI18N


    NbJMICompletionJavaDoc(ExtEditorUI extEditorUI) {
        super(extEditorUI);
    }

    private JMIUtils getJMIUtils() {
        if (jmiUtils == null)
            jmiUtils = JMIUtils.get(extEditorUI.getDocument());
        return jmiUtils;
    }

    protected NbJavaJMISyntaxSupport getJMISyntaxSupport() {
        if (jmiSup == null) {
            SyntaxSupport sup = Utilities.getSyntaxSupport(extEditorUI.getComponent());
            jmiSup = (NbJavaJMISyntaxSupport)sup.get(NbJavaJMISyntaxSupport.class);
        }
        return jmiSup;
    }

    protected Object getCurrentContent() {
        return convert(super.getCurrentContent());
    }

    protected Comparator getContentComparator() {
        if (contentComparator == null)
            contentComparator = new ContentComparator();
        return contentComparator;
    }

    protected String findProperClass(String name, String pkgName) {
        JavaClass jc = getJMIUtils().getExactClass(name, pkgName);
        return jc != null ? jc.getName() : null;
    }

    protected ParsingThread setInRequestProcessorThread(final Object content){
        JMIParsingThread pt = new JMIParsingThread(content);
        RequestProcessor.getDefault().post(pt);
        return pt;
    }

    /** Returns true if javadoc is mounted in FS */
    public boolean isExternalJavaDocMounted(){
        Object currentContent = getCurrentContent();
        if (currentContent instanceof URL || currentContent instanceof String){
            try{
                if (lastBase == null || lastBase.length() == 0) return false;
                FileObject fo = URLMapper.findFileObject(new URL(lastBase));
                if (fo != null) return true;
            }catch(MalformedURLException mue){
                mue.printStackTrace();
                return false;
            }
        }
        URL urls[] = getJMISyntaxSupport().getJavaDocURLs(currentContent);
        return (urls == null || urls.length < 1) ? false : true;
    }
    
    /** Opens source of current javadoc in editor */
    public void goToSource(){
        Object currentContent = getCurrentContent();
        if (getJMISyntaxSupport().openSource(currentContent, true) == null){
            extEditorUI.getCompletion().setPaneVisible(false);
        }
    }

    /** Returns JavaDoc popup pane */
    public JavaDocPane getJavaDocPane(){
        Completion completion = extEditorUI.getCompletion();
        if (completion != null){
            JavaDocPane jdp = completion.getJDCPopupPanel().getJavaDocPane();
            if (jdp instanceof NbJMIScrollJavaDocPane)return jdp;
        }
        
        if (pane == null){
            pane = new NbJMIScrollJavaDocPane(extEditorUI);
        }
        return pane;
     }

    private CompletionJavaDoc.JavaDocTagItem[] getJavaDocTags(List jdt){
        CompletionJavaDoc.JavaDocTagItem ret [] = new CompletionJavaDoc.JavaDocTagItem[jdt.size()];
        int i = 0;
        for (Iterator it = jdt.iterator(); it.hasNext(); i++) {
            TagValue tv = (TagValue) it.next();
            ret[i] = new JavaDocTagItemImpl(tv.getDefinition().getName(), tv.getValue());
        }
        return ret;
    }

    protected boolean isGoToSourceEnabled(){
        return goToSourceEnabled;
    }
    
    /** Opens javadoc in external browser */
    public void openInExternalBrowser(){
        Object currentContent = getCurrentContent();

        if (currentContent instanceof URL){
            org.openide.awt.HtmlBrowser.URLDisplayer.getDefault().showURL((URL)currentContent);
        }else if (currentContent instanceof String){
            URL url = mergeRelLink(lastBase,(String)currentContent);
            if (url!=null){
                org.openide.awt.HtmlBrowser.URLDisplayer.getDefault().showURL(url);
            }
        }
        
        URL[] urls = getJMISyntaxSupport().getJavaDocURLs(currentContent);

        if (urls != null && urls.length > 0) {
            org.openide.awt.HtmlBrowser.URLDisplayer.getDefault().showURL(urls[0]); // show first URL
        }
    }
    
    private List parseMethodTypes(String parameters, ClassDefinition cls){
        ArrayList ret = new ArrayList();
        if (parameters == null) return ret;
        StringTokenizer st = new StringTokenizer(parameters, ","); //NOI18N
        while (st.hasMoreTokens()) {
            String param = st.nextToken();
            param = param.trim();
            param = (param.indexOf(" ") > 0) ? param.substring(0, param.indexOf(" ")) : param; //NOI18N
            int arrDepth = 0;
            Type typ = null;
            if (param.indexOf("[") > 0) { // NOI18N
                int idx = param.indexOf("["); // NOI18N
                param = param.substring(0, idx); //NOI18N
                while (idx >= 0) {
                    arrDepth++;
                    idx = param.indexOf("[", idx); // NOI18N
                }
            }
            if (param.indexOf(".") < 0) { // NOI18N
                typ = getJMISyntaxSupport().getTypeFromName(param, true, cls instanceof JavaClass ? (JavaClass)cls : null, true);
            } else {
                typ = getJMIUtils().getExactClass(param);
                if (typ == null)
                    typ = getJMIUtils().getExactClass(param, getJMIUtils().getPackageName(cls));
            }
            if (typ != null) {
                while (arrDepth > 0) {
                    typ = getJMIUtils().resolveArray(typ);
                    arrDepth--;
                }
            }
            if (typ == null)
                return null;
            ret.add(typ);
        }

        return ret;
    }

    public Object parseLink(String link, String clsFQN, String pkgName) {
        return parseLink(link, getJMIUtils().getExactClass(clsFQN));
    }

    public Object parseLink(String link, ClassDefinition cls) {
        link = link.trim();
        ClassDefinition linkClass = null;

        getJMIUtils().beginTrans(false);
        try {
            if (cls != null && !cls.isValid())
                cls = null;
            if (link.indexOf("#") > -1 ) { //NOI18N
                if (link.startsWith("#")) { //NOI18N
                    /* Referencing a member of the current class i.e:
                     * @see  #field
                     * @see  #method(Type, Type,...)
                     * @see  #method(Type argname, Type argname,...)
                     */
                    if (cls == null)
                        return null;
                    linkClass = cls;
                } else {
                    /* Referencing another class in the current or imported packages
                     * @see  Class#field
                     * @see  Class#method(Type, Type,...)
                     * @see  Class#method(Type argname, Type argname,...)
                     * @see  package.Class#field
                     * @see  package.Class#method(Type, Type,...)
                     * @see  package.Class#method(Type argname, Type argname,...)
                     */
                    String refCls = link.substring(0, link.indexOf("#")); //NOI18N
                    if (refCls.indexOf(".") < 0){ // NOI18N
                        Type typ = getJMISyntaxSupport().getTypeFromName(refCls, true, cls instanceof JavaClass ? (JavaClass)cls : null, true);
                        if (typ instanceof ClassDefinition)
                            linkClass = (ClassDefinition)typ;
                    } else {
                        if (refCls.endsWith(".html")) { //NOI18N
                            refCls = urlToFqn(refCls);
                        }
                        linkClass = getJMIUtils().getExactClass(refCls);
                        if (linkClass == null)
                            linkClass = getJMIUtils().getExactClass(refCls, getJMIUtils().getPackageName(cls));
                    }
                    if (linkClass == null)
                        return null;
                }

                linkClass = JMIUtils.getSourceElementIfExists(linkClass);

                String memberLink = link.substring(link.indexOf("#")+1) ;  //NOI18N
                if (link.indexOf("(")>0){ //NOI18N
                //method or constructor
                    String memberName = memberLink.substring(0,memberLink.indexOf("(")); //NOI18N
                    String memberParams = null;
                    if (link.indexOf(")")>0){ //NOI18N
                        memberParams = link.substring(link.indexOf("(")+1,link.indexOf(")"));  //NOI18N
                    }
                    List params = parseMethodTypes(memberParams, cls);
                    if (params == null)
                        return null;
                    if (memberName.equals(linkClass instanceof JavaClass ? ((JavaClass)linkClass).getSimpleName() : linkClass.getName()))
                        return linkClass.getConstructor(params, true);
                    else
                        return linkClass.getMethod(memberName, params, true);
                } else {
                //field or method or constructor
                    String memberName = (memberLink.indexOf(" ")>0) ? memberLink.substring(0,memberLink.indexOf(" ")) : memberLink; //NOI18N
                    return findMember(linkClass, memberName, cls);
                }
            } else {
                /* no member available, it can be package or class i.e:
                 * @see  Class
                 * @see  package.Class
                 * @see  package
                 */
                String refCls = (link.indexOf(" ")>0) ? link.substring(0, link.indexOf(" ")) : link; //NOI18N
                if (refCls.indexOf(".") < 0){ // NOI18N
                    Type typ = getJMISyntaxSupport().getTypeFromName(refCls, true, cls instanceof JavaClass ? (JavaClass)cls : null, true);
                    if (typ instanceof ClassDefinition)
                        linkClass = (ClassDefinition)typ;
                } else {
                    if (refCls.endsWith(".html")) { //NOI18N
                        refCls = urlToFqn(refCls);
                    }
                    linkClass = getJMIUtils().getExactClass(refCls);
                    if (linkClass == null)
                        linkClass = getJMIUtils().getExactClass(refCls, getJMIUtils().getPackageName(cls));
                }
                if (linkClass != null)
                    return JMIUtils.getSourceElementIfExists(linkClass);
                return getJMIUtils().getExactPackage(refCls);
            }
        } finally {
            getJMIUtils().endTrans(false);
        }
    }
    
    /** 
     * jbecicka:
     * javadoc contains links in form ../../java/lang/Object.html
     * I don't understand why this link needs to be converted to JavaClass.
     * JavaClass is later converted back to url
     */
    private static String urlToFqn(String url) {
        // "../.." must be removed to resolve class correctly
        url = url.replaceAll("(\\.\\./)*","");          //NOI18N
        
        //".html" must be removed
        url = url.substring(0,url.indexOf(".html")); //NOI18N
        
        //dots must be here instead of slashes
        url = url.replace('/','.');                     //NOI18N
        return url;
    }
    
    private Object findMember(ClassDefinition linkClass, String memberName, ClassDefinition cls) {
        if (!membersCache.containsKey(linkClass)) {
            Iterator it = getJMIUtils().findFeatures(linkClass, memberName, true, false, cls instanceof JavaClass ? (JavaClass)cls : null, false, false, null, false, false, false, false).iterator();
            return it.hasNext() ? it.next() : null;
        }
        List ret = (List)membersCache.get(linkClass);
        if (ret == null) {
            ret = getJMIUtils().findFeatures(linkClass, "", false, false, cls instanceof JavaClass ? (JavaClass)cls : null, false, false, null, false, false, false, false); //NOI18N
            membersCache.put(linkClass, ret);
        }
        for (Iterator it = ret.iterator(); it.hasNext();) {
            Feature f = (Feature)it.next();
            if (memberName.equals(f instanceof JavaClass ? ((JavaClass)f).getSimpleName() : f.getName()))
                return f;
        }
        return null;
    }
    
     /** Prepares raw javadoc content with given javadoc parameters
     *  @param cls root class
     *  @param member String representation of field or method
     *  @param content raw javadoc content text
     *  @param tags javadoc tag items array
     */
    public String prepareJavaDocContent(ClassDefinition cls, String member, String content, CompletionJavaDoc.JavaDocTagItem tags[]){
        JMIUtils jmiUtils = getJMIUtils();
        String pkgName = jmiUtils.getPackageName(cls);
        return prepareJavaDocContent(cls.getName(), pkgName, member, content, tags);
    }

    private Object convert(Object content) {
        if ((content instanceof NbJMIResultItem)){
            Object obj = ((NbJMIResultItem)content).getAssociatedObject();
            if (obj!=null){
                content = obj;
            }
        }
        getJMIUtils().beginTrans(false);
        try {
            if (content instanceof Element && !((Element)content).isValid())
                return null;
            if (content instanceof Feature)
                content = JMIUtils.getDefintion((Feature)content);
            if (content instanceof ClassDefinition)
                content = JMIUtils.getSourceElementIfExists((ClassDefinition)content);
            return content;
        } finally {
            getJMIUtils().endTrans(false);
        }
    }

    private class ContentComparator implements Comparator {
        public ContentComparator () {}
        
        public int compare(Object o1, Object o2) {
            Object obj1 = convert(o1);
            Object obj2 = convert(o2);
            return obj1 != null && obj1.equals(obj2) ? 0 : 1;
        }
    }


    class JMIParsingThread extends ParsingThread {
        
        public JMIParsingThread(Object content){
            super(content);
        }
        
        private String wrapClass(JavaClass clazz) {
            if (clazz==null || clazz.getName().length()==0) return "";
            URL[] urls = getJMISyntaxSupport().getJavaDocURLs(clazz);
            if (urls.length > 0 && urls[0]!=null){
                return "<font size='+0'><b><A href='"+urls[0].toString()+"'>"+clazz.getName()+"</A></b></font>"; // NOI18N
            }
            return "";
        }
        
        /** Retireves class from javadoc URL and wraps it to the anchor tag format */
        private  String wrapClass(String url){
            if (url==null) return "";
            String parent = url;
            int hashIndex = parent.lastIndexOf("#"); //NOI18N
            if (hashIndex>0){
                parent = parent.substring(0, hashIndex);
            }
            
            StringBuffer sb  = new StringBuffer(url);
            int htmlIndex = sb.indexOf(".html"); //NOI18N
            if (htmlIndex>0){
                sb.delete(htmlIndex, sb.length());
            }
            
            for (int i=0; i<sb.length()-1; i++){
                if (sb.charAt(i)=='/'){
                    String subStr = sb.substring(i+1);
                    subStr = subStr.replace('/','.');
                    if (getJMIUtils().getExactClass(subStr) != null){
                        return "<font size='+0'><b><A href='"+parent+"'>"+subStr+"</A></b></font>"; //NOI18N
                    }
                }
            }
            return "";
        }
        
        private boolean tryMountedJavaDoc(URL url){
            if (url==null){
                URL[] urls = getJMISyntaxSupport().getJavaDocURLs(content);
                if (urls.length > 0 && urls[0]!=null){
                    url = urls[0];
                }else{
                    return false;
                }
            }
            String urlStr = url.toString();

            ClassDefinition clazz = null;
            String header = null;
            if (content instanceof JavaClass) {
                clazz = (JavaClass)content;
                header = getClassHeader((JavaClass)content);
            } else if (content instanceof CallableFeature || content instanceof Field){
                clazz = ((Feature)content).getDeclaringClass();
            }
            String clazzInfo = clazz instanceof JavaClass ? wrapClass((JavaClass)clazz) : wrapClass(urlStr);
            if (header != null)
                clazzInfo += "<PRE>" + header + "</PRE>"; //NOI18N

            String textFromURL = HTMLJavadocParser.getJavadocText(url, content instanceof JavaPackage);

            if (textFromURL!=null && textFromURL.length()>0){
                if (!textFromURL.toUpperCase().startsWith("<DL>") && //NOI18N
                   (!textFromURL.toUpperCase().startsWith("<PRE>"))) //NOI18N
                    textFromURL = "<BLOCKQUOTE>" + textFromURL + "</BLOCKQUOTE>"; //NOI18N                
                String retrievedText = clazzInfo + textFromURL;
                lastBase = getLastBase(urlStr);
                goToSourceEnabled = false;
                showJavaDoc(retrievedText);
                return true;
            }
            
            return false;
        }
        
        private void setClass(JavaClass cls){
            if (cls == null) {
                if (tryMountedJavaDoc(null)) return;
                goToSourceEnabled = false;
                javaDocNotFound();
                return;
            } else {
                if (isFromLibrary(cls) && tryMountedJavaDoc(null)) return;
            }
            JavaDoc jd = cls.getJavadoc();
            boolean notFound = jd == null || (jd.getText().length() == 0 && jd.getTags().size() == 0);
            String preparedText = prepareJavaDocContent(cls, getClassHeader(cls),
            notFound ? CONTENT_NOT_FOUND : jd.getText(),
            notFound ? null : getJavaDocTags(jd.getTags()));
            if (notFound && tryMountedJavaDoc(null)) return;
            showJavaDoc(preparedText);
        }

        private boolean isFromLibrary(Element el) {
            assert el != null;
            JavaModelPackage jmp = (JavaModelPackage)el.refImmediatePackage();
            if (jmp == null)
                return false;
            CodebaseClass cbClass = jmp.getCodebase();
            Iterator it = cbClass.refAllOfClass().iterator();
            if (it.hasNext()) {
                Codebase cb=(Codebase)it.next();
                return cb.isLibrary();
            } else {
                return false;
            }
        }

        private void setFeature(Feature feature, String displayText){
            if (feature == null) {
                if (tryMountedJavaDoc(null)) return;
                goToSourceEnabled = false;                
                javaDocNotFound();
                return;
            } else {
                if (isFromLibrary(feature) && tryMountedJavaDoc(null)) return;
            }
            
            if (feature.getDeclaringClass() instanceof Array && "length".equals(feature.getName())){ // NOI18N
                showJavaDoc(ARRAY_LENGTH_FIELD_JAVADOC);
                return;
            }
            
            JavaDoc jd = feature.getJavadoc();
            String jdText = ""; //NOI18N
            List jdTags = new ArrayList();
            
            if (jd!=null){
                jdTags = jd.getTags();
                jdText = jd.getText();  
            }
            
            if (feature instanceof Method) {
                if (((jdTags.size() == 0 && jdText.length() == 0) || jdText.indexOf("{@inheritDoc}") > -1)) { //NOI18N
                    DataObject dObj = NbEditorUtilities.getDataObject(extEditorUI.getDocument());
                    FileObject fo = dObj != null ? dObj.getPrimaryFile() : null;
                    JavaMetamodel.getManager().setClassPath(fo, true);
                    Iterator parentIt = JavaModelUtil.getOverriddenMethods((Method) feature).iterator();

                    while(((jdTags.size() == 0 && jdText.length() == 0) || jdText.indexOf("{@inheritDoc}") > -1) && parentIt.hasNext()) { //NOI18N
                        JavaDoc parentJD = ((Method)parentIt.next()).getJavadoc();
                        if (parentJD!=null){
                            if (jdText.length() == 0){
                                jdText = parentJD.getText();
                            } else {
                                jdText = jdText.replaceAll("\\{@inheritDoc\\}", parentJD.getText()); //NOI18N
                            }                        
                            if (jdTags.size() == 0){
                                jdTags = parentJD.getTags();
                            }
                        }
                    }
                }
            }
            
            boolean notFound = jdText.length() == 0 && jdTags.size() == 0;
            String preparedText = prepareJavaDocContent(feature.getDeclaringClass(),
            displayText,        
            (notFound) ? CONTENT_NOT_FOUND : jdText,
            (notFound) ? null : getJavaDocTags(jdTags));
            if (notFound && tryMountedJavaDoc(null)) return;
            showJavaDoc(preparedText);
        }
        
        private String getHyperlinkedTypeName(Type typ) {
            TypeReference tRef = null;
            try {
                tRef = JavaModelUtil.createTypeReferenceFromType(typ);
                return getHyperlinkedTypeName(tRef);
            } finally {
                if (tRef != null)
                    tRef.refDelete();
            }            
        }
        
        private String getHyperlinkedTypeName(TypeReference typ) {
            StringBuffer sb = new StringBuffer();
            if (typ instanceof ArrayReference) {
                sb.append(getHyperlinkedTypeName(typ.getParent()));
                for (int i = 0; i < ((ArrayReference)typ).getDimCount(); i++) {
                    sb.append("[]"); //NOI18N
                }
            } else if (typ instanceof MultipartId) {
                NamedElement el = typ.getElement();
                if (el instanceof JavaClass && !(el instanceof TypeParameter)) {
                    sb.append(createAnchor(((JavaClass)el).getSimpleName(), el.getName(), jmiUtils.getPackageName((JavaClass)el)));
                } else {
                    sb.append(el.getName());
                }
                Iterator args = ((MultipartId)typ).getTypeArguments().iterator();
                if (args.hasNext()) {
                    sb.append("&lt;"); //NOI18N
                    while(args.hasNext()) {
                        TypeArgument arg = (TypeArgument)args.next();
                        if (arg instanceof WildCard) {
                            sb.append('?'); //NOI18N                            
                            TypeReference bound = ((WildCard)arg).getBoundName();
                            if (bound != null) {
                                sb.append(((WildCard)arg).isLower() ? " extends " : " super "); //NOI18N
                                sb.append(getHyperlinkedTypeName(bound));
                            }
                        } else if (arg instanceof TypeReference) {
                            sb.append(getHyperlinkedTypeName((TypeReference)arg));
                        }
                        if (args.hasNext())
                            sb.append(", "); //NOI18N
                    }
                    sb.append("&gt;"); //NOI18N
                }                
            } else if (typ != null) {
                sb.append(typ.getName());
            }
            return sb.toString();
        }
        
        private String getTypeName(JavaClass jc) {
            StringBuffer sb = new StringBuffer();
            if (jc instanceof TypeParameter) {
                sb.append(jc.getSimpleName());
                boolean hasExtends = false;
                JavaClass sup = ((TypeParameter)jc).getSuperClass();
                if (sup != null && !"java.lang.Object".equals(sup.getName())) {
                    sb.append(" extends "); //NOI18N
                    sb.append(getHyperlinkedTypeName(sup));
                    hasExtends = true;
                }
                for (Iterator it = ((TypeParameter)jc).getInterfaces().iterator(); it.hasNext();) {
                    if (!hasExtends) {
                        sb.append(" extends "); // NOI18N
                        hasExtends = true;
                    } else {
                        sb.append(" & "); // NOI18N
                    }
                    sb.append(getHyperlinkedTypeName((JavaClass)it.next()));
                }
            } else {
                sb.append("<b>"); //NOI18N
                sb.append(jc.getSimpleName());
                sb.append("</b>"); //NOI18N
                Iterator params = jc.getTypeParameters().iterator();
                if (params.hasNext()) {
                    sb.append("&lt;"); //NOI18N
                    while(params.hasNext()) {
                        sb.append(getTypeName((TypeParameter)params.next()));
                        if (params.hasNext())
                            sb.append(", "); //NOI18N
                    }
                    sb.append("&gt;"); //NOI18N
                }
            }
            return sb.toString();            
        }
        
        private String getDocumentedAnnotations(AnnotableElement element) {
            StringBuffer sb = new StringBuffer();
            for (Iterator it = element.getAnnotations().iterator(); it.hasNext();) {
                Annotation ann = (Annotation)it.next();
                AnnotationType type = ann.getType();
                if (type != null) {
                    for (Iterator itt = type.getAnnotations().iterator(); itt.hasNext();) {
                        AnnotationType meta = (AnnotationType)((Annotation)itt.next()).getType();
                        if (meta != null && "java.lang.annotation.Documented".equals(meta.getName())) { // NOI18N
                            sb.append('@'); //NOI18N
                            sb.append(getHyperlinkedTypeName(type));
                            List values = ann.getAttributeValues();
                            if (!values.isEmpty()) {
                                try {
                                    StringBuffer vsb = new StringBuffer();
                                    for (Iterator ittt = values.iterator(); ittt.hasNext();) {
                                        AttributeValue av = (AttributeValue)ittt.next();
                                        vsb.append(av.getDefinition().getName());
                                        vsb.append('='); //NOI18N
                                        PositionBounds bounds = JavaMetamodel.getManager().getElementPosition(av.getValue());
                                        if (bounds != null) {
                                            vsb.append(bounds.getText());
                                        } else {
                                            vsb = null;
                                            break;
                                        }
                                        if (ittt.hasNext())
                                            vsb.append(", "); //NOI18N
                                    }
                                    if (vsb != null) {
                                        sb.append('('); // NOI18N
                                        sb.append(vsb);
                                        sb.append(')'); // NOI18N
                                    }
                                } catch (Exception ex) {
                                }
                            }
                            sb.append("<br>"); // NOI18N
                        }
                    }
                }
            }
            return sb.toString();
        }

        private String getClassHeader(JavaClass jc){
            StringBuffer sb = new StringBuffer();
            sb.append(getDocumentedAnnotations(jc));
            sb.append(Modifier.toString(jc.getModifiers()));
            if (jc instanceof AnnotationType) {
                sb.append(" @interface "); //NOI18N
            } else if (jc.isInterface()) {
                sb.append(" "); //NOI18N
            } else {
                sb.append(" class "); //NOI18N
            }
            sb.append(getTypeName(jc));
            JavaClass sup = jc.getSuperClass();
            if (sup != null && (!jc.isInterface() || sup.isInterface())) {
                sb.append("\nextends "); //NOI18N
                sb.append(getHyperlinkedTypeName(sup));
            }
            if (!(jc instanceof AnnotationType)) {
                Iterator it = jc.getInterfaces().iterator();
                if (it.hasNext())
                    sb.append("\nimplements ");//NOI18N
                while (it.hasNext()) {
                    JavaClass iface = (JavaClass) it.next();
                    sb.append(getHyperlinkedTypeName(iface));
                    if (it.hasNext())
                        sb.append(", "); //NOI18N
                }
            }
            return sb.toString();
        }

        private String getFieldHeader(Field f){
            StringBuffer sb = new StringBuffer();
            sb.append(getDocumentedAnnotations(f));
            sb.append(Modifier.toString(f.getModifiers()));
            sb.append(" "); // NOI18N
            sb.append(getHyperlinkedTypeName(f.getType()));
            sb.append(" <b>"); //NOI18N
            sb.append(f.getName());
            sb.append("</b>"); //NOI18N
            return sb.toString();
        }

        private String getAttributeHeader(Attribute attr){
            StringBuffer sb = new StringBuffer();
            sb.append(getDocumentedAnnotations(attr));
            sb.append(Modifier.toString(attr.getModifiers()));
            sb.append(" "); // NOI18N
            sb.append(getHyperlinkedTypeName(attr.getType()));
            sb.append(" <b>"); //NOI18N
            sb.append(attr.getName());
            sb.append("</b>"); //NOI18N
            return sb.toString();
        }

        private String getCallableFeatureHeader(CallableFeature cf){
            StringBuffer sb = new StringBuffer();
            sb.append(getDocumentedAnnotations(cf));
            int len = sb.length();
            sb.append(Modifier.toString(cf.getModifiers()));            
            sb.append(" "); // NOI18N
            Iterator tp = cf.getTypeParameters().iterator();
            if (tp.hasNext()) {
                sb.append("&lt;"); //NOI18N
                while(tp.hasNext()) {
                    sb.append(getTypeName((TypeParameter)tp.next()));
                    if (tp.hasNext())
                        sb.append(", "); //NOI18N
                }
                sb.append("&gt; "); //NOI18N
            }
            if (cf instanceof Method){
                sb.append(getHyperlinkedTypeName(cf.getType()));
                sb.append(" <b>"); //NOI18N
                sb.append(cf.getName());
            } else {
                sb.append("<b>"); //NOI18N
                sb.append(((JavaClass)cf.getType()).getSimpleName());
            }
            sb.append("</b>("); //NOI18N
            len = getHtmlTextLength(sb.substring(len));
            List params = cf.getParameters();
            for (Iterator it = params.iterator(); it.hasNext();) {
                Parameter param = (Parameter) it.next();
                sb.append(getHyperlinkedTypeName(param.getType()));
                if (param.isVarArg())
                    sb.append("..."); // NOI18N
                String paramName = param.getName();
                if (paramName != null && paramName.length() > 0)
                    sb.append(" "); // NOI18N
                sb.append(paramName);
                if (it.hasNext()) {
                    sb.append(",\n"); //NOI18N
                    for (int i = 0; i < len; i++)
                        sb.append(' '); //NOI18N
                }
            }
            sb.append(")"); //NOI18N
            Iterator excs = cf.getExceptions().iterator();
            if (excs.hasNext())
                sb.append("\n\t\tthrows ");//NOI18N
            while (excs.hasNext()) {
                sb.append(getHyperlinkedTypeName((JavaClass)excs.next()));
                if (excs.hasNext())
                    sb.append(", "); //NOI18N
            }
            return sb.toString();
        }
        
        private int getHtmlTextLength(String text) {
            int len = 0;
            boolean insideTag = false;
            boolean insideAmp = false;
            for (int i = 0; i < text.length(); i++) {
                switch (text.charAt(i)) {
                    case '<': //NOI18N
                        insideTag = true;
                        continue;
                    case '&': //NOI18N
                        insideAmp = true;
                        continue;
                    case '>': //NOI18N
                        insideTag = false;
                        continue;
                    case ';': //NOI18N
                        insideAmp = false;
                }
                if (!insideTag && !insideAmp)
                    len++;
            }
            return len;
        }
        
        public void run() {
            getJMIUtils().beginTrans(false);
            try {
                if (content instanceof Element && !((Element)content).isValid()) {
                    goToSourceEnabled = false;
                    if (!tryMountedJavaDoc(null)) javaDocNotFound();
                    return;
                }
                goToSourceEnabled = true;
                if(content instanceof JavaClass) {
                    membersCache.put(content, null);
                    setClass((JavaClass)content);
                    membersCache.clear();
                } else if(content instanceof Field) {
                    membersCache.put(((Field)content).getDeclaringClass(), null);
                    setFeature((Field)content, getFieldHeader((Field)content));
                    membersCache.clear();
                } else if(content instanceof CallableFeature){
                    membersCache.put(((CallableFeature)content).getDeclaringClass(), null);
                    setFeature((CallableFeature)content, getCallableFeatureHeader((CallableFeature)content));
                    membersCache.clear();
                } else if (content instanceof Attribute) {
                    membersCache.put(((Attribute)content).getDeclaringClass(), null);
                    setFeature((Feature)content, getAttributeHeader((Attribute)content));
                    membersCache.clear();
                } else if (content instanceof URL){
                    URL u = (URL)content;
                    // filter outgoing requests. Dangerous in open MDR transaction !!!
                    if ("http".equals(u.getProtocol())) return; //NOI18N
                    
                    if (!tryMountedJavaDoc((URL)content)){
                        goToSourceEnabled = false;
                        javaDocNotFound();
                    }
                    if (addToHistory) addToHistory(content);
                } else if (content instanceof String){
                    goToSourceEnabled = false;
                    String strCon = (String) content;
                    URL url = mergeRelLink(lastBase, strCon);
                    if (url == null) return;
                    // filter outgoing requests. Dangerous in open MDR transaction !!!
                    if ("http".equals(url.getProtocol())) return; //NOI18N
                    
                    if (addToHistory) addToHistory(url);
                    if (!tryMountedJavaDoc(url)) javaDocNotFound();
                } else if (content instanceof NbJMIResultItem.VarResultItem){
                    NbJMIResultItem.VarResultItem varItem = (NbJMIResultItem.VarResultItem)content;
                    String itemText = varItem.getItemText();
                    Type varType = varItem.getType();
                    if ("class".equals(itemText) && varType!=null && "java.lang.Class".equals(varType.getName())){ //NOI18N
                        showJavaDoc(CLASS_CONSTANT_JAVADOC);
                        return;
                    }
                    javaDocNotFound();
                }
                else{
                    goToSourceEnabled = false;
                    if (!tryMountedJavaDoc(null)) javaDocNotFound();
                }
            } finally {
                getJMIUtils().endTrans(false);
            }
        }
        
    }    
    
}
