/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.java;

import org.netbeans.api.editor.completion.Completion;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.Formatter;
import org.netbeans.editor.Utilities;
import org.netbeans.editor.ext.CompletionQuery;
import org.netbeans.editor.ext.ExtFormatter;
import org.netbeans.editor.ext.java.JCExpression;
import org.netbeans.editor.ext.java.JavaCompletion;
import org.netbeans.editor.ext.java.JavaSettingsNames;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.editor.util.CharSequenceUtilities;
import org.netbeans.lib.editor.util.swing.DocumentUtilities;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.spi.editor.completion.*;
import org.netbeans.spi.editor.completion.support.AsyncCompletionTask;

import org.openide.util.RequestProcessor;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.Graphics;
import java.lang.reflect.Modifier;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.swing.text.BadLocationException;
import javax.swing.text.JTextComponent;
import javax.swing.*;


/**
 *
 * @author  Dusan Balek
 */

public abstract class NbJMIResultItem implements CompletionQuery.ResultItem, CompletionItem {

    protected int selectionStartOffset = -1;
    protected int selectionEndOffset = -1;

    protected int substituteOffset = -1;

    public abstract String getItemText();
    
    protected abstract Object getAssociatedObject();

    protected static Color getTypeColor(Type typ) {
        return (typ instanceof PrimitiveType) ? NbJMIPaintComponent.KEYWORD_COLOR : NbJMIPaintComponent.TYPE_COLOR;
    }
    
    public void setSubstituteOffset(int substituteOffset) {
        this.substituteOffset = substituteOffset;
    }
    
    public boolean substituteCommonText(JTextComponent c, int offset, int len, int subLen) {
        // [PENDING] not enough info in parameters...
        // commonText
        // substituteExp
        return false;
    }
    
    public boolean substituteText(JTextComponent c, int offset, int len, boolean shift) {
        BaseDocument doc = (BaseDocument)c.getDocument();
        String text = getItemText();

        if (text != null) {
            if (toAdd != null && !toAdd.equals("\n")) // NOI18N
                text += toAdd;
            // Update the text
            doc.atomicLock();
            try {
                CharSequence textToReplace = DocumentUtilities.getText(doc, offset, len);
                if (CharSequenceUtilities.textEquals(text, textToReplace)) return false;
                
                doc.remove(offset, len);
                doc.insertString(offset, text, null);
                if (selectionStartOffset >= 0) {
                    c.select(offset + selectionStartOffset,
                    offset + selectionEndOffset);
                }
            } catch (BadLocationException e) {
                // Can't update
            } finally {
                doc.atomicUnlock();
            }
            return true;

        } else {
            return false;
        }
    }
    
    public Component getPaintComponent(javax.swing.JList list, boolean isSelected, boolean cellHasFocus) {
        Component ret = getPaintComponent(isSelected);
        if (ret==null) return null;
        if (isSelected) {
            ret.setBackground(list.getSelectionBackground());
            ret.setForeground(list.getSelectionForeground());
        } else {
            ret.setBackground(list.getBackground());
            ret.setForeground(list.getForeground());
        }
        ret.getAccessibleContext().setAccessibleName(getItemText());
        ret.getAccessibleContext().setAccessibleDescription(getItemText());
        return ret;
    }
    
    public abstract Component getPaintComponent(boolean isSelected);

    public int getPreferredWidth(Graphics g, Font defaultFont) {
        Component renderComponent = getPaintComponent(false);
        return renderComponent.getPreferredSize().width;
    }

    public void render(Graphics g, Font defaultFont, Color defaultColor,
    Color backgroundColor, int width, int height, boolean selected) {
        Component renderComponent = getPaintComponent(selected);
        renderComponent.setFont(defaultFont);
        renderComponent.setForeground(defaultColor);
        renderComponent.setBackground(backgroundColor);
        renderComponent.setBounds(0, 0, width, height);
        ((NbJMIPaintComponent)renderComponent).paintComponent(g);
    }
    
    public String toString() {
        return getItemText();
    }

    // CompletionItem implementation

    public static final String COMPLETION_SUBSTITUTE_TEXT= "completion-substitute-text"; //NOI18N

    static String toAdd;

    public void processKeyEvent(KeyEvent evt) {
        if (evt.getID() == KeyEvent.KEY_TYPED) {
            Completion completion = Completion.get();
            switch (evt.getKeyChar()) {
                case ' ':
                    if (evt.getModifiers() == 0) {
                        completion.hideCompletion();
                        completion.hideDocumentation();
                    }
                    break;
                case ';':
                case ',':
                    completion.hideCompletion();
                    completion.hideDocumentation();
                case '.':
                    if (defaultAction((JTextComponent)evt.getSource(), Character.toString(evt.getKeyChar()))) {
                        evt.consume();
                        break;
                    }
            }
        }
    }

    public CharSequence getSortText() {
        return getItemText();
    }
    
    public CharSequence getInsertPrefix() {
        return getItemText();
    }

    public CompletionTask createDocumentationTask() {
        return new AsyncCompletionTask(new JavaCompletionProvider.DocQuery(this),
            org.netbeans.editor.Registry.getMostActiveComponent());
    }
    
    public CompletionTask createToolTipTask() {
        return null;
    }

    public boolean instantSubstitution(JTextComponent c) {
        Completion completion = Completion.get();
        completion.hideCompletion();
        completion.hideDocumentation();
        defaultAction(c);
        return true;
    }

    public void defaultAction(JTextComponent component) {
        Completion completion = Completion.get();
        completion.hideCompletion();
        completion.hideDocumentation();
        defaultAction(component, "");
    }
    
    boolean defaultAction(JTextComponent component, String addText) {
        int substOffset = substituteOffset;
        if (substOffset == -1)
            substOffset = component.getCaret().getDot();
        NbJMIResultItem.toAdd = addText;
        return substituteText(component, substOffset, component.getCaret().getDot() - substOffset, false);
    }

    public static class VarResultItem extends NbJMIResultItem {

        private Type type;
        private String typeName;
        private Color typeColor;
        private String varName;
        private int modifiers;

        private static NbJMIPaintComponent.NbFieldPaintComponent fieldComponent = null;

        public VarResultItem(String varName, Type type, int modifiers){
            this.type = type;
            this.varName = varName;
            this.modifiers = modifiers | JavaCompletion.LOCAL_MEMBER_BIT;
            this.typeName = JMIUtils.getTypeName(type, false, false);
            this.typeColor = getTypeColor(type);
        }

        public String getItemText() {
            return varName;
        }

        public Component getPaintComponent(boolean isSelected) {
            if (fieldComponent == null) {
                fieldComponent = new NbJMIPaintComponent.NbFieldPaintComponent(true);
            }
            fieldComponent.setTypeName(typeName);
            fieldComponent.setTypeColor(typeColor);
            fieldComponent.setFieldName(varName);
            fieldComponent.setModifiers(modifiers);
            fieldComponent.setSelected(isSelected);
            return fieldComponent;
        }

        protected Object getAssociatedObject() {
            return this;
        }

        public Type getType() {
            return type;
        }

        public int getSortPriority() {
            return 200;
        }
        
        public String toString() {
            String mods = Modifier.toString(modifiers) + " "; // NOI18N
            return (mods.length() > 1 ? mods : "") + typeName + " " + varName; // NOI18N
        }
    }

    public static class FieldResultItem extends NbJMIResultItem{

        private Field fld;
        private String typeName;
        private Color typeColor;
        private String fldName;
        private int modifiers;
        private boolean isDeprecated;
        private JavaClass owner;
        private ClassDefinition context;

        private static NbJMIPaintComponent.NbFieldPaintComponent fieldComponent = null;
        
        public FieldResultItem(Field fld, JavaClass owner, ClassDefinition context){
            this.fld = fld;
            this.fldName = fld.getName();
            this.modifiers = fld.getModifiers();
            if (fld.getDeclaringClass() == (context instanceof ParameterizedType ? ((ParameterizedType)context).getDefinition() : context)) {
                this.modifiers |= JavaCompletion.LOCAL_MEMBER_BIT;
            }
            Type type=fld.getType();
            this.typeName = JMIUtils.getTypeName(type, false, false);
            this.typeColor = getTypeColor(type);
            this.isDeprecated = fld.isDeprecated();
            this.owner = owner;
            this.context = owner != null ? context : null;
        }
        
        public String getItemText() {
            return fldName;
        }
                
        public String getTypeName() {
            return typeName;
        }
        
        public int getModifiers() {
            return modifiers;
        }
        
        public String getFieldName() {
            return fldName;
        }
        
        public Component getPaintComponent(boolean isSelected) {
            if (fieldComponent == null) {
                fieldComponent = new NbJMIPaintComponent.NbFieldPaintComponent(false);
            }
            fieldComponent.setTypeName(typeName);
            fieldComponent.setFieldName(fldName);
            fieldComponent.setTypeColor(typeColor);
            fieldComponent.setModifiers(modifiers);
            fieldComponent.setSelected(isSelected);
            fieldComponent.setDeprecated(isDeprecated);
            return fieldComponent;
        }
        
        public boolean substituteText(final JTextComponent c, final int offset, final int len, final boolean shift) {
            if (context == null)
                return super.substituteText(c, offset, len, shift);
            RequestProcessor.getDefault().post(new Runnable() {
                public void run() {
                    final BaseDocument doc = (BaseDocument)c.getDocument();
                    final StringBuffer sb = new StringBuffer();
                    JMIUtils jmiUtils = JMIUtils.get(doc);
                    jmiUtils.beginTrans(true);
                    try {
                        sb.append(context.isValid() && owner.isValid() ? JavaModelUtil.resolveImportsForClass(context, owner).getName() : typeName);
                    } finally {
                        jmiUtils.endTrans(false);
                    }
                    sb.append("."); //NOI18N
                    sb.append(fldName);
                    SwingUtilities.invokeLater(new Runnable() {
                        public void run() {
                            String text = sb.toString();
                            if (text != null && text.length() > 0) {
                                int offset = c.getCaret().getDot() - len;
                                doc.atomicLock();
                                try {
                                    doc.remove(offset, len);
                                    doc.insertString(offset, text, null);
                                } catch (BadLocationException e) {
                                    // Can't update
                                } finally {
                                    doc.atomicUnlock();
                                }
                            }                    
                        }
                    });
                }
            });
            return true;
        }
        
        protected Object getAssociatedObject() {
            return fld;
        }
        
        public int getSortPriority() {
            return 300;
        }

        public String toString() {
            String mods = Modifier.toString(modifiers) + " "; // NOI18N
            return (mods.length() > 1 ? mods : "") + typeName + " " + fldName; // NOI18N
        }
    }
    
    public static class MethodResultItem extends CallableFeatureResultItem {
        
        private static NbJMIPaintComponent.NbMethodPaintComponent mtdComponent = null;
        
        public MethodResultItem(Method mtd, JCExpression substituteExp, ClassDefinition context) {
            super(mtd, substituteExp, context);
        }

        public Component getPaintComponent(boolean isSelected) {
            if (mtdComponent == null) {
                mtdComponent = new NbJMIPaintComponent.NbMethodPaintComponent();
            }
            mtdComponent.setFeatureName(getName());
            mtdComponent.setModifiers(getModifiers());
            mtdComponent.setTypeName(getTypeName());
            mtdComponent.setTypeColor(getTypeColor());
            mtdComponent.setParams(getParams());
            mtdComponent.setExceptions(getExceptions());
            mtdComponent.setDeprecated(isDeprecated());
            mtdComponent.setSelected(isSelected);
            mtdComponent.setActiveParameterIndex(getActiveParameterIndex());
            return mtdComponent;
        }
        
        public int getSortPriority() {
            return isEnclosingCall() ? 10 : 500;
        }

        public String toString() {
            String mods = Modifier.toString(getModifiers()) + " "; // NOI18N
            return (mods.length() > 1 ? mods : "") + getTypeName() + " " + getName() + printParams(true) + printExceptions(); // NOI18N
        }
    }
    
    public static class ConstructorResultItem extends CallableFeatureResultItem {
        
        private static NbJMIPaintComponent.NbConstructorPaintComponent ctrComponent = null;

        public ConstructorResultItem(Constructor con, JCExpression substituteExp) {
            super(con, substituteExp, null);
        }

        public String getName() {
            return getTypeName();
        }

        public Component getPaintComponent(boolean isSelected) {
            if (ctrComponent == null) {
                ctrComponent = new NbJMIPaintComponent.NbConstructorPaintComponent();
            }
            ctrComponent.setFeatureName(getName());
            ctrComponent.setModifiers(getModifiers());
            ctrComponent.setParams(getParams());
            ctrComponent.setExceptions(getExceptions());
            ctrComponent.setDeprecated(isDeprecated());
            ctrComponent.setSelected(isSelected);
            ctrComponent.setActiveParameterIndex(getActiveParameterIndex());
            return ctrComponent;
        }

        public int getSortPriority() {
            return isEnclosingCall() ? 5 : 400;
        }

        public String toString() {
            String mods = Modifier.toString(getModifiers()) + " "; // NOI18N
            return (mods.length() > 1 ? mods : "") + getName() + printParams(true) + printExceptions();
        }
    }

    public abstract static class CallableFeatureResultItem extends NbJMIResultItem {
        
        JCExpression substituteExp;
        private CallableFeature cf;
        private List params = new ArrayList();
        private List excs = new ArrayList();
        private int modifiers;
        private String cfName, typeName;
        private Color typeColor;
        private boolean isDeprecated;
        private int activeParameterIndex = -1;
        private int varArgIndex = -1;

        public CallableFeatureResultItem(CallableFeature cf, JCExpression substituteExp, ClassDefinition context) {
            this.cf = cf;
            this.substituteExp = substituteExp;
            this.modifiers = cf.getModifiers();
            if (cf.getDeclaringClass() == (context instanceof ParameterizedType ? ((ParameterizedType)context).getDefinition() : context)) {
                modifiers |= JavaCompletion.LOCAL_MEMBER_BIT;
            }
            cfName = cf.getName();
            Type tp=cf.getType();
            typeName = JMIUtils.getTypeName(tp, false, false);
            typeColor = getTypeColor(tp);
            isDeprecated = cf.isDeprecated();
            for (Iterator it = cf.getParameters().iterator(); it.hasNext();) {
                Parameter prm = (Parameter) it.next();
                Type type = prm.getType();
                params.add(new ParamStr(type.getName(), JMIUtils.getTypeName(type, false, false), prm.getName(), prm.isVarArg(), getTypeColor(type)));
                if (prm.isVarArg())
                    varArgIndex = params.size() - 1;
            }
            for (Iterator it = cf.getExceptions().iterator(); it.hasNext();) {
                JavaClass ex = (JavaClass) it.next();
                excs.add(new ExcStr(ex.getSimpleName(), getTypeColor(ex)));
            }
        }

        public void processKeyEvent(KeyEvent evt) {
            super.processKeyEvent(evt);
            if (!evt.isConsumed() && evt.getID() == KeyEvent.KEY_TYPED) {
                Completion completion = Completion.get();
                switch (evt.getKeyChar()) {
                    case '(':
                        completion.hideCompletion();
                        completion.hideDocumentation();
                        if (defaultAction((JTextComponent)evt.getSource(), Character.toString(evt.getKeyChar()))) {
                            evt.consume();
                            break;
                        }
                }
            }
        }
        
        public String getItemText() {
            return getName();
        }

        public String getTypeName() {
            return typeName;
        }
        
        public Color getTypeColor() {
            return typeColor;
        }

        public int getModifiers() {
            return modifiers;
        }
        
        public boolean isDeprecated() {
            return isDeprecated;
        }
        
        public String getName() {
            return cfName;
        }
        
        /** Returns List of ParamStr */
        public List getParams() {
            return params;
        }        
        
        /** Returns List of ExcStr */
        public List getExceptions() {
            return excs;
        }
        
        public CharSequence getSortText() {
            return getName() + "#" + getParamsCountString() + "#" + printParams(false); //NOI18N
        }
        
        private String getParamsCountString() {
            int size = params.size();
            return (size < 10 ? "0" : "") + size; //NOI18N
        }
        
        protected Object getAssociatedObject() {
            return cf;
        }
        
        int getActiveParameterIndex() {
            return activeParameterIndex;
        }
        
        /**
         * If set to value different than -1 it marks that
         * this component renders an outer enclosing constructor/method
         * and the given index is the index of the active parameter
         * which is being completed as an inner expression.
         */
        void setActiveParameterIndex(int activeParamIndex) {
            this.activeParameterIndex = activeParamIndex;
        }
        
        /**
         * Check whether this paint component renders an outer method/constructor
         * which should be rendered in grey with black active parameter.
         *
         * @return true if this paint component renders outer method/constructor
         *  or false otherwise.
         */
        boolean isEnclosingCall() {
            return (activeParameterIndex != -1);
        }

        public boolean substituteText(JTextComponent c, int offset, int len, boolean shift) {
            BaseDocument doc = (BaseDocument)c.getDocument();
            String text = null;
            boolean addParams = true;
            JCExpression exp = substituteExp;
            while(exp != null) {
                if (exp.getExpID() == JCExpression.IMPORT) {
                    addParams = false;
                    break;
                }
                exp = exp.getParent();
            }

            switch ((substituteExp != null) ? substituteExp.getExpID() : -1) {
            case JCExpression.METHOD:
                // no subst
                break;

            case JCExpression.METHOD_OPEN:
                int parmsCnt = params.size();
                if (parmsCnt == 0) {
                    if (getActiveParameterIndex() == -1) { // not showing active parm
                        try {
                            int fnwpos = Utilities.getFirstNonWhiteFwd(doc, offset + len);
                            if (fnwpos > -1 && doc.getChars(fnwpos, 1)[0] == ')') { // NOI18N
                                text = doc.getText(offset + len, fnwpos + 1 - offset - len);
                                len = fnwpos + 1 - offset;
                            }
                        } catch (BadLocationException e) {
                        }
                        if (text == null)
                            text = ")"; // NOI18N
                    }

                } else { // one or more parameters
                    int activeParamIndex = getActiveParameterIndex();
                    if (activeParamIndex != -1) { // Active parameter being shown
                        boolean substed = false;
                        if (activeParamIndex < parmsCnt) {
                            String paramName = ((ParamStr)params.get(activeParamIndex)).getName();
                            if (paramName != null) {
                                try {
                                    // Fill in the parameter's name
                                    doc.insertString(c.getCaretPosition(), paramName, null);
                                    substed = true;
                                } catch (BadLocationException e) {
                                    // Can't insert
                                }
                            }
                        }
                        return substed;
                    }
                    int ind = substituteExp.getParameterCount() - 1;
                    boolean addSpace = false;
                    Formatter f = doc.getFormatter();
                    if (f instanceof ExtFormatter) {
                        Object o = ((ExtFormatter)f).getSettingValue(JavaSettingsNames.JAVA_FORMAT_SPACE_AFTER_COMMA);
                        if ((o instanceof Boolean) && ((Boolean)o).booleanValue()) {
                            addSpace = true;
                        }
                    }
                    try {
                        if (addSpace && (ind == 0 || (offset > 0 && Character.isWhitespace(DocumentUtilities.getText(doc, offset - 1, 1).charAt(0))))) {
                            addSpace = false;
                        }
                    } catch (BadLocationException e) {
                    }

                    boolean isVarArg = parmsCnt > 0 ? ((ParamStr)params.get(parmsCnt - 1)).isVarArg() : false;
                    if (ind < parmsCnt || isVarArg) {
                        text = addSpace ? " " : ""; // NOI18N
                    }
                }
                break;

            default:
                text = getItemText();
                boolean addSpace = false;
                boolean addClosingParen = false;
                Formatter f = doc.getFormatter();
                if (f instanceof ExtFormatter) {
                    Object o = ((ExtFormatter)f).getSettingValue(JavaSettingsNames.JAVA_FORMAT_SPACE_BEFORE_PARENTHESIS);
                    if ((o instanceof Boolean) && ((Boolean)o).booleanValue()) {
                        addSpace = true;
                    }
                    o = ((ExtFormatter)f).getSettingValue(JavaSettingsNames.PAIR_CHARACTERS_COMPLETION);
                    if ((o instanceof Boolean) && ((Boolean)o).booleanValue()) {
                        addClosingParen = true;
                    }
                }

                if (addParams) {
                    String paramsText = null;
                    try {
                        int fnwpos = Utilities.getFirstNonWhiteFwd(doc, offset + len);
                        if (fnwpos > -1 && fnwpos <= Utilities.getRowEnd(doc, offset + len) && doc.getChars(fnwpos, 1)[0] == '(') { // NOI18N
                            paramsText = doc.getText(offset + len, fnwpos + 1 - offset - len);
                            if (addSpace && paramsText.length() < 2)
                                text += ' '; // NOI18N
                            len = fnwpos + 1 - offset;
                            text += paramsText;
                            toAdd = null; // do not add '.', ',', ';'
                        }
                    } catch (BadLocationException e) {
                    }
                    if (paramsText == null) {
                        if (addSpace) {
                            text += ' '; // NOI18N
                        }
                        text += '('; // NOI18N
                        if (params.size() > 0) {
                            selectionStartOffset = selectionEndOffset = text.length();
                            Completion completion = Completion.get();
                            completion.hideCompletion();
                            completion.hideDocumentation();
                            completion.showToolTip();
                        }
                        if (addClosingParen)
                            text += ")"; // NOI18N
                    } else {
                        try {
                            int fnwpos = Utilities.getFirstNonWhiteFwd(doc, offset + len);
                            if (fnwpos > -1 && doc.getChars(fnwpos, 1)[0] == ')') { // NOI18N
                                paramsText = doc.getText(offset + len, fnwpos + 1 - offset - len);
                                len = fnwpos + 1 - offset;
                                if (params.size() > 0) {
                                    selectionStartOffset = selectionEndOffset = text.length();
                                }
                                text += paramsText;
                            }
                        } catch (BadLocationException e) {
                        }
                    }
                }
                break;
            }

            if (text != null) {
                if (toAdd != null && !toAdd.equals("\n") && !"(".equals(toAdd)) // NOI18N
                    text += toAdd;
                // Update the text
                doc.atomicLock();
                try {
                    CharSequence textToReplace = DocumentUtilities.getText(doc, offset, len);
                    if (CharSequenceUtilities.textEquals(text, textToReplace)) {
                        c.setCaretPosition(offset + len);
                        return false;
                    }
                    doc.remove(offset, len);
                    doc.insertString(offset, text, null);
                    if (selectionStartOffset >= 0) {
                        c.select(offset + selectionStartOffset,
                        offset + selectionEndOffset);
                    } else if ("(".equals(toAdd)) { // NOI18N
                        int index = text.lastIndexOf(')');
                        if (index > -1) {
                            c.setCaretPosition(offset + index);
                        }
                    }
                } catch (BadLocationException e) {
                    // Can't update
                } finally {
                    doc.atomicUnlock();
                }
                return true;
            } else {
                return false;
            }
        }

        List createParamsList() {
            List ret = new ArrayList();
            for (Iterator it = params.iterator(); it.hasNext();) {
                StringBuffer sb = new StringBuffer();
                ParamStr ps = (ParamStr)it.next();
                sb.append(ps.getSimpleTypeName());
                if (ps.isVarArg()) {
                    sb.append("..."); // NOI18N
                }
                String name = ps.getName();
                if (name != null && name.length() > 0) {
                    sb.append(" "); // NOI18N
                    sb.append(name);
                }
                if (it.hasNext()) {
                    sb.append(", "); // NOI18N
                }
                ret.add(sb.toString());
            }
            return ret;
        }

        int getCurrentParamIndex() {
            int idx = 0;
            if (substituteExp != null && substituteExp.getExpID() == JCExpression.METHOD_OPEN)
                idx = substituteExp.getParameterCount() - 1;
            if (varArgIndex > -1 && varArgIndex < idx)
                idx = varArgIndex;
            return idx;
        }

        protected String printParams(boolean includeParamNames) {
            StringBuffer sb = new StringBuffer();
            sb.append("("); // NOI18N
            for (Iterator it = params.iterator(); it.hasNext();) {
                ParamStr ps = (ParamStr)it.next();
                sb.append(ps.getSimpleTypeName());
                if (ps.isVarArg()) {
                    sb.append("..."); // NOI18N
                }
                if (includeParamNames) {
                    String name = ps.getName();
                    if (name != null && name.length() > 0) {
                        sb.append(" "); // NOI18N
                        sb.append(name);
                    }
                }
                if (it.hasNext()) {
                    sb.append(", "); // NOI18N
                }
            }
            sb.append(")"); // NOI18N
            return sb.toString();
        }

        protected String printExceptions() {
            StringBuffer sb = new StringBuffer();
            if (excs.size() > 0) {
                sb.append(" throws "); // NOI18N
                for (Iterator it = excs.iterator(); it.hasNext();) {
                    ExcStr ex = (ExcStr) it.next();
                    sb.append(ex.getName());
                    if (it.hasNext()) {
                        sb.append(", "); // NOI18N
                    }
                }
            }
            return sb.toString();
        }
    }
    
    public static class PackageResultItem extends NbJMIResultItem {
        
        private boolean displayFullPackagePath;
        private JavaPackage pkg;
        private String pkgName;
        private static NbJMIPaintComponent.NbPackagePaintComponent pkgComponent = null;
        
        public PackageResultItem(JavaPackage pkg, boolean displayFullPackagePath){
            this.pkg = pkg;
            this.displayFullPackagePath = displayFullPackagePath;
            this.pkgName = pkg.getName();
        }
        
        public String getItemText() {
            return displayFullPackagePath ? pkgName : pkgName.substring(pkgName.lastIndexOf('.') + 1);
        }
        
        public Component getPaintComponent(boolean isSelected) {
            if (pkgComponent == null) {
                pkgComponent = new NbJMIPaintComponent.NbPackagePaintComponent();
            }
            pkgComponent.setSelected(isSelected);
            pkgComponent.setPackageName(pkgName);
            pkgComponent.setDisplayFullPackagePath(displayFullPackagePath);
            return pkgComponent;
        }

        public int getSortPriority() {
            return 700;
        }
        
        protected Object getAssociatedObject() {
            return pkg;
        }
    }
    
    public static class ClassResultItem extends NbJMIResultItem {
        
        private JavaClass cls;
        private boolean isInterface;
        private boolean isDeprecated;
        boolean addImport;
        boolean generateClassSkeleton;
        private String fqName = null;
        private String name = null;
        private static NbJMIPaintComponent.NbInterfacePaintComponent interfaceComponent = null;

        private static NbJMIPaintComponent.NbClassPaintComponent classComponent = null;
        private static NbJMIPaintComponent.NbEnumPaintComponent enumComponent = null;
        private static NbJMIPaintComponent.NbAnnotationPaintComponent annotationComponent = null;
        private static final boolean autoImportDisabled = Boolean.getBoolean("org.netbeans.java.editor.disableAutoImport"); // NOI18N
        private static final boolean autoGenerationDisabled = Boolean.getBoolean("org.netbeans.java.editor.disableAutoClassSkeletonGeneration"); // NOI18N
        private boolean displayFQN;

        public ClassResultItem(JavaClass cls, boolean displayFQN, boolean addImport, boolean generateClassSkeleton){
            this.cls = cls;
            this.addImport = addImport && !autoImportDisabled;
            this.name = cls.getSimpleName();
            this.displayFQN = displayFQN;
            if (displayFQN || this.addImport) {
                this.fqName = cls.getName();
                int idx = this.fqName.indexOf('<'); //NOI18N
                if (idx >= 0)
                    this.fqName = this.fqName.substring(0, idx);
                idx = this.fqName.lastIndexOf('.'); //NOI18N
                this.fqName = idx >= 0 ? " (" + this.fqName.substring(0, idx) + ")"  : ""; //NOI18N
            } else {
                this.fqName = ""; //NOI18N
            }
            this.isInterface = cls.isInterface();
            this.isDeprecated = cls.isDeprecated();
            this.addImport = addImport && !autoImportDisabled;
            this.generateClassSkeleton = generateClassSkeleton && !autoGenerationDisabled;
        }
        
        public boolean substituteText(final JTextComponent c, final int offset, final int len, final boolean shift) {
            final BaseDocument doc = (BaseDocument)c.getDocument();
            String text = generateClassSkeleton && cls instanceof ParameterizedType ? null : getItemText();
            int toAddDelta = 0;
            final boolean makeConstructor = "(".equals(toAdd); // NOI18N
            boolean ret = true;
            if (text != null) {
                if (toAdd != null && !toAdd.equals("\n")) { // NOI18N
                    text += toAdd;
                    if (makeConstructor) {
                        text += ")"; // NOI18N
                    }
                    toAddDelta = toAdd.length();
                }
                // Update the text
                doc.atomicLock();
                try {
                    CharSequence textToReplace = DocumentUtilities.getText(doc, offset, len);
                    if (CharSequenceUtilities.textEquals(text, textToReplace)) {
                        ret = false;
                    } else {
                        doc.remove(offset, len);
                        doc.insertString(offset, text, null);
                        if (makeConstructor)
                            c.setCaretPosition(c.getCaretPosition() - 1);
                    }
                } catch (BadLocationException e) {
                    // Can't update
                } finally {
                    doc.atomicUnlock();
                }
            }
            
            final int toAddDeltaResult = toAddDelta;
            
            RequestProcessor.getDefault().post(new Runnable() {
                public void run() {
                    final StringBuffer sb = new StringBuffer();
                    JMIUtils jmiUtils = JMIUtils.get(doc);
                    jmiUtils.beginTrans(true);
                    try {
                        if (cls.isValid()) {
                            Map cache = new HashMap();
                            NbJavaJMISyntaxSupport ssup = (NbJavaJMISyntaxSupport)doc.getSyntaxSupport().get(NbJavaJMISyntaxSupport.class);
                            JavaClass ctx = ssup.getJavaClass(c.getCaretPosition());
                            if (addImport && ctx != null && !cls.isInner()) {
                                JavaClass jc = cls instanceof ParameterizedType ? ((ParameterizedType)cls).getDefinition() : cls;
                                MultipartId mpid = JavaModelUtil.resolveImportsForClass(ctx, jc);
                                if (!jc.getSimpleName().equals(mpid.getName())) {
                                    doc.atomicLock();
                                    try {
                                        int pos = c.getCaretPosition() - toAddDeltaResult - name.length();
                                        doc.remove(pos, name.length());
                                        doc.insertString(pos, mpid.getName(), null);
                                    } catch (BadLocationException ble) {
                                    } finally {
                                        doc.atomicUnlock();
                                    }
                                    cache.put(jc, Boolean.TRUE);
                                } else {
                                    cache.put(jc, Boolean.FALSE);
                                }
                            }
                            if (!makeConstructor && checkClassSkeletonAutoGeneration()) {
                                sb.append("() {\n"); //NOI18N
                                List methods = jmiUtils.findMethods(cls, "", false, false, null, false, false, null, false, true); //NOI18N
                                for (Iterator it = methods.iterator(); it.hasNext();) {
                                    Method mtd = (Method)it.next();
                                    int mods = mtd.getModifiers();
                                    if (Modifier.isAbstract(mods)) {
                                        sb.append(Modifier.toString(mods & ~(Modifier.NATIVE | Modifier.ABSTRACT | Modifier.SYNCHRONIZED)));
                                        sb.append(' '); //NOI18N
                                        Type typ = mtd.getType();
                                        sb.append(JMIUtils.getTypeName(typ, typ instanceof JavaClass && useFQN((JavaClass)typ, ctx, cache), true));
                                        sb.append(' '); //NOI18N
                                        sb.append(mtd.getName());
                                        sb.append('('); //NOI18N
                                        for (Iterator itt = mtd.getParameters().iterator(); itt.hasNext();) {
                                            Parameter prm = (Parameter) itt.next();
                                            typ = prm.getType();
                                            sb.append(JMIUtils.getTypeName(typ, typ instanceof JavaClass && useFQN((JavaClass)typ, ctx, cache), true));
                                            sb.append(' '); //NOI18N
                                            sb.append(prm.getName());
                                            if (prm.isVarArg())
                                                sb.append("..."); //NOI18N
                                            if (itt.hasNext())
                                                sb.append(", "); //NOI18N
                                        }
                                        sb.append(')'); //NOI18N
                                        List exs = mtd.getExceptions();
                                        if (!exs.isEmpty())
                                            sb.append(" throws "); //NOI18N
                                        for (Iterator itt = exs.iterator(); itt.hasNext();) {
                                            JavaClass ex = (JavaClass) itt.next();
                                            sb.append(JMIUtils.getTypeName(ex, useFQN(ex, ctx, cache), true));
                                            if (itt.hasNext())
                                                sb.append(','); //NOI18N
                                        }
                                        sb.append(" {\n}\n"); //NOI18N
                                    }
                                }
                                sb.append('}'); //NOI18N
                            }
                        }
                    } finally {
                        jmiUtils.endTrans(false);
                    }
                    SwingUtilities.invokeLater(new Runnable() {
                        public void run() {
                            String skeleton = sb.toString();
                            if (skeleton != null && skeleton.length() > 0) {
                                doc.atomicLock();
                                try {
                                    int startOffset = c.getCaret().getDot();
                                    doc.insertString(startOffset, skeleton, null);
                                    int endOffset = c.getCaret().getDot();
                                    doc.getFormatter().reformat(doc, startOffset, endOffset);
                                } catch (BadLocationException e) {
                                    // Can't update
                                } finally {
                                    doc.atomicUnlock();
                                }
                            }                    
                        }
                    });
                }
            });
            return ret;
        }
        
        private boolean useFQN(JavaClass cls, JavaClass ctx, Map cache) {
            cls = (JavaClass)JMIUtils.getDefintion(cls);
            if (cls instanceof TypeParameter)
                cls = ((TypeParameter)cls).getSuperClass();
            Boolean b = (Boolean)cache.get(cls);
            if (b == null) {
                if (ctx != null) {
                    MultipartId mpid = JavaModelUtil.resolveImportsForClass(ctx, cls);                
                    b = Boolean.valueOf(!cls.getSimpleName().equals(mpid.getName()));
                } else {
                    b = Boolean.FALSE;
                }
                cache.put(cls, b);
            }
            return b.booleanValue();
        }

        public boolean substituteTextSimple(final JTextComponent c, int offset, int len, boolean shift) {
            return super.substituteText(c, offset, len, shift);            
        }
       /* 
        public boolean substituteText(JTextComponent c, int offset, int len, boolean shift) {
            boolean ret = super.substituteText(c, offset, len, shift);
            if (addImport) {
                final NbJavaJMIFastImport fastImport = new NbJavaJMIFastImport(c);
                RequestProcessor.getDefault().post(new Runnable() {
                    public void run() {
                        fastImport.autoImport(ClassResultItem.this);
                    }
                });
            }
            return ret;
        }        
        */

        public void processKeyEvent(KeyEvent evt) {
            if (evt.getID() == KeyEvent.KEY_PRESSED
                    && evt.getKeyCode() == KeyEvent.VK_ENTER
                    && evt.getModifiers() == InputEvent.CTRL_MASK
            ) {
                substituteTextSimple((JTextComponent)evt.getSource());
                evt.consume();
            }
            
            if (!evt.isConsumed()) {
                super.processKeyEvent(evt);
            }
        }
            
        private void substituteTextSimple(JTextComponent component) {
            int substOffset = substituteOffset;
            if (substOffset == -1)
                substOffset = component.getCaretPosition();
            substituteTextSimple(
                    component, substOffset, component.getCaretPosition() - substOffset, false);
            Completion.get().hideCompletion();
        }
        
        public boolean checkAutoImport(final JTextComponent c) {
            if (addImport)
                return new NbJavaJMIFastImport(c).checkAutoImport(ClassResultItem.this);
            return false;
        }
        
        private boolean checkClassSkeletonAutoGeneration() {
            return generateClassSkeleton && (Modifier.isAbstract(cls.getModifiers()) || Modifier.isInterface(cls.getModifiers()));
        }
        
        public String getItemText() {
            return name;
        }

        public boolean instantSubstitution(JTextComponent c) {
            boolean ret = !(checkAutoImport(c) || checkClassSkeletonAutoGeneration());
	    if (ret)
	        super.instantSubstitution(c);
	    return ret;
        }
        
        public CharSequence getSortText() {
            return name + fqName;
        }
        
        public Component getPaintComponent(boolean isSelected) {
            if (cls instanceof AnnotationType) {
                if (annotationComponent == null) {
                    annotationComponent = new NbJMIPaintComponent.NbAnnotationPaintComponent();
                }
                annotationComponent.setSelected(isSelected);
                annotationComponent.setDeprecated(isDeprecated);
                annotationComponent.setSimpleClassName(name);
                annotationComponent.setFQName(fqName);
                if (displayFQN){
                    annotationComponent.setCls(cls);
                }
                return annotationComponent;
            } else if (cls instanceof JavaEnum) {
                if (enumComponent == null) {
                    enumComponent = new NbJMIPaintComponent.NbEnumPaintComponent();
                }
                enumComponent.setSelected(isSelected);
                enumComponent.setDeprecated(isDeprecated);
                enumComponent.setSimpleClassName(name);
                enumComponent.setFQName(fqName);
                if (displayFQN){
                    enumComponent.setCls(cls);
                }
                return enumComponent;
            } else if (isInterface) {
                if (interfaceComponent == null) {
                    interfaceComponent = new NbJMIPaintComponent.NbInterfacePaintComponent();
                }
                interfaceComponent.setSelected(isSelected);
                interfaceComponent.setDeprecated(isDeprecated);
                interfaceComponent.setSimpleClassName(name);
                interfaceComponent.setFQName(fqName);
                if (displayFQN){
                    interfaceComponent.setCls(cls);
                }
                return interfaceComponent;
            } else {
                if (classComponent == null) {
                    classComponent = new NbJMIPaintComponent.NbClassPaintComponent();
                }
                classComponent.setSelected(isSelected);
                classComponent.setDeprecated(isDeprecated);
                classComponent.setSimpleClassName(name);
                classComponent.setFQName(fqName);
                if (displayFQN){
                    classComponent.setCls(cls);
                }
                return classComponent;
            }
        }
        
        protected Object getAssociatedObject() {
            return cls;
        }

        public int getSortPriority() {
            return 600;
        }

    }

    public static class AnnotationResultItem extends ClassResultItem {

        private int size = 0;
        private int defaultMembersCnt = 0;

        public AnnotationResultItem(AnnotationType annType, boolean displayFQN, boolean addImport) {
            super(annType, displayFQN, addImport, false);
            for (Iterator it = annType.getContents().iterator(); it.hasNext();) {
                Object o = it.next();
                if (o instanceof Attribute) {
                    Attribute attr = (Attribute) o;
                    if (attr.getDefaultValueText() != null)
                        defaultMembersCnt++;
                    size++;
                }
            }
        }

        public boolean substituteText(JTextComponent c, int offset, int len, boolean shift) {
            if (defaultMembersCnt != size) {
                if (toAdd.length() == 0) {
                    toAdd = "("; //NOI18N
                } else if (!"(".equals(toAdd)) { // NOI18N
                    toAdd = "()" + toAdd; //NOI18N
                }
            }
            return super.substituteText(c, offset, len, shift);
        }        
    }

    public static class AttributeResultItem extends NbJMIResultItem {

        private Attribute attr;
        private String attrName;
        private String typeName;
        private Color typeColor;
        private String defaultValueText;
        private static NbJMIPaintComponent.NbAttributePaintComponent attrComponent = null;

        public AttributeResultItem(Attribute attr) {
            this.attr = attr;
            this.attrName = attr.getName();
            Type type = attr.getType();
            this.typeName = JMIUtils.getTypeName(type, false, false);
            this.typeColor = getTypeColor(type);
            this.defaultValueText = attr.getDefaultValueText();
        }

        public String getItemText() {
            return attrName + "="; //NOI18N
        }

        public String getAttrName() {
            return attrName;
        }

        public String getTypeName() {
            return typeName;
        }

        public String getDefaultValueText() {
            return defaultValueText;
        }

        public Component getPaintComponent(boolean isSelected) {
            if (attrComponent == null) {
                attrComponent = new NbJMIPaintComponent.NbAttributePaintComponent();
            }
            attrComponent.setAttrName(attrName);
            attrComponent.setTypeName(typeName);
            attrComponent.setTypeColor(typeColor);
            attrComponent.setDefaultValueText(defaultValueText);
            attrComponent.setSelected(isSelected);
            return attrComponent;
        }

        protected Object getAssociatedObject() {
            return attr;
        }
        
        public int getSortPriority() {
            return 100;
        }

    }

    public static class StringResultItem extends NbJMIResultItem {

        private String str;
        private static NbJMIPaintComponent.NbStringPaintComponent stringComponent = null;

        public StringResultItem(String str) {
            this.str = str;
        }

        public String getItemText() {
            return str;
        }

        public Component getPaintComponent(boolean isSelected) {
            if (stringComponent == null) {
                stringComponent = new NbJMIPaintComponent.NbStringPaintComponent();
            }
            stringComponent.setSelected(isSelected);
            stringComponent.setString(str);
            return stringComponent;
        }
        
        public int getSortPriority() {
            return 50;
        }        

        protected Object getAssociatedObject() {
            return str;
        }
    }
    
    static class ParamStr {
        private String type, simpleType, prm;
        private boolean isVarArg;
        private Color typeColor;
        public ParamStr(String type, String simpleType, String prm, boolean isVarArg, Color typeColor) {
            this.type = type;
            this.simpleType = simpleType;
            this.prm = prm;
            this.isVarArg = isVarArg;
            this.typeColor = typeColor;
        }
        
        public String getTypeName() {
            return type;
        }
        
        public String getSimpleTypeName() {
            return simpleType;
        }

        public String getName() {
            return prm;
        }
        
        public boolean isVarArg() {
            return isVarArg;
        }
        
        public Color getTypeColor() {
            return typeColor;
        }
    }
    
    static class ExcStr {
        private String name;
        private Color typeColor;
        public ExcStr(String name, Color typeColor) {
            this.name = name;
            this.typeColor = typeColor;
        }
        
        public String getName() {
            return name;
        }
        
        public Color getTypeColor() {
            return typeColor;
        }
    }
}
