/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.java;

import java.util.*;
import java.lang.reflect.Modifier;
import javax.swing.text.JTextComponent;
import javax.swing.text.BadLocationException;

import org.netbeans.editor.ext.CompletionQuery;
import org.netbeans.editor.ext.ExtFormatter;
import org.netbeans.editor.ext.Completion;
import org.netbeans.editor.ext.ExtEditorUI;
import org.netbeans.editor.ext.java.*;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.Formatter;
import org.netbeans.editor.Utilities;
import org.netbeans.editor.TokenItem;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.editor.util.CharSequenceUtilities;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.lib.editor.util.swing.DocumentUtilities;

/**
*  Java completion query working with java JMI interfaces.
*
* @author Dusan Balek
* @version 1.00
*/

public class NbJavaJMICompletionQuery extends NbJavaCompletionQuery
implements JavaCompletionQuery.EnclosingResultProcessor {

    private static JMIItemFactory jmiItemFactory = new DefaultJMIItemFactory();

    public NbJavaJMICompletionQuery(boolean isJava15) {
        setJava15(isJava15);
    }

    protected CompletionQuery.Result getResult(JTextComponent component, JavaSyntaxSupport sup, boolean openingSource, int offset, JCExpression exp) {
        Completion completion = ((ExtEditorUI)Utilities.getEditorUI(component)).getCompletion();
        boolean autoPopup = completion != null ? completion.provokedByAutoPopup : false;
        JMIUtils utils = JMIUtils.get(getBaseDocument());
        utils.beginTrans(false);
        try {
            ((JMManager) JMManager.getManager()).setSafeTrans(true);
            Context ctx = new Context(component, (NbJavaJMISyntaxSupport)sup.get(NbJavaJMISyntaxSupport.class), openingSource, offset, utils, autoPopup);
            boolean ok = ctx.resolveExp(exp);
            return ok ? ctx.result : null;
        } finally {
            utils.endTrans(false);
        }
    }

    protected CompletionQuery.Result getResult(JTextComponent component, JavaSyntaxSupport sup,
    boolean openingSource, int offset, JCExpression exp, JCExpression exp2) {
        Completion completion = ((ExtEditorUI)Utilities.getEditorUI(component)).getCompletion();
        boolean autoPopup = completion != null ? completion.provokedByAutoPopup : false;
        JMIUtils utils = JMIUtils.get(getBaseDocument());
        utils.beginTrans(false);
        try {
            ((JMManager) JMManager.getManager()).setSafeTrans(true);
            Context ctx = new Context(component, (NbJavaJMISyntaxSupport)sup.get(NbJavaJMISyntaxSupport.class), openingSource, offset, utils, autoPopup);
            boolean ok = ctx.resolveExp(exp);
            return ok ? ctx.result : null;
        } finally {
            utils.endTrans(false);
        }
    }
    
    public void processEnclosingResult(
    Result enclosingResult, JCExpression enclosingExp, Result regularResult) {

        int addIndex = 0;
        for (Iterator it = enclosingResult.getData().iterator(); it.hasNext();) {
            Object elem = it.next();
            if (elem instanceof NbJMIResultItem.CallableFeatureResultItem) {
                NbJMIResultItem.CallableFeatureResultItem callable
                        = (NbJMIResultItem.CallableFeatureResultItem)elem;
                callable.setActiveParameterIndex(callable.getCurrentParamIndex());
                // Insert the outer result at the begining of the result
                regularResult.getData().add(addIndex++, callable);
            }
        }
    }

    class Context {

        /** Text component */
        private JTextComponent component;

        /** Syntax support for the given document */
        private NbJavaJMISyntaxSupport sup;

        /** Whether the query is performed to open the source file. It has slightly
        * different handling in some situations.
        */
        private boolean openingSource;

        /** End position of the scanning - usually the caret position */
        private int endOffset;

        /** If set to true true - find the type of the result expression.
        * It's stored in the lastType variable or lastPkg if it's a package.
        * The result variable is not populated.
        * False means that the code completion output should be collected.
        */
        private boolean findType;

        /** Whether currently scanning either the package or the class name
        * so the results should limit the search to the static fields and methods.
        */
        private boolean staticOnly = true;

        /** Last package found when scanning dot expression */
        private JavaPackage lastPkg;

        /** Last type found when scanning dot expression */
        private Type lastType;

        /** Result list when code completion output is generated */
        private JavaResult result;

        /** Helper flag for recognizing constructors */
        private boolean isConstructor;

        private boolean isImport;
        private boolean isStaticImport;

        private boolean isGeneric;
        private Collection typeBounds;

        private boolean isAnnotation;
        private boolean isAnnotationOpen;

        private JavaClass curCls;
        private boolean isThisContext;
        private boolean superCall;

        /** True when code completion is invoked by auto popup. In such case, code completion returns no result
         * after "new ". To get a result, code completion has to be invoked manually (using Ctrl-Space). */ // NOI18N
        private boolean autoPopup;

        /** Finder associated with this Context. */
        private JMIUtils jmiUtils = null;

        public Context(JTextComponent component, NbJavaJMISyntaxSupport sup, boolean openingSource, int endOffset, JMIUtils utils, boolean autoPopup) {

            this.component = component;
            this.sup = sup;
            this.openingSource = openingSource;
            this.endOffset = endOffset;
            this.jmiUtils = utils;
            this.curCls = sup.getJavaClass(endOffset);
            if (this.curCls == null) {
                this.curCls = sup.getTopJavaClass();
            }
            this.autoPopup = autoPopup;
        }

        public void setFindType(boolean findType) {
            this.findType = findType;
        }

        protected Object clone() {
            return new Context(component, sup, openingSource, endOffset, jmiUtils, autoPopup);
        }

        private int getSwitchOffset(int caseOffset) {
            int offset = -1;
            try {
                JavaFormatSupport jfs = new JavaFormatSupport(null);
                TokenItem itm = jfs.findSwitch(sup.getTokenChain(caseOffset, caseOffset + 4));
                if (itm != null) {
                    int off = itm.getNext().getOffset();
                    off = Utilities.getFirstNonWhiteFwd(getBaseDocument(), off);
                    int[] block = sup.findMatchingBlock(off, true);
                    if (block != null) {
                        offset = block[1] - 1;
                    }
                }
            } catch (Exception e) {
            }
            return offset;
        }

        private String formatType(Type type, boolean useFullName,
                                  boolean appendDot, boolean appendStar) {
            StringBuffer sb = new StringBuffer();
            if (type != null) {
                if (type instanceof AnnotationType)
                    sb.append('@'); //NOI18N
                if (useFullName) {
                    if (type instanceof ParameterizedType) {
                        ClassDefinition decl = ((ParameterizedType)type).getDeclaringClass();
                        if (decl != null) {
                            sb.append(formatType(decl, useFullName, true, false));
                        } else {
                            String s = ((ParameterizedType)type).getDefinition().getName();
                            sb.append(s.substring(0, s.lastIndexOf('.') + 1));
                        }
                        sb.append(((ParameterizedType)type).getSimpleName());
                    } else {
                        sb.append(type.getName());
                    }
                } else {
                    sb.append(type instanceof JavaClass ? ((JavaClass)type).getSimpleName() : type.getName());
                }
            }
            if (appendDot) {
                sb.append('.'); // NOI18N
            }
            if (appendStar) {
                sb.append('*'); // NOI18N
            }
            return sb.toString();
        }

        private Type resolveType(JCExpression exp, boolean propagateIsStatic) {
            Context ctx = (Context)clone();
            ctx.setFindType(true);
            ctx.isGeneric = isGeneric;
            Type typ = null;
            if (ctx.resolveExp(exp)) {
                typ = ctx.lastType;
                if (propagateIsStatic) {
                    staticOnly = ctx.staticOnly;
                }
            }
            if (JMIUtils.debugCompletionFailures && typ instanceof UnresolvedClass) {
                System.err.println("resolveType: unresolved type " + typ.getName() + " returned when resolving " + exp); // NOI18N
                Thread.dumpStack();
            }
            return typ;
        }

        boolean resolveExp(JCExpression exp) {
            boolean lastDot = false; // dot at the end of the whole expression?
            boolean ok = true;

            switch (exp.getExpID()) {
            case JCExpression.DOT_OPEN: // Dot expression with the dot at the end
                lastDot = true;
                // let it flow to DOT
            case JCExpression.DOT: // Dot expression
                int parmCnt = exp.getParameterCount(); // Number of items in the dot exp

                for (int i = 0; i < parmCnt && ok; i++) { // resolve all items in a dot exp
                    ok = resolveItem(exp.getParameter(i), (i == 0),
                                     (!lastDot && i == parmCnt - 1)
                                    );
                    isThisContext = superCall;
                    superCall = false;
                    if (JMIUtils.debugCompletionFailures && !ok && !findType) {
                        System.err.println("resolveItem: failed when resolving " + exp.getParameter(i) + "\n    lastType=" + lastType + "\n    lastPkg=" + lastPkg); // NOI18N
                        Thread.dumpStack();
                    }

                }

                if (ok && lastDot) { // Found either type or package help
                    // Need to process dot at the end of the expression
                    int tokenCntM1 = exp.getTokenCount() - 1;
                    int substPos = exp.getTokenOffset(tokenCntM1) + exp.getTokenLength(tokenCntM1);
                    if (lastType != null) { // Found type
                        if (lastType instanceof ClassDefinition) {
                            List res = new ArrayList();
                            if (openingSource) {
                                res.add(lastType);
                            } else if (isAnnotationOpen) {
                                res.add(getJMIItemFactory().createVarResultItem("class", jmiUtils.getExactClass("java.lang.Class"), Modifier.PUBLIC | Modifier.STATIC | Modifier.FINAL)); // NOI18N
                                res.addAll(findConstMembers(lastType, "", false, curCls));
                            } else if (!isConstructor && (!isImport || isStaticImport)){ // not source-help
                                if (staticOnly && !isImport)
                                    res.add(getJMIItemFactory().createVarResultItem("class", jmiUtils.getExactClass("java.lang.Class"), Modifier.PUBLIC | Modifier.STATIC | Modifier.FINAL)); // NOI18N
                                res.addAll(findFieldsMethodsAndInnerClasses(lastType, "", false, false, curCls, staticOnly || isStaticImport, isThisContext, exp, null, false, false)); // NOI18N
                            } else {
                                res.addAll(findInnerClasses(lastType, "", false, false, curCls, isThisContext, true, true, isConstructor));
                            }
                            // Get all fields and methods of the cls
                            result = new JavaResult(component, res, formatType(lastType, true, true, true),
                                    exp, substPos, 0, lastType instanceof JavaClass ? JMIUtils.getSourceElementIfExists((JavaClass)lastType) : curCls);
                        } else if (lastType instanceof PrimitiveType) {
                            List res = Collections.singletonList(getJMIItemFactory().createVarResultItem("class", jmiUtils.getExactClass("java.lang.Class"), Modifier.PUBLIC | Modifier.STATIC | Modifier.FINAL)); // NOI18N
                            result = new JavaResult(component, res, formatType(lastType, true, true, true),
                                    exp, substPos, 0, curCls);                            
                        }
                    } else { // Found package (otherwise ok would be false)
                        String searchPkg = lastPkg.getName() + '.';
                        List res = new ArrayList();
                        if (openingSource) {
                            res.add(lastPkg); // return only the package
                        } else {
                            if (isAnnotation) {
                                res.addAll(jmiUtils.findAnnotations(lastPkg, "", false, curCls, false));
                            } else {
                                CharSequence text = null;
                                try {
                                    int firstTokenIdx = exp.getTokenOffset(0);
                                    int cmdStartIdx = sup.getLastCommandSeparator(firstTokenIdx);
                                    if (cmdStartIdx < 0) {
                                        text = DocumentUtilities.getText(sup.getDocument(), 0, firstTokenIdx);
                                        cmdStartIdx = CharSequenceUtilities.lastIndexOf(text, 0x0A);
                                        if (cmdStartIdx != -1) {
                                            text = text.subSequence(0, cmdStartIdx + 1);
                                        }
                                    } else {
                                        text = DocumentUtilities.getText(sup.getDocument(), cmdStartIdx, firstTokenIdx - cmdStartIdx);
                                    }
                                } catch (BadLocationException e) {
                                    // ignore and provide full list of items
                                }

                                if (text != null) {
                                    int index = CharSequenceUtilities.indexOf(text, "package"); // NOI18N
                                    if (index == -1 || text.length() <= index + 8 || !Character.isWhitespace(text.charAt(index + 7))) {
                                        res.addAll(jmiUtils.findClasses(lastPkg, "", false, false, isImport, curCls, !isGeneric, isConstructor)); // package classes
                                    }
                                }
                            }
                        }
                        res.addAll(jmiUtils.findPackages(searchPkg, false, false, !isGeneric)); // find all subpackages
                        if (isGeneric)
                            res = filterGenericTypeArguments(res);
                        result = new JavaResult(component, res, (isAnnotation ? "@" : "") + searchPkg + '*', // NOI18N
                                                exp, substPos, 0, curCls);
                    }
                }
                break;


            case JCExpression.NEW: // 'new' keyword
                if (!autoPopup) {
                    List res = findInnerClasses(curCls, "", false, true, curCls, false, true, true, true);
                    res.addAll(jmiUtils.getImportedInnerClasses("", false, curCls, true, true, true)); // NOI18N
                    res.addAll(jmiUtils.findClasses(null, "", false, false, false, curCls, true, true)); // Find all classes by name // NOI18N
                    result = new JavaResult(component, res, "*", exp, endOffset, 0, curCls); // NOI18N
                }
                break;

            case JCExpression.CASE:
                int off = getSwitchOffset(exp.getTokenOffset(0));
                if (off > -1) {
                    Result r = NbJavaJMICompletionQuery.this.query(component, off, sup, true);
                    if (r != null && r.getData().size() > 0) {
                        Object itm = r.getData().get(0);
                        Type typ = null;
                        if (itm instanceof NbJMIResultItem.VarResultItem) {
                            typ = ((NbJMIResultItem.VarResultItem)itm).getType();
                        } else {
                            if (itm instanceof NbJMIResultItem)
                                itm = ((NbJMIResultItem)itm).getAssociatedObject();
                            if (itm instanceof TypedElement) {
                                typ = ((TypedElement)itm).getType();
                            }
                        }
                        if (typ != null) {
                            if (typ instanceof ParameterizedType)
                                typ = ((ParameterizedType)typ).getDefinition();
                            String name = ""; // NOI18N
                            int constOffset = endOffset;
                            int constLen = 0;
                            JCExpression eexp = exp.getParameter(0);
                            if (eexp.getExpID() == JCExpression.VARIABLE) {
                                name = eexp.getTokenText(0);
                                constOffset = eexp.getTokenOffset(0);
                                constLen = eexp.getTokenLength(0);
                            }
                            if (typ instanceof JavaEnum) {
                                List res = jmiUtils.findEnumConstants((JavaEnum)typ, name, false, true, true);
                                result = new JavaResult(component, res, formatType(typ, true, false, false), exp, constOffset, constLen, curCls);
                                break;
                            }
                        }
                    }
                }
                exp = exp.getParameter(0);

            default: // The rest of the situations is resolved as a singleton item
                resolveItem(exp, true, true);
                break;
            }

            return ok;
        }

        /** Resolve one item from the expression connected by dots.
        * @param item expression item to resolve
        * @param first whether this expression is the first one in a dot expression
        * @param last whether this expression is the last one in a dot expression
        */
        boolean resolveItem(JCExpression item, boolean first, boolean last) {
            boolean cont = true; // whether parsing should continue or not
            boolean methodOpen = false; // helper flag for unclosed methods

            switch (item.getExpID()) {
            case JCExpression.GENERIC_WILD_CHAR:
                lastType = jmiUtils.resolveType("java.lang.Object"); // NOI18N
                break;

            case JCExpression.CONSTANT: // Constant item
                if (first) {
                    lastType = jmiUtils.resolveType(item.getType()); // Get the constant type
                    staticOnly = false;
                } else { // Not the first item in a dot exp
                    cont = false; // impossible to have constant inside the expression
                }
                break;

            case JCExpression.VARIABLE: // Variable or special keywords
                switch (item.getTokenID(0).getNumericID()) {
                    case JavaTokenContext.THIS_ID: // 'this' keyword
                        if (first) { // first item in expression
                            if (curCls != null) {
                                lastType = curCls;
                                staticOnly = false;
                            }
                        } else { // 'something.this'
                            staticOnly = false;
                        }
                        break;

                    case JavaTokenContext.SUPER_ID: // 'super' keyword
                        if (first) { // only allowed as the first item
                            JavaClass cls = curCls;
                            if (cls != null) {
                                cls = cls.getSuperClass();
                                if (cls != null) {
                                    lastType = cls;
                                    staticOnly = false;
                                    superCall = true;
                                }
                            }
                        } else {
                            cont = false;
                        }
                        break;

                    case JavaTokenContext.CLASS_ID: // 'class' keyword
                        if (!first) {
                            lastType = jmiUtils.getExactClass("java.lang.Class"); // NOI18N
                            staticOnly = false;
                        } else {
                            cont = false;
                        }
                        break;

                    default: // Regular constant
                        String var = item.getTokenText(0);
                        int varPos = item.getTokenOffset(0);
                        if (first) { // try to find variable for the first item
                            if (last && !findType) { // both first and last item
                                List res = new ArrayList();
                                if (isAnnotation) {
                                    res.addAll(jmiUtils.findAnnotations(null, var, openingSource, openingSource ? null : curCls, false));
                                } else if (isAnnotationOpen) {
                                    res.addAll(findConstMembers(curCls, var, openingSource, openingSource ? null : curCls));
                                    if (lastType instanceof JavaEnum) {
                                        List l = jmiUtils.findEnumConstants((JavaEnum)lastType, var, false, false, true);
                                        for (Iterator it = l.iterator(); it.hasNext();)
                                            res.add(jmiItemFactory.createFieldResultItem((Field)it.next(), (JavaEnum)lastType, curCls));
                                    } else if (lastType instanceof PrimitiveType && ((PrimitiveType)lastType).getKind() == PrimitiveTypeKindEnum.BOOLEAN) { //NOI18N
                                        if (jmiUtils.startsWith("true", var)) //NOI18N
                                            res.add(jmiItemFactory.createStringResultItem("true")); //NOI18N
                                        if (jmiUtils.startsWith("false", var)) //NOI18N
                                            res.add(jmiItemFactory.createStringResultItem("false")); //NOI18N
                                    }
                                } else if (curCls != null && !isGeneric && !isConstructor && !isImport) {
                                    Collection local = sup.getLocalVariableNames(var, varPos, openingSource);
                                    for (Iterator it = local.iterator(); it.hasNext();) {
                                        String name = (String) it.next();
                                        Type t = (Type)sup.findType(name, varPos);
                                        if (t != null)
                                            res.add(getJMIItemFactory().createVarResultItem(name, t, 0));
                                    }
                                    res.addAll(findFieldsMethodsAndInnerClasses(curCls, var, openingSource, true, curCls, sup.isStaticBlock(varPos), true, item, local, true, false));
                                    res.addAll(findStaticallyImportedFeatures(var, openingSource, curCls, true)); // add static imports
                                }
                                if (var.length() > 0 && !isImport) {
                                    if (openingSource) {
                                        res.add(sup.getTypeFromName(var, true, curCls, false));
                                    } else if (!isAnnotation) {
                                        if (isGeneric || isConstructor)
                                            res.addAll(findInnerClasses(curCls, var, false, true, curCls, true, !isGeneric, !isGeneric, isConstructor));
                                        res.addAll(jmiUtils.getImportedInnerClasses(var, false, curCls, true, !isGeneric, isConstructor));
                                        res.addAll(jmiUtils.findClasses(null, var, false, false, false, curCls, !isGeneric, isConstructor)); // add matching classes
                                    }
                                }
                                res.addAll(jmiUtils.findPackages(var, openingSource, false, true)); // add matching packages
                                if (isGeneric)
                                    res = filterGenericTypeArguments(res);
                                result = new JavaResult(component, res, (isAnnotation ? "@" : "") + var + '*', item, curCls); // NOI18N
                            } else { // not last item or finding type
                                lastType = (Type)sup.findType(var, varPos);
                                if (lastType != null) { // variable found
                                    if (isGeneric)
                                        lastType = null;
                                    else
                                        staticOnly = false;
                                    if (JMIUtils.debugCompletionFailures && lastType instanceof UnresolvedClass) {
                                        System.err.println("resolveItem: unresolved type " + lastType.getName() + " returned when resolving type of variable " + var); // NOI18N
                                        Thread.dumpStack();
                                    }
                                } else { // no variable found
                                    lastPkg = jmiUtils.getExactPackage(var); // try package
                                    if (lastPkg == null) { // not package, let's try class name
                                        Type typ = sup.getTypeFromName(var, true, curCls, !openingSource);
                                        if (typ == null) {
                                            List sf = jmiUtils.getStaticallyImportedFields(var, true, openingSource ? null : curCls, true);
                                            if (sf.size() > 0) {
                                                typ = ((Field)sf.get(0)).getType();
                                                staticOnly = false;
                                            }
                                        }
                                        if (typ != null) {
                                            lastType = typ;
                                        } else { // class not found
                                            cont = false;
                                        }
                                    }
                                }
                            }
                        } else { // not the first item
                            if (lastType != null) { // last was type
                                if (lastType instanceof ClassDefinition) {
                                    if (findType || !last) {
                                        boolean inner = false;
                                        if (staticOnly && lastType instanceof JavaClass) { // can be inner class
                                            JavaClass cls = ((JavaClass)lastType).getInnerClass(var, true); // NOI18N
                                            if (cls != null) {
                                                lastType = cls;
                                                inner = true;
                                            }
                                        }
                                        
                                        if (!inner) { // not inner class name
                                            if (lastType instanceof ClassDefinition) {
                                                List fldList = jmiUtils.findFields(lastType, var, true, false, curCls, staticOnly, isThisContext, false, false);
                                                if (fldList.size() > 0) { // match found
                                                    Field fld = (Field)fldList.get(0);
                                                    lastType = fld.getType();
                                                    staticOnly = false;
                                                } else { // no match found
                                                    lastType = null;
                                                    cont = false;
                                                }
                                            } else {
                                                cont = false;
                                            }
                                        }
                                    } else { // last and searching for completion output
                                        List res = new ArrayList();
                                        if (isAnnotationOpen) {
                                            if (jmiUtils.startsWith("class", var)) // NOI18N
                                                res.add(getJMIItemFactory().createVarResultItem("class", jmiUtils.getExactClass("java.lang.Class"), Modifier.PUBLIC | Modifier.STATIC | Modifier.FINAL)); // NOI18N
                                            res.addAll(findConstMembers(lastType, var, openingSource, curCls));
                                        } else if (!isImport || isStaticImport) {
                                            if (!isImport && staticOnly && jmiUtils.startsWith("class", var)) // NOI18N
                                                res.add(getJMIItemFactory().createVarResultItem("class", jmiUtils.getExactClass("java.lang.Class"), Modifier.PUBLIC | Modifier.STATIC | Modifier.FINAL)); // NOI18N
                                            res.addAll(findFieldsMethodsAndInnerClasses(lastType, var, openingSource, false, curCls, staticOnly || isStaticImport, isThisContext, item, null, false, false));
                                        } else {
                                            res.addAll(findInnerClasses(lastType, var, openingSource, false, curCls, isThisContext, true, true, false));
                                        }
                                        result = new JavaResult(
                                                component,
                                                res,
                                                formatType(lastType, true, true, false) + var + '*',
                                                item,
                                                lastType instanceof JavaClass ? JMIUtils.getSourceElementIfExists((JavaClass)lastType) : curCls);
                                    }
                                } else if (lastType instanceof PrimitiveType) {
                                    List res = Collections.singletonList(getJMIItemFactory().createVarResultItem("class", jmiUtils.getExactClass("java.lang.Class"), Modifier.PUBLIC | Modifier.STATIC | Modifier.FINAL)); // NOI18N
                                    result = new JavaResult(component, res, formatType(lastType, true, true, true), item, curCls);
                                }
                            } else { // currently package
                                String searchName = lastPkg.getName() + '.' + var;
                                if (findType || !last) {
                                    lastPkg = jmiUtils.getExactPackage(searchName);
                                    if (lastPkg == null) { // package doesn't exist
                                        ClassDefinition cls = jmiUtils.getExactClass(searchName);
                                        if (cls != null) {
                                            lastType = cls;
                                        } else {
                                            lastType = null;
                                            cont = false;
                                        }
                                    }
                                } else { // last and searching for completion output
                                    if (last) { // get all matching fields/methods/packages
                                        String searchPkg = lastPkg.getName() + '.' + var;
                                        List res = new ArrayList();
                                        if (!isAnnotation) {
                                           res.addAll(jmiUtils.getImportedInnerClasses(var, false, curCls, isThisContext, !isGeneric, isConstructor));
                                            res.addAll(jmiUtils.findClasses(lastPkg, var, false, false, false, curCls, !isGeneric, isConstructor)); // matching classes
                                        }
                                        if (isAnnotation || isImport) {
                                            res.addAll(jmiUtils.findAnnotations(lastPkg, var, false, openingSource ? null : curCls, !isGeneric));
                                        }
                                        res.addAll(jmiUtils.findPackages(searchPkg, false, false, !isGeneric)); // find matching subpackages
                                        if (isGeneric)
                                            res = filterGenericTypeArguments(res);
                                        result = new JavaResult(component, res, (isAnnotation ? "@" : "") + searchPkg + '*', item, curCls); // NOI18N
                                    }
                                }
                            }
                        }
                        break;

                }
                break;

            case JCExpression.ARRAY:
                Type arrType = resolveType(item.getParameter(0), true);
                cont = false;
                if (arrType != null) { // must be type
                    if (item.getParameterCount() == 2) { // index in array follows
                        Type arrIdxType = resolveType(item.getParameter(1), false);
                        if (arrType instanceof Array && arrIdxType != null && arrIdxType instanceof PrimitiveType && ((PrimitiveType)arrIdxType).getKind().equals(PrimitiveTypeKindEnum.INT)) {
                            lastType = ((Array)arrType).getType();
                            staticOnly = false;
                            cont = true;
                        }
                    } else if (staticOnly) { // no index, increase array depth
                        lastType = jmiUtils.resolveArray(arrType);
                        if ("new".equals(item.getTokenText(0))) // NOI18N
                            staticOnly = false;
                        cont = true;
                    }
                }
                break;

            case JCExpression.GENERIC_TYPE:
                lastType = resolveType(item.getParameter(0), false);
                if (lastType instanceof JavaClass && !((JavaClass)lastType).getTypeParameters().isEmpty()) {
                    List params = new ArrayList(item.getParameterCount() - 1);
                    for (int i = 1; i < item.getParameterCount(); i++)
                        params.add(resolveType(item.getParameter(i), false));
                    lastType = jmiUtils.resolveParameterizedType((JavaClass)lastType, params);
                }
                if (isConstructor && !findType && lastType instanceof ParameterizedType) {
                    List ret = Collections.singletonList(new NbJMIResultItem.ClassResultItem((ParameterizedType)lastType, true, true, true));
                    result = new JavaResult(component, ret, formatType(lastType, false, false, true), item, endOffset, 0, curCls);
                }
                break;

            case JCExpression.INSTANCEOF:
                lastType = jmiUtils.resolveType("boolean"); // NOI18N
                break;

            case JCExpression.OPERATOR:
                if (!findType) {
                    List res = new ArrayList();
                    int pos = item.getTokenOffset(0);
                    Collection local = sup.getLocalVariableNames("", pos, openingSource); // NOI18N
                    for (Iterator it = local.iterator(); it.hasNext();) {
                        String name = (String) it.next();
                        Type t = (Type)sup.findType(name, pos);
                        if (t != null)
                            res.add(getJMIItemFactory().createVarResultItem(name, t, 0));
                    }
                    res.addAll(findFieldsMethodsAndInnerClasses(curCls, "", openingSource, true, curCls, sup.isStaticBlock(pos), true, item, local, true, false)); // NOI18N
                    res.addAll(findStaticallyImportedFeatures("", openingSource, curCls, true)); // NOI18N
                    res.addAll(jmiUtils.findPackages("", openingSource, false, true)); // NOI18N

                    result = new JavaResult(component, res, "*", item, endOffset, 0, curCls); // NOI18N
                }

                switch (item.getTokenID(0).getNumericID()) {
                    case JavaTokenContext.EQ_ID: // Assignment operators
                    case JavaTokenContext.PLUS_EQ_ID:
                    case JavaTokenContext.MINUS_EQ_ID:
                    case JavaTokenContext.MUL_EQ_ID:
                    case JavaTokenContext.DIV_EQ_ID:
                    case JavaTokenContext.AND_EQ_ID:
                    case JavaTokenContext.OR_EQ_ID:
                    case JavaTokenContext.XOR_EQ_ID:
                    case JavaTokenContext.MOD_EQ_ID:
                    case JavaTokenContext.LSHIFT_EQ_ID:
                    case JavaTokenContext.RSSHIFT_EQ_ID:
                    case JavaTokenContext.RUSHIFT_EQ_ID:
                        if (item.getParameterCount() > 0) {
                            lastType = resolveType(item.getParameter(0), false);
                            staticOnly = false;
                        }
                        break;

                    case JavaTokenContext.LT_ID: // Binary, result is boolean
                    case JavaTokenContext.GT_ID:
                    case JavaTokenContext.LT_EQ_ID:
                    case JavaTokenContext.GT_EQ_ID:
                    case JavaTokenContext.EQ_EQ_ID:
                    case JavaTokenContext.NOT_EQ_ID:
                    case JavaTokenContext.AND_AND_ID: // Binary, result is boolean
                    case JavaTokenContext.OR_OR_ID:
                        lastType = jmiUtils.resolveType("boolean"); // NOI18N
                        break;

                    case JavaTokenContext.LSHIFT_ID: // Always binary
                    case JavaTokenContext.RSSHIFT_ID:
                    case JavaTokenContext.RUSHIFT_ID:
                    case JavaTokenContext.MUL_ID:
                    case JavaTokenContext.DIV_ID:
                    case JavaTokenContext.AND_ID:
                    case JavaTokenContext.OR_ID:
                    case JavaTokenContext.XOR_ID:
                    case JavaTokenContext.MOD_ID:

                    case JavaTokenContext.PLUS_ID:
                    case JavaTokenContext.MINUS_ID:
                    case JavaTokenContext.COLON_ID:
                        switch (item.getParameterCount()) {
                        case 2:
                            Type typ1 = resolveType(item.getParameter(0), false);
                            Type typ2 = resolveType(item.getParameter(1), false);
                            if (typ1 != null && typ2 != null) {
                                lastType = sup.getCommonType(typ1, typ2);
                                staticOnly = false;
                            }
                            break;

                        case 1: // get the only one parameter
                            lastType = resolveType(item.getParameter(0), false);
                            staticOnly = false;
                            break;
                        }
                        break;

                    case JavaTokenContext.QUESTION_ID:
                        if (item.getParameterCount() >= 2) {
                            lastType = resolveType(item.getParameter(1), false); // should be colon
                            staticOnly = false;
                        }
                        break;
                }
                break;

            case JCExpression.UNARY_OPERATOR:
                if (item.getParameterCount() > 0) {
                    lastType = resolveType(item.getParameter(0), false);
                }
                break;

            case JCExpression.CONVERSION:
                lastType = resolveType(item.getParameter(0), false);
                staticOnly = false;
                break;

            case JCExpression.TYPE:
                lastType = jmiUtils.resolveType(item.getType());
                break;

            case JCExpression.PARENTHESIS:
                cont = resolveItem(item.getParameter(0), first, last);
                break;

            case JCExpression.IMPORT: // import statement
                isImport = true;
                if (item.getParameterCount() == 2) {
                    isStaticImport = true;
                    cont = resolveExp(item.getParameter(1));
                } else if (item.getParameterCount() == 1) {
                    cont = resolveExp(item.getParameter(0));
                }
                break;

            case JCExpression.GENERIC_TYPE_OPEN:
                isGeneric = true;
                int lastIdx = item.getParameterCount() - 1;
                Type typ = resolveType(item.getParameter(0), false);
                List args = typ instanceof GenericElement ? ((GenericElement)typ).getTypeParameters() : Collections.EMPTY_LIST;
                if (args.size() > 0) {
                    List tList = getTypeList(item, 1, false);
                    JCExpression lastParam = item.getParameter(lastIdx);
                    if (lastParam.getTokenCount() > 0 && lastParam.getTokenText(0).length() == 0) {
                        lastIdx--;
                        if (tList != null)
                            tList.remove(tList.size() - 1);
                    }
                    if (lastIdx < args.size() && lastIdx < item.getTokenCount()) {
                        List tmp = jmiUtils.findMatchingTypes((TypeParameter) args.get(lastIdx), null, true);
                        tmp.addAll(jmiUtils.findPackages("", openingSource, false, true)); // add matching packages
                        String parmStr = formatTypeList(tList, true);
                        result = new JavaResult(component, tmp, formatType(typ, true, false, false) + '<' + parmStr + '>', item, endOffset, 0, curCls);
                    } else if (lastIdx <= args.size() && lastIdx >= item.getTokenCount()) {
                        TypeParameter tp = ((TypeParameter)args.get(lastIdx - 1));
                        typeBounds = new ArrayList();
                        typeBounds.add(tp.getSuperClass());
                        typeBounds.addAll(tp.getInterfaces());
                        resolveExp(item.getParameter(lastIdx));
                    }
                } else {
                    isGeneric = false;
                    resolveExp(item.getParameter(1));
                }
                break;

            case JCExpression.CONSTRUCTOR: // constructor can be part of a DOT expression
                isConstructor = true;
                cont = resolveExp(item.getParameter(0));
                staticOnly = false;
                break;

            case JCExpression.ANNOTATION:
                if (item.getParameterCount() == 0) { // "@"
                    List res = jmiUtils.findAnnotations(null, "", false, openingSource ? null : curCls, false);
                    res.addAll(jmiUtils.findPackages("", false, false, true)); // add matching packages
                    result = new JavaResult(component, res, "@*", item, endOffset, 0, curCls); // NOI18N
                } else { // "@..."
                    isAnnotation = true;
                    cont = resolveExp(item.getParameter(0));
                }
                break;

            case JCExpression.ANNOTATION_OPEN:
                Type aType = resolveType(item.getParameter(0), false);
                if (aType instanceof AnnotationType) {
                    AnnotationType annType = (AnnotationType) aType;
                    int parCnt = item.getParameterCount();
                    if (parCnt == 1 || parCnt == item.getTokenCount()) {
                        Set usedMemberNames = getAnnotationMemberNames(item);
                        List res = findAnnotationMembers(annType, "", usedMemberNames); //NOI18N
                        if (parCnt == 1 && res.size() == 1) {
                            lastType = ((Attribute)((NbJMIResultItem.AttributeResultItem)res.get(0)).getAssociatedObject()).getType();
                            if (lastType instanceof ParameterizedType)
                                lastType = ((ParameterizedType)lastType).getDefinition();
                            res.addAll(findConstMembers(curCls, "", false, openingSource ? null : curCls)); //NOI18N
                            res.addAll(jmiUtils.findPackages("", false, false, false)); //NOI18N
                            if (lastType instanceof JavaEnum) {
                                List l = jmiUtils.findEnumConstants((JavaEnum)lastType, "", false, false, true);
                                for (Iterator it = l.iterator(); it.hasNext();)
                                    res.add(jmiItemFactory.createFieldResultItem((Field)it.next(), (JavaEnum)lastType, curCls));
                            } else if (lastType instanceof PrimitiveType && ((PrimitiveType)lastType).getKind() == PrimitiveTypeKindEnum.BOOLEAN) { //NOI18N
                                res.add(jmiItemFactory.createStringResultItem("true")); //NOI18N
                                res.add(jmiItemFactory.createStringResultItem("false")); //NOI18N
                            }
                        }
                        result = new JavaResult(component, res, formatType(annType, true, false, false) + "(*)", item, endOffset, 0, curCls); // NOI18N
                    } else if (parCnt > 1) {
                        JCExpression lastMember = item.getParameter(parCnt - 1);
                        switch (lastMember.getExpID()) {
                            case JCExpression.OPERATOR:
                                if (lastMember.getParameterCount() == 2) {
                                    isAnnotationOpen = true;
                                    List annMembers = jmiUtils.findAnnotationMembers(annType, lastMember.getParameter(0).getTokenText(0), true, false, false);
                                    if (annMembers.size() > 0) {
                                        lastType = ((Attribute)annMembers.get(0)).getType();
                                        if (lastType instanceof ParameterizedType)
                                            lastType = ((ParameterizedType)lastType).getDefinition();
                                    }
                                    resolveExp(lastMember.getParameter(1));
                                }
                                break;
                            case JCExpression.VARIABLE:
                                Set usedMemberNames = getAnnotationMemberNames(item);
                                String name = lastMember.getTokenText(0);
                                List res = findAnnotationMembers(annType, name, usedMemberNames);
                                List lst = jmiUtils.findAnnotationMembers(annType, "", false, false, false); //NOI18N
                                if (lst.size() == 1) {
                                    lastType = ((Attribute)lst.get(0)).getType();
                                    if (lastType instanceof ParameterizedType)
                                        lastType = ((ParameterizedType)lastType).getDefinition();
                                    res.addAll(findConstMembers(curCls, name, false, openingSource ? null : curCls));
                                    if (name.length() > 0)
                                        res.addAll(jmiUtils.findClasses(null, name, false, false, true, curCls, false, false));
                                    res.addAll(jmiUtils.findPackages(name, false, false, false));
                                    if (lastType instanceof JavaEnum) {
                                        List l = jmiUtils.findEnumConstants((JavaEnum)lastType, name, false, false, true);
                                        for (Iterator it = l.iterator(); it.hasNext();)
                                            res.add(jmiItemFactory.createFieldResultItem((Field)it.next(), (JavaEnum)lastType, curCls));
                                    } else if (lastType instanceof PrimitiveType && ((PrimitiveType)lastType).getKind() == PrimitiveTypeKindEnum.BOOLEAN) { //NOI18N
                                        if (jmiUtils.startsWith("true", name)) //NOI18N
                                            res.add(jmiItemFactory.createStringResultItem("true")); //NOI18N
                                        if (jmiUtils.startsWith("false", name)) //NOI18N
                                            res.add(jmiItemFactory.createStringResultItem("false")); //NOI18N
                                    }
                                }
                                result = new JavaResult(component, res, formatType(annType, true, false, false) + "(*)", lastMember, curCls); // NOI18N
                                break;
                            case JCExpression.DOT:
                            case JCExpression.DOT_OPEN:
                                isAnnotationOpen = true;
                                resolveExp(lastMember);
                                break;
                        }
                    }
                }
                break;

            case JCExpression.METHOD_OPEN:
                methodOpen = true;
                // let it flow to method
            case JCExpression.METHOD:
                JCExpression mtdNameExp = item.getParameter(0);
                String mtdName = mtdNameExp.getTokenText(0);
                List genericParams = null;

                if (isConstructor && "<".equals(mtdName) && mtdNameExp.getParameterCount() > 0) { //NOI18N
                    mtdName = mtdNameExp.getParameter(0).getTokenText(0);
                    if (mtdNameExp.getParameterCount() > 1)
                        genericParams = getTypeList(mtdNameExp, 1, false);
                }

                // this() invoked, offer constructors
                if( ("this".equals(mtdName)) && (item.getTokenCount()>0) ){ //NOI18N
                    if (curCls != null) {
                        isConstructor = true;
                        mtdName = curCls.getName();
                    }
                }

                // super() invoked, offer constructors for super class
                if( ("super".equals(mtdName)) && (item.getTokenCount()>0) ){ //NOI18N
                    JavaClass cls = curCls;
                    if (cls != null) {
                        cls = cls.getSuperClass();
                        if (cls != null) {
                            if (cls instanceof ParameterizedType)
                                cls = ((ParameterizedType)cls).getDefinition();
                            isConstructor = true;
                            superCall = true;
                            mtdName = cls.getName();
                        }
                    }
                }

                if (isConstructor) { // Help for the constructor
                    Type type = null;
                    if (first) {
                        if (lastType instanceof JavaClass) {
                            Iterator iter = jmiUtils.findInnerClasses(lastType, mtdName, openingSource, true, 
                                    sup.getJavaClass(item.getTokenOffset(0)), false, false, false, false).iterator();
                            type = iter.hasNext() ? (Type)iter.next() : null;
                        } else {
                            type = sup.getTypeFromName(mtdName, true, sup.getJavaClass(item.getTokenOffset(0)), !openingSource);
                        }
                        
                    } else { // not first
                        if ((last)&&(lastPkg != null)) { // valid package
                            type = jmiUtils.getExactClass(mtdName, lastPkg.getName());
                        } else if (lastType != null) {
                            if(last){ // inner class
                                type = jmiUtils.getExactClass(mtdName, lastType.getName());
                            }else{
                                type = lastType;
                            }
                        }
                    }
                    if (type instanceof JavaClass) {
                        lastType = (type instanceof ParameterizedType && genericParams == null) ? type : jmiUtils.resolveParameterizedType((JavaClass)type, genericParams);
                        List ctrList = jmiUtils.findConstructors(lastType, openingSource ? null : curCls, superCall);
                        List typeList = getTypeList(item, 1, true);
                        ctrList = sup.filterMethods(ctrList, typeList, methodOpen);
                        if (ctrList.size() > 0 && last && !findType) {
                            String parmStr = formatTypeList(typeList, methodOpen);
                            result = new JavaResult(component, ctrList, formatType(lastType, true, false, false) + '(' + parmStr + ')', item, lastType instanceof JavaClass ? JMIUtils.getSourceElementIfExists((JavaClass) lastType) : curCls);
                        }
                        staticOnly = false;
                    } else {
                        isConstructor = false;
                    }
                }
                if (!isConstructor){
                    // Help for the method
                    if (first) {
                        if (curCls != null) {
                            lastType = curCls;
                            isThisContext = true;
                        }
                    }
                    if (lastType != null) {
                        List mtdList = jmiUtils.findMethods(lastType, mtdName, true, first, openingSource ? null : curCls, false, isThisContext, null, false, last && !findType);
                        mtdList.addAll(jmiUtils.getStaticallyImportedMethods(mtdName, true, openingSource ? null : curCls, isThisContext));
                        List typeList = getTypeList(item, 1, true);
                        mtdList = sup.filterMethods(mtdList, typeList, methodOpen);
                        if (mtdList.size() > 0) {
                            if (last && !findType) {
                                String parmStr = formatTypeList(typeList, methodOpen);
                                result = new JavaResult(component, mtdList,
                                                        formatType(lastType, true, true, false) + mtdName + '(' + parmStr + ')',
                                                        item, lastType instanceof JavaClass ? JMIUtils.getSourceElementIfExists((JavaClass) lastType) : curCls);
                            } else {
                                lastType = ((Method)mtdList.get(0)).getType();
                                staticOnly = false;
                            }
                        } else {
                            lastType = null; // no method found
                            cont = false;
                        }
                    } else { // package.method() is invalid
                        lastPkg = null;
                        cont = false;
                    }
                }
                isConstructor = false;
                break;
            }

            if (lastType == null && lastPkg == null) { // !!! shouldn't be necessary
                cont = false;
            }

            return cont;
        }

        private List filterGenericTypeArguments(List res) {
            if (typeBounds != null && typeBounds.size() > 0) {
                List tmp = new ArrayList();
                for (Iterator it = res.iterator(); it.hasNext();) {
                    Object o = it.next();
                    boolean toAdd = true;
                    if (o instanceof JavaClass) {
                        for (Iterator itt = typeBounds.iterator(); itt.hasNext();) {
                            JavaClass bound = (JavaClass) itt.next();
                            if (!jmiUtils.isAssignable((JavaClass)o, bound)) {
                                toAdd = false;
                                break;
                            }
                        }
                    }
                    if (toAdd) {
                        tmp.add(o);
                    }
                }
                res = tmp;
            }
            return res;
        }

        private Set getAnnotationMemberNames(JCExpression item) {
            HashSet names = new HashSet();
            int parmCnt = item.getParameterCount();
            if (parmCnt > 1) {
                for (int i = 1; i < parmCnt; i++) {
                    JCExpression parm = item.getParameter(i);
                    if (parm.getExpID() == JCExpression.OPERATOR && parm.getParameterCount() > 0)
                        names.add(parm.getParameter(0).getTokenText(0));
                }
            }
            return names;
        }

        private List getTypeList(JCExpression item, int firstChildIdx, boolean allowNullItems) {
            int parmCnt = item.getParameterCount();
            ArrayList typeList = new ArrayList();
            if (parmCnt > firstChildIdx) { // will try to filter by parameters
                for (int i = firstChildIdx; i < parmCnt; i++) {
                    JCExpression parm = item.getParameter(i);
                    Type typ = resolveType(parm, false);
                    if (typ == null && !allowNullItems)
                        return null;
                    typeList.add(typ);
                }
            }
            return typeList;
        }

        private String formatTypeList(List typeList, boolean methodOpen) {
            StringBuffer sb = new StringBuffer();
            if (typeList == null) {
                sb.append('?'); // NOI18N
            } else {
                if (typeList.size() > 0) {
                    int cntM1 = typeList.size() - 1;
                    for (int i = 0; i <= cntM1; i++) {
                        Type t = (Type)typeList.get(i);
                        if (t != null) {
                            sb.append(formatType(t, false, false, false));
                        } else {
                            sb.append('?'); // NOI18N
                        }
                        if (i < cntM1) {
                            sb.append(", "); // NOI18N
                        }
                    }
                    if (methodOpen) {
                        sb.append(", *"); // NOI18N
                    }
                } else { // no parameters
                    if (methodOpen) {
                        sb.append("*"); // NOI18N
                    }
                }
            }
            return sb.toString();
        }

        private List findFieldsMethodsAndInnerClasses(Type type, String name, boolean exactMatch, boolean inspectOuterClasses, JavaClass context, boolean staticContext, boolean thisContext, JCExpression exp, Collection localVarNames, boolean simple, boolean isInstanceCreationCtx) {
            List ret = new ArrayList();
            for (Iterator it = jmiUtils.findFeatures(type, name, exactMatch, inspectOuterClasses, context, staticContext, thisContext, exp, true, true, isInstanceCreationCtx, true).iterator(); it.hasNext();) {
                Object f = it.next();
                if (f instanceof NbJMIResultItem.MethodResultItem ||
                    f instanceof NbJMIResultItem.FieldResultItem && (localVarNames == null || !localVarNames.contains(((NbJMIResultItem.FieldResultItem)f).getFieldName())) ||
                    ((staticContext || simple) && f instanceof NbJMIResultItem.ClassResultItem))
                    ret.add(f);
            }
            return ret;
        }

        private List findInnerClasses(Type type, String name, boolean exactMatch, boolean inspectOuterClasses, JavaClass context, boolean thisContext, boolean createResultItems, boolean preferSources, boolean isInstanceCreationCtx) {
            return jmiUtils.findInnerClasses(type, name, exactMatch, inspectOuterClasses, context, thisContext, createResultItems, preferSources, false);
        }

        private List findStaticallyImportedFeatures(String name, boolean exactMatch, JavaClass context, boolean isThisContext) {
            List ret = jmiUtils.getStaticallyImportedFields(name, exactMatch, context, isThisContext);
            ret.addAll(jmiUtils.getStaticallyImportedMethods(name, exactMatch, context, isThisContext));
            return ret;
        }

        private List findAnnotationMembers(AnnotationType type, String name, Set namesToSkip) {
            List members = jmiUtils.findAnnotationMembers(type, name, false, true, true);
            if (namesToSkip.size() == 0)
                return members;
            List ret = new ArrayList();
            for (Iterator it = members.iterator(); it.hasNext();) {
                NbJMIResultItem.AttributeResultItem attr = (NbJMIResultItem.AttributeResultItem)it.next();
                if (!namesToSkip.contains(attr.getAttrName()))
                    ret.add(attr);
            }
            return ret;
        }

        private List findConstMembers(Type type, String name, boolean exactMatch, JavaClass context) {
            List flds = jmiUtils.findFields(type, name, exactMatch, true, context, true, false, false, true);
            flds.addAll(jmiUtils.getStaticallyImportedFields(name, exactMatch, context, false));
            List ret = new ArrayList();
            for (Iterator it = flds.iterator(); it.hasNext();) {
                Field field = (Field) it.next();
                if (Modifier.isFinal(field.getModifiers()))
                    ret.add(field);
            }
            return ret;
        }
    }


    public static class JavaResult extends CompletionQuery.AbstractResult {

        /** Expression to substitute */
        private JCExpression substituteExp;

        /** Starting position of the text to substitute */
        private int substituteOffset;

        /** Length of the text to substitute */
        private int substituteLength;

        /** Component to update */
        private JTextComponent component;

        public JavaResult(JTextComponent component, List data, String title,
                   JCExpression substituteExp, ClassDefinition context) {
            this(component, data, title, substituteExp, substituteExp.getTokenOffset(0),
                 substituteExp.getTokenLength(0), context);
        }

        public JavaResult(JTextComponent component, List data, String title,
                   JCExpression substituteExp, int substituteOffset,
                   int substituteLength, ClassDefinition context) {
            super(convertData(data, substituteExp, context, substituteOffset), title);
            this.component = component;
            this.substituteExp = substituteExp;
            this.substituteOffset = substituteOffset;
            this.substituteLength = substituteLength;
        }

        private static List convertData(List dataList, JCExpression substituteExp, ClassDefinition context, int substituteOffset){
            Iterator iter = dataList.iterator();
            List ret = new ArrayList();
            while (iter.hasNext()){
                Object obj = iter.next();
                NbJMIResultItem item;
                if (obj instanceof CompletionQuery.ResultItem){
                    item = (NbJMIResultItem)obj;
                }else{
                    item = getJMIItemFactory().createResultItem(obj, substituteExp, context, false);
                }
                if (item != null) {
                    item.setSubstituteOffset(substituteOffset);
                    ret.add(item);
                }
            }
            return ret;
        }


        protected JTextComponent getComponent(){
            return component;
        }

        protected int getSubstituteLength(){
            return substituteLength;
        }

        protected int getSubstituteOffset(){
            return substituteOffset;
        }

        protected JCExpression getSubstituteExp(){
            return substituteExp;
        }

        /** Get the text that is normally filled into the text if enter is pressed. */
        protected String getMainText(Object dataItem) {
            String text = null;
            if (dataItem instanceof NbJMIResultItem){
                dataItem = ((NbJMIResultItem)dataItem).getAssociatedObject();
            }
            if (dataItem instanceof NamedElement) {
                text = ((NamedElement)dataItem).getName();
                if (dataItem instanceof JavaPackage) {
                    text = text.substring(text.lastIndexOf('.') + 1);
                }
            }
            return text;
        }

        /** Get the text that is common to all the entries in the query-result */
        protected String getCommonText(String prefix) {
            List data = getData();
            int cnt = data.size();
            int prefixLen = prefix.length();
            String commonText = null;
            for (int i = 0; i < cnt; i++) {
                String mainText = getMainText(data.get(i));
                if (mainText != null && mainText.startsWith(prefix)) {
                    mainText = mainText.substring(prefixLen);
                    if (commonText == null) {
                        commonText = mainText;
                    }
                    // Get largest common part
                    int minLen = Math.min(mainText.length(), commonText.length());
                    int commonInd;
                    for (commonInd = 0; commonInd < minLen; commonInd++) {
                        if (mainText.charAt(commonInd) != commonText.charAt(commonInd)) {
                            break;
                        }
                    }
                    if (commonInd != 0) {
                        commonText = commonText.substring(0, commonInd);
                    } else {
                        return null; // no common text
                    }
                }
            }
            return prefix + ((commonText != null) ? commonText : ""); // NOI18N
        }

        /** Update the text in response to pressing TAB key.
        * @return whether the text was successfully updated
        */
        public boolean substituteCommonText(int dataIndex) {
            List data = getData();
            if( data.size() == 0 ){
                return false;
            }

            Object obj = getData().get( dataIndex );
            if (obj instanceof CompletionQuery.ResultItem){
                //return super.substituteCommonText(dataIndex); [PENDING]
                // how to get getCommonText to CompletionQuery.ResultItem ???
            }

            BaseDocument doc = (BaseDocument)component.getDocument();
            try {
                String prefix = doc.getText(substituteOffset, substituteLength);
                String commonText = getCommonText(prefix);
                if (commonText != null) {
                    if(substituteExp!=null){
                        if( (substituteExp.getExpID()==JCExpression.METHOD_OPEN) || (substituteExp.getExpID()==JCExpression.METHOD) )
                            return true;
                    }
                    doc.atomicLock();
                    try {
                        doc.remove(substituteOffset, substituteLength);
                        doc.insertString(substituteOffset, commonText, null);
                    } finally {
                        doc.atomicUnlock();
                    }
                }
            } catch (BadLocationException e) {
                // no updating
            }
            return true;
        }

        /** Update the text in response to pressing ENTER.
        * @return whether the text was successfully updated
        */
        public boolean substituteText(int dataIndex, boolean shift ) {

            Object actData = getData().get( dataIndex );
            if (actData instanceof CompletionQuery.ResultItem){
                return ((CompletionQuery.ResultItem)actData).substituteText( component, substituteOffset, substituteLength, shift );
            }

            // the rest part of code is here only for backward compatibility...
            // it should be removed later if all data will be CompletionQuery.ResultItem

            BaseDocument doc = (BaseDocument)component.getDocument();
            String text = null;
            int selectionStartOffset = -1;
            int selectionEndOffset = -1;
            Object replacement = getData().get(dataIndex);

            if (replacement instanceof JavaPackage) {
                text = ((JavaPackage)replacement).getName();
                text = text.substring(text.lastIndexOf('.') + 1);
            } else if (replacement instanceof JavaClass) {
                text = ((JavaClass)replacement).getName();
            } else if (replacement instanceof Field) {
                text = ((Field)replacement).getName();

            } else if (replacement instanceof CallableFeature) {
                CallableFeature mtd = (CallableFeature)replacement;
                switch ((substituteExp != null) ? substituteExp.getExpID() : -1) {
                case JCExpression.METHOD:
                    // no substitution
                    break;

                case JCExpression.METHOD_OPEN:
                    List parms = mtd.getParameters();
                    if (parms.size() == 0) {
                        text = ")"; // NOI18N
                    } else { // one or more parameters
                        int ind = substituteExp.getParameterCount();
                        boolean addSpace = false;
                        Formatter f = doc.getFormatter();
                        if (f instanceof ExtFormatter) {
                            Object o = ((ExtFormatter)f).getSettingValue(JavaSettingsNames.JAVA_FORMAT_SPACE_AFTER_COMMA);
                            if ((o instanceof Boolean) && ((Boolean)o).booleanValue()) {
                                addSpace = true;
                            }
                        }

                        try {
                            if (addSpace && (ind == 0 || (substituteOffset > 0
                                                          && Character.isWhitespace(DocumentUtilities.getText(doc, substituteOffset - 1, 1).charAt(0))))
                               ) {
                                addSpace = false;
                            }
                        } catch (BadLocationException e) {
                        }

                        if (ind < parms.size()) {
                            text = addSpace ? " " : ""; // NOI18N
                            selectionStartOffset = text.length();
                            text += ((Parameter)parms.get(ind)).getName();
                            selectionEndOffset = text.length();
                        }
                    }
                    break;

                default:
                    text = getMainText(replacement);
                    boolean addSpace = false;
                    Formatter f = doc.getFormatter();
                    if (f instanceof ExtFormatter) {
                        Object o = ((ExtFormatter)f).getSettingValue(JavaSettingsNames.JAVA_FORMAT_SPACE_BEFORE_PARENTHESIS);
                        if ((o instanceof Boolean) && ((Boolean)o).booleanValue()) {
                            addSpace = true;
                        }
                    }

                    if (addSpace) {
                        text += ' ';
                    }
                    text += '(';

                    parms = mtd.getParameters();
                    if (parms.size() > 0) {
                        selectionStartOffset = text.length();
                        text += ((Parameter)parms.get(0)).getName();
                        selectionEndOffset = text.length();
                    } else {
                        text += ")"; // NOI18N
                    }
                    break;
                }

            }

            if (text != null) {
                // Update the text
                doc.atomicLock();
                try {
                    CharSequence textToReplace = DocumentUtilities.getText(doc, substituteOffset, substituteLength);
                    if (CharSequenceUtilities.textEquals(text, textToReplace)) return false;

                    doc.remove(substituteOffset, substituteLength);
                    doc.insertString(substituteOffset, text, null);
                    if (selectionStartOffset >= 0) {
                        component.select(substituteOffset + selectionStartOffset,
                                         substituteOffset + selectionEndOffset);
                    }
                } catch (BadLocationException e) {
                    // Can't update
                } finally {
                    doc.atomicUnlock();
                }
            }

            return true;
        }

    }

    protected void setJMIItemFactory(JMIItemFactory itemFactory){
        jmiItemFactory = itemFactory;
    }

    public static JMIItemFactory getJMIItemFactory(){
        return jmiItemFactory;
    }

    public interface JMIItemFactory{
        public NbJMIResultItem.VarResultItem createVarResultItem(String varName, Type type, int modifiers);
        public NbJMIResultItem.PackageResultItem createPackageResultItem(JavaPackage pkg);
        public NbJMIResultItem.ClassResultItem createClassResultItem(JavaClass cls, boolean addImport, boolean generateClassSkeleton);
        public NbJMIResultItem.FieldResultItem createFieldResultItem(Field fld, JavaClass owner, ClassDefinition context);
        public NbJMIResultItem.MethodResultItem createMethodResultItem(Method mtd, JCExpression substituteExp, ClassDefinition context);
        public NbJMIResultItem.ConstructorResultItem createConstructorResultItem(Constructor ctr, JCExpression substituteExp);
        public NbJMIResultItem.AttributeResultItem createAttributeResultItem(Attribute attr);
        public NbJMIResultItem.StringResultItem createStringResultItem(String str);
        public NbJMIResultItem createResultItem(Object obj, JCExpression substituteExp, ClassDefinition context, boolean isInstanceCreationCtx);
    }

    public static class DefaultJMIItemFactory implements JMIItemFactory{

        public NbJMIResultItem.VarResultItem createVarResultItem(String varName, Type type, int modifiers) {
            return new NbJMIResultItem.VarResultItem(varName, type, modifiers);
        }

        public NbJMIResultItem.PackageResultItem createPackageResultItem(JavaPackage pkg){
            return new NbJMIResultItem.PackageResultItem(pkg, false);
        }
        public NbJMIResultItem.ClassResultItem createClassResultItem(JavaClass cls, boolean addImport, boolean generateClassSkeleton){
            return cls instanceof AnnotationType ? new NbJMIResultItem.AnnotationResultItem((AnnotationType)cls, false, addImport) : new NbJMIResultItem.ClassResultItem(cls, false, addImport, generateClassSkeleton);
        }
        public NbJMIResultItem.FieldResultItem createFieldResultItem(Field fld, JavaClass owner, ClassDefinition context){
            return new NbJMIResultItem.FieldResultItem(fld, owner, context);
        }
        public NbJMIResultItem.MethodResultItem createMethodResultItem(Method mtd, JCExpression substituteExp, ClassDefinition context){
            return new NbJMIResultItem.MethodResultItem(mtd, substituteExp, context);
        }
        public NbJMIResultItem.ConstructorResultItem createConstructorResultItem(Constructor ctr, JCExpression substituteExp){
            return new NbJMIResultItem.ConstructorResultItem(ctr, substituteExp);
        }
        public NbJMIResultItem.AttributeResultItem createAttributeResultItem(Attribute attr) {
            return new NbJMIResultItem.AttributeResultItem(attr);
        }
        public NbJMIResultItem.StringResultItem createStringResultItem(String str) {
            return new NbJMIResultItem.StringResultItem(str);
        }

        public NbJMIResultItem createResultItem(Object obj, JCExpression substituteExp, ClassDefinition context, boolean isInstanceCreationCtx){
            if (obj instanceof JavaPackage) {
                return createPackageResultItem((JavaPackage)obj);
            } else if (obj instanceof JavaClass) {
                return createClassResultItem((JavaClass)obj, true, isInstanceCreationCtx);
            } else if (obj instanceof Field) {
                return createFieldResultItem((Field)obj, null, context);
            } else if (obj instanceof Method) {
                return createMethodResultItem((Method)obj, substituteExp, context);
            } else if (obj instanceof Constructor) {
                return createConstructorResultItem((Constructor)obj, substituteExp);
            } else if (obj instanceof Attribute) {
                return createAttributeResultItem((Attribute)obj);
            } else if (obj instanceof String) {
                return createStringResultItem((String)obj);
            }
            return null;
        }

    }
}
