/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.java;

import java.text.MessageFormat;
import java.util.*;
import javax.swing.text.JTextComponent;

import org.netbeans.editor.Utilities;
import org.netbeans.editor.ext.java.JavaFastImport;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.editor.options.BaseOptions;
import org.netbeans.modules.java.editor.options.JavaOptions;

/**
 *
 * @author Dusan Balek
 * @version 1.0
 */

public class NbJavaJMIFastImport extends JavaFastImport {
    
    private static final int OK = 0;
    private static final int AMBIGUOUS = 1;
    private static final int CLASHING = 2;

    private JMIUtils jmiUtils;
    private NbJavaJMISyntaxSupport support;

    public NbJavaJMIFastImport(JTextComponent target) {
        super(target);
        jmiUtils = JMIUtils.get(Utilities.getDocument(target));
        support = (NbJavaJMISyntaxSupport)Utilities.getSyntaxSupport(target).get(NbJavaJMISyntaxSupport.class);
    }
    
    protected List findClasses(String exp, int importType) {
        List ret = new ArrayList();
        jmiUtils.beginTrans(false);
        try {
            List classes = jmiUtils.findClasses(null, exp, true, true, true, support.getTopJavaClass(), false, false);
            if (importType == 1){ // package import
                TreeSet ts = new TreeSet(JMIUtils.getNaturalMemberNameComparator());
                for (int i=0; i < classes.size(); i++){
                    JavaPackage pkg = (JavaPackage)((JavaClass)classes.get(i)).getResource().refImmediateComposite();
                    if (pkg != null){
                        ts.add(pkg);
                    }
                }
                Iterator it = ts.iterator();
                while (it.hasNext()) {
                    JavaPackage pkg = (JavaPackage)it.next();
                    ret.add(new NbJMIResultItem.PackageResultItem(pkg, true));
                }
            } else {
                for (Iterator it = classes.iterator(); it.hasNext();) {
                    ret.add(new NbJMIResultItem.ClassResultItem((JavaClass) it.next(), true, false, false));
                }
            }
        } finally {
            jmiUtils.endTrans(false);
        }
        return ret;
    }

    protected void setFastImportSettings(int importType){
        Class kitClass = Utilities.getKitClass(target);
        if (kitClass != null) {
            BaseOptions bop = BaseOptions.getOptions(kitClass);
            if (bop instanceof JavaOptions){
                ((JavaOptions)bop).setFastImportSelection(importType);
            }
        }
    }
    
    protected void updateImport(Object item) {
        if (item instanceof NbJMIResultItem.PackageResultItem || item instanceof NbJMIResultItem.ClassResultItem) {
            String fqn = null;
            jmiUtils.beginTrans(true);
            try {
                Resource resource = support.getResource();
                if (resource != null) {
                    if (item instanceof NbJMIResultItem.ClassResultItem){
                        JavaClass cls = (JavaClass)((NbJMIResultItem.ClassResultItem)item).getAssociatedObject();
                        int status = checkImport(cls, resource);
                        if (isClassAlreadyImported(cls, resource, status)) {
                            Utilities.setStatusText(target,
                                    MessageFormat.format(
                                             org.openide.util.NbBundle.getMessage(NbJavaJMIFastImport.class,"NJFI_CLASS_ALREADY_IMPORTED"), //NOI18N
                                             new Object[] {cls.getName()}));
                        } else {
                            addImport(cls, resource);
                        }
                    }else if (item instanceof NbJMIResultItem.PackageResultItem){
                        JavaPackage pkg = (JavaPackage)((NbJMIResultItem.PackageResultItem)item).getAssociatedObject();
                        if (isPackageAlreadyImported(pkg, resource)) {
                            Utilities.setStatusText(target,
                                    MessageFormat.format(
                                            org.openide.util.NbBundle.getMessage(NbJavaJMIFastImport.class,"NJFI_PACKAGE_ALREADY_IMPORTED"), //NOI18N
                                            new Object[] {pkg.getName()}));
                        } else {
                            addImport(pkg, resource);
                        }
                    }
                }
            } finally {
                jmiUtils.endTrans(false);
            }
            if (fqn != null)
                pasteFQN(fqn);
        }
    }
    
    boolean checkAutoImport(Object item) {
        if (item instanceof NbJMIResultItem.ClassResultItem) {
            jmiUtils.beginTrans(true);
            try {
                Resource resource = support.getResource();
                if (resource != null) {
                    JavaClass cls = (JavaClass)((NbJMIResultItem.ClassResultItem)item).getAssociatedObject();
                    if (!cls.isValid())
                        return false;
                    int status = checkImport(cls, resource);
                    if (status == CLASHING)
                        return true;
                    return !isClassAlreadyImported(cls, resource, status);
                }
            } finally {
                jmiUtils.endTrans(false);
            }
        }
        return false;
    }

     boolean autoImport(JavaClass cls, boolean pasteFQN) {
        jmiUtils.beginTrans(true);
        boolean fail = true;
        try {
            if (!cls.isValid())
                return true;
            Resource resource = support.getResource();
            if (resource != null) {
                int status = checkImport(cls, resource);
                if (status == CLASHING) {
                    if (pasteFQN)
                        pasteFQN(cls.getName());
                    fail = false;
                    return false;
                }
                if (!isClassAlreadyImported(cls, resource, status)) {
                    addImport(cls, resource);
                }
            }
            fail = false;
            return true;
        } finally {
            jmiUtils.endTrans(fail);
        }
    }
    
    protected String getItemFQN(Object item) {
        if (item instanceof NbJMIResultItem.ClassResultItem) {
            JavaClass cls = (JavaClass)((NbJMIResultItem.ClassResultItem)item).getAssociatedObject();
            return cls.getName();
        }
        return super.getItemFQN(item);
    }

    protected boolean isInnerClass(Object item) {
        if (item instanceof NbJMIResultItem.ClassResultItem) {
            JavaClass cls = (JavaClass)((NbJMIResultItem.ClassResultItem)item).getAssociatedObject();
            return cls.isInner();
        }
        return super.isInnerClass(item);
    }

    private boolean isClassAlreadyImported(JavaClass cls, Resource resource, int status) {
        String pkgName = jmiUtils.getPackageName(cls);
        if (!cls.isInner()) {
            if (("java.lang".equals(pkgName) || resource.getPackageName().equals(pkgName)) && status == OK) //NOI18N
                return true;
            for (Iterator it = resource.getImports().iterator(); it.hasNext();) {
                Import imp = (Import) it.next();
                if (imp.isOnDemand()) {
                    if (imp.getImportedElements().contains(cls) && status == OK)
                        return true;
                }
            }
        } else {
            if (status == OK && resource == cls.getResource())
                return true;
        }
        for (Iterator it = resource.getImports().iterator(); it.hasNext();) {
            Import imp = (Import) it.next();            
            if (!imp.isOnDemand() && cls.equals(imp.getImportedNamespace()))
                return true;
        }
        return false;
    }
    
    private int checkImport(JavaClass cls, Resource resource) {
        int status = OK;
        String simpleName = cls.getSimpleName();
        for (Iterator it = resource.getImports().iterator(); it.hasNext();) {
            Import imp = (Import) it.next();
            if (imp.isOnDemand()) {
                if (status == OK) {
                    for (Iterator itt = imp.getImportedElements().iterator(); itt.hasNext();) {
                        NamedElement el = (NamedElement)itt.next();
                        if (el instanceof JavaClass && simpleName.equals(((JavaClass)el).getSimpleName()) && !el.equals(cls)) {
                            status = AMBIGUOUS;
                            break;
                        }
                    }
                }
            } else {
                NamedElement el = imp.getImportedNamespace();
                if (el instanceof JavaClass && simpleName.equals(((JavaClass)el).getSimpleName()) && !el.equals(cls))
                    return CLASHING;
            }
        }
        return status;
    }

    private boolean isPackageAlreadyImported(JavaPackage pkg, Resource resource) {
        String pkgName = pkg.getName();
        if ("java.lang".equals(pkgName) || resource.getPackageName().equals(pkgName)) //NOI18N
            return true;
        for (Iterator it = resource.getImports().iterator(); it.hasNext();) {
            Import imp = (Import) it.next();            
            if (imp.getImportedNamespace().equals(pkg))
                return true;
        }
        return false;
    }

    private void addImport(NamedElement element, Resource resource) {
        String name = element.getName();
        ListIterator it = resource.getImports().listIterator();
        while (it.hasNext()) {
            Import imp = (Import) it.next();
            String impName = imp.getName();
            if (impName != null && name.compareTo(impName) < 0) {
                it.previous();
                break;
            }
        }
        ImportClass proxy = ((JavaModelPackage)resource.refOutermostPackage()).getImport();
        it.add(proxy.createImport(name, null, false, element instanceof JavaPackage));
    }
}
