/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.java;

import java.lang.reflect.Modifier;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;
import java.awt.Toolkit;
import javax.swing.event.DocumentEvent;

import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.ext.java.JavaSyntaxSupport;
import org.netbeans.editor.ext.java.JCExpression;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;

import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.nodes.Node;
import org.openide.text.PositionBounds;


/**
* Support methods for syntax analyzes working with java JMI interfaces.
*
* @author Dusan Balek, Martin Roskanin
* @version 1.00
*/

public class NbJavaJMISyntaxSupport extends NbJavaSyntaxSupport {

    private JMIUtils jmiUtils;
    private HashMap featuresAtPosMap = new HashMap();
            
    public synchronized JMIUtils getJMIUtils() {
	if (jmiUtils == null) {
	    jmiUtils = JMIUtils.get(getDocument());
	}
        return jmiUtils;
    }

    public NbJavaJMISyntaxSupport(BaseDocument doc) {
        super(doc);
        setJava15(true);
    }

    public JavaClass getJavaClass(int pos) {
        Feature f = getFeatureAtPos(pos, false);
        return (f == null || !f.isValid()) ? null : f instanceof JavaClass ? (JavaClass)f : (JavaClass)f.getDeclaringClass();
    }

    public boolean isStaticBlock(int pos) {
        Feature f = getFeatureAtPos(pos, false);
        return (f != null && f.isValid()) ? (f.getModifiers() & Modifier.STATIC) != 0 : false;
    }

    protected int getMethodStartPosition(int pos) {
        Feature f = getFeatureAtPos(pos, false);
        return f instanceof CallableFeature ? JavaMetamodel.getManager().getElementPosition(f).getBegin().getOffset() : -1;
    }

    /**
     * Find a feature at given position.
     *
     * @param pos position within a document
     * @param includingJavadoc if true and pos is inside a Javadoc, the feature corresponding to
     *                         the Javadoc is returned.
     * @return the feature found.
     */
    
    public Feature getFeatureAtPos(int pos, boolean includingJavadoc) {
        Integer posI = new Integer(pos);
        if (featuresAtPosMap.containsKey(posI)) {
            Element e = (Element)featuresAtPosMap.get(posI);
            if (e instanceof JavaDoc)
                return (Feature)e.refImmediateComposite();
            return (Feature)e;
        }
        Resource rsc = getResource();
        if (rsc == null) {
            featuresAtPosMap.put(posI, null);
            return null;
        }
        JavaClass jcls = null;
        Collection els = rsc.getClassifiers();
        boolean cont = true;
        while (cont) {
            cont = false;
            Object[] features = els.toArray();
            for (int i=0; i<features.length; i++) {
                Feature el = (Feature) features[i];
                PositionBounds bounds = JavaMetamodel.getManager().getElementPosition(el);
                if (bounds.getBegin().getOffset() < pos && bounds.getEnd().getOffset() > pos) {
                    if (el instanceof JavaClass) {
                        jcls = (JavaClass)el;
                        els = jcls.getFeatures();
                        cont = true;
                        break;
                    } else {
                        featuresAtPosMap.put(posI, el);
                        return el;
                    }
                }
                JavaDoc jd = el.getJavadoc();
                
                if (includingJavadoc && jd != null) {
                    PositionBounds jbounds = JavaMetamodel.getManager().getElementPosition(jd);
                    if (jbounds.getBegin().getOffset() < pos && jbounds.getEnd().getOffset() > pos) {
                        featuresAtPosMap.put(posI, jd);
                        return el;
                    }
                }
                
            }
        }
        featuresAtPosMap.put(posI, jcls);
        return jcls;
    }
    
    protected void documentModified(DocumentEvent evt) {
        super.documentModified(evt);
        featuresAtPosMap.clear();
    }
    
    public Object findType(String varName, int varPos) {
        Object obj = super.findType(varName, varPos);
        if (!(obj instanceof JavaSyntaxSupport.JavaVariable))
            return obj;
        JCExpression typeExp = ((JavaSyntaxSupport.JavaVariable)obj).getTypeExpression();
        JCExpression varExp = ((JavaSyntaxSupport.JavaVariable)obj).getVariableExpression();
        Type type = getType(typeExp);
        if (type != null)
            type = processArrayDepth(type, getArrayDepth(varExp));
        return type;
    }

    public int findLocalDeclarationPosition(String varName, int varPos) {
        Object obj = super.findType(varName, varPos);
        if (!(obj instanceof JavaSyntaxSupport.JavaVariable))
            return super.findLocalDeclarationPosition(varName, varPos);
        else
            return getExpressionPos(((JavaSyntaxSupport.JavaVariable)obj).getTypeExpression());
    }

    public Collection getLocalVariableNames(String namePrefix, int pos, boolean exactMatch) {
        Map varMap = getLocalVariableMap(pos);
        if (varMap != null) {
            if (namePrefix != null && namePrefix.length() > 0) {
                return getJMIUtils().filterNames(varMap.keySet(), namePrefix, exactMatch);
            } else if (!exactMatch) {
                return varMap.keySet();
            }
        }
        return Collections.EMPTY_LIST;
    }
    
    
    public Collection getLocalVariableNamesOfType(int offset, Type type) {
        Map localVarMap = getLocalVariableMap(offset);
        Collection varNames = getLocalVariableNames("", offset, false);
        List matchingVarNames = new ArrayList();
        for (Iterator it = varNames.iterator(); it.hasNext();) {
            String varName = (String)it.next();
            // setJava15(true) called in constructor -> OffsetJavaVariable
            OffsetJavaVariable var = (OffsetJavaVariable)localVarMap.get(varName);
            Type varType = getType(var.getTypeExpression());
            // Will the following cover primitive types
            if (isSubType(varType, type)) {
                matchingVarNames.add(varName);
            }
        }
        return matchingVarNames;
    }
    
    public Collection getGlobalVariableNamesOfType(int offset, Type type) {
        Map globalVarMap = getGlobalVariableMap(offset);
        List matchingVarNames = new ArrayList();
        for (Iterator it = globalVarMap.keySet().iterator(); it.hasNext();) {
            String varName = (String)it.next();
            // setJava15(true) called in constructor -> OffsetJavaVariable
            Type varType = (Type)globalVarMap.get(varName);
            // Will the following cover primitive types
            if (isSubType(varType, type)) {
                matchingVarNames.add(varName);
            }
        }
        return matchingVarNames;
    }
    
    public Collection getLocalVariableNamesOfArrayType(int offset) {
        Map localVarMap = getLocalVariableMap(offset);
        Collection varNames = getLocalVariableNames("", offset, false);
        List matchingVarNames = new ArrayList();
        for (Iterator it = varNames.iterator(); it.hasNext();) {
            String varName = (String)it.next();
            // setJava15(true) called in constructor -> OffsetJavaVariable
            OffsetJavaVariable var = (OffsetJavaVariable)localVarMap.get(varName);
            Type varType = getType(var.getTypeExpression());
            // Will the following cover primitive types
            if (varType instanceof Array) {
                matchingVarNames.add(varName);
            }
        }
        return matchingVarNames;
    }
    
    public Collection getGlobalVariableNamesOfArrayType(int offset) {
        Map globalVarMap = getGlobalVariableMap(offset);
        List matchingVarNames = new ArrayList();
        for (Iterator it = globalVarMap.keySet().iterator(); it.hasNext();) {
            String varName = (String)it.next();
            // setJava15(true) called in constructor -> OffsetJavaVariable
            Type varType = (Type)globalVarMap.get(varName);
            // Will the following cover primitive types
            if (varType instanceof Array) {
                matchingVarNames.add(varName);
            }
        }
        return matchingVarNames;
    }
    
    private static boolean isSubType(Type type, Type ofType) {
        boolean subType = type.equals(ofType);
        if (!subType && (type instanceof ClassDefinition) && (ofType instanceof ClassDefinition)) {
            // Try to cast to class and check subtype
            subType = ((ClassDefinition)type).isSubTypeOf((ClassDefinition)ofType);
        }
        return subType;
    }        

    private int getExpressionPos(JCExpression exp) {
        return Math.min(exp.getTokenCount() > 0 ? exp.getTokenOffset(0) : Integer.MAX_VALUE, exp.getParameterCount() > 0 ? getExpressionPos(exp.getParameter(0)) : Integer.MAX_VALUE);
    }

    private int getArrayDepth(JCExpression exp) {
        return exp.getExpID() == JCExpression.ARRAY ? exp.getTokenCount() / 2 : 0;
    }

    private String getTypeName(JCExpression exp) {
        switch (exp.getExpID()) {
            case JCExpression.VARIABLE:
            case JCExpression.TYPE:
                return exp.getTokenText(0);

            case JCExpression.DOT:
                StringBuffer sb = new StringBuffer();
                for(int i = 0; i < exp.getParameterCount(); i++) {
                    sb.append(getTypeName(exp.getParameter(i)));
                    if (i < exp.getParameterCount() - 1)
                        sb.append('.'); // NOI18N
                }
                return sb.toString();

            case JCExpression.GENERIC_TYPE:
                return getTypeName(exp.getParameter(0));

            default:
                throw new IllegalStateException();
        }
    }

    private Type getType(JCExpression typeExp) {
        Type type = null;
        switch (typeExp.getExpID()) {
            case JCExpression.GENERIC_WILD_CHAR:
                type = getJMIUtils().resolveType("java.lang.Object"); // NOI18N
                break;

            case JCExpression.ARRAY:
                type = processArrayDepth(getType(typeExp.getParameter(0)), getArrayDepth(typeExp));
                break;

            case JCExpression.GENERIC_TYPE:
                type = getType(typeExp.getParameter(0));
                if (type instanceof JavaClass && !((JavaClass)type).getTypeParameters().isEmpty()) {
                    List params = new ArrayList(typeExp.getParameterCount() - 1);
                    for (int i = 1; i < typeExp.getParameterCount(); i++)
                        params.add(getType(typeExp.getParameter(i)));
                    type = getJMIUtils().resolveParameterizedType((JavaClass)type, params);
                }
                break;

            case JCExpression.DOT:
                type = getType(typeExp.getParameter(0));
                if (type != null && !(type instanceof UnresolvedClass)) {
                    int i = 1;
                    while(type instanceof ClassDefinition && i < typeExp.getParameterCount()) {
                        type = ((ClassDefinition)type).getInnerClass(getTypeName(typeExp.getParameter(i++)), true);
                        if (type instanceof JavaClass && !((JavaClass)type).getTypeParameters().isEmpty())
                            type = getJMIUtils().resolveParameterizedType((JavaClass)type, null);
                    }
                    break;
                }
                type = getJMIUtils().resolveType(getTypeName(typeExp)); // try FQN

            default:
                String typeName = getTypeName(typeExp);
                // Maybe it's an inner class or a type parameter. Stick an outerClass before it ...
                JavaClass outerCls = getJavaClass(typeExp.getTokenOffset(0));
                if (outerCls != null){
                    JavaClass innerClass = outerCls.getInnerClass(typeName, true); //NOI18N
                    if (innerClass != null){
                        type = innerClass;
                    } else {
                        for (Iterator it = outerCls.getTypeParameters().iterator(); it.hasNext();) {
                            TypeParameter tp = (TypeParameter) it.next();
                            if (tp.getName().equals(typeName)) {
                                type = tp;
                                break;
                            }
                        }
                    }
                }
                if (type == null) {
                    Feature f = getFeatureAtPos(typeExp.getTokenOffset(0), false);
                    if (f instanceof CallableFeature) {
                        for (Iterator it = ((CallableFeature)f).getTypeParameters().iterator(); it.hasNext();) {
                            TypeParameter tp = (TypeParameter) it.next();
                            if (tp.getName().equals(typeName)) {
                                type = tp;
                                break;
                            }
                        }
                    }
                }
                if (type == null){
                    type = getTypeFromName(typeName, true, outerCls, true);
                }
                if (type instanceof JavaClass && !((JavaClass)type).getTypeParameters().isEmpty())
                    type = getJMIUtils().resolveParameterizedType((JavaClass)type, null);
                if (type == null)
                    type = getJMIUtils().resolveType(typeName);
        }
        return type;
    }

    private Type processArrayDepth(Type typ, int arrayDepth) {
        while (arrayDepth > 0) {
            typ = getJMIUtils().resolveArray(typ);
            arrayDepth--;
        }
        return typ;
    }

    public Type getTypeFromName(String name, boolean searchByName, JavaClass context, boolean useContext) {

        // try to resolve as a primitive type or a FQN class
        Type ret = getJMIUtils().resolveType(name);
        if (ret != null && !(ret instanceof UnresolvedClass))
            return ret;

        // try to resolve as an inner class
        JavaClass topClass = getTopJavaClass();
        if (context != null) {
            ret = context.getInnerClass(name, true);
        } else if (topClass != null) {
            ret = topClass.getInnerClass(name, true);
        } else {
            ret = null;
        }
        if (ret != null)
            return ret;

        // check imports and package content
        JavaClass importedClass = getJMIUtils().getImportedClass(name, topClass, useContext ? context : null, getResource());
        if (importedClass != null) return importedClass;

        if (searchByName) {
            List clsList = getJMIUtils().findClasses(null, name, true, false, true, useContext ? context : null, false, false);
            int maxWeight = 0;
            for (Iterator it = clsList.iterator(); it.hasNext();) {
                JavaClass javaClass = (JavaClass) it.next();
                int weight = getJMIUtils().getDefaultSelectionWeight(javaClass);
                if (weight > maxWeight) {
                    maxWeight = weight;
                    ret = javaClass;
                }
            }
            return ret;
        }

        return null;
    }

    public Resource getResource() {
        return getJMIUtils().getResource();
    }

    public JavaClass getTopJavaClass(Resource res) {
        if (res != null) {
            Iterator cls = res.getClassifiers().iterator();
            if (cls.hasNext())
                return (JavaClass)cls.next();
        }
        return null;
    }

    public JavaClass getTopJavaClass() {
        return getTopJavaClass(getResource());
    }

    protected Map buildGlobalVariableMap(int pos) {
        refreshClassInfo();
        JavaClass cls = getJavaClass(pos);
        if (cls == null)
            cls = getTopJavaClass();
        if (cls != null) {
            HashMap varMap = new HashMap();
            List fldList = getJMIUtils().findFields(cls, "", false, true, null, false, false, false, false); // NOI18N
            for (int i = fldList.size() - 1; i >= 0; i--) {
                Field fld = (Field)fldList.get(i);
                varMap.put(fld.getName(), fld.getType());
            }
            return varMap;
        }
        return null;
    }

    public Type getCommonType(Type typ1, Type typ2) {
        if (getJMIUtils().isAssignable(typ1, typ2)) {
            return typ1;
        } else if (getJMIUtils().isAssignable(typ2, typ1)) {
            return typ2;
        } else {
            return null;
        }
    }

    public List filterMethods(List methodList, List parmTypeList,
                              boolean acceptMoreParameters) {
        if (parmTypeList == null) {
            return Collections.EMPTY_LIST;
        }

        List ret = new ArrayList();
        int parmTypeCnt = parmTypeList.size();
        for (Iterator it = methodList.iterator(); it.hasNext();) {
            CallableFeature m = (CallableFeature) it.next();
            List methodParms = m.getParameters();
            int methodParmsCnt = methodParms.size();
            boolean isVarArg = methodParmsCnt > 0 ? ((Parameter)methodParms.get(methodParmsCnt - 1)).isVarArg() : false;
            if ((methodParmsCnt == parmTypeCnt && (!acceptMoreParameters || methodParmsCnt == 0))
            || (acceptMoreParameters && methodParmsCnt > parmTypeCnt)
            || (isVarArg && methodParmsCnt <= parmTypeCnt)
            ) {
                boolean accept = true;
                boolean bestMatch = !acceptMoreParameters;
                for (int i = 0; accept && i < parmTypeCnt; i++) {
                    Type mpt = ((Parameter)methodParms.get(i < methodParmsCnt ? i : methodParmsCnt - 1)).getType();
                    Type t = (Type)parmTypeList.get(i);
                    if ((t!=null && !t.isValid()) || !mpt.isValid()) {
                        accept = false;
                        break;
                    }
                    if (t != null) {
                        if (!getJMIUtils().isEqualType(t, mpt)) {
                            bestMatch = false;
                            if (!getJMIUtils().isAssignable(t, mpt)) {
                                if (acceptMoreParameters || !isVarArg || i != methodParmsCnt - 1 || methodParmsCnt != parmTypeCnt
                                || !(t instanceof Array) || !getJMIUtils().isAssignable(((Array)t).getType(), mpt)) {
                                    accept = false;
                                    break;
                                }
                            }
                        }
                    } else if (mpt instanceof PrimitiveType) {
                        accept = false;
                        break;
                    } else {
                        bestMatch = false;
                    }
                }

                if (accept) {
                    if (bestMatch) {
                        ret.clear();
                    }
                    ret.add(m);
                    if (bestMatch) {
                        break;
                    }
                }

            }
        }
        return ret;
    }

    /** Returns true if the given class is in the import statement directly or
     *  indirectly (package.name.*) */
    public boolean isImported(JavaClass cls){
        JavaClass importedClass = getJMIUtils().getImportedClass(cls.getSimpleName(), getTopJavaClass(), null, getResource());
        return (importedClass != null && importedClass.equals(cls));
    }

    /**
     * @return non-null [startOffset, endOffset] pair defining bounds
     *  of the import section.
     *  <br>
     *  If there are no imports the bounds are
     *  [Integer.MAX_VALUE, Integer.MIN_VALUE].
     */
    public int[] getImportSectionBounds() {
        Resource resource = getResource();
        int startOffset = Integer.MAX_VALUE;
        int endOffset = Integer.MIN_VALUE;
        if (resource != null){
            for (Iterator it = resource.getImports().iterator(); it.hasNext();) {
                Import imp = (Import)it.next();
                PositionBounds pb = JavaMetamodel.getManager().getElementPosition(imp);
                startOffset = Math.min(startOffset, pb.getBegin().getOffset());
                endOffset = Math.max(endOffset, pb.getEnd().getOffset());
            }
        }
        return new int[] { startOffset, endOffset };
    }

    public URL[] getJavaDocURLs(Object obj) {
        ArrayList urlList = new ArrayList();
        if (obj instanceof NbJMIResultItem){
            obj = ((NbJMIResultItem)obj).getAssociatedObject();
        }
        if (obj instanceof Element) {
            JavaModel.getJavaRepository().beginTrans(false);
            try {
                if (((Element)obj).isValid()) {
                    if (obj instanceof JavaPackage) {
                        JavaPackage pkg = (JavaPackage)obj;
                        URL u = getDocFileObjects(pkg.getName(), PACKAGE_SUMMARY);
                        if (u != null) {
                            urlList.add(u);
                        }
                    } else if (obj instanceof JavaClass) {
                        JavaClass cls = (JavaClass)obj;
                        URL u = getDocFileObjects(cls.getName(), null);
                        if (u != null) {
                            urlList.add(u);
                        }
                    } else if (obj instanceof CallableFeature) { // covers JCMethod too
                        CallableFeature ctr = (CallableFeature)obj;
                        ClassDefinition cls = ctr.getDeclaringClass();
                        URL url = getDocFileObjects(cls.getName(), null);
                        if (url != null) {
                            StringBuffer sb = new StringBuffer("#"); // NOI18N
                            sb.append(ctr instanceof Constructor ? JMIUtils.getTypeName(ctr.getType(), false, false) : ctr.getName());
                            sb.append('(');
                            List parms = ctr.getParameters();
                            int cntM1 = parms.size() - 1;
                            for (int j = 0; j <= cntM1; j++) {
                                String name = ((Parameter)parms.get(j)).getType().getName();
                                int idx = name.indexOf('<');
                                if (idx > -1)
                                    name = name.substring(0, idx);
                                sb.append(name);
                                if (j < cntM1) {
                                    sb.append(", "); // NOI18N
                                }
                            }
                            sb.append(')');
                            try {
                                urlList.add(new URL(url.toExternalForm() + sb));
                            } catch (MalformedURLException e) {
                                ErrorManager.getDefault().log(ErrorManager.ERROR, e.toString());
                            }
                        }
                    } else if (obj instanceof Field) {
                        Field fld = (Field)obj;
                        ClassDefinition cls = fld.getDeclaringClass();
                        if (cls != null) {
                            URL u = getDocFileObjects(cls.getName(), null);
                            if (u != null) {
                                try {
                                    urlList.add(new URL(u.toExternalForm() + '#' + fld.getName())); //NOI18N
                                } catch (MalformedURLException e) {
                                    ErrorManager.getDefault().log(ErrorManager.ERROR, e.toString());
                                }
                            }
                        }
                    } else if (obj instanceof Attribute) {
                        Attribute attr = (Attribute)obj;
                        ClassDefinition cls = attr.getDeclaringClass();
                        if (cls != null) {
                            URL u = getClassDocFileObjects(cls);
                            if (u != null) {
                                try {
                                    urlList.add(new URL(u.toExternalForm() + '#' + attr.getName() + "()")); //NOI18N
                                } catch (MalformedURLException e) {
                                    ErrorManager.getDefault().log(ErrorManager.ERROR, e.toString());
                                }
                            }
                        }
                    }
                }
            } finally {
                JavaModel.getJavaRepository().endTrans();
            }
        }

        URL[] ret = new URL[urlList.size()];
        urlList.toArray(ret);
        return ret;
    }

    public String openSource(Object item, boolean findDeclaration) {

        Element elem = null;
        Type typ = null;

        if (item instanceof JavaPackage) {
            if (!findDeclaration) {
                String pkgName = ((JavaPackage)item).getName();
                if ( (pkgName==null) || (pkgName.length() <= 0) ) return null;
                FileObject fo = findResource(pkgName.replace('.', '/'));
                if (fo != null) {
                    DataObject dob = getDataObject(fo);
                    if (dob != null) {
                        Node node = dob.getNodeDelegate();
                        if (node != null) {
                            org.openide.nodes.NodeOperation.getDefault().explore(node); // explore the package
                            return null;
                        }
                    }
                }
                return pkgName;
            }
        } else if (item instanceof JavaClass) {
            elem = typ = (JavaClass)item;
        } else if (item instanceof Field) {
            if (findDeclaration) {
                elem = (Field)item;
                typ = ((Field)elem).getDeclaringClass();
            } else {
                typ = ((Field)item).getType();
                if (typ instanceof JavaClass)
                    elem = JMIUtils.getSourceElementIfExists((JavaClass)typ);
                else
                    Toolkit.getDefaultToolkit().beep();
            }
        } else if (item instanceof CallableFeature) {
            elem = (CallableFeature)item;
            typ = ((CallableFeature)item).getDeclaringClass();
        }

        if (elem != null) {
            if (JMIUtils.openElement(elem))
                return null;
        }

        return typ instanceof JavaClass ? typ.getName() : null;
    }

    private URL getClassDocFileObjects(final ClassDefinition cls) {
        if (cls instanceof Array) {
            return null;
        }
        String pkgName = cls.getResource().getPackageName();
        String clsName = pkgName.length() > 0?
                cls.getName().substring(pkgName.length() + 1):
                cls.getName();
        URL u = getDocFileObjects(pkgName, clsName);
        return u;
    }

}
