/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.editor.codetemplates;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import javax.jmi.reflect.InvalidObjectException;
import javax.swing.text.JTextComponent;
import org.netbeans.editor.SyntaxSupport;
import org.netbeans.editor.Utilities;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.MultipartId;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.jmi.javamodel.TypeReference;
import org.netbeans.lib.editor.codetemplates.spi.CodeTemplateInsertRequest;
import org.netbeans.lib.editor.codetemplates.spi.CodeTemplateParameter;
import org.netbeans.lib.editor.codetemplates.spi.CodeTemplateProcessor;
import org.netbeans.lib.editor.codetemplates.spi.CodeTemplateProcessorFactory;
import org.netbeans.modules.editor.java.NbJavaJMISyntaxSupport;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
/**
 * Implementation of java code template processor.
 *
 * @author  Miloslav Metelka
 */
public class JavaCodeTemplateProcessor implements CodeTemplateProcessor {
    
    private static final String INSTANCEOF_HINT = "instanceof"; // NOI18N
    
    private static final String INDEX_HINT = "index"; // NOI18N
    
    private static final String ARRAY_HINT = "array"; // NOI18N
    
    private static final String TYPE_HINT = "type"; // NOI18N
    
    private CodeTemplateInsertRequest request;
    
    JavaCodeTemplateProcessor(CodeTemplateInsertRequest request) {
        this.request = request;
    }
    
    public void parameterValueChanged(CodeTemplateParameter masterParameter, boolean typingChange) {
        if (!typingChange) {
            
        }
    }

    public void updateDefaultValues() {
        JTextComponent component = request.getComponent();
        int offset = component.getCaretPosition();
        List typeHints = new ArrayList();
        for (Iterator masterParamsIt = request.getMasterParameters().iterator(); masterParamsIt.hasNext();) {
            CodeTemplateParameter master = (CodeTemplateParameter)masterParamsIt.next();
            Map hints = master.getHints();
            if (!hints.isEmpty()) {
                String instanceOfType = (String)hints.get(INSTANCEOF_HINT);
                if (instanceOfType != null) {
                    SyntaxSupport syntaxSupport = Utilities.getSyntaxSupport(component);
                    if (syntaxSupport instanceof NbJavaJMISyntaxSupport) {
                        NbJavaJMISyntaxSupport jss = (NbJavaJMISyntaxSupport)syntaxSupport;
                        Type type = jss.getTypeFromName(instanceOfType, false, jss.getJavaClass(offset), true);
                        Collection varNames = jss.getLocalVariableNamesOfType(offset, type);
                        boolean resolved = false;
                        for (Iterator varNamesIt = varNames.iterator(); varNamesIt.hasNext();) {
                            String varName = (String)varNamesIt.next();
                            master.setValue(varName);
                            resolved = true;
                            break;
                        }
                        if (!resolved) {
                            // Try global vars
                            varNames = jss.getGlobalVariableNamesOfType(offset, type);
                            for (Iterator varNamesIt = varNames.iterator(); varNamesIt.hasNext();) {
                                String varName = (String)varNamesIt.next();
                                master.setValue(varName);
                                resolved = true;
                                break;
                            }
                        }
                    }
                }

                if (hints.get(TYPE_HINT) != null) {
                    typeHints.add(master);
                }
                
                if ("true".equals(hints.get(INDEX_HINT))) { // NOI18N
                    SyntaxSupport syntaxSupport = Utilities.getSyntaxSupport(component);
                    if (syntaxSupport instanceof NbJavaJMISyntaxSupport) {
                        NbJavaJMISyntaxSupport jss = (NbJavaJMISyntaxSupport)syntaxSupport;
                        boolean attemptLetters = true;
                        int attemptIndex = 0;
                        while (true) {
                            // Try either "i", "j", "k" or "i2", "i3" etc.
                            String attemptVarName = attemptLetters
                                    ? String.valueOf((char)('i' + attemptIndex))
                                    : "i" + String.valueOf(attemptIndex); // NOI18N
                            Collection varNames = jss.getLocalVariableMap(offset).keySet();
                            boolean varUsed = false;
                            for (Iterator varNamesIt = varNames.iterator(); varNamesIt.hasNext();) {
                                String varName = (String)varNamesIt.next();
                                if (attemptVarName.equals(varName)) {
                                    varUsed = true;
                                    break;
                                }
                            }

                            if (!varUsed) {
                                master.setValue(attemptVarName);
                                break;
                            } else if ("z".equals(attemptVarName)) { // NOI18N
                                attemptLetters = false;
                                attemptIndex = 2; // Continue by "i2", "i3" etc.
                            } else {
                                attemptIndex++;
                            }
                        }
                    }
                }

                if ("true".equals(hints.get(ARRAY_HINT))) { // NOI18N
                    SyntaxSupport syntaxSupport = Utilities.getSyntaxSupport(component);
                    if (syntaxSupport instanceof NbJavaJMISyntaxSupport) {
                        NbJavaJMISyntaxSupport jss = (NbJavaJMISyntaxSupport)syntaxSupport;
                        Collection localVarNames = jss.getLocalVariableNamesOfArrayType(offset);
                        boolean resolved = false;
                        for (Iterator varNamesIt = localVarNames.iterator(); varNamesIt.hasNext();) {
                            String varName = (String)varNamesIt.next();
                            master.setValue(varName);
                            resolved = true;
                            break;
                        }
                        if (!resolved) {
                            // Try global vars
                            Collection globalVarNames = jss.getGlobalVariableNamesOfArrayType(offset);
                            for (Iterator varNamesIt = globalVarNames.iterator(); varNamesIt.hasNext();) {
                                String varName = (String)varNamesIt.next();
                                Collection localVars = jss.getLocalVariableNames(varName, offset, true);
                                if (localVars.size() > 0) {
                                    // there is a local variable of the same name, we must ensure that the member variable is accessed correctly
                                    try {
                                        JavaClass javaClass = jss.getJavaClass(offset);
                                        // locate jmi object corresponding to the array type variable named 'varName'
                                        ClassDefinition jc = javaClass;
                                        Field field = null;
                                        while (field == null && jc != null) {
                                            field = jc.getField(varName, true);
                                            if (field == null && jc instanceof JavaClass) {
                                                jc = ((JavaClass) jc).getDeclaringClass();
                                            }
                                        }
                                        if (field != null) {
                                            if (Modifier.isStatic(field.getModifiers())) {
                                                varName = ((JavaClass)field.getDeclaringClass()).getSimpleName() + '.' + varName;
                                            } else if (javaClass != jc) {
                                                varName = ((JavaClass) jc).getSimpleName() + ".this." + varName; // NOI18N
                                            } else {
                                                varName = "this." + varName; // NOI18N
                                            }
                                        }
                                    } catch (InvalidObjectException e) {
                                    }
                                }
                                master.setValue(varName);
                                resolved = true;
                                break;
                            }
                        }
                    }
                }
            }
        } // for
        
        // processing of type hints is required to be postponed since it adds imports and changes 
        // source code which causes that the other hints cannot retrieve the needed information correctly
        for (Iterator iter = typeHints.iterator(); iter.hasNext(); ) {
            CodeTemplateParameter master = (CodeTemplateParameter)iter.next();
            Map hints = master.getHints();
            String typeHint = (String)hints.get(TYPE_HINT);
            SyntaxSupport syntaxSupport = Utilities.getSyntaxSupport(component);
            if (syntaxSupport instanceof NbJavaJMISyntaxSupport) {
                NbJavaJMISyntaxSupport jss = (NbJavaJMISyntaxSupport)syntaxSupport;
                JavaClass jc = jss.getJavaClass(offset);
                Type type = jss.getTypeFromName(typeHint, false, jc, true);
                TypeReference id = JavaModelUtil.resolveImportsForType(jc, type);
                String varName = id instanceof MultipartId ? multipartIdToName((MultipartId) id) : typeHint;
                master.setValue(varName);
            }
        }
    }

    private static String multipartIdToName(MultipartId id) {
        LinkedList list = new LinkedList();
        while (id != null) {
            list.addFirst(id.getName());
            id = id.getParent();
        }
        StringBuffer buf = new StringBuffer();
        for (Iterator iter = list.iterator(); iter.hasNext();) {
            buf.append((String)iter.next());
            if (iter.hasNext())
                buf.append('.');
        }
        return buf.toString();
    }
    
    public void release() {
        
    }


    public static final class Factory implements CodeTemplateProcessorFactory {
        
        public CodeTemplateProcessor createProcessor(CodeTemplateInsertRequest request) {
            return new JavaCodeTemplateProcessor(request);
        }
        
    }

}

