/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Leon Chiver. All Rights Reserved.
 */

package org.netbeans.modules.java.hints;

import java.lang.reflect.Modifier;
import java.util.*;
import javax.jmi.reflect.RefObject;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.openide.ErrorManager;


/**
 * @author leon chiver
 */
final class DeclarationInfo {
    
    private static ErrorManager ERR = ErrorManager.getDefault().getInstance("org.netbeans.modules.java.hints"); // NOI18N
    
    private static Map primitiveTypeWeights;
    
    static {
        primitiveTypeWeights = new HashMap();
        primitiveTypeWeights.put("byte", new Integer(0)); // NOI18N
        primitiveTypeWeights.put("char", new Integer(1)); // NOI18N
        primitiveTypeWeights.put("short", new Integer(2)); // NOI18N
        primitiveTypeWeights.put("int", new Integer(3)); // NOI18N
        primitiveTypeWeights.put("long", new Integer(4)); // NOI18N
        primitiveTypeWeights.put("float", new Integer(5)); // NOI18N
        primitiveTypeWeights.put("double", new Integer(6)); // NOI18N
    }

    // TODO - send a patch for JMIUtils to handle names of inner classes right, 
    // then use directly JMIUtils 
    public static String getSimpleName(Type type) {
        if (type instanceof JavaClass) {
            JavaClass clazz = (JavaClass) type;
            String name = clazz.getSimpleName();
            ClassDefinition declaring = clazz.getDeclaringClass();
            String declaringName = getSimpleName(declaring);
            if (declaringName != null) {
                return declaringName + "." + name; // NOI18N
            } else {
                return name;
            }
        } else if (type instanceof Array) {
            return getSimpleName(((Array) type).getType()) + "[]"; // NOI18N
        } else if (type != null) {
            return type.getName();
        } else {
            return null;
        }
    }
    
    private static String getPackageName(ClassDefinition jcls) {
        return jcls.getResource().getPackageName();
    }
        
    private static void collectMethods(ClassDefinition clazz, MethodMatcher matcher, 
            List/*<Method>*/ l, Set visited) {
        String n = clazz.getName();
        if (visited.contains(n)) {
            return;
        }
        visited.add(n);
        Object[] f = clazz.getContents().toArray();
        for (int i = 0; i < f.length; i++) {
            if (f[i] instanceof Method) {
                Method m = (Method) f[i];
                if (!containsMethod(l, m) && matcher.matches(m)) {
                    l.add(m);
                }
            }
        }
        ClassDefinition supr = clazz.getSuperClass();
        if (supr != null) {
            collectMethods(supr, matcher, l, visited);
        }
        // Get the methods of all interfaces
        List intfList = clazz.getInterfaces();
        JavaClass[] intf = (JavaClass[]) intfList.toArray(new JavaClass[0]);
        for (int i = 0; i < intf.length; i++) {
            collectMethods(intf[i], matcher, l, visited);
        }
    }

    public static boolean isAssignableFrom(Type t1, Type t2) {
        if (t1 instanceof PrimitiveType && t2 instanceof PrimitiveType) {
            String n1 = ((PrimitiveType) t1).getName();
            String n2 = ((PrimitiveType) t2).getName();
            if (n1.equals("void") || n2.equals("void")) { // NOI18N
                return false;
            }
            if (n1.equals(n2)) {
                return true;
            }
            if (n1.equals("boolean") || n2.equals("boolean")) { // NOI18N
                return false;
            }
            int n1w = ((Integer) primitiveTypeWeights.get(n1)).intValue();
            int n2w = ((Integer) primitiveTypeWeights.get(n2)).intValue();
            return n1w >= n2w;
        } else if (t1 instanceof ClassDefinition && t2 instanceof ClassDefinition) {
            ClassDefinition d1 = (ClassDefinition) t1;
            ClassDefinition d2 = (ClassDefinition) t2;
            return d1.getName().equals(d2.getName()) || d2.isSubTypeOf(d1);
        }
        return false;
    }
    
    private static boolean containsMethod(List l, Method m) {
        for (Iterator it = l.iterator(); it.hasNext();) {
            if (m.signatureEquals((Method) it.next())) {
                return true;
            }
        }
        return false;
    }
    
    public static boolean isFeatureAccessibleFrom(Feature f, ClassDefinition clazz, Expression featureUser) {
        int featureModif = f.getModifiers();
        // Public
        if ((Modifier.PUBLIC & featureModif) != 0) {
            return true;
        }
        // Private
        ClassDefinition featureClass = f.getDeclaringClass();
        if ((Modifier.PRIVATE & featureModif) != 0) {
            // Is the feature inside a inner class of the accessing class?
            return isSameOrInnerClassOf(f.getDeclaringClass(), clazz);
        }
        // Protected
        // Here we have to check the feature user. For protected access it has to be
        // this or null. For super it's also null
        if ((Modifier.PROTECTED & featureModif) != 0 && 
                (featureUser == null || featureUser instanceof ThisExpression)) {
            return getPackageName(featureClass).equals(getPackageName(clazz)) ||
                    clazz.isSubTypeOf(featureClass);
        }
        // Package access
        return getPackageName(featureClass).equals(getPackageName(clazz));
    }
    
    public static boolean isSameOrInnerClassOf(ClassDefinition inner, ClassDefinition clazz) {
        ClassDefinition current = inner;
        while (inner != null) {
            if (inner.getName().equals(clazz.getName())) {
                return true;
            }
            if (inner instanceof JavaClass) {
                inner = ((JavaClass) inner).getDeclaringClass();
            } else {
                inner = null;
            }
        }
        return false;
    }
    
    public static ClassDefinition getContainingClass(Element element) {
        Object o = element;
        while (o != null && o instanceof RefObject && !(o instanceof ClassDefinition)) {
            o = ((RefObject) o).refImmediateComposite();
        }
        return (o instanceof ClassDefinition) ? (ClassDefinition) o : null;
    }
    
    public static Method getMethodByNameAndParamTypeNames(ClassDefinition def, String name, String[] paramTypeNames) {
        Feature[] f = (Feature[]) def.getFeatures().toArray(new Feature[0]);
        for (int i = 0; i < f.length; i++) {
            if (f[i] instanceof Method) {
                Method m = (Method) f[i];
                if (!m.getName().equals(name)) {
                    continue;
                }
                List paramList = m.getParameters();
                if (paramList.size() != paramTypeNames.length) {
                    continue;
                }
                Parameter[] p = (Parameter[]) m.getParameters().toArray(new Parameter[0]);
                for (int j = 0; j < p.length; j++) {
                    if (!p[j].getType().getName().equals(paramTypeNames[j])) {
                        continue;
                    }
                }
                return m;
            }
        }
        return null;
    }
    
    public static List/*<Method>*/ getMethodsByNameAndParamCount(ClassDefinition def, String name, int paramCount) {
        List l = new ArrayList();
        Set visited = new HashSet();
        collectMethods(def, new MethodNameAndParamCountMatcher(name, paramCount), l, visited);
        return l;
    }

    public static List/*<Constructor>*/ getConstructorsByParamCount(ClassDefinition def, int paramCount) {
        Feature[] f = (Feature[]) def.getFeatures().toArray(new Feature[0]);
        List matches = new ArrayList();
        for (int i = 0; i < f.length; i++) {
            Feature feat = f[i];
            if (feat instanceof Constructor) {
                Constructor c = (Constructor) feat;
                if (c.getParameters().size() == paramCount) {
                    matches.add(c);
                }
            }
        }
        return matches;
    }
    
    public static List/*<Method>*/ getAbstractMethods(ClassDefinition def) {
        Set visitedClasses = new HashSet();
        List abstractMethods = new ArrayList();
        List visitedMethods = new ArrayList();
        collectMethods(def, new AbstractMethodMatcher(visitedMethods), abstractMethods, visitedClasses);
        return abstractMethods;
    }
    
    protected static Type computeType(Element symbol) {
        Element element=symbol;
        Element previousElement;
        TypedElement typedEl=null;
        
        do {
            previousElement=element;
            element = (Element)element.refImmediateComposite();
        } while (element != null && 
                !(element instanceof Assignment) && 
                !(element instanceof LocalVariable) &&
                !(element instanceof Field) &&
                !(element instanceof ReturnStatement) &&
                !(element instanceof Invocation) &&
                !(element instanceof InfixExpression) &&
                !(element instanceof PrefixExpression) &&
                !(element instanceof SwitchStatement) &&
                !(element instanceof Condition) &&
                !(element instanceof NewArrayExpression));
        if (element instanceof ReturnStatement) {
            Feature f=JavaModelUtil.getDeclaringFeature(element);
            
            typedEl=(Method)f;
        } else if (element instanceof Assignment) {
            Assignment assign=(Assignment)element;
            Expression rightSide=assign.getRightSide();
            Expression leftSide=assign.getLeftSide();
            
            if (previousElement.equals(rightSide))
                typedEl=leftSide;
            else
                typedEl=rightSide;
        } else if (element instanceof Variable) {
            typedEl=(Variable)element;
        } else if (element instanceof Invocation) {
            Invocation inv = (Invocation)element;
            CallableFeature callFeature=(CallableFeature)inv.getElement();
            int parIndex=inv.getParameters().indexOf(previousElement);
            
            if (callFeature!=null && parIndex>=0) {
                List parameters=callFeature.getParameters();
                
                if (parameters.size()>parIndex) {
                    typedEl=(Parameter)parameters.get(parIndex);
                }
            }
        } else if (element instanceof InfixExpression) {
            InfixExpression expr=(InfixExpression)element;
            Expression rightSide=expr.getRightSide();
            Expression leftSide=expr.getLeftSide();
            if (previousElement.equals(rightSide))
                typedEl=leftSide;
            else
                typedEl=rightSide;
        } else if (element instanceof PrefixExpression) {
            TypeClass typeClass = ((JavaModelPackage) element.refImmediatePackage()).getType();
            PrefixExpression expr=(PrefixExpression)element;
            if (OperatorEnum.NOT.equals(expr.getOperator())) {
                return typeClass.resolve("boolean"); // NOI18N
            } else {
                return typeClass.resolve("int"); // NOI18N
            }
        } else if (element instanceof SwitchStatement) {
            SwitchStatement swtch = (SwitchStatement) element;
            
            typedEl = swtch.getExpression();
        } else if (element instanceof Condition) {
            Condition cond = (Condition) element;
            if (cond.getExpression().equals(previousElement)) {
                TypeClass typeClass = ((JavaModelPackage) element.refImmediatePackage()).getType();
                return typeClass.resolve("boolean"); // NOI18N
            }
        } else if (element instanceof NewArrayExpression) {
            TypeClass typeClass = ((JavaModelPackage) element.refImmediatePackage()).getType();
            return typeClass.resolve("int");
        }
        if (typedEl!=null) {
            Type t=typedEl.getType();
            
            if (t!=null) {
                return t;
            }
        }
        return comupteDefaultType(symbol);
    }

    private static Type comupteDefaultType(Element symbol) {
        String typeString;
        TypeClass typeClass=((JavaModelPackage)symbol.refImmediatePackage()).getType();
        
        if (symbol instanceof Invocation) {
            typeString="void"; // NOI18N
        } else {
            typeString="java.lang.Object"; // NOI18N
        }
        return typeClass.resolve(typeString);
    }

    static boolean canbeCastTo(Type fromType, Type toType) {
	if (toType instanceof PrimitiveType && fromType instanceof PrimitiveType) {
	    return true;
	}
	if (toType instanceof Array && fromType instanceof Array) {
	    Array toArray=(Array)toType;
	    Array fromArray=(Array)fromType;
	    
	    return canbeCastTo(fromArray.getType(),toArray.getType());
	}
	if (toType instanceof ClassDefinition && fromType instanceof ClassDefinition) {
	    ClassDefinition toClass=(ClassDefinition)toType;
	    ClassDefinition fromClass=(ClassDefinition)fromType;
	    
            if (fromClass instanceof TypeParameter) {
                List classes=new ArrayList(1);
                Iterator clIt;
                
                classes.add(fromClass.getSuperClass());
                classes.addAll(fromClass.getInterfaces());
                for (clIt=classes.iterator();clIt.hasNext();) {
                    if (canbeCastTo((JavaClass)clIt.next(),toClass))
                        return true;
                }
                return false;
            }
            if (toClass.isSubTypeOf(fromClass))
		return true;
	}
	return false;
    }
    
    private static interface MethodMatcher {
        boolean matches(Method m);
    }
    
    private static class MethodNameAndParamCountMatcher implements MethodMatcher {
        
        String name;
        
        int paramCount;
        
        MethodNameAndParamCountMatcher(String name, int paramCount) {
            this.name = name;
            this.paramCount = paramCount;
        }
        
        public boolean matches(Method m) {
            return m.getName().equals(name) && m.getParameters().size() == paramCount;
        }
    }
    
    private static class AbstractMethodMatcher implements MethodMatcher {
        
        List checkedMethods;
        
        AbstractMethodMatcher(List checkedMethods) {
            this.checkedMethods = checkedMethods;
        }
        
        public boolean matches(Method m) {
            boolean abstr = Modifier.isAbstract(m.getModifiers());
            if (abstr && containsMethod(checkedMethods, m)) {
                abstr = false;
            }
            checkedMethods.add(m);
            return abstr;
        }
        
    }
    
    private static class ReturnTypeMatcher implements MethodMatcher {
        
        Type returnType;

        ReturnTypeMatcher(Type returnType) {
            this.returnType = returnType;
        }
        
        public boolean matches(Method m) {
            Type actMethodType = m.getType();
            boolean matches = false;
            if (actMethodType instanceof PrimitiveType && returnType instanceof PrimitiveType) {
                String pActMethodType = ((PrimitiveType) actMethodType).getName();
                String pSearchedMethodType = ((PrimitiveType) returnType).getName();
                if (pSearchedMethodType.equals("boolean") && pActMethodType.equals("boolean")) { // NOI18N
                    return true;
                } else if (!pActMethodType.equals("void") && !pActMethodType.equals("boolean") && // NOI18N
                        !pSearchedMethodType.equals("boolean")) { // NOI18N
                    int actWeight = ((Integer) primitiveTypeWeights.get(pActMethodType)).intValue();
                    int searchedWeight = ((Integer) primitiveTypeWeights.get(pSearchedMethodType)).intValue();
                    if (searchedWeight >= actWeight) {
                        return true;
                    }
                }
            } else if (actMethodType instanceof ClassDefinition && returnType instanceof ClassDefinition) {
                ClassDefinition cdActMethodType = (ClassDefinition) actMethodType;
                ClassDefinition cdSearchedMethodType = (ClassDefinition) returnType;
                if (isAssignableFrom(cdSearchedMethodType, cdActMethodType)) {
                    return true;
                }
            }
            return false;
        }
    }

}
