/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.hints;

import java.lang.reflect.Modifier;
import java.util.*;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.Utilities;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.editor.java.JMIUtils;
import org.netbeans.modules.editor.hints.spi.HintsProvider;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.internalapi.ParsingListener;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceImpl;
import org.openide.ErrorManager;
import org.openide.cookies.EditorCookie;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.text.PositionBounds;

/**
 * @author Jan Lahoda
 * @author leon chiver
 */
public final class JavaHintsProvider extends HintsProvider implements ParsingListener {
    
    private static ErrorManager ERR = ErrorManager.getDefault().getInstance("org.netbeans.modules.java.hints"); // NOI18N
    
    private static final String INCOMPATIBLE_TYPES = "prob.found.req"; // NOI18N
    private static final String CANT_RESOLVE = "cant.resolve"; // NOI18N
    private static final String CANT_RESOLVE_LOCATION = "cant.resolve.location"; // NOI18N
    private static final String CANT_APPLY_SYMBOL = "cant.apply.symbol"; // NOI18N
    private static final String UNREPORTED_EXCEPTION_ERROR = "unreported.exception.need.to.catch.or.throw"; // NOI18N
    private static final String VAR_MIGHT_NOT_HAVE_BEEN_INITIALIZED = "var.might.not.have.been.initialized"; // NOI18N
    private static final String DOES_NOT_OVERRIDE_ABSTRACT = "does.not.override.abstract"; // NOI18N
    private static final String ABSTRACT_CANT_BE_INSTANTIATED = "abstract.cant.be.instantiated"; // NOI18N
    private static final String DOESNT_EXIST = "doesnt.exist"; // NOI18N
    
    private static final String KINDNAME_CONSTRUCTOR = "kindname.constructor";  // NOI18N
    private static final String KINDNAME_VARIABLE = "kindname.variable";  // NOI18N
    private static final String KINDNAME_METHOD = "kindname.method";  // NOI18N
    
    private static final int TabInc = 8;
    
    /** Creates a new instance of JavaHintsProvider */
    public JavaHintsProvider() {
        JavaMetamodel.getManager().addParsingListener(this);
    }
        
    public List getHints(Document doc, int offset) {
        if (doc instanceof BaseDocument) {
            try {
                BaseDocument baseDoc = (BaseDocument) doc;
                List result = new ArrayList();
                Resource resource = null;
                JMIUtils utils = null;
                List errors;
                JavaModel.getJavaRepository().beginTrans(false);
                try {
                    utils = JMIUtils.get(baseDoc);
                    resource = utils.getResource();
                    
                    if (resource == null)
                        return Collections.EMPTY_LIST;
                    errors = resource.getErrors();
                } finally {
                    JavaModel.getJavaRepository().endTrans();
                }
		int lineNumber = Utilities.getLineOffset(baseDoc, offset) + 1;
                
                int realStart = Utilities.getFirstNonWhiteFwd(baseDoc, Utilities.getRowStart(baseDoc, offset));
                int realEnd = Utilities.getFirstNonWhiteBwd(baseDoc, Utilities.getRowEnd(baseDoc, offset));
                if (realStart > realEnd) {
                    return Collections.EMPTY_LIST;
                }
                if (errors.size() == 0) {
                    return Collections.EMPTY_LIST;
                }
                offset = Math.min(Math.max(offset, realStart), realEnd);
                
                Set/*<String>*/ handledDescriptions = new HashSet/*<String>*/();
                JavaMetamodel m = JavaMetamodel.getManager();
                JavaModel.getJavaRepository().beginTrans(false);
                try {                    
                    if (!resource.isValid())
                        return Collections.EMPTY_LIST;
                    JavaModel.setClassPath(resource);
                    if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                        Element el = resource.getElementByOffset(offset);
                        
                        ERR.log("el=" + el); // NOI18N
                    }                                        
                    for (Iterator i = errors.iterator(); i.hasNext(); ) {
                        ErrorInfo error = (ErrorInfo) i.next();
                        if (error.getLineNumber() != lineNumber)
                            continue;
                        
                        String description = error.getDescription();
                        
                        if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                            ERR.log("description = " + description ); // NOI18N
                            ERR.log("line = " + error.getLineNumber() ); // NOI18N
                            ERR.log("column = " + error.getColumn() ); // NOI18N
                        }
                        
                    if (handledDescriptions.contains(description)) {
                            //sometimes the errors are doubled
                            continue;
                        }
                        
                        handledDescriptions.add(description);
                        
                        for (int cntr = 0; cntr < creators.length; cntr++) {
                            if (error.getErrorId().equals(creators[cntr].getErrorId())) {
                                creators[cntr].createHint(utils, resource, doc, error, result);
                            }
                        }
                    }
                    Element e = resource.getElementByOffset(offset);
                    if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                        ERR.log("element = " + e);
                    }
                    if (e instanceof Field) {
                        Field f = (Field) e;
                        for (int i = 0; i < fieldHintCreators.length; i++) {
                            fieldHintCreators[i].createHint(utils, resource, doc, f, offset, result);
                        }
                    } else {
                        Feature f = JavaModelUtil.getDeclaringFeature(e);
                        if (f instanceof CallableFeature) {
                            CallableFeature feature = (CallableFeature) f;
                            PositionBounds bounds = m.getElementPosition(feature);
                            if (feature instanceof Method &&
                                bounds.getBegin().getOffset() <= offset && 
                                bounds.getEnd().getOffset() >= offset) 
                            {
                                for (int i = 0; i < methodHintCreators.length; i++) {
                                    methodHintCreators[i].createHint(utils, resource, doc, (Method) feature, offset, result);
                                }
                            }
                        }
                    }
                } catch (Exception e) {
                    // report only exceptions from code, do not consider problems
                    // when user edited the code during hints computation
                    if (resource == null || !m.isModified(m.getFileObject(resource))) {
                        throw new RuntimeException(e);
                    }
                } finally {
                    JavaModel.getJavaRepository().endTrans();
                }
                if (m.isModified(m.getFileObject(resource))) {
                    return Collections.EMPTY_LIST;
                }
                return result;
            } catch (BadLocationException e) {
                //in case that the document is about to be closed 
                //the BLE can be throws from some editor utility classes
                //so we can swallow it in this case
            }
        }
        
        return Collections.EMPTY_LIST;
    }
    
    private static abstract class HintCreator {
        public abstract String getErrorId();
        public abstract void   createHint(JMIUtils utils, Resource resource, Document doc, ErrorInfo error, List result);
    }
    
    private static abstract class FieldHintCreator {
        public abstract void createHint(JMIUtils utils, Resource resource, Document doc, Field field, int offset, List result);
    }
    
    private static abstract class MethodHintCreator {
        public abstract void createHint(JMIUtils utils, Resource resource, Document doc, Method method, int offset, List result);
    }
    
    private static boolean isConstant(Field f, boolean publik) {
        int constModif = Modifier.STATIC | Modifier.FINAL;
        if (publik) {
            constModif |= Modifier.PUBLIC;
        }
        return (f.getModifiers() & constModif) == constModif;
    }
    
    private static boolean isFinal(Field f) {
        return (f.getModifiers() & Modifier.FINAL) == Modifier.FINAL;
    }
        
    private static class InitializeVariableHintCreator extends HintCreator {
        
        public String getErrorId() {
            return JavaHintsProvider.VAR_MIGHT_NOT_HAVE_BEEN_INITIALIZED;
        }
        
        public void createHint(JMIUtils utils, Resource resource, Document doc, ErrorInfo error, List result) {
            String varName = (String)error.getArguments().get(0);
            VariableAccess access = (VariableAccess) getElementByOffset(resource, getErrorOffset(doc, error), varName);
            if (access != null && varName.equals(access.getName())) {
                result.add(new InitializeVariableHint(access));
            }
        }
        
    }
    
    private static class UnreportedExceptionHintCreator extends HintCreator {
        
        public String getErrorId() {
            return JavaHintsProvider.UNREPORTED_EXCEPTION_ERROR;
        }
        
        public void createHint(JMIUtils utils, Resource resource, Document doc, ErrorInfo error, List result) {
            if (isReadOnly(resource)) {
                return;
            }
            ClassDefinition def = utils.getExactClass((String)error.getArguments().get(0));
            
            int errorOffset = getErrorOffset(doc, error);
            Element el = resource.getElementByOffset(errorOffset);
            Feature f = JavaModelUtil.getDeclaringFeature(el);
            CallableFeature cf = f instanceof CallableFeature ? (CallableFeature) f : null;
            // not applicable for non-callable, e.g. mainWin w = new mainWin();
            // where constructor mainWin() throws some exception.
            if (cf != null) {
                if (def != null && def instanceof JavaClass) {
                    cf = (CallableFeature) f;
                    result.add(new AddThrowsClauseJavaHint(cf, (JavaClass) def));
                }
                try {
                    result.add(new TryWrapperJavaHint(doc.createPosition(errorOffset), cf));
                } catch (BadLocationException e) {
                    ErrorManager.getDefault().notify(e);
                }
            }
        }
    }
        
    private static class IncompatibleTypesHintCreator extends HintCreator {

        public String getErrorId() {
            return JavaHintsProvider.INCOMPATIBLE_TYPES;
        }
        
        public void createHint(JMIUtils utils, Resource resource, Document doc, ErrorInfo error, List result) {
            if (isReadOnly(resource)) {
                return;
            }
            //potentially missing cast:
            if (ERR.isLoggable(ErrorManager.INFORMATIONAL))
                ERR.log("potentially missing cast"); // NOI18N
            
            int errorOffset = getErrorOffset(doc, error);
            Element errElement = resource.getElementByOffset(errorOffset);
            if (errElement instanceof ConditionalExpression) {
                Type t = DeclarationInfo.computeType(errElement);
                ConditionalExpression expr = (ConditionalExpression) errElement;
                addCastHint(t, expr.getTruePart(), result);
                addCastHint(t, expr.getFalsePart(), result);
            } else {
                // there are cases like 'attempting to use incompatible return
                // type'. Such cases has to be filtered.
                if (errElement instanceof Expression) {
                    Type t = DeclarationInfo.computeType(errElement);
                    addCastHint(t, (Expression) errElement, result);
                }
            }
        }
        
        public static void addCastHint(Type toType, Expression castedElement, List result) {
            Type fromType = castedElement.getType();
            if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                ERR.log("toType=" + toType); // NOI18N
                ERR.log("fromType=" + fromType); // NOI18N
            }
            if (toType instanceof ClassDefinition && fromType instanceof ClassDefinition) {
                ClassDefinition toClass = (ClassDefinition) toType;
                ClassDefinition fromClass = (ClassDefinition) fromType;

                if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                    ERR.log("toClass=" + toClass.getName()); // NOI18N
                    ERR.log("fromClass=" + fromClass.getName()); // NOI18N
                }
                if (!fromClass.isSubTypeOf(toClass) && DeclarationInfo.canbeCastTo(fromClass,toClass)) {
                    result.add(new AddCastJavaHint(getElementName(castedElement), (ClassDefinition) toClass, castedElement));
                }
            }
        }
        
        private static String getElementName(Element elem) {
            String elementName;
            if (elem instanceof Invocation) {
                Invocation i = (Invocation) elem;
                elementName = i.getName();
                if (i.getParameters().isEmpty()) {
                    elementName += "()"; // NOI18N
                } else {
                    elementName += "(...)"; // NOI18N
                }
                if (elem instanceof NewClassExpression) {
                    elementName = "new " + elementName; // NOI18N
                } else if (elem instanceof MethodInvocation) {
                    elementName = "..." + elementName; // NOI18N
                }
            } else if (elem instanceof NamedElement) {
                elementName = "..." + ((NamedElement) elem).getName(); // NOI18N
            } else if (elem instanceof ThisExpression) {
                elementName = "this"; // NOI18N
            } else if (elem instanceof StringLiteral) {
                elementName = ((StringLiteral) elem).getValue();
            } else if (elem instanceof BooleanLiteral) {
                elementName = String.valueOf(((BooleanLiteral) elem).isValue());
            } else if (elem instanceof CharLiteral) {
                elementName = "'" + ((CharLiteral) elem).getValue() + "'"; // NOI18N
            } else if (elem instanceof DoubleLiteral) {
                elementName = String.valueOf(((DoubleLiteral) elem).getValue()) + "d"; // NOI18N
            } else if (elem instanceof FloatLiteral) {
                elementName = String.valueOf(((FloatLiteral) elem).getValue()) + "f"; // NOI18N
            } else if (elem instanceof IntLiteral) {
                elementName = String.valueOf(((IntLiteral) elem).getValue());
            } else if (elem instanceof LongLiteral) {
                elementName = String.valueOf(((LongLiteral) elem).getValue()) + "l"; // NOI18N
            } else if (elem instanceof NullLiteral) {
                elementName = "null"; // NOI18N
            } else if (elem instanceof ArrayAccess) {
                elementName = getElementName(((ArrayAccess) elem).getArray()) + "[]"; // NOI18N
            } else {
                // TODO - what now?
                elementName = "..."; // NOI18N
            }
            return elementName;
        }
        
    }
    
    private static class ImplementAbstractMethodsHintCreator extends HintCreator {
        
        public String getErrorId() {
            return JavaHintsProvider.DOES_NOT_OVERRIDE_ABSTRACT;
        }
        
        public void createHint(JMIUtils utils, Resource resource, Document doc, ErrorInfo error, List result) {
            List arguments = error.getArguments();
            String subClassName = (String)arguments.get(0);
            String className = (String) arguments.get(2);
            if (className.equals(subClassName)) {
                Type t = JavaModel.getDefaultExtent().getType().resolve(className);
                if (t instanceof JavaClass) {
                    result.add(new MakeClassAbstractJavaHint((JavaClass) t));
                }
                return;
            }
            JavaMetamodel.getManager().setClassPath(JavaModel.getFileObject(resource), true);
            int errorOffset = getErrorOffset(doc, error);
            Element e = resource.getElementByOffset(errorOffset);
            if (e != null && e instanceof ClassDefinition) {
                ClassDefinition def = (ClassDefinition) e;
                String name = def.getName();
                if (name == null) {
                    // anonymous inner class
                    JavaClass jc;
                    List list = def.getInterfaces();
                    if (list.size() > 0) {
                        jc = (JavaClass) list.get(0);
                    } else {
                        jc = def.getSuperClass();
                    }
                    if (jc instanceof ParameterizedType) {
                        jc = ((ParameterizedType)jc).getDefinition();
                    }

                    name = jc.getName();
                    subClassName = className;
                }
                if (name.equals(subClassName)) {
                    List methods = DeclarationInfo.getAbstractMethods(def);
                    if (!methods.isEmpty()) {
                        result.add(new ImplementMethodJavaHint(def, methods));
                    }
                }
            }
            JavaModel.setClassPath(resource);
        }
    }
    
    
    private static class ImplementAbstractMethods2HintCreator extends HintCreator {
        
        public String getErrorId() {
            return JavaHintsProvider.ABSTRACT_CANT_BE_INSTANTIATED;
        }
        
        public void createHint(JMIUtils utils, Resource resource, Document doc, ErrorInfo error, List result) {
            if (isReadOnly(resource)) {
                return;
            }
            if (ERR.isLoggable(ErrorManager.INFORMATIONAL))
                ERR.log("pontentially unimplemented methods from: " + error.getArguments().get(0)); // NOI18N
            
            int errorOffset = getErrorOffset(doc, error);
            NewClassExpression expression = (NewClassExpression) resource.getElementByOffset(errorOffset);
            JavaMetamodel.getManager().setClassPath(JavaModel.getFileObject(resource), true);
            Type t = ((JavaModelPackage) resource.refImmediatePackage()).getType().resolve((String)error.getArguments().get(0));
            JavaClass clazz = t instanceof JavaClass ? (JavaClass) t : null;
            if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                ERR.log("expression=" + expression); // NOI18N
                ERR.log("clazz=" + clazz); // NOI18N
            }

            if (expression != null && clazz != null) {
                List unimplementedMethods = new ArrayList();

                for (Iterator methods = clazz.getContents().iterator(); methods.hasNext(); ) {
                    Feature f = (Feature) methods.next();

                    if (f instanceof Method) {
                        Method method = (Method) f;

                        if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                            ERR.log("method=" + method); // NOI18N
                        }

                        int modifiers = method.getModifiers();

                        if (Modifier.isInterface(modifiers) || Modifier.isAbstract(modifiers)) {
                            if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                                ERR.log("is abstract"); // NOI18N
                            }

                            unimplementedMethods.add(method);
                        }
                    }
                }

                if (!unimplementedMethods.isEmpty()) {
                    if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                        ERR.log("unimplementedMethods=" + unimplementedMethods); // NOI18N
                    }
                    result.add(new ImplementMethodJavaHint2(expression, unimplementedMethods));
                }
            }
        }
    }

    public static class IncorrectArgumentsHintCreator extends HintCreator {
        
        public String getErrorId() {
            return JavaHintsProvider.CANT_APPLY_SYMBOL;
        }
        
        public void createHint(JMIUtils utils, Resource resource, Document doc, ErrorInfo error, List result) {
            List arg = error.getArguments();
            String methodName = (String)arg.get(0);
            int left = methodName.indexOf('(');
            if (left != -1) {
                methodName = methodName.substring(0, left);
            }
            String className  = (String)arg.get(1);
            String arguments  = (String)arg.get(3);
            
            boolean isMethod = !className.endsWith("."+methodName); // NOI18N
            boolean isStaticMethodCall = false;
            
            BaseDocument bd = (BaseDocument) doc;
            
            if (!isMethod && !isStaticMethodCall) {
                return;
            }
            ClassDefinition clazz = utils.getExactClass(className);
            if (clazz == null)
                return;
            int errorOffset = getErrorOffset(doc, error);
            
            addCastsOrInvokeMethods(resource, clazz, doc, errorOffset, methodName, arguments, result);
            handleUnresolvedHints(utils, resource, errorOffset, clazz, 
                    methodName, false, isMethod, isStaticMethodCall, result);
        }
        
        private static void addCastsOrInvokeMethods(Resource res, ClassDefinition clazz, 
                Document doc, int errorOffset, String methodName, String expectedArgs, List result) {
            StringTokenizer tokExp = new StringTokenizer(expectedArgs, ","); // NOI18N
            String[] expectedTypeNames = new String[tokExp.countTokens()];
            for (int i = 0; i < expectedTypeNames.length; i++) {
                expectedTypeNames[i] = tokExp.nextToken();
            }
            // Get the declared method
            CallableFeature f = DeclarationInfo.getMethodByNameAndParamTypeNames(clazz, methodName, expectedTypeNames);
            // Find the invocation using the arguments with the wrong types
            Invocation inv = (Invocation) getElementByOffset(res, errorOffset, methodName);
            // Did we find them?
            if (inv != null && f != null) {
                Expression[] invParams = (Expression[]) inv.getParameters().toArray(new Expression[0]);
                Parameter[] expectedParams = (Parameter[]) f.getParameters().toArray(new Parameter[0]);
                if (invParams.length != expectedParams.length) {
                    return;
                }
                // Loop through all params, check if invocation param matches the method param, and
                // if not propose a cast, or invoke a method returning the proper type.
                for (int i = 0; i < invParams.length; i++) {
                    Type invType = invParams[i].getType();
                    Type expType = expectedParams[i].getType();
                    if (invType!=null && expType!=null && !expType.getName().equals(invType.getName())) {
                        IncompatibleTypesHintCreator.addCastHint(expType, invParams[i], result);
                    }
                }
            }
        }
        
    }

    private static class UnresolvedSymbolCreator extends HintCreator {
        
        public String getErrorId() {
            return JavaHintsProvider.CANT_RESOLVE_LOCATION;
        }
        
        public void createHint(JMIUtils utils, Resource resource, Document doc, ErrorInfo error, List result) {
            ////////////////////////////////////////////////////////////////////////////
            // Add Import HInt creator: return "cannot find symbol.*symbol  : class(.*)\n.*"; // NOI18N
            ////////////////////////////////////////////////////////////////////////////
            if (isReadOnly(resource)) {
                return;
            }
            List arguments = error.getArguments();
            String type = (String) arguments.get(0);
            String invName = (String) arguments.get(1);
            String className = (String) arguments.get(5);
            if (!KINDNAME_VARIABLE.equals(type) && !KINDNAME_CONSTRUCTOR.equals(type)) { // NOI18N
                //seems like missing import candidate:
                addImportVariants(utils, resource, invName, result);
            }
            ////////////////////////////////////////////////////////////////////////////
            //CastParamHintCreator "cannot find symbol\nsymbol  : (method|constructor) (.*)\nlocation: class (.*)"; // NOI18N
            ////////////////////////////////////////////////////////////////////////////
            ClassDefinition clazz = utils.getExactClass(className); // todo: remove this!
            if (KINDNAME_METHOD.equals(type)) { // NOI18N
                NamedElement el = getElementByOffset(resource, getErrorOffset(doc, error), invName);
                if (el != null && el instanceof MethodInvocation) {
                    MethodInvocation inv = (MethodInvocation) el;
                    List invParams = inv.getParameters();
                    int paramCount = invParams.size();
                    if (paramCount > 0) { // We cannot help here with a cast (nothing to cast) if paramCount == 0
                        clazz = getParentClass(inv);
                        List methods = DeclarationInfo.getMethodsByNameAndParamCount(clazz, invName, paramCount);
                        ClassDefinition invClass = DeclarationInfo.getContainingClass(inv);
                        for (Iterator it = methods.iterator(); it.hasNext();) {
                            Method m = (Method) it.next();
                            if (DeclarationInfo.isFeatureAccessibleFrom(m, invClass, inv.getParentClass())) {
                                checkParamsAndAddHint(invParams, m.getParameters(), result);
                            }
                        }
                    }
                }
            } else if (KINDNAME_CONSTRUCTOR.equals(type)) { // NOI18N
                NewClassExpression expr = (NewClassExpression) getElementByOffset(resource, getErrorOffset(doc, error), invName);
                if (expr != null) {
                    List invParams = expr.getParameters();
                    int paramCount = invParams.size();
                    if (paramCount != 0) { // We cannot help here with a cast (nothing to cast) if paramCount == 0
                        clazz = (ClassDefinition) expr.getClassName().getElement();
                        List constructors = DeclarationInfo.getConstructorsByParamCount(clazz, paramCount);
                        ClassDefinition newExprClass = DeclarationInfo.getContainingClass(expr);
                        for (Iterator it = constructors.iterator(); it.hasNext();) {
                            Constructor c = (Constructor) it.next();
                            if (DeclarationInfo.isFeatureAccessibleFrom(c, newExprClass, null)) {
                                checkParamsAndAddHint(invParams, c.getParameters(), result);
                            }
                        }
                    }
                }
            }
            ////////////////////////////////////////////////////////////////////////////
            // UnresolvedElemementHintCreator "cannot find symbol\nsymbol  : (method|variable) (.*)\nlocation: class (.*)"; // NOI18N
            ////////////////////////////////////////////////////////////////////////////
            String typeString = (String) arguments.get(0);
            String symbol     = (String) arguments.get(1);
            String params     = (String) arguments.get(2);
            boolean isVariable = KINDNAME_VARIABLE.equals(typeString); // NOI18N
            boolean isMethod = KINDNAME_METHOD.equals(typeString); // NOI18N
            boolean isStaticMethodCall = false;
            boolean isField = false;
            
            BaseDocument bd = (BaseDocument) doc;
            
            int errorOffset = getErrorOffset(doc, error);
            
            if (isVariable) {
                javax.swing.text.Element le = bd.getParagraphElement(errorOffset);
                String line = null;
                try {
                    int so = le.getStartOffset();
                    line = doc.getText(so, le.getEndOffset() - so - 1);
                } catch (BadLocationException ex) {
                    ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, ex);
                }
                if (line != null) {
                    isStaticMethodCall = line.indexOf(symbol + ".") != -1; // NOI18N
                    isField = !isStaticMethodCall;
                }
            }
            
            if (isField || isMethod || isStaticMethodCall) {
                handleUnresolvedHints(utils, resource, errorOffset, clazz, symbol, isField, isMethod, isStaticMethodCall, result);
            }
        }

        private ClassDefinition getParentClass(final MethodInvocation inv) {
            ClassDefinition clazz;
            PrimaryExpression parent = inv.getParentClass();
            if (parent != null) {
                clazz = (ClassDefinition) parent.getType();
            } else {
                clazz = JavaModelUtil.getDeclaringFeature(inv).getDeclaringClass();
            }
            if (inv.isHasSuper()) {
                clazz = clazz.getSuperClass();
            }
            return clazz;
        }
        
        private void checkParamsAndAddHint(List invParams, List actualParams, List result) {
            int index = -1;
            int sz = invParams.size();
            for (int i = 0; i < sz; i++) {
                Expression invParamI = (Expression) invParams.get(i);
                Parameter actualParamI = (Parameter) actualParams.get(i);
                if (!DeclarationInfo.isAssignableFrom(actualParamI.getType(),invParamI.getType())) {
                    if (index != -1) {
                        // TODO - more than one parameters do not match. Do not propose hints, as it would 
                        // be too confusing. Check how competing IDE's handle this situation
                        return;
                    }
                    index = i;
                }
            }
            if (index != -1) {
                Type correctType = ((Parameter) actualParams.get(index)).getType();
                Expression wrongExpr = (Expression) invParams.get(index);
                IncompatibleTypesHintCreator.addCastHint(correctType, wrongExpr, result);
            }
        }
    }
    
    private static void handleUnresolvedHints(JMIUtils utils, Resource resource, int errorOffset, ClassDefinition clazz, 
            String symbol, boolean isField, boolean isMethod, boolean isStaticMethodCall, List result) {
        if ((isField || isMethod) && clazz != null && !isReadOnly(clazz.getResource())) {
            NamedElement unresolvedElement=getElementByOffset(resource,errorOffset,symbol);
            // we were unable to obtain element where error occured,
            // seems like source was modified. Do not compute hint.
            if (unresolvedElement != null && unresolvedElement.getResource() != null) {
                if (isField) {
                    boolean proposeLocalVar;
                    Feature feature = JavaModelUtil.getDeclaringFeature(unresolvedElement);

                    result.add(new CreateFieldJavaHint(unresolvedElement,clazz));
                    if (unresolvedElement instanceof MultipartId) {
                        MultipartId id = (MultipartId) unresolvedElement;
                        proposeLocalVar = id.getParent() == null;
                    } else {
                        proposeLocalVar = false;
                    }                
                    if (feature instanceof BehavioralFeature && proposeLocalVar) {
                        if (feature instanceof CallableFeature) {
                            result.add(new CreateParameterJavaHint(unresolvedElement));
                        }
                        result.add(new CreateLocVarJavaHint(unresolvedElement));
                    }
                } else if (isMethod) {
                    if (unresolvedElement instanceof Invocation) {
                        result.add(new CreateMethodJavaHint(unresolvedElement,clazz));
                    }
                }
            }
        } else if (isStaticMethodCall) {
            addImportVariants(utils, resource, symbol, result);
        }
    }
    
    private static class UnresolvedSymbolCreatorNoLocation extends HintCreator {

        public String getErrorId() {
            return JavaHintsProvider.CANT_RESOLVE;
        }
        
        public void createHint(JMIUtils utils, Resource resource, Document doc, ErrorInfo error, List result) {
            ////////////////////////////////////////////////////////////////////////////
            // Add Import HInt creator: return "cannot find symbol.*symbol  : class(.*)\n.*"; // NOI18N
            ////////////////////////////////////////////////////////////////////////////
            if (isReadOnly(resource)) {
                return;
            }
            //seems like missing import candidate:
            addImportVariants(utils, resource, (String)error.getArguments().get(1), result);
        }
    }

    private static class PackageDoesNotExistCreator extends HintCreator {
        
        public String getErrorId() {
            return JavaHintsProvider.DOESNT_EXIST;
        }

        public void createHint(JMIUtils utils, Resource resource, Document doc, ErrorInfo error, List result) {
            /////////////////////////
            // Package does not exist
            /////////////////////////
            if (isReadOnly(resource)) {
                return;
            }
            //seems like missing import candidate:
            addImportVariants(utils, resource, (String)error.getArguments().get(0), result);
        }
    }
    
    private static HintCreator[] creators = new HintCreator[] {
        //new AddImportHintCreator(),
        //new AddImportHintCreator2(),
        new UnreportedExceptionHintCreator(),
        new IncompatibleTypesHintCreator(),
        new IncorrectArgumentsHintCreator(),
        //new CastParamHintCreator(),
        new ImplementAbstractMethodsHintCreator(),
        new ImplementAbstractMethods2HintCreator(),
        // new UnresolvedElemementHintCreator(),
        new InitializeVariableHintCreator(),
        new UnresolvedSymbolCreator(),
        new UnresolvedSymbolCreatorNoLocation(),
        new PackageDoesNotExistCreator()
    };
    
    private static FieldHintCreator[] fieldHintCreators = new FieldHintCreator[] {
    };
    
    private static MethodHintCreator[] methodHintCreators = new MethodHintCreator[] {
    };
    
    private static void addImportVariants(JMIUtils utils, Resource resource, String className, List result) {
        List classes = findClasses(utils, className, getTopJavaClass(resource));
        
        for (Iterator it = classes.iterator(); it.hasNext(); ) {
            JavaClass c = (JavaClass) it.next();
            ImportJavaHint importHint = new ImportJavaHint(resource, c);
            if (!result.contains(importHint)) {
                result.add(0, importHint);
            }
        }
    }
    
    private static JavaClass getTopJavaClass(Resource res) {
        if (res != null) {
            Iterator cls = res.getClassifiers().iterator();
            if (cls.hasNext())
                return (JavaClass)cls.next();
        }
        return null;
    }
    
    private static boolean isReadOnly(Resource res) {
        return res == null || (res.getStatus() & ResourceImpl.IS_FROM_CLASSFILE) != 0 ||
                !JavaMetamodel.getManager().getFileObject(res).canWrite();
    }
    
    public void resourceParsed(Resource rsc) {
        DataObject od = JavaMetamodel.getManager().getDataObject(rsc);

        if (od!=null) {
            EditorCookie ec = (EditorCookie) od.getCookie(EditorCookie.class);

            firePropertyChangeEvent(PROP_REFRESH, null, ec.getDocument());
        }
    }
    
    /** 
     * Find classes by simple name.
     *
     * @param name simple name of the class. The package name must be omitted.
     * @return list of the matching classes
     */
    public static List/*<JavaClass>*/ findClasses(JMIUtils utils, String name, JavaClass context) 
    {
        TreeSet ret = new TreeSet(utils.getNaturalMemberNameComparator());
        ClassPath cp = JavaMetamodel.getManager().getClassPath();
        FileObject[] cpRoots = cp.getRoots();
        for (int i = 0; i < cpRoots.length; i++) {
            assert cpRoots[i] != null : "Classpath root is null"; // NOI18N
            ClassIndex ci = ClassIndex.getIndex(JavaModel.getJavaExtent(cpRoots[i]));
            // #63476: NPE - hotfix
            // Shouldn't be guaranteed that cpRoot[i] and related ci cannot be null?
            if (ci == null) {
                ErrorManager.getDefault().log(
                          ErrorManager.INFORMATIONAL, 
                          "Index not found for " + cpRoots[i] // NOI18N
                        );
                continue;
            }
            for (Iterator it = ci.getClassesBySimpleName(name).iterator(); it.hasNext();) {
                JavaClass javaClass = (JavaClass) it.next();
                if (!javaClass.isInner() && utils.isAccessible(javaClass, context)) {
                    ret.add(javaClass);
                }
            }
        }
        return new ArrayList(ret);
    }
    
    public static int getErrorOffset(Document doc, ErrorInfo error) {
        javax.swing.text.Element lineRoot = ((BaseDocument) doc).getParagraphElement(0).getParentElement();
        javax.swing.text.Element line = lineRoot.getElement(error.getLineNumber() - 1);
        String s = null;
        int lineOffset = line.getStartOffset();
        try {
            s = doc.getText(lineOffset, line.getEndOffset() - lineOffset);
        } catch (BadLocationException e) {
            ErrorManager.getDefault().notify(e);
        }
        int origColumn = error.getColumn() - 1; 
        int column = 0;
        int newColumn;
        for (newColumn = 0; column < origColumn; newColumn++) {
            if (s.charAt(newColumn) == '\t')
                column = (column / TabInc * TabInc) + TabInc;
            else
                ++column;
        }
        return lineOffset + newColumn;
    }

    // fix for incorrect error position 
    private static NamedElement getElementByOffset(Resource resource, int offset, String symbol) {
        Element el=(Element)resource.getElementByOffset(offset);
        int index=symbol.indexOf('(');
        
        if (index!=-1)
            symbol=symbol.substring(0,index);
        for (int i=0;i<2;i++,el=(Element)el.refImmediateComposite()) {
            if (el instanceof NamedElement) {
                NamedElement nel=(NamedElement)el;
                if (symbol.equals(nel.getName()))
                    return nel;
            }
        }
        return null;
    }
    
}
