/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore;
import java.util.ArrayList;
import java.util.HashMap;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.mdr.util.TransactionMutex;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceClassImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.SemiPersistentElement;
import org.openide.ErrorManager;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.netbeans.mdr.NBMDRepositoryImpl;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.modules.javacore.jmiimpl.javamodel.BehavioralFeatureImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.EnumConstantImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.FieldImpl;
import org.netbeans.modules.javacore.internalapi.ExternalChange;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.classpath.FilterClassPathImplementation;
import org.netbeans.api.java.classpath.ClassPath;
import org.openide.filesystems.FileObject;
import javax.swing.SwingUtilities;
import org.netbeans.modules.javacore.jmiimpl.javamodel.AttributeImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.AttributeValueImpl;

/**
 *
 * @author Martin Matula
 */
public class ExclusiveMutex extends TransactionMutex {
    private static final boolean DEBUG = false;
    
    private boolean changes = false;
    private boolean fail = false;
    private Thread thread = null;
    private int counter = 0;
    private int modCount = 1;
    private boolean isSafeTrans = false;

    private Set newObjects = new HashSet();

    // set of changed resources
    private Set changedRscSet = new HashSet();
    private Set changedExternalSet = new HashSet();
    private Set undoElementsSet = new HashSet();
    
    private Set upToDateElements = new HashSet();
    private Set invalidClasses = new HashSet();

    // list of BehavioralElements, which has to be reinitialized after commit
    private List initBodyQueue;
    
    // cache used by MDRParser for scopes
    private Map parserCache;

    private Set needParsing = new HashSet();
    private Set needDelete = new HashSet();
    private Set needParsingRW = new HashSet();
    private Set needTSUpdate = new HashSet();
    private Set priorityThreads = null;

    private ClassPath classPath = null;
    private List/*ClassPath*/ searchScope = null;
    private volatile boolean isSwingWaiting = false;
    private JMManager manager = (JMManager) JMManager.getManager();

    private boolean modificationsDisabled = false;

    /** Creates a new instance of ExclusiveMutex */
    public ExclusiveMutex(Object p1, Object p2, Object p3) {
        super(p1, p2, p3);
    }

    // can be called only from within a transaction
    public boolean isSwingWaiting() {
        return isSwingWaiting;
    }
    
    public int getModCount() {
        return modCount;    
    }
    
    public boolean isSafeTrans() {
        return isSafeTrans;
    }
    
    void setSafeTrans(boolean safeTrans) {
        this.isSafeTrans = safeTrans;
    }
    
    public synchronized void addPriorityThread() {
        if (priorityThreads == null) {
            priorityThreads = new HashSet();
        }
        priorityThreads.add(Thread.currentThread());
    }
    
    public synchronized void enter(boolean writeAccess) {
        Thread thread = Thread.currentThread();

        // make sure no document is locked upon starting a new transaction
        assert (counter > 0 && this.thread == thread) || JMManager.getDocumentLocksCounter() == null || JMManager.getDocumentLocksCounter().get() == null : "Document was locked before starting the MDR transaction."; // NOI18N
        
        if (this.thread != thread) {
            if (SwingUtilities.isEventDispatchThread()) addPriorityThread();
            boolean isPriorityThread = priorityThreads != null && priorityThreads.remove(thread);
            while (!(counter == 0 && (!isSwingWaiting || isPriorityThread))) {
                try {
                    isSwingWaiting |= isPriorityThread;
    //                if (isPriorityThread) {
    //                    System.err.println("Priority thread waiting: " + thread.toString());
    //                    Thread.dumpStack();
    //                }
                    this.wait();
                } catch (InterruptedException e) {
                    ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, e);
                }
            }
    //            if (writeAccess && endTransInProgress) {
    //                throw new IllegalStateException("Cannot do modifications while commit/rollback in progress.");
    //            }
            this.thread = thread;

            if (isPriorityThread) {
    //            System.out.print(".");
                if (priorityThreads != null && priorityThreads.isEmpty()) {
                    priorityThreads = null;
                }
                isSwingWaiting = priorityThreads != null;
            }
            
            modCount++;
            if (modCount == 0) modCount = 1;

            if (JMManager.TRANS_DEBUG) {
                Thread.dumpStack(); 
            }
        }
        
        counter++;
        
        if (writeAccess) {
            if (!changes) {
                changes = true;
                start();
            }
        }
        
        if (counter == 1) {
            boolean success = false;
            try {
                parseIfNeeded(writeAccess);
                success = true;
            } finally {
                if (!success) {
                    try {
                        if (changes && counter == 1) {
                            end(true);
                            notifyElements();
                            ClassIndex.rollback();
                        }
                    } catch (RuntimeException x) {
                        // throw the original exception rather than x
                        ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, x);
                    } finally {
                        finalizeTrans();
                    }
                }
            }
            clearClassPath();
            clearParserCache();
        }
    }

    private void clearClassPath() {
        classPath = null;
        searchScope = null;
    }
    
    private void clearParserCache() {
        parserCache=null;
    }

    private void parseIfNeeded(boolean writeAccess) {
        boolean isEmpty;
        FileObject[] fos;
        
        synchronized (needDelete) {
            isEmpty = needDelete.isEmpty();
        }
        
        if (!isEmpty) {
            synchronized (needDelete) {
                fos = (FileObject[]) needDelete.toArray(new FileObject[needDelete.size()]);
                if (DEBUG) {
                    System.err.println("cleaning needDelete"); // NOI18N
                    Thread.dumpStack();
                }
                needDelete.clear();
            }
            for (int i = 0; i < fos.length; i++) {
                FileObject fo = fos[i];
                ResourceImpl res;
                
                FileObject cpRoot = Util.getCPRoot(fo);
                if (cpRoot == null) return; // ignore fileobjects that are not visible from our merged classpath
                
                JavaModelPackage modelPckg = manager.resolveJavaExtent(cpRoot);
                if (modelPckg==null) return;
                Resource resource = (ResourceImpl) ((ResourceClassImpl) modelPckg.getResource()).resolveResource(
                        manager.getResourceName(fo), false, false);
                
                if (resource!=null) {
                    resource.refDelete();
                }
            }
        }

        synchronized (needParsingRW) {
            isEmpty = needParsingRW.isEmpty();
        }
        if (!isEmpty) {
            synchronized (needParsingRW) {
                fos = (FileObject[]) needParsingRW.toArray(new FileObject[needParsingRW.size()]);
                needParsingRW.clear();
            }

            for (int i = 0; i < fos.length; i++) {
                FileObject fobj=fos[i];

                if (!fobj.isValid()) continue;
                JavaModel.setClassPath(fobj);
                RepositoryUpdater.getDefault().createOrUpdateResource(fobj);
		needParsing.remove(fobj);
            }
        }

        if (writeAccess) {
            synchronized (needParsing) {
                isEmpty = needParsing.isEmpty();
            }
            
            if (!isEmpty) {
                synchronized (needParsing) {
                    fos = (FileObject[]) needParsing.toArray(new FileObject[needParsing.size()]);
                    if (DEBUG) {
                        System.err.println("cleaning needParsing"); // NOI18N
                        Thread.dumpStack();
                    }
                    needParsing.clear();
                }
                for (int i = 0; i < fos.length; i++) {
                    FileObject fobj=fos[i];
                    ResourceImpl res;
                    
                    if (!fobj.isValid()) continue;
                    JavaModel.setClassPath(fobj);
                    res = (ResourceImpl) ((JMManager) JavaMetamodel.getManager()).getResource(fobj, false);
                    if (res != null) {
                        res.updateFromFileObject(fobj,true);
                    }
                }
            }
        }
        
        synchronized (needTSUpdate) {
            isEmpty = needTSUpdate.isEmpty();
        }
        if (!isEmpty) {
            synchronized (needTSUpdate) {
                fos = (FileObject[]) needTSUpdate.toArray(new FileObject[needTSUpdate.size()]);
                needTSUpdate.clear();
            }
            
            for (int i = 0; i < fos.length; i++) {
                FileObject fo = fos[i];
                if (!fo.isValid()) continue;
                RepositoryUpdater.updateTimeStamp(fo);
            }
        }
    }
    
    public Thread getThread() {
        return thread;
    }
    
    public void addModified(FileObject fo) {
        if (DEBUG) {
            System.out.println("adding " + fo + " to needParsing"); // NOI18N
            Thread.dumpStack();
        }
        synchronized (needParsing) {
            needParsing.add(fo); 
        }
    }
    
    void addDeleted(FileObject fo) {
        synchronized (needDelete) {
            needDelete.add(fo);
        }
    }
    
    void addModifiedRW(FileObject fo) {
        synchronized (needParsingRW) {
            synchronized (needParsing) {
                needParsingRW.add(fo);
                needParsing.remove(fo);
            }
        }
    }
    
    void addUpdateTS(FileObject fo) {
        synchronized (needTSUpdate) {
            needTSUpdate.add(fo);
        }
    }
    
    public synchronized boolean leave(boolean fail) {
        assert thread == Thread.currentThread() : "Cannot end transaction from a different thread!"; // NOI18N
        if (fail) {
            JMManager.getLog().notify(ErrorManager.INFORMATIONAL,new Exception("rollback!!!")); // NOI18N
        }
        boolean result = false;
        try {
            if (changes) {
                this.fail |= fail;
            } else {
                if (fail) throw new RuntimeException("Cannot fail in read mode."); // NOI18N
            }

            if (counter == 1) {
                result = true;
                if (changes) {
//                    endTransInProgress = true;
                    if (this.fail) {
                        end(true);
                        notifyElements();
                        ClassIndex.rollback();
                    } else {
                        notifyElements();
                        end(false);
                        ClassIndex.commit();
                    }
                }
            }
        } catch (RuntimeException x) {
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, x);
            this.fail = true;
            try {
                end(true);
            } catch (RuntimeException e) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, e);
            }
            try {
                notifyElements();
            } catch (RuntimeException e) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, e);
            }
            try {
                ClassIndex.rollback();
            } catch (RuntimeException e) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, e);
            }
        } finally {
            finalizeTrans();
        }
        return result;
    }
    
    private void finalizeTrans() {
        if (counter > 0) {
            counter--;
            if (counter == 0) {
                this.modificationsDisabled = false;
                this.isSafeTrans = false;
                this.thread = null;
                this.fail = false;
                changes = false;
                clearClassPath();
                clearParserCache();
                this.notifyAll();
            }
        } else {
            throw new RuntimeException("Error: leave() without enter()."); // NOI18N
        }
    }

    void setClassPath(List/*<ClassPath>*/ cp, boolean preferSources) {
        classPath = FilterClassPathImplementation.createClassPath(cp, preferSources);
    }
    
    public void setSearchScope(List/*<ClassPath>*/ cp) {
        searchScope = cp;
    }
    
    List/*<ClassPath>*/ getSearchScope() {
        return searchScope;
    }
    
    public ClassPath getClassPath() {
        return classPath;
    }
    
    public synchronized Map getParserCache() {
        if (parserCache==null)
            parserCache=new HashMap();
        return parserCache;
    }
    
    public void invalidateAtCommit(Element element) {
        invalidClasses.add(element);
    }

    public boolean pendingChanges() {
        return changes;
    }

    public boolean willFail() {
        return fail;
    }

    public void registerChange(ResourceImpl resource) {
        changedRscSet.add(resource);
    }
    
    public void registerPersisted(SemiPersistentElement element) {
        upToDateElements.add(element._getMofId());
    }

    public void unregisterChange(ResourceImpl resource) {
        changedRscSet.remove(resource);
    }

    public void registerExtChange(ExternalChange change) {
        changedExternalSet.add(change);
    }

    public void registerUndoElement(ExternalChange change) {
        undoElementsSet.add(change);
    }

    public void addNew(Element h) {
        newObjects.add(h);
    }

    public void removeNew(Element h) {
        newObjects.remove(h);
    }

    public void addBFeatureToInitQueue(Object bf) {
        initBodyQueue.add(bf);
    }

    private void notifyElements() {
        if (!upToDateElements.isEmpty()) {
            for (Iterator it = upToDateElements.iterator(); it.hasNext();) {
                Object element = ((NBMDRepositoryImpl) JavaMetamodel.getDefaultRepository()).getByMofId((MOFID) it.next());
                if (element != null) {
                    ((SemiPersistentElement) element).clearPersist(this.fail);
                }
            }
            if (!this.fail) {
                upToDateElements.clear();
            }
        }
        if (!modificationsDisabled && (!changedRscSet.isEmpty() || !changedExternalSet.isEmpty() || !undoElementsSet.isEmpty())) {
            JavaMetamodel.getUndoManager().transactionStarted();
        }

        try {
            if (!modificationsDisabled && !this.fail) {
                for (Iterator it = undoElementsSet.iterator(); it.hasNext();) {
                    JavaMetamodel.getUndoManager().addItem((ExternalChange) it.next());
                }
            }
            initBodyQueue = new ArrayList();
            if (this.fail || !modificationsDisabled) {
                if (!changedRscSet.isEmpty()) {
                    JMManager.initIndentation();
                    for (Iterator resIt = changedRscSet.iterator(); resIt.hasNext(); ) {
                        ResourceImpl resource = (ResourceImpl) resIt.next();
                        if (this.fail) {
                            resource.rollbackChanges();
                        } else {
                            resource.commitChanges();
                            if (!newObjects.isEmpty()) {
                                throw new RuntimeException("Some objects were not added to a resource or deleted before commit."); // NOI18N
                            }
                        }
                    }
                }
            }

            if (!this.fail) {
                for (Iterator it = invalidClasses.iterator(); it.hasNext();) {
                    Element el = (Element) it.next();
                    if (el.isValid()) {
                        el.refDelete();    
                    }
                    it.remove();
                }
                ResourceImpl resource;
                Iterator resIt;            
                for (resIt = changedRscSet.iterator(); resIt.hasNext(); ) {
                    resource = (ResourceImpl) resIt.next();                
                    resource.parseResource();
                }
                for (resIt = changedRscSet.iterator(); resIt.hasNext(); ) {
                    resource = (ResourceImpl) resIt.next();                
                    resource.commitConfirmed();
                    FileObject fo = ((JMManager) JavaMetamodel.getManager()).getFileObject(resource);
                    if (DEBUG) System.out.println("removing " + fo + " from needParsing"); // NOI18N
                    synchronized (needParsing) {
                        needParsing.remove(fo);
                    }
                    synchronized (needParsing) {
                        needParsingRW.remove(fo);
                    }
                }
            }
            while (!initBodyQueue.isEmpty()) {
                Object bf = initBodyQueue.remove(0);
                if (bf instanceof BehavioralFeatureImpl) {
                    ((BehavioralFeatureImpl)bf).initBody();
                } else if (bf instanceof FieldImpl) {
                    ((FieldImpl)bf).initInitValue();
                } else if (bf instanceof EnumConstantImpl) {
                    ((EnumConstantImpl) bf).initInitValue();
                } else if (bf instanceof AttributeImpl) {
                    ((AttributeImpl) bf).initDefaultValue();
                } else {
                    ((AttributeValueImpl) bf).initInitValue();
                }
            }
        
            initBodyQueue=null;        

            if (!this.fail && !modificationsDisabled) {
                for (Iterator it = changedExternalSet.iterator(); it.hasNext(); ) {
                    ExternalChange change = (ExternalChange) it.next();
                    JavaMetamodel.getUndoManager().addItem(change);
                    change.performExternalChange();
                }
            }
        } finally {
            if (!modificationsDisabled && (!changedRscSet.isEmpty() || !changedExternalSet.isEmpty() || !undoElementsSet.isEmpty())) {
                JavaMetamodel.getUndoManager().transactionEnded(this.fail);
            }
            changedExternalSet.clear();
            undoElementsSet.clear();
            changedRscSet.clear();
            newObjects.clear();
        }
    }

    public void disableModifications() {
        modificationsDisabled = true;
    }
    
    boolean isModified(FileObject fileObject) {
        synchronized(needParsing) {
            return needParsing.contains(fileObject);
        }
    }
}
