/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore;

import java.util.AbstractSequentialList;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.NoSuchElementException;

/**
 *
 * @author  Martin Matula
 */
public class LazyImmutableList extends AbstractSequentialList {
    private LazyIterator internal;
    private boolean complete = false;
    private final Object[] array;
    private int nextIndex = 0;
    
    /** Creates a new instance of LazyImmutableList */
    public LazyImmutableList(LazyIterator internalIterator) {
        internal = internalIterator;
        array = new Object[internalIterator.maxEstimatedSize()];
    }
    
    public int size() {
        while (internal != null) {
            nextItem();
        }
        return nextIndex;
    }
    
    public ListIterator listIterator(int index) {
        return new LazyImmutableIterator(index);
    }
    
    public int maxEstimatedSize() {
        return internal == null ? nextIndex : internal.maxEstimatedSize();
    }
    
    private void nextItem() {
        if (internal != null && internal.hasNext()) {
            array[nextIndex] = internal.next();
            nextIndex++;
        } else {
            internal = null;
        }
    }
    
    private class LazyImmutableIterator implements ListIterator {
        private int currentIndex;
        
        public LazyImmutableIterator(int index) {
            currentIndex = index;
        }

        public boolean hasNext() {
            while (nextIndex <= currentIndex && internal != null) {
                nextItem();
            }
            return nextIndex > currentIndex;
        }

        public int previousIndex() {
            return currentIndex - 1;
        }

        public Object previous() {
            if (hasPrevious()) {
                currentIndex--;
                return array[currentIndex];
            }
            throw new NoSuchElementException();
        }

        public boolean hasPrevious() {
            return currentIndex > 0;
        }

        public Object next() {
            if (hasNext()) {
                currentIndex++;
                return array[currentIndex - 1];
            }
            throw new NoSuchElementException();
        }

        public int nextIndex() {
            return currentIndex;
        }

        public void add(Object o) {
            throw new UnsupportedOperationException();
        }

        public void set(Object o) {
            throw new UnsupportedOperationException();
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
    
    public static abstract class LazyIterator implements Iterator {
        public final void remove() {
            throw new UnsupportedOperationException();
        }
        
        protected abstract int maxEstimatedSize();
    }
}
