/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore;

import java.util.*;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.internalapi.GuardedQuery;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.jmiimpl.javamodel.*;
import org.openide.filesystems.FileObject;
import org.openide.util.NbBundle;

/**
 * TryWrapper can encapsulate block with a try statement
 * @author Jan Becicka
 */
public class TryWrapper {
    /**
     * element which will be replaced by try statement
     */    
    private Element elementToReplace;
    /**
     * parent of elementToReplace
     */    
    private Element replaceParent;
    
    private JavaModelPackage model;

    private List tryBody;
    
    /**
     * TryWrapper can encapsulate block with a try statement
     * @param fo FileObject where try will be added
     * @param from begin index
     * @param to end index
     */    
    public TryWrapper(FileObject fo, int from, int to) { 
        this(JavaModelUtil.getSelectedStatements(JavaMetamodel.getManager().getResource(fo), from, to));
    }

    public TryWrapper(List/*<Statement>*/ statements) {
        if (statements==null) {
            throw new GeneralException(NbBundle.getMessage(TryWrapper.class, "MSG_CannotEncapsulate"));
        }
        tryBody = new ArrayList(statements);
        
        elementToReplace = (Element) tryBody.get(0);
        if (GuardedQuery.isSectionGuarded(elementToReplace.getResource(), JavaMetamodel.getManager().getElementPosition(elementToReplace))) {
            throw new GeneralException(NbBundle.getMessage(TryWrapper.class, "MSG_CannotEncapsulateGuardedBlock"));
        }
        replaceParent = (Element) elementToReplace.refImmediateComposite();
        model = (JavaModelPackage) elementToReplace.refOutermostPackage();
    }
    
    private void check(List body, List declaredVariables) {
        Iterator it = body.iterator();
        while (it.hasNext()) {
            Object o = it.next(); 
            if (o instanceof LocalVarDeclaration) {
                LocalVarDeclaration decl = (LocalVarDeclaration) o;
                declaredVariables.addAll(decl.getVariables());
            }
        }
    }
    
    private ExpressionStatement createAssignment(LocalVariable var) {
        VariableAccess access = model.getVariableAccess().createVariableAccess(var.getName(),null, false);
        InitialValue initValue = var.getInitialValue();
        var.setInitialValue(null);
        Assignment a = model.getAssignment().createAssignment(access, OperatorEnum.ASSIGN, (Expression) initValue);
        ExpressionStatement s = model.getExpressionStatement().createExpressionStatement(a);
        return s;
    }
    
    private Set getUsedVariables(List decl, List statements) {
        Set used = new LinkedHashSet();
        LinkedList l = new LinkedList();
        l.addAll(statements);
        while (!l.isEmpty()) {
            Element e = (Element) l.removeFirst();
            if (e instanceof VariableAccess) {
                Iterator i = decl.iterator();
                while (i.hasNext()) {
                    LocalVariable var = (LocalVariable) i.next();
//                    if (((VariableAccess) e).getElement().equals(var)) {
//                        used.add(var);
//                    }
                    if (((VariableAccess) e).getName().equals(var.getName())) {
                        used.add(var);
                    }
                }
            }
            l.addAll(e.getChildren());
        }
        return used;
    }
    
    /**
     * wrap given selection with a try statement
     * @return TryStatement
     */    
    public TryStatement wrap() {
        //try block
        List decl = new ArrayList();
        check(tryBody, decl);
        
        List catches = createCatches(getExceptions(tryBody));
        
        //empty finally block
        StatementBlock finallyBlock = !catches.isEmpty()?null:model.getStatementBlock().createStatementBlock();
        
        TryStatement tryStatement = model.getTryStatement().createTryStatement();
        tryStatement.getCatches().addAll(catches);
        tryStatement.setFinalizer(finallyBlock);
        
        if (replaceParent instanceof Feature) {
            ArrayList l = new ArrayList();
            l.add(tryStatement);
            replaceParent.replaceChild(elementToReplace, model.getStatementBlock().createStatementBlock(l));
            StatementBlock tryBlock = model.getStatementBlock().createStatementBlock(tryBody);
            tryStatement.setBody(tryBlock);
        } else {
            List ll = new ArrayList();
            ll.addAll(replaceParent.getChildren());
            ll.remove(elementToReplace);
            replaceParent.replaceChild(elementToReplace, tryStatement);
            ArrayList tryBody2 = new ArrayList(tryBody);
            tryBody.remove(elementToReplace);
            for (Iterator i = tryBody.iterator(); i.hasNext() ; ) {
                Element e = (Element) i.next();
                replaceParent.replaceChild(e, null);
                ll.remove(e);
            }
            StatementBlock tryBlock = model.getStatementBlock().createStatementBlock(tryBody2);
            tryStatement.setBody(tryBlock);
            Set l = getUsedVariables(decl, ll);
            List body;
            if (replaceParent instanceof StatementBlock) {
                // commands are mostly closed in statement blocks,
                body = ((StatementBlock) replaceParent).getStatements();
            } else {
                // but they can be also in if/else statements outside the
                // block, e.g. 
                // if (b == true) throw new IllegalArgumentException
                body = Collections.singletonList(tryStatement);
            }
            int index = body.indexOf(tryStatement);
            body.addAll(index, separateDeclsAndInits(l, tryBlock));
            // leave all the other declarations!!!
        }
        return tryStatement;
    }
    
    /**
     * Separetes declarations and initializers of variables. This separation
     * is used when initial value of local variable in declaration throws
     * exception and variable is used outside target try block. In such a case,
     * we have to create new declaration before the try statement, but
     * initializer has to be inside as it can throw exception. It is obvious
     * from example:
     * <code>
     *   FileInputStream fis = new FileInputStream(new File("jezFile"));
     *   if (fis == null) return;
     * </code>
     * Consider you are wrapping first line of code with declaring <tt>fis</tt>
     * variable. Constructor can throw <tt>java.io.IOException</tt>, so the
     * code has to be changed to:
     * <code>
     *   FileInputStream fis;
     *   try {
     *       fis = new FileInputStream(new File("jezFile"));
     *   } catch (FileNotFoundException ex) {
     *       ex.printStackTrace();
     *   }
     *   if (fis == null) return;
     * </code>
     * Variable <tt>fis</tt> is used outside the try block, it has to be
     * declared before.
     *
     * The method also solve cases when more then one variable is declared
     * in a row, i.e.
     * <code>
     *   FileInputStream fis = new FileInputStream(new File("jezFile")),
     *                   fis2 = new FileInputStream(new File("nejezFile"));
     * </code>
     *
     * @param   usedVars set of local variable which are used outside the
     *          try block and needs separation of declaration and init values
     * @param   addInitializersTo statement block, where the initializers
     *          separated from declaration will be added. From the example
     *          above, newly created assignement
     *          <tt>fis = new FileInputStream(new File("jezFile"));</tt>
     *          is added.
     * @return  list of newly created declarations which has to be added
     *          before the try block. From the example above,
     *          <tt>FileInputStream fis;</tt> is added to the list.
     *
     */
    private List separateDeclsAndInits(Set usedVars, StatementBlock addInitializersTo) {
        List decl = new ArrayList();
        Iterator varsIt = usedVars.iterator();
        Set separated = new HashSet(2);
        List statements = addInitializersTo.getStatements();
        while (varsIt.hasNext()) {
            LocalVariable var = (LocalVariable) varsIt.next();
            LocalVarDeclaration parent = (LocalVarDeclaration) var.refImmediateComposite();
            // work only on set of declarations. All the variables from
            // one declaration are solved toghether. So next variable
            // from the same declaration was solved already, skip it.
            if (!separated.contains(parent)) {
                separated.add(parent);
                LocalVarDeclaration lvdCopy = (LocalVarDeclaration) parent.duplicate();
                // operate with all variables in declaration
                LocalVariable[] vars = (LocalVariable[]) lvdCopy.getVariables().toArray(new LocalVariable[0]);
                boolean removeInitVals = false;
                for (int varInDecl = 0; varInDecl < vars.length; varInDecl++) {
                    for (Iterator varsIt2 = usedVars.iterator(); varsIt2.hasNext(); ) {
                        LocalVariable lv_used = (LocalVariable) varsIt2.next();
                        if (lv_used.getName().equals(vars[varInDecl].getName())) {
                            // if we have found the first used variable declared
                            // in specified declaration and used outside the
                            // try block, we have to separate declaration and
                            // initialization. Set the flag.
                            removeInitVals = true;
                            break;
                        }
                    }
                    // if any of initial value from local variable declaration
                    // can throw exception, the variable is initialized
                    // inside the try block in assignement expression, outside
                    // the declaration.
                    if (removeInitVals) {
                        List newAssignments = new ArrayList();
                        int parentIndex = statements.indexOf(parent);
                        
                        for (int j = 0; j < vars.length; j++) {
                            if (vars[j].getInitialValue() != null) {
                                newAssignments.add(createAssignment(vars[j]));
                                vars[j].setInitialValue(null);
                            }
                        }
                        statements.addAll(parentIndex,newAssignments); 
                        break;
                    }
                }
                // if any of declarations was separated from its initializers
                // (initializer of variable throws exception), put the newly
                // created declaration to the return list.
                if (removeInitVals) {
                    decl.add(lvdCopy);
                    parent.refDelete();
                }
            }
        }
        return decl;
    }

    /**
     * creates catch statements from List of Exception names
     * @param ex List of ElementReferences
     * @return List of Catch statements
     */    
    private List createCatches(Collection ex) {
        List catches = new ArrayList();
        String parName = "ex";	// NOI18N

        for (Iterator i = ex.iterator(); i.hasNext();) {
            MultipartId e = (MultipartId) i.next();
            Parameter par = model.getParameter().createParameter(parName, null, false, e, 0, false);
            MultipartId var = model.getMultipartId().createMultipartId(parName, null, null);
            MethodInvocation m = model.getMethodInvocation().createMethodInvocation("printStackTrace", Collections.EMPTY_LIST, var, false); //NOI18N
            Statement st = model.getExpressionStatement().createExpressionStatement(m);
            StatementBlock block = model.getStatementBlock().createStatementBlock(Collections.singletonList(st));
            
            catches.add(model.getCatch().createCatch(par, block));
        }
        return catches;
    }
    
    /**
     *
     * @param b Statement block to search
     * @return Exception names of all callable features in statement blcok
     */    
    private Collection getExceptions(List b) {
        Set exSet = JavaModelUtil.getExceptionsFromStatements(b);
        Map exMap = new TreeMap();
        Collection result = new ArrayList();
        Collection exceptions;
        int index = 0;
        int exs = exSet.size();
                
        for (Iterator i = exSet.iterator(); i.hasNext(); index++) {
            JavaClass c = (JavaClass) i.next();
            int id = -(getExceptionLevel(c)*exs+index);
            
            exMap.put(new Integer(id),c);
        }
        exceptions = exMap.values();
        for (Iterator i = exceptions.iterator(); i.hasNext();) {
            JavaClass c = (JavaClass) i.next();
            result.add(JavaModelUtil.resolveImportsForClass(replaceParent, c));
        }
        return result;
    }
    
    private int getExceptionLevel(JavaClass ex) {
        if (ex.getName().equals("java.lang.Object")) {  // to prevent endless loop 
            return 0;                                   // for unresolved classes
        }
        if (ex.getName().equals("java.lang.Throwable")) {
            return 0;
        }
        return getExceptionLevel(ex.getSuperClass())+1;
    }
    
}
