/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.internalapi;

import java.awt.Container;
import java.awt.Dialog;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.MessageFormat;
import javax.swing.JComponent;
import javax.swing.SwingConstants;
import javax.swing.border.EmptyBorder;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.mdr.MDRManager;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.jmi.util.MetamodelManager;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.JavaCoreModule;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.text.PositionBounds;
import java.util.ArrayList;
import java.util.List;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import org.netbeans.modules.javacore.UndoProgressListener;
import org.openide.DialogDescriptor;
import org.openide.DialogDisplayer;
import org.openide.util.NbBundle;
import org.openide.util.RequestProcessor;

/**
 * @author Martin Matula
 */
public abstract class JavaMetamodel extends MetamodelManager {
    public static final String NAME_REPOSITORY = "org.netbeans.java"; // NOI18N

    private static JavaMetamodel instance = null;
    private static MDRepository defaultRepository = null;
    private static UndoManager undoManager = null;

    protected static final ArrayList listeners = new ArrayList();

    protected JavaMetamodel(MDRepository repository, Class cls) {
        super(repository, cls);
    }

    public static synchronized JavaMetamodel getManager() {
        if (instance == null) {
            instance = new JMManager(getDefaultRepository());
//            instance.resolveCodebases();
        }
        return instance;
    }

    /** WARNING: This method returns null if no classpath is set during the transaction. */
    public abstract ClassPath getClassPath();
    
    public abstract void setClassPath(ClassPath cp);
    public abstract void setClassPath(List/*<ClassPath>*/ cps);
    public abstract void setClassPath(Resource rsc);
    public abstract void setClassPath(FileObject fo);
    public abstract void setClassPath(FileObject fo, boolean preferSources);
    public abstract Resource getResource(FileObject fo);
    public abstract Resource getResource(FileObject cpRoot, String name);
    public abstract JavaPackage getPackage(FileObject fo);
    public abstract FileObject getFileObject(Resource r);

    public abstract DataObject getDataObject(Resource r);
    public abstract void registerUndoElement(ExternalChange ch);
    public abstract void registerExtChange(ExternalChange ch);

    public abstract boolean isElementGuarded(Element elem);
    
    /** Returns JavaModelPackage for a given cpRoot. If the extent does not exist, it will be created.
     *
     * @param cpRoot
     * @return
     */
    public abstract JavaModelPackage resolveJavaExtent(FileObject cpRoot);

    /** Tries to find an extent for a given classpath root. If the extent does not exist, returns null.
     *
     * @param cpRoot
     * @return
     */
    public abstract JavaModelPackage getJavaExtent(FileObject cpRoot);

    public abstract JavaModelPackage getDefaultExtent();
    public abstract PositionBounds getElementPosition(Element element);
    public abstract PositionBounds getElementPosition(Element element, boolean inclDoc);
    public abstract Element getElementByOffset(FileObject fo, int offset);
    public abstract ProgressSupport getProgressSupport();
    public abstract void addModified(FileObject fobj);
    
    /**
     * Tests if the file is modified and has to be re-parsed. This means, that
     * content of the file (document) does not correspond to data available
     * in the storage.
     *
     * @param   fo  file representing the document
     * @return  true, if the file does not correspond to data in the storage,
     *          false, if file content corresponds to data in the storage
     */
    public abstract boolean isModified(FileObject fo);
    
    public abstract PositionBounds getElementPartPosition(Element element, ElementPartKind part, int position);
    /**
     * @return true if classpath scanning is progress <br>
     *         false otherwise
     */
    public abstract boolean isScanInProgress();
    /**
     * Waits for scanning finish. Goes through all registered path roots and
     * looks for its extent. When there is extent for all path roots,
     * method exits. The method blocks the caller.
     * Currently there isn't any timeout.
     * @return false if no waiting was needed (scanning was not in progress)
     * <br>true if scanning was in progress and thread waited
     */
    public abstract boolean waitScanFinished();

    public JavaModelPackage getJavaExtent(Element element) {
        return (JavaModelPackage) element.refImmediatePackage();
    }

    protected abstract void resolveCodebases();

    public static synchronized MDRepository getDefaultRepository() {
        if (JavaMetamodel.defaultRepository == null) {
            // find repository
            defaultRepository = MDRManager.getDefault().getRepository(NAME_REPOSITORY);
            if (defaultRepository == null) {
                throw new IllegalStateException("FATAL: Repository " + NAME_REPOSITORY + " not found."); // NOI18N
            }
        }
        return defaultRepository;
    }

    public synchronized static UndoManager getUndoManager() {
        if (undoManager == null) {
            undoManager = new UndoManager(new UndoProgressListener());
        }
        return undoManager;
    }

    public static void addParsingListener(ParsingListener listener) {
        synchronized (listeners) {
            listeners.add(listener);
        }
    }

    public static void removeParsingListener(ParsingListener listener) {
        synchronized (listeners) {
            listeners.remove(listener);
        }
    }
    
    /**
     * This is a helper method to provide support for delaying invocations of actions
     * depending on java model. See <a href="http://java.netbeans.org/ui/waitscanfinished.html">UI Specification</a>.
     * <br>Behavior of this method is following:<br>
     * If classpath scanning is not in progress, runnable's run() is called. <br>
     * If classpath scanning is in progress, modal cancellable notification dialog with specified
     * tile is opened.
     * </ul>
     * As soon as classpath scanning finishes, this dialog is closed and runnable's run() is called.
     * This method must be called in AWT EventQueue. Runnable is performed in AWT thread.
     * 
     * @param runnable Runnable instance which will be called.
     * @param actionName Title of wait dialog.
     * @return true action was cancelled <br>
     *         false action was performed
     */
    public boolean invokeAfterScanFinished(final Runnable runnable , final String actionName) {
        assert SwingUtilities.isEventDispatchThread();
        if (isScanInProgress()) {
            final ActionPerformer ap = new ActionPerformer(runnable);
            ActionListener listener = new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    ap.cancel();
                    waitTask.cancel();
                }
            };
            JLabel label = new JLabel(getString("MSG_WaitScan"), javax.swing.UIManager.getIcon("OptionPane.informationIcon"), SwingConstants.LEFT);
            label.setBorder(new EmptyBorder(12,12,11,11));
            DialogDescriptor dd = new DialogDescriptor(label, actionName, true, new Object[]{getString("LBL_CancelAction", new Object[]{actionName})}, null, 0, null, listener);
            waitDialog = DialogDisplayer.getDefault().createDialog(dd);
            waitDialog.pack();
            waitTask = RequestProcessor.getDefault().post(ap);
            waitDialog.setVisible(true);
            waitTask = null;
            waitDialog = null;
            return ap.hasBeenCancelled();
        } else {
            runnable.run();
            return false;
        }
    }
    private Dialog waitDialog = null;
    private RequestProcessor.Task waitTask = null;
    
    private static String getString(String key) {
        return NbBundle.getMessage(JavaCoreModule.class, key);
    }
    
    private static String getString(String key, Object values) {
        return new MessageFormat(getString(key)).format(values);
    }

    private class ActionPerformer implements Runnable {
        private Runnable action;
        private boolean cancel = false;

        ActionPerformer(Runnable a) {
            this.action = a;
        }
        
        public boolean hasBeenCancelled() {
            return cancel;
        }
        
        public void run() {
            waitScanFinished();
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    if (!cancel) {
                        waitDialog.setVisible(false);
                        action.run();
                    }
                }
            });
        }
        
        public void cancel() {
            assert SwingUtilities.isEventDispatchThread();
            // check if the scanning did not finish during cancel
            // invocation - in such case do not set cancel to true
            // and do not try to hide waitDialog window
            if (waitDialog != null) {
                cancel = true;
                waitDialog.setVisible(false);
            }
        }
    }
}
