/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.javacore.internalapi;

import java.util.*;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ArrayCloneMethod;
import org.netbeans.modules.javacore.jmiimpl.javamodel.JavaClassImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.MetadataElement;
import org.netbeans.modules.javacore.jmiimpl.javamodel.MethodImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ParameterizedTypeImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.SemiPersistentElement;
import org.netbeans.modules.javacore.parser.Scope;
import org.netbeans.modules.javacore.parser.TypeRef;

/**
 *
 * @author Tomas Hurka
 */
public class JavaModelUtil {
    
    private JavaModelUtil() {
    }

    /**
     * Analyze list of statements and returns all unhandled exceptions in this list.
     * @param statements List of {@link Statement} to be analyzed
     * @return Set of unhandled exceptions. The Set contains {@link JavaClass}  
     */
    public static Set getExceptionsFromStatements(List statements) {
        return computeAllExceptions(statements, new HashSet());
    }
    
    private static Set computeAllExceptions(List elements,Set ignoredExceptions) {
        Iterator elIt=elements.iterator();
        Set foundExceptions=new HashSet();
        
        while(elIt.hasNext()) {
            Object el=elIt.next();
            
            if (el instanceof Invocation) {
                Invocation inv=(Invocation)el;
                CallableFeature cf=(CallableFeature)inv.getElement();
                
                if (cf!=null) {
                    List exceptions=cf.getExceptions();

                    if (!exceptions.isEmpty()) {
                        Iterator exIt=exceptions.iterator();

                        while(exIt.hasNext()) {
                            addException((JavaClass)exIt.next(), foundExceptions, ignoredExceptions);
                        }
                    }
                }
            } else if (el instanceof MultipartId) {
                continue;
            } else if (el instanceof ThrowStatement) {
                ThrowStatement throwSt=(ThrowStatement)el;
                Expression ex=throwSt.getExpression();

                if (ex!=null) addException((JavaClass)ex.getType(), foundExceptions, ignoredExceptions);
            } else if (el instanceof TryStatement) {
                TryStatement trySt=(TryStatement)el;
                StatementBlock body=trySt.getBody();
                
                if (body!=null) {
                    List catches=trySt.getCatches();
                    Iterator catchIt=catches.iterator();
                    Set localIgnores=new HashSet(ignoredExceptions);
                    StatementBlock finalizer=trySt.getFinalizer();

                    while (catchIt.hasNext()) {
                        Catch c=(Catch)catchIt.next();
                        JavaClass ex=(JavaClass)c.getParameter().getType();
                        
                        addException(ex, localIgnores, Collections.EMPTY_SET);
                    }
                    foundExceptions.addAll(computeAllExceptions(Collections.singletonList(body), localIgnores));
                    foundExceptions.addAll(computeAllExceptions(catches, ignoredExceptions));
                    if (finalizer!=null)
                        foundExceptions.addAll(computeAllExceptions(Collections.singletonList(finalizer), ignoredExceptions));
                    return foundExceptions;
                }
            }
            foundExceptions.addAll(computeAllExceptions(((Element)el).getChildren(), ignoredExceptions));
        }
        return foundExceptions;
    }

    private static void addException(final JavaClass ex, final Set foundExceptions, final Set ignoredExceptions) {
        Iterator jclsIt;
        
        if (ex==null || ex instanceof UnresolvedClass)
            return;
        if (ignoredExceptions.contains(ex) || foundExceptions.contains(ex))
            return;
        jclsIt=ignoredExceptions.iterator();
        while(jclsIt.hasNext()) {
            JavaClass igEx=(JavaClass)jclsIt.next();
            
            if (ex.isSubTypeOf(igEx)) {
                return;
            }
        }
        jclsIt=foundExceptions.iterator();
        while(jclsIt.hasNext()) {
            JavaClass fex=(JavaClass)jclsIt.next();
            
            if (ex.isSubTypeOf(fex)) {
                return;
            }
        }
        foundExceptions.add(ex);
    }

    /**
     * Computes List of {@link Statement} specified by start and end offset. 
     * Handles all cases where start and end offset are between statements 
     * as well as cases where start and end statements are form different context.
     * @param rsc {@link Resource} of the source file.
     * @param startOffset start offset of the selection
     * @param endOffset end offset of the selection
     * @return List of {@link Statement} or null if the selection specified by startOffset and endOffset is incorrect
     */
    public static List getSelectedStatements(Resource rsc,int startOffset,int endOffset) {
        Element firstStComposite;
        Element lastStComposite;
        int lastStEndOffset;
        List selectedStatements;
        Statement firstStatement=getStatement(rsc, startOffset);
        Statement lastStatement=getStatement(rsc, endOffset-1);
        if (firstStatement==null || lastStatement==null) {
            return null; // ERR_ExtractMethodWrongSelection"              
        }
        firstStatement=adjustFirstStatement(startOffset,firstStatement);
        lastStatement=adjustLastStatement(endOffset,lastStatement);
        if (firstStatement==null || lastStatement==null) {
            return null; // ERR_ExtractMethodWrongSelection
        }
        firstStComposite=(Element)firstStatement.refImmediateComposite();
        lastStComposite=(Element)lastStatement.refImmediateComposite();
        lastStEndOffset=lastStComposite.getEndOffset();
        while (!firstStComposite.equals(lastStComposite)) {
            if (!(lastStComposite instanceof Statement) || lastStComposite.getEndOffset()!=lastStEndOffset) {
                return null; // ERR_ExtractMethodWrongList                
            }
            lastStatement=(Statement)lastStComposite;
            lastStComposite=(Element)lastStatement.refImmediateComposite();
        }
        if (firstStatement.equals(lastStatement)) {
            selectedStatements=Collections.singletonList(firstStatement);
        } else {
            List statements;
            if (firstStComposite instanceof StatementBlock) {
                statements = ((StatementBlock)firstStComposite).getStatements();
            } else {
                statements = ((Case)firstStComposite).getStatements();
            }
            int firstIndex=statements.indexOf(firstStatement);
            Iterator sIt=statements.listIterator(firstIndex);
            Object st;
            selectedStatements=new ArrayList();
            do {
                st=sIt.next();
                selectedStatements.add(st);
                
            } while(!st.equals(lastStatement));
        }
        return selectedStatements;
    }

    private static Statement getStatement(Resource rsc,int offset) {
        Element el=rsc.getElementByOffset(offset);
        int elStart,elEnd;

        if (el instanceof Feature || el instanceof Resource)
            return null;
        elStart=el.getStartOffset();
        elEnd=el.getEndOffset();
        while(el!=null && !(el instanceof Statement)) {
            el = (Element)el.refImmediateComposite();
            if (el instanceof Feature || (el.getStartOffset()!=elStart && el.getEndOffset()!=elEnd))
                return null;
        }
        return (Statement)el;
    }
    
    private static Statement adjustFirstStatement(int startOffset,Statement s) {
        JavaMetamodel model=JavaMetamodel.getManager();
        Iterator chIt;
        
        if (model.getElementPosition(s).getBegin().getOffset()==startOffset)
            return s;
        chIt=s.getChildren().iterator();
        while(chIt.hasNext()) {
            Element el=(Element)chIt.next();
            
            if (el instanceof Statement) {
                if (model.getElementPosition(el).getBegin().getOffset()>startOffset) {
                    return (Statement)el;
                }
            }
        }
        return null;
    }
    
    private static Statement adjustLastStatement(int endOffset,Statement s) {
        JavaMetamodel model=JavaMetamodel.getManager();
        ListIterator chIt;
        List children;
        
        if (model.getElementPosition(s).getEnd().getOffset()==endOffset)
            return s;
        children=s.getChildren();
        chIt=children.listIterator(children.size());
        while(chIt.hasPrevious()) {
            Element el=(Element)chIt.previous();
            
            if (el instanceof Statement) {
                if (model.getElementPosition(el).getEnd().getOffset()<endOffset) {
                    return (Statement)el;
                }
            }
        }
        return null;
    }

    /**
     * This helper method takes {@link Element} as a scope and {@link JavaClass} and creates 
     * {@link MultipartId} and possibly adds appropriate {@link Import} statement to 
     * {@link Resource}. Returned {@link MultipartId} contains simple name of resolved class 
     * if this does not introduce name conflict in source file.
     * @param scope scope defines place where jcls should be resolved.
     * @param jcls {@link JavaClass} which will be put in to source code in place defined by scope
     * @return Returns {@link MulitpartId} which resolves to jcls.
     */
    public static MultipartId resolveImportsForClass(Element scope,JavaClass jcls) {
        return (MultipartId)typeToTypeReference(scope,jcls);
    }
    
    /**
     * This helper method takes {@link Element} as a scope and {@link Type} and creates 
     * {@link TypeReference} and possibly adds appropriate {@link Import} statement to 
     * {@link Resource}. Returned {@link TypeReference} contains simple name of resolved class 
     * if this does not introduce name conflict in source file or TypeReference of primitive type
     * if {@link Type} is primitive type.
     * @param scope scope defines place where type should be resolved.
     * @param type {@link Type} which will be put in to source code in place defined by scope
     * @return Returns {@link TypeReference} which resolves to type.
     */
    public static TypeReference resolveImportsForType(Element scope,Type type) {
        return (TypeReference)typeToTypeReference(scope,type);
    }
    
    private static Element typeToTypeReference(Element scope,Type type) {
        JavaModelPackage jpck;
        MultipartIdClass mpidClass;
        
        if (type == null) {
            return null;
        }
        scope=unwrapElement(scope);
        jpck=(JavaModelPackage)scope.refImmediatePackage();
        mpidClass=jpck.getMultipartId();
        if (type instanceof TypeParameter) {
            return mpidClass.createMultipartId(type.getName(),null,null);
        } else if (type instanceof ParameterizedType) {
            ParameterizedTypeImpl paramType = (ParameterizedTypeImpl) type;
            Type typePars[] = (Type[])paramType.getParameters().toArray(new Type[0]);
            TypeArgument args[] = new TypeArgument[typePars.length];
            int i = 0;
            for (; i<typePars.length; i++) {
                args[i] = (TypeArgument)typeToTypeReference(scope,typePars[i]);
                int status = paramType.getWildCardStatus(i);
                if (status != 0) {
                    args[i] = jpck.getWildCard().createWildCard(status == 1, status == 3 ? null : (MultipartId) args[i]);
                }
            }
            MultipartId parentName=(MultipartId) typeToTypeReference(scope,paramType.getDeclaringClass());
            JavaClass def=paramType.getDefinition();
            String name;
            
            if  (parentName==null) {
                MultipartId defId=createImportsForClass(scope,def);
                
                if (typePars.length==0)
                    return defId;
                name=defId.getName();
            } else
                name=def.getSimpleName();
            return mpidClass.createMultipartId(name, parentName, Arrays.asList(args));
        } else if (type instanceof Array) {
            int dimCount = 0;
            Type currType = type;
            while (currType instanceof Array) {
                dimCount++;
                currType = ((Array) currType).getType();
            }
            return jpck.getArrayReference().createArrayReference(null,(MultipartId)typeToTypeReference(scope,currType), dimCount);
        } else if (type instanceof JavaClass) {
            if (type instanceof JavaClassImpl && ((JavaClassImpl)type).isTransient())  // local class
                return mpidClass.createMultipartId(type.getName(),null,null);
            return createImportsForClass(scope,(JavaClass)type);
        } else if (type instanceof PrimitiveType) {
            return mpidClass.createMultipartId(type.getName(),null,null);
        }
        throw new IllegalArgumentException("Unable to convert to typeref: " + type.getClass().getName()); // NOI18N
    }
    
    private static MultipartId createImportsForClass(Element scope,JavaClass type) {
        JavaModelPackage jpck=(JavaModelPackage)scope.refImmediatePackage();
        MultipartIdClass mpidClass=jpck.getMultipartId();

        if (type instanceof UnresolvedClass) {
            return mpidClass.createMultipartId(type.getName(),null,null);            
        } else {
            Scope sc=Scope.computeTypeScope(scope);
            String simpleName=type.getSimpleName();
            Object resolvedClass=sc.lookup(simpleName);
            Import imp;

            if (resolvedClass!=null) {
                if (type.getName().equals(resolvedClass)) {   // correct class found - use simple name
                    return mpidClass.createMultipartId(simpleName,null,null);
                }
                // name confilict - use FQN
                return mpidClass.createMultipartId(type.getName(),null,null);
            }
            // class not found use simple name and add import
            imp=jpck.getImport().createImport(type.getName(),null,false,false);
            scope.getResource().addImport(imp);
            return mpidClass.createMultipartId(simpleName,null,null);
        }
    }    
    
    /**
     * This method returns the closest {@link Feature}, which contains 
     * {@link Element} element.
     * @param element {@link Element} where to start
     * @return {@link Feature} or <CODE>null</CODE> if element is 
     * not part of any {@link Feature} (e.g. {@link Import})
     */
    public static Feature getDeclaringFeature(Element element) {
        while (!(element instanceof Feature) && element!=null) {
            element=(Element)element.refImmediateComposite();
        }
        return (Feature)element;
    }
    
    /**
     * This helper method takes {@link Element} as a scope and original {@link Element}. It duplicates 
     * original element and resolves types in duplicated element according to scope. 
     * It can add appropriate {@link Import} statement to 
     * {@link Resource}. Returned {@link Element} contains simple names of resolved classes 
     * if this does not introduce name conflict in source file.
     * @param scope scope defines place where duplicated element should be placed.
     * @param original {@link Element} which will be put in to source code in place defined by scope
     * @return Returns duplicated {@link Element} with correctly resolved classes.
     */
    public static Element duplicateInScope(Element scope,Element original) {
        scope=unwrapElement(scope);
        JavaModelPackage pck=(JavaModelPackage)scope.refImmediatePackage();
        MetadataElement newElement=(MetadataElement)unwrapElement(original).duplicate(pck);

        newElement.fixImports(scope,original);
        return newElement;
    }

    private static MetadataElement unwrapElement(Element el) {
        if (el instanceof MetadataElement)
            return (MetadataElement)el;
        if (el instanceof ParameterizedType) {
            return (MetadataElement)((ParameterizedType)el).getDefinition();
        }
        if (el instanceof ParameterizedTypeImpl.Wrapper) {
            return (MetadataElement)((ParameterizedTypeImpl.Wrapper)el).getWrappedObject();
        }
        throw new IllegalArgumentException("Unknown type "+el.getClass());
    }
    
    /**
     * This helper method computes and returns collection of overridden methods
     * @param method for which overridden methods will be computed
     * @return Collection of overridden methods. If there are no overridden methods empty list is returned. 
     */
    public static Collection getOverriddenMethods(Method method) {
        if (method instanceof ArrayCloneMethod) {
            ArrayList result = new ArrayList(1);
            result.add(
                    ((JavaClass)JavaModel.getDefaultExtent().getType()
                    .resolve("java.lang.Object")).getMethod("clone",Collections.EMPTY_LIST, false)); //NOI18N
            return result;
        }
        MethodImpl mImpl=(MethodImpl)unwrapElement(method);
        
        return mImpl.getOverriddenMethods();
    }
    /**
     * This method creates {@link TypeReference} based on parameter {@link Type} type. New instance of 
     * TypeReference is created in default JavaModelPackage returned by JavaModel.getDefaultExtent().
     * @param type {@link Type} used to generate new instance of {@link TypeReference}
     * @return new instance of {@link TypeReference} representing parameter type
     */
    public static TypeReference createTypeReferenceFromType(Type type) {
        JavaModelPackage extent = JavaModel.getDefaultExtent();
        TypeRef typeRef = SemiPersistentElement.typeToTypeRef(type);
        
        return (TypeReference)SemiPersistentElement.typeRefToTypeReference(extent, typeRef, 0);
    }
}
