/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.util.Collection;
import java.util.List;
import javax.jmi.reflect.ConstraintViolationException;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.lib.java.parser.Token;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.parser.*;
import org.openide.util.Utilities;

/**
 * Represents an attribute of an annotation. Consider annotation
 * <code>
 * public @interface RequestForEnhancement {
 *     int      id();
 *     String   synopsis();
 *     String   engineer() default "[unassigned]"
 *     String   date();    default "[unimplemented]"
 * }
 * </code>
 *
 * Such an annotation contains four instances of the Attribute element. They
 * are named id, synopsis, engineer and date. Two elements have not the default
 * value. (Calling of getDefaultValue() method returns null.)
 *
 * @author  Pavel Flaska
 */
public abstract class AttributeImpl extends FeatureImpl implements Attribute {
    
    InitialValue defaultValue;
    String defaultValueText;
    TypeReference typeName;
    
    boolean defaultValueInited;
    protected TypeRef type;
    
    private static final ElementInfo DEFAULT_INFO = new AttributeInfo(null, 0, null, 0, null, null, null);

    private boolean elementsInited;

    /** Creates a new instance of AttributeImpl */
    public AttributeImpl(StorableObject s) {
        super(s);
        defaultValueInited = false;
    }
    
    protected void matchPersistent(ElementInfo info) {
        super.matchPersistent(info);
        AttributeInfo newInfo = (AttributeInfo) info;
        
        if (!isPersisted()) {
            setPersisted(true);
            persist();
            setTypeRef(newInfo.type);
            persistChildren(getPersistentList("annotations", super_getAnnotations()), ((FeatureInfo) info).annotations);
        } else {
            if (!Utilities.compareObjects(newInfo.type, getTypeRef())) {
                setType(resolveType(newInfo.type));
            }
            processMembers(getAnnotations(), newInfo.annotations);
        }
    }
    
    protected void matchElementInfo(ElementInfo info) {
        super.matchElementInfo(info);
        resetASTElements();
    }
    
    /**
     * Returns the default value of the annotation attribute.
     * In the following example, the value is represented by "[unassigned]".
     *
     * <code>
     * String   engineer() default "[unassigned]"
     * </code>
     * 
     * @return  InitialValue default attribute's value
     */
    public InitialValue getDefaultValue() {
        checkUpToDate();
        if (!defaultValueInited) {
            initDefaultValue();
        }
        return defaultValue;
    }
    
    /**
     * Sets the default value of the annotation attribute.
     * {@link #getDefaultValue} for description of annotation attribute.
     *
     * @param  initial value tree representing new default value
     */
    public void setDefaultValue(InitialValue newValue) {
        if (!defaultValueInited) {
            initDefaultValue();
            defaultValueInited = true;
        }
        changeChild(defaultValue, newValue);
        defaultValue = newValue;
        defaultValueText = null;
        objectChanged(CHANGED_INITIAL_VALUE);
    }
    
    /**
     * Returns the default value of the annotation attribute in string
     * representation. In the following example, the value is represented 
     * by "[unassigned]".
     *
     * <code>
     * String   engineer() default "[unassigned]"
     * </code>
     * 
     * @return  InitialValue default attribute's value
     */
    public String getDefaultValueText() {
        checkUpToDate();
        if (isChanged(CHANGED_INITIAL_VALUE)) {
            if (defaultValue != null) {
                throw new ConstraintViolationException(null, null, 
                    "Cannot ask for default value text after the default value was changed."); // NOI18N
            }
            return defaultValueText;
        } else {
            return extractDefaultValueText();
        }
    }
    
    /**
     * Sets the default value of annotation attribute. 
     * See {@link #getDefaultText} for description of annotation attribute.
     *
     * @param newValue New value to be set.
     */
    public void setDefaultValueText(String newValue) {
        if (!elementsInited) {
            initASTElements();
        }
        if (defaultValueInited && defaultValue != null) {
            changeChild(defaultValue, null);
        }
        objectChanged(CHANGED_INITIAL_VALUE);
        defaultValueText = newValue;
        defaultValue = null;
        defaultValueInited = true;
    }
    
    public List getChildren() {
        List result = super.getChildren();
        addIfNotNull(result, getTypeName());
        if (!isChanged(CHANGED_INITIAL_VALUE) || defaultValueText == null) {
            addIfNotNull(result, getDefaultValue());
        }
        return result;
    }
    
    protected List getInitedChildren() {
        List result = super.getInitedChildren();
        if (elementsInited) {
            addIfNotNull(result, getTypeName());
            if (defaultValueInited) {
                if (!isChanged(CHANGED_INITIAL_VALUE) || defaultValueText == null) {
                    addIfNotNull(result, getDefaultValue());
                }
            }
        }
        return result;
    }
    
    /**
     * Returns the value of type. You will get resolved Type model element.
     * Method can return null value on new attributes created by 
     * createAttribute() method on its class proxy.
     *
     * @return value of type or null if new attribute has not set type
     */
    public Type getType() {
        checkUpToDate();
        return resolveType(getTypeRef());
    }

    private void fireTypeNameChange(TypeReference typeReference) {
        Object oldValue = null;
        Object newValue = null;
        if (elementsInited && !disableChanges) {
            oldValue = getTypeName();
            newValue = typeReference;
        }
        fireAttrChange("typeName", oldValue, newValue); // NOI18N
    }
    
    /**
     * Sets the value of type. You have to provide resolved model element.
     * The method guarantee the sync values between type and typeName.
     *
     * @param newValue new value to be set to type
     */
    public void setType(Type newValue) {
        TypeRef tr = typeToTypeRef(newValue);
        TypeReference typeReference = (TypeReference) typeRefToTypeReference(tr, 0);
        fireTypeNameChange(typeReference);
        _setTypeName(typeReference, tr);
    }

    public TypeReference getTypeName() {
        checkUpToDate();
        if (!elementsInited) {
            initASTElements();
        }
        return typeName;
    }

    public void setTypeName(TypeReference typeName) {
        _setTypeName(typeName, typeReferenceToTypeRef(typeName, 0));
    }

    private void _setTypeName(TypeReference typeName, TypeRef typeRef) {
        if (!disableChanges) {
            objectChanged(CHANGED_TYPE);
            changeChild(getTypeName(), typeName);
            this.typeName = typeName;
        }
        setTypeRef(typeRef);
    }

    public Collection getValues() {
        throw new UnsupportedOperationException("getValues()"); // NOI18N
    }
    
    /**
     * Returns the source representation of the attribute element. If the
     * element is not changed, it cuts the representation from the original
     * source text. Otherwise it creates it.
     *
     * @return  source text representation of the instance
     */
    public String getSourceText() {
        String origElem;
        if ((origElem = checkChange()) != null)
            return origElem;
        
        StringBuffer buf = new StringBuffer(27);
        buf.append('\n'); // todo (#pf): use options for leaving empty line before the element
        generateNewJavaDoc(buf);
        buf.append(getIndentation());
        buf.append(((MetadataElement) getTypeName()).getSourceText());
        buf.append(' ').append(getName());
        buf.append(formatElementPart(MetadataElement.PAR_OPEN_BRACKET));
        buf.append(formatElementPart(MetadataElement.PAR_CLOSE_BRACKET));
        String defValText = defaultValue == null ? defaultValueText :
            ((MetadataElement) getDefaultValue()).getSourceText();
        if (defValText != null) {
            buf.append(" default "); // todo (#pf): use formatter! // NOI18N
            buf.append(defValText);
        }
        buf.append(';');
        
        return buf.toString();
    }
    
    /**
     * Returns a list of changes which have been done on element.
     *
     * @param  diff diff elements representing changes (list to add to)
     */
    public void getDiff(List diff) {
        ASTree tree = getASTree();
        ASTProvider parser = getParser();
        ASTree[] children = tree.getSubTrees();

       // javadoc print
        replaceJavaDoc(diff);
        // modifiers
        if (isChanged(CHANGED_MODIFIERS | CHANGED_ANNOTATION)) {
            diffModifiers(diff, children[1], parser);
        } else if (children[0] != null) {
            AttributeInfo astInfo=(AttributeInfo)getElementInfo();
            getCollectionDiff(diff, parser, CHANGED_ANNOTATION, astInfo.annotations, getAnnotations(), parser.getToken(children[0].getLastToken()).getEndOffset(), " "); // NOI18N
        }
        // type
        getChildDiff(diff, getParser(), tree.getSubTrees()[1], (MetadataElement) getTypeName(), CHANGED_TYPE);
        // name
        if (isChanged(CHANGED_NAME)) {
            Token identifier = (Token) tree.getSubTrees()[2];
            int startOffset = identifier.getStartOffset();
            int endOffset = identifier.getEndOffset();
            diff.add(new DiffElement(startOffset, endOffset, getName()));
        }
        if (isChanged(CHANGED_INITIAL_VALUE)) {
            String newText = defaultValue == null ? defaultValueText :
                ((MetadataElement) getDefaultValue()).getSourceText();
            ASTree initValT = extractDefaultValue();
            // there was an default value in original source
            int startOffset, endOffset;
            endOffset = parser.getToken(tree.getLastToken()).getStartOffset();
            if (initValT != null) {
                // changing old default value
                int initValFirstToken = initValT.getFirstToken();
                if (newText != null) {
                    startOffset = parser.getToken(initValFirstToken).getStartOffset();
                } else {
                    startOffset = parser.getToken(initValFirstToken - 2).getEndOffset();
                }
            } else {
                if (newText == null) return;
                // adding new default value
                newText = " default " + newText; // NOI18N
                startOffset = endOffset;
            }
            diff.add(new DiffElement(startOffset, endOffset, newText == null ? "" : newText));
        } else if (isChanged(CHANGED_CHILDREN)) {
            // children changed in initial value.
            MetadataElement defaultValue = (MetadataElement) getDefaultValue();
            if (defaultValue.isChanged()) {
                defaultValue.getDiff(diff);
            }
        }
    }
    
    ////////////////////////////////////////////////////////////////////////////
    protected ElementInfo getDefaultInfo() {
        return DEFAULT_INFO;
    }
    
    protected void setData(List annotations,
                           String javadocText,
                           JavaDoc javadoc,
                           TypeReference typeName,
                           InitialValue defaultValue,
                           String defaultValueText) 
    {
        super.setData(annotations, javadocText, javadoc);
        if (defaultValueText == null) {
            changeChild(null, defaultValue);
            this.defaultValue = defaultValue;
        } else {
            if (defaultValue != null) {
                throw new ConstraintViolationException(null, null, "Cannot set both defaultValue and defaultValueText."); // NOI18N
            }
            this.defaultValueText = defaultValueText;
        }
        defaultValueInited = true;
        changeChild(null, typeName);
        this.typeName = typeName;
        elementsInited = true;
    }

    protected void initChildren() {
        childrenInited = false;
        super.initChildren();
        childrenInited = true;
        if (elementsInited) {
            initASTElements();
        }
        if (defaultValueInited) {
            JMManager.getTransactionMutex().addBFeatureToInitQueue(this);
        }
    }
    
    protected void initASTElements() {
        elementsInited = false;
        if (!childrenInited) {
            initChildren();
        }
        AttributeInfo info = (AttributeInfo) getElementInfo();
        ASTree tree = info.getTypeAST(this);
        typeName = (TypeReference) initOrCreate(typeName, tree);
        //info.setSemanticInfo(this);
        elementsInited = true;
    }
    
    protected void resetChildren() {
        super.resetChildren();
        resetASTElements();
    }

    protected void resetASTElements() {
        if (defaultValue != null) {
            InitialValue temp = defaultValue;
            defaultValue = null;
            changeChild(temp, null);
            temp.refDelete();
        }
        if (typeName != null) {
            TypeReference temp = typeName;
            typeName = null;
            changeChild(temp, null);
            temp.refDelete();
        }
        defaultValueInited = false;
        elementsInited = false;
    }
    
    public Element duplicate(JavaModelPackage targetExtent) {
        InitialValue defVal;
        String defValText;
        
        if (isChanged(CHANGED_INITIAL_VALUE) && defaultValueText != null) {
            defVal = null;
            defValText = defaultValueText;
        } else {
            defVal = (InitialValue) duplicateElement(getDefaultValue(), targetExtent);
            defValText = null;
        }
        
        return targetExtent.getAttribute().createAttribute(
                getName(),
                duplicateList(getAnnotations(), targetExtent),
                getModifiers(),
                null,
                (JavaDoc) duplicateElement(getJavadoc(), targetExtent),
                (TypeReference) duplicateElement(getTypeName(), targetExtent),
                defVal,
                defValText
               );
    }
    
    public void initDefaultValue() {
        defaultValueInited = false;
        if (!elementsInited) {
            initASTElements();
        }
        AttributeInfo defaultValInfo = (AttributeInfo) getElementInfo();
        if (defaultValInfo != null) {
            defaultValInfo.doAttribution(this);
            defaultValue = (InitialValue) initOrCreate(defaultValue, extractDefaultValue());
        }
        defaultValueInited = true;
    }
    
    ////////////////////////////////////////////////////////////////////////////
    // PRIVATE MEMBERS
    ////////////////////////////////////////////////////////////////////////////
    private ASTree extractDefaultValue() {
        ASTree defValTree = getASTree().getSubTrees()[3];
        return defValTree == null ? null : defValTree.getSubTrees()[0];
    }

    private String extractDefaultValueText() {
        ASTProvider parser = getParser();
        if (parser == null)
            return null;
        ASTree attributeDefaultValue = extractDefaultValue();
        if (attributeDefaultValue == null)
            return null;
        return parser.getText(attributeDefaultValue);
    }

    protected void _delete() {
        // --- delete components -------------------------------------------
        if (elementsInited) {
            deleteChild(typeName);
        }
        if (defaultValueInited) {
            deleteChild(defaultValue);
        }
        // --- delete links -----------------------------------------------
        // no links to delete
        // [TODO] should Throws association be notified?
        // --- call super -------------------------------------------------
        super._delete();
    }
    
}
