/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.util.ArrayList;
import java.util.Arrays;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.mdr.handlers.BaseObjectHandler;
import javax.jmi.reflect.RefObject;
import javax.jmi.reflect.TypeMismatchException;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.lang.reflect.Array;

/**
 *
 * @author Martin Matula
 */
public class FeaturesList implements List {
    private ClassDefinition javaClass;

    public FeaturesList(ClassDefinition javaClass) {
        this.javaClass = javaClass;
    }

    private void lock() {
        lock(false);
    }

    private void lock(boolean readWrite) {
        ((BaseObjectHandler) javaClass).repository().beginTrans(readWrite);
    }

    private void unlock() {
        unlock(false);
    }

    private void unlock(boolean fail) {
        ((BaseObjectHandler) javaClass).repository().endTrans(fail);
    }

    public int size() {
        lock();
        try {
            int size = 0;
            Object features[]=javaClass.getContents().toArray();
            for (int i=0;i<features.length;i++) {
                Object temp = features[i];
                if (temp instanceof FieldGroup) {
                    size += ((FieldGroup) temp).getFields().size();
                } else {
                    size++;
                }
            }
            return size;
        } finally {
            unlock();
        }
    }

    public boolean isEmpty() {
        lock();
        try {
            return javaClass.getContents().isEmpty();
        } finally {
            unlock();
        }
    }

    public boolean contains(Object o) {
        lock();
        try {
            if (o instanceof FieldGroup) {
                return false;
            }
            return javaClass.equals(((RefObject) o).refImmediateComposite());
        } finally {
            unlock();
        }
    }

    public Iterator iterator() {
        return listIterator();
    }

    public Object[] toArray() {
        lock();
        try {
            Object features[]=javaClass.getContents().toArray();
            List result=new ArrayList(100);
            boolean hasFieldGroup=false;
            
            for (int i=0;i<features.length;i++) {
                Object feature=features[i];
                
                if (feature instanceof FieldGroup) {
                    FieldGroup group=(FieldGroup)feature;
                    
                    hasFieldGroup=true;
                    result.addAll(Arrays.asList(group.getFields().toArray()));
                } else
                    result.add(feature);
            }
            if (hasFieldGroup)
                return result.toArray();
            return features;
        } finally {
            unlock();
        }
    }

    public Object[] toArray(Object a[]) {
        Object arr[]=toArray();
        int size = arr.length;
        if (a.length < size) {
            a = (Object[]) Array.newInstance(a.getClass().getComponentType(), size);
        }
        System.arraycopy(arr, 0, a, 0, size);
        if (a.length > size)
            a[size] = null;
        return a;
    }

    public boolean add(Object o) {
        if (o instanceof FieldGroup) throw new TypeMismatchException(Feature.class, o, null);
        boolean fail = true;
        lock(true);
        try {
            boolean result = javaClass.getContents().add(o);
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }

    public boolean remove(Object o) {
        if (!(o instanceof Feature)) return false;
        boolean fail = true;
        lock(true);
        try {
            boolean result = false;
            Object parent = ((Feature) o).refImmediateComposite();
            if (javaClass.equals(parent)) {
                result = javaClass.getContents().remove(o);
            } else if (parent instanceof FieldGroup
                    && javaClass.equals(((FieldGroup) parent).refImmediateComposite())) {
                result = ((FieldGroup) parent).getFields().remove(o);
                separate((Field) o, (FieldGroup) parent);
            }

            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }

    private void separate(Field field, FieldGroup fg) {
        field.setModifiers(((FieldGroupImpl)fg).getSourceModifiers());
        field.setTypeName((TypeReference) fg.getTypeName().duplicate());
    }

    public boolean containsAll(Collection c) {
        lock();
        try {
            for (Iterator it = c.iterator(); it.hasNext();) {
                if (!contains(it.next())) {
                    return false;
                }
            }
            return true;
        } finally {
            unlock();
        }
    }

    public boolean addAll(Collection c) {
        boolean fail = true;
        lock(true);
        try {
            boolean result = false;
            for (Iterator it = c.iterator(); it.hasNext();) {
                result |= add(it.next());
            }
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }

    public boolean addAll(int index, Collection c) {
        boolean fail = true;
        lock(true);
        try {
            ListIterator cm = listIterator(index);
            for (Iterator it = c.iterator(); it.hasNext();) {
                cm.add(it.next());
            }
            fail = false;
            return !c.isEmpty();
        } finally {
            unlock(fail);
        }
    }

    public boolean removeAll(Collection c) {
        boolean fail = true;
        lock(true);
        try {
            boolean result = false;
            for (Iterator it = c.iterator(); it.hasNext();) {
                result |= remove(it.next());
            }
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }

    public boolean retainAll(Collection c) {
        boolean fail = true;
        lock(true);
        try {
            boolean result = false;
            for (Iterator it = iterator(); it.hasNext();) {
                if (!c.contains(it.next())) {
                    it.remove();
                    result = true;
                }
            }
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }

    public void clear() {
        boolean fail = true;
        lock(true);
        try {
            javaClass.getContents().clear();
            fail = false;
        } finally {
            unlock(fail);
        }
    }

    public Object get(int index) {
        lock();
        try {
            return listIterator(index).next();
        } finally {
            unlock();
        }
    }

    public Object set(int index, Object element) {
        boolean fail = true;
        lock(true);
        try {
            ListIterator it = listIterator(index);
            Object result = it.next();
            it.set(element);
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }

    public void add(int index, Object element) {
        boolean fail = true;
        lock(true);
        try {
            listIterator(index).add(element);
            fail = false;
        } finally {
            unlock(fail);
        }
    }

    public Object remove(int index) {
        boolean fail = true;
        lock(true);
        try {
            ListIterator it = listIterator(index);
            Object result = it.next();
            it.remove();
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }

    public int indexOf(Object o) {
        lock();
        try {
            for (ListIterator it = listIterator(); it.hasNext();) {
                if (it.next().equals(o)) {
                    return it.previousIndex();
                }
            }
            return -1;
        } finally {
            unlock();
        }
    }

    public int lastIndexOf(Object o) {
        lock();
        try {
            int result = -1;
            for (ListIterator it = listIterator(); it.hasNext();) {
                if (it.next().equals(o)) {
                    result = it.previousIndex();
                }
            }
            return result;
        } finally {
            unlock();
        }
    }

    public ListIterator listIterator() {
        lock();
        try {
            return new FeatureListIterator(javaClass.getContents().listIterator());
        } finally {
            unlock();
        }
    }

    public ListIterator listIterator(int index) {
        lock();
        try {
            ListIterator result = listIterator();
            while (result.nextIndex() < index) {
                result.next();
            }
            return result;
        } finally {
            unlock();
        }
    }

    public boolean equals(Object o) {
        lock();
        try {
            if (o == this) return true;
            if (!(o instanceof List)) return false;

            ListIterator e1 = listIterator();
            ListIterator e2 = ((List) o).listIterator();
            while(e1.hasNext() && e2.hasNext()) {
                Object o1 = e1.next();
                Object o2 = e2.next();
                if (o1 != o2 && !o1.equals(o2))
                    return false;
            }
            return !(e1.hasNext() || e2.hasNext());
        } finally {
            unlock();
        }
    }

    public int hashCode() {
        lock();
        try {
            int hashCode = 1;
            Iterator it = iterator();
            while (it.hasNext()) {
                hashCode = 31 * hashCode + it.next().hashCode();
            }
            return hashCode;
        } finally {
            unlock();
        }
    }

    public String toString() {
        lock();
        try {
            StringBuffer buf = new StringBuffer();
            buf.append("["); // NOI18N

            Iterator i = iterator();
            boolean hasNext = i.hasNext();
            while (hasNext) {
                Object o = i.next();
                buf.append(o == this ? "(this Collection)" : String.valueOf(o)); // NOI18N
                hasNext = i.hasNext();
                if (hasNext)
                    buf.append(", "); // NOI18N
            }

            buf.append("]"); // NOI18N
            return buf.toString();
        } finally {
            unlock();
        }
    }

    // [TODO] implement me
    public List subList(int fromIndex, int toIndex) {
        throw new UnsupportedOperationException();
    }

    private class FeatureListIterator implements ListIterator {
        private final ListIterator contents;
        private ListIterator fieldGroup = null;
        private int index = 0;
        private boolean forward = true;
        private boolean lastForward = true;
        private FieldGroup parentGroup = null;
        private Field currentField = null;

        public FeatureListIterator(ListIterator contents) {
            this.contents = contents;
        }

        public boolean hasNext() {
            lock();
            try {
                if (!forward) {
                    contents.next();
                    forward = true;
                }
                return contents.hasNext() || (fieldGroup != null && fieldGroup.hasNext());
            } finally {
                unlock();
            }
        }

        public Object next() {
            lock();
            try {
                if (fieldGroup != null) {
                    if (fieldGroup.hasNext()) {
                        index++;
                        lastForward = true;
                        return currentField = (Field) fieldGroup.next();
                    } else {
                        fieldGroup = null;
                        parentGroup = null;
                        if (!forward) contents.next();
                    }
                }
                Object next = contents.next();
                if (next instanceof FieldGroup) {
                    ListIterator it = ((FieldGroup) next).getFields().listIterator();
                    ListIterator temp = fieldGroup;
                    FieldGroup tmpGroup = parentGroup;
                    fieldGroup = it;
                    parentGroup = (FieldGroup) next;
                    forward = true;
                    Object result;
                    try {
                        result = currentField = (Field) next();
                    } catch (RuntimeException e) {
                        fieldGroup = temp;
                        parentGroup = tmpGroup;
                        throw e;
                    }
                    index++;
                    lastForward = true;
                    return result;
                }
                index++;
                lastForward = true;
                return next;
            } finally {
                unlock();
            }
        }

        public boolean hasPrevious() {
            lock();
            try {
                if (forward) {
                    contents.previous();
                    forward = false;
                }
                return contents.hasPrevious() || (fieldGroup != null && fieldGroup.hasPrevious());
            } finally {
                unlock();
            }
        }

        public Object previous() {
            lock();
            try {
                if (fieldGroup != null) {
                    if (fieldGroup.hasPrevious()) {
                        index--;
                        lastForward = false;
                        return currentField = (Field) fieldGroup.previous();
                    } else {
                        fieldGroup = null;
                        parentGroup = null;
                        if (forward) contents.previous();
                    }
                }
                Object prev = contents.previous();
                if (prev instanceof FieldGroup) {
                    ListIterator it = ((FieldGroup) prev).getFields().listIterator();
                    while (it.hasNext()) it.next();
                    forward = false;
                    ListIterator temp = fieldGroup;
                    FieldGroup tmpGroup = parentGroup;
                    fieldGroup = it;
                    parentGroup = (FieldGroup) prev;
                    Object result;
                    try {
                        result = currentField = (Field) previous();
                    } catch (RuntimeException e) {
                        fieldGroup = temp;
                        parentGroup = tmpGroup;
                        throw e;
                    }
                    index--;
                    lastForward = false;
                    return result;
                }
                index--;
                lastForward = false;
                return prev;
            } finally {
                unlock();
            }
        }

        public int nextIndex() {
            return index;
        }

        public int previousIndex() {
            return index - 1;
        }

        public void remove() {
            boolean fail = true;
            lock(true);
            try {
                if (fieldGroup != null) {
                    fieldGroup.remove();
                    separate(currentField, parentGroup);
                    if (!(fieldGroup.hasNext() || fieldGroup.hasPrevious())) {
                        contents.remove();
                        fieldGroup = null;
                    }
                } else {
                    contents.remove();
                }
                if (lastForward) {
                    index--;
                }
                fail = false;
            } finally {
                unlock(fail);
            }
        }

        public void set(Object o) {
            boolean fail = true;
            lock(true);
            try {
                remove();
                add(o);
                fail = false;
            } finally {
                unlock(fail);
            }
        }

        public void add(Object o) {
            if (o instanceof FieldGroup) throw new TypeMismatchException(Feature.class, o, null);
            boolean fail = true;
            lock(fail);
            try {
                if (fieldGroup == null || !fieldGroup.hasNext()) {
                    contents.add(o);
                } else if (!fieldGroup.hasPrevious()) {
                    contents.previous();
                    contents.add(o);
                    contents.next();
                } else {
                    FieldGroup fg;
                    if (forward) {
                        fg = (FieldGroup) contents.previous();
                    } else {
                        fg = (FieldGroup) contents.next();
                    }
                    forward = !forward;
                    JavaModelPackage pkg = (JavaModelPackage) javaClass.refImmediatePackage();
                    FieldGroup newGroup = pkg.getFieldGroup().createFieldGroup(null, null, ((FieldGroupImpl)fg).getSourceModifiers(),
                            null, null, null, null);
                    newGroup.setType(fg.getType());
                    while (fieldGroup.hasNext()) {
                        Object field = fieldGroup.next();
                        fieldGroup.remove();
                        newGroup.getFields().add(field);
                    }
                    if (!forward) {
                        contents.next();
                    }
                    contents.add(o);
                    contents.add(newGroup);
                    fieldGroup = newGroup.getFields().listIterator();
                    forward = true;
                    index++;
                }
                fail = false;
            } finally {
                unlock(fail);
            }
        }
    }
}
