/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.lang.reflect.Modifier;
import java.util.*;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.lib.java.parser.Token;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.parser.ElementInfo;
import org.netbeans.modules.javacore.parser.ASTProvider;
import org.openide.util.Utilities;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.parser.MDRParser;

/**
 * Implementation of Import object instance interface.
 *
 * @author  Martin Matula
 * @author  Pavel Flaska
 */
public abstract class ImportImpl extends SemiPersistentElement implements Import {
    private static final ElementInfo DEFAULT_INFO = new ElementInfo(null, ElementInfo.SINGLE_IMPORT_TYPE, null);

    private String name = null;
    private boolean isStatic = false;
    private boolean isOnDemand = false;

    private MultipartId identifier = null;

    /** Creates a new instance of ImportImpl */
    public ImportImpl(StorableObject s) {
        super(s);
    }

    /** Should be overriden by elements that have persistent attributes.
     * They should implement this method to compare values in newly passed AST info with
     * values of the persistent attributes and if the values do not match, they should be
     * updated and proper events should be fired.
     */
    protected void matchPersistent(ElementInfo info) {
        // override to not match name (which is matched by default)
    }

    protected ElementInfo getDefaultInfo() {
        return DEFAULT_INFO;
    }

    /** The method has to make sure that the AST infos of children are also updated.
     */
    protected void matchElementInfo(ElementInfo newInfo) {
        super.matchElementInfo(newInfo);

        ElementInfo refInfo=getElementInfo();

        if (!Utilities.compareObjects(refInfo.name, newInfo.name))
            setName(newInfo.name);
        resetChildren();
    }

    protected void resetChildren() {
        super.resetChildren();
        if (identifier != null) {
            MultipartId temp = identifier;
            identifier = null;
            temp.refDelete();
            childrenInited = false;
        }
    }

    public String getName() {
        if (isChanged(CHANGED_NAME)) {
            return name;
        } else {
            return getElementInfo().name;
        }
    }

    public void setName(String name) {
        objectChanged(CHANGED_NAME);
        this.name = name;
        MultipartId newValue = createIdentifier();
        changeChild(getIdentifier(), newValue);
        this.identifier = newValue;
    }
    
    private MultipartId createIdentifier() {
        if (name == null) return null;
        MultipartIdClass proxy = ((JavaModelPackage) refImmediatePackage()).getMultipartId();
        return proxy.createMultipartId(name, null, null);
    }
    
    void setData(String name, boolean onDemand, boolean isStatic, MultipartId identifier) {
        this.name = name;
        if (identifier == null) {
            this.identifier = createIdentifier();
        } else {
            this.identifier = identifier;
            if (name == null) {
                name = identifier.getName();
            }
        }
        changeChild(null, this.identifier);
        this.isOnDemand = onDemand;
        this.isStatic = isStatic;
        childrenInited = true;
    }

    public boolean isStatic() {
        if (isChanged(CHANGED_IS_STATIC)) {
            return isStatic;
        } else {
            ASTree tree=getASTree();
            
            if (tree==null)
                return false;
            return tree.getSubTrees()[0]!=null;
        }
    }

    public void setStatic(boolean s) {
        objectChanged(CHANGED_IS_STATIC);
        isStatic = s;
    }
    
    public boolean isOnDemand() {
        if (isChanged(CHANGED_IS_ON_DEMAND)) {
            return isOnDemand;
        } else {
            return getElementInfo().infoType == ElementInfo.IMPORT_ON_DEMAND_TYPE;
        }
    }

    public void setOnDemand(boolean s) {
        objectChanged(CHANGED_IS_ON_DEMAND);
        isOnDemand = s;
    }

    protected ASTree getPartTree(ElementPartKind part) {
        if (ElementPartKindEnum.NAME.equals(part)) {
            return getASTree().getSubTrees()[1];
        }
        throw new IllegalArgumentException("Invalid part for this element: " + part); // NOI18N
    }

    String getRawText() {
        StringBuffer buf = new StringBuffer();
        buf.append("import "); // NOI18N
        if (isStatic())
            buf.append("static "); // NOI18N
        MetadataElement id = (MetadataElement)getIdentifier();
        buf.append(id == null ? getName() : id.getSourceText());
        if (isOnDemand()) {
            buf.append(".*"); // NOI18N
        }
        buf.append(';');
        return buf.toString();
    }

    public void getDiff(List diff) {
        ASTProvider parser = getParser();
        ASTree tree = getASTree();
        ASTree[] children = tree.getSubTrees();
        
        if (isChanged(CHANGED_IS_STATIC)) {
            int startOffset, endOffset = parser.getToken(children[1].getFirstToken()).getStartOffset();
            if (isStatic) {
                if (children[0] == null) {
                    startOffset = endOffset;
                    diff.add(new DiffElement(startOffset, endOffset, "static ")); // NOI18N
                }
            }
            else {
                if (children[0] != null) {
                    startOffset = parser.getToken(children[0].getFirstToken()).getStartOffset();
                    diff.add(new DiffElement(startOffset, endOffset, ""));
                }
            }
        }
        getChildDiff(diff, parser, children[1], (MetadataElement) getIdentifier(), CHANGED_NAME);

        if (isChanged(CHANGED_IS_ON_DEMAND)) {
            int startOffset = parser.getToken(children[1].getLastToken()).getEndOffset();
            int endOffset = parser.getToken(tree.getLastToken()).getStartOffset();
            diff.add(new DiffElement(startOffset, endOffset, isOnDemand() ? ".*" : "")); // NOI18N
        }
    }

    public void replaceChild(Element oldElement, Element newElement) {
        if (childrenInited && oldElement.equals(identifier)) {
            setIdentifier((MultipartId) newElement);
            return;
        }
    }

    public MultipartId getIdentifier() {
        checkUpToDate();
        if (!childrenInited) {
            initChildren();
        }
        return identifier;
    }

    public void setIdentifier(MultipartId newValue) {
        objectChanged(CHANGED_NAME);
        changeChild(getIdentifier(), newValue);
        this.identifier = newValue;
        this.name = newValue != null ? ((MultipartIdImpl)newValue).getSourceText() : ""; // NOI18N
    }

    protected void initChildren() {
        childrenInited = false;
        ElementInfo info = getElementInfo();
        ASTree tree = info.getTypeAST(this);
        if (identifier != null) {
            ((TransientElement) identifier).init(tree);
        } else {
            identifier = (MultipartId) createElement(tree, ((JavaModelPackage) refImmediatePackage()).getMultipartId());
        }
        childrenInited = true;
    }

    public List getChildren() {
        List l = new ArrayList(1);
        addIfNotNull(l, getIdentifier());
        return l;
    }
    
    public NamedElement getImportedNamespace() {
        JavaModelPackage model = (JavaModelPackage) refImmediatePackage();
        MultipartId id = getIdentifier();
        if (id == null) return null;
        return id.getElement();
    }
    
    public Collection getReferences() {
        return Collections.EMPTY_LIST;
    }

    private String getSignature(Feature feature) {
        if (feature instanceof Method) {
            StringBuffer sig=new StringBuffer();
            Object[] pars=((Method)feature).getParameters().toArray();
            
            sig.append(feature.getName());
            for(int i=0;i<pars.length;i++) {
                Parameter par=(Parameter)pars[i];
                
                sig.append(par.getType().getName());
                sig.append(';');
            }
            sig.append('.');
            return sig.toString();
        }
        return feature.getName();        
    }
    
    private void getAllStaticWithName(JavaClass jcls,String name,Map fMap, Set visited) {
        if (!visited.add(jcls)) return;
        JavaClass superJavaClass=jcls.getSuperClass();
        Iterator ifaceIt=jcls.getInterfaces().iterator();
        Iterator fIt;
        
        if (superJavaClass!=null)
            getAllStaticWithName(superJavaClass, name, fMap, visited);
        while (ifaceIt.hasNext()) {
            JavaClass ifaceClass=(JavaClass)ifaceIt.next();
            
            if (ifaceClass!=null)
                getAllStaticWithName(ifaceClass, name, fMap, visited);
        }
        Object[] features=jcls.getFeatures().toArray();
        for(int i=0;i<features.length;i++) {
            Feature feature=(Feature)features[i];
            
            if (!Modifier.isStatic(feature.getModifiers()))
                continue;
            if (name!=null && !name.equals(feature.getName()))
                continue;
            if (feature instanceof Field || feature instanceof Method) {
                fMap.put(getSignature(feature), feature);
            }
        }
    }
    
    public Collection/*<org.netbeans.jmi.javamodel.NamedElement>*/ getImportedElements() {
        if (isStatic()) {
            Map imported=new HashMap();
            
            if (isOnDemand()) {
                getAllStaticWithName((JavaClass)getImportedNamespace(),null,imported, new HashSet());
            } else {
                MDRParser parser=getParser();
                ASTree tree=getPartTree(ElementPartKindEnum.NAME);
                ASTree parts[]=tree.getSubTrees();
                Object symbol=parser.getSemanticInfo(parts[0], this);
                
                if (symbol instanceof JavaClass) {
                    String name=(String)((Token)parts[1]).getValue();
                    
                    getAllStaticWithName((JavaClass)symbol, name, imported, new HashSet());
                }
            }
            return imported.values();
        } else if (isOnDemand()) {
            NamedElement el=getImportedNamespace(); 
            Collection classes=new ArrayList();
            
            if (el instanceof JavaPackage) {
                Iterator rIt=((JavaPackage)el).getResources().iterator();
                
                while(rIt.hasNext()) {
                    Iterator topIt=((Resource)rIt.next()).getClassifiers().iterator();
                    
                    while(topIt.hasNext()) {
                        JavaClass jcls=(JavaClass)topIt.next();
                        
                        if (Modifier.isPublic(jcls.getModifiers())) {
                            classes.add(jcls);
                        }
                    }
                }
            } else if ((el instanceof JavaClass) && !(el instanceof UnresolvedClass)) {
                JavaClass jcls=(JavaClass)el;
                ClassIndex index=ClassIndex.getIndex((JavaModelPackage)jcls.refImmediatePackage());
                Iterator innerIt=index.getClassesByFQNPrefix(jcls.getName().concat(".")).iterator(); // NOI18N
                
                while(innerIt.hasNext()) {
                    JavaClass inner=(JavaClass)innerIt.next();
                    
                    if (inner.getDeclaringClass().equals(jcls)) {
                        classes.add(inner);
                    }
                }
            }
            return classes;
        } else {
            return Collections.singletonList(getImportedNamespace());
        }
    }
    
    protected void hardRefParent(boolean enabled) {
        // do not hardref parent - this object is transient
    }

    protected void parentChanged() {
        // do nothing on parentChanged - this object is transient
    }

    public Element duplicate(JavaModelPackage targetExtent) {
        return targetExtent.getImport().createImport(
                getName(),
                (MultipartId) duplicateElement(getIdentifier(), targetExtent),
                isStatic(),
                isOnDemand()
               );
    }
    
    protected void _delete() {
        // --- delete components -------------------------------------------
        if (childrenInited) {
            deleteChild(identifier);
        }
        // --- delete links -----------------------------------------------
        // no links to delete
        // [TODO] should Throws association be notified?
        // --- call super -------------------------------------------------
        super._delete();
    }
}
