/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.util.*;
import javax.jmi.model.MofClass;
import javax.jmi.model.NameNotFoundException;
import javax.jmi.reflect.ConstraintViolationException;
import javax.jmi.reflect.InvalidObjectException;
import javax.jmi.reflect.RefFeatured;
import javax.jmi.reflect.RefObject;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.ElementPartKind;
import org.netbeans.jmi.javamodel.JavaPackage;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.mdr.handlers.InstanceHandler;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;

/**
 *
 * @author  Martin Matula
 */
public abstract class JavaPackageImpl extends InstanceHandler implements JavaPackage {
    String name = null;
    private final boolean isProxy;
    private final SubPackagesCollection subpackages;
    private final ResourcesCollection resources;
    private final HashSet representedPackages;
    private boolean simpleDelete = false;
    
    private boolean isValid = true;

    /** Creates a new instance of PackageImpl */
    public JavaPackageImpl(StorableObject s) {
        super(s);
        if (s instanceof DeferredObject) {
            subpackages = new SubPackagesCollection();
            resources = new ResourcesCollection();
            representedPackages = new HashSet();
            isProxy = true;
        } else {
            subpackages = null;
            resources = null;
            representedPackages = null;
            isProxy = false;
        }
    }

    public String getName() {
        if (isProxy) {
            return name;
        } else {
            return super_getName();
        }
    }

    protected abstract String super_getName();

    public void setName(String name) {
        RefObject nameAttr = null;
        try {
            nameAttr = ((MofClass) refMetaObject()).lookupElementExtended("name"); // NOI18N
        } catch (NameNotFoundException e) {
            // ignore
        }
        throw new ConstraintViolationException(this, nameAttr, "Name attribute is readonly."); // NOI18N
    }

    public Collection getSubPackages() {
        return isProxy ? subpackages : super_getSubPackages();
    }

    public Collection getResources() {
        return isProxy ? resources : super_getResources();
    }
    
    public Resource getResource() {
        return null;
    }

    protected abstract Collection super_getSubPackages();
    protected abstract Collection super_getResources();

    void addSubPackage(JavaPackageImpl pkg) {
//        System.out.println("setting parent of: " + pkg.getName() + " to: " + getName());
        if (isProxy) {
            try {
                ((StorableObject) pkg._getDelegate()).setComposite(_getDelegate(), null, null);
            } catch (StorageException e) {
                ErrorManager.getDefault().notify(e);
            }
            subpackages.addSubPackage(pkg);
        } else {
            super_getSubPackages().add(pkg);
        }
    }

    void removeRealPackage(JavaPackageImpl real) {
        if (!isProxy) throw new IllegalStateException();
        representedPackages.remove(real);
        if (representedPackages.isEmpty()) {
            refDelete();
        }
    }

    void addRealPackage(JavaPackageImpl real) {
        if (!isProxy) throw new IllegalStateException();
        if (!representedPackages.add(real)) {
            throw new IllegalStateException();
        }
    }
    
    public boolean isValid() {
        return isValid;
    }

    protected void _delete() {
        if (!isProxy) {
            if (!simpleDelete) ((JavaPackageClassImpl) refClass()).removeRealPackage(this);
            super._delete();
        } else {
            JavaPackageClassImpl.removePackage(this);
            for (Iterator it = representedPackages.iterator(); it.hasNext();) {
                JavaPackageImpl pck = (JavaPackageImpl) it.next();
                pck.delete();
            }
            representedPackages.clear();
            JavaPackageImpl parent = (JavaPackageImpl) super._immediateComposite();
            if (parent != null) {
                parent.subpackages.removeSubPackage(this);
            }
        }
        isValid = false;
    }
    
    private void delete() {
        simpleDelete = true;
        try {
            refDelete();
        } finally {
            simpleDelete = false;
        }
    }

    protected RefFeatured _immediateComposite() {
        JavaPackage composite = (JavaPackage) super._immediateComposite();
        return composite == null ? null : ((JavaPackageClassImpl) refClass()).resolvePackage(composite.getName(), true);
    }

    protected RefFeatured _outermostComposite() {
        JavaPackage composite = (JavaPackage) super._outermostComposite();
        return ((JavaPackageClassImpl) refClass()).resolvePackage(composite.getName(), true);
    }

    public Collection getReferences() {
        Resource[] res = findReferencedResources();
        UsageFinder finder = new UsageFinder(this);
        return finder.getUsers(res);
    }

    private Resource[] findReferencedResources() {
        StringTokenizer tokenizer = new StringTokenizer(getName(), "."); //NOI18N
        String part = tokenizer.nextToken();
        List result = new ArrayList(Arrays.asList((Object[]) ClassIndex.findResourcesForIdentifier(part)));

        while (tokenizer.hasMoreTokens()) {
            part = tokenizer.nextToken();
            result.retainAll(Arrays.asList((Object[]) ClassIndex.findResourcesForIdentifier(part)));
        }
        return (Resource[]) result.toArray(new Resource[result.size()]);
    }

    boolean shouldInclude() {
        return !(getResources().isEmpty() && getSubPackages().isEmpty());
    }

    private static class SubPackagesCollection extends AbstractCollection {
        private HashSet subpackages = new HashSet();

        private SubPackagesCollection() {
        }
        
        private void _lock(boolean write) {
            JavaMetamodel.getDefaultRepository().beginTrans(write);
        }
        
        private void _unlock() {
            JavaMetamodel.getDefaultRepository().endTrans(false);
        }

        public int size() {
            _lock(false);
            try {
                int size = 0;
                for (Iterator it = iterator(); it.hasNext(); size++, it.next());
                return size;
            } finally {
                _unlock();
            }
        }

        public boolean isEmpty() {
            _lock(false);
            try {
                return !iterator().hasNext();
            } finally {
                _unlock();
            }
        }

        public boolean contains(Object o) {
            _lock(false);
            try {
                return subpackages.contains(o) && ((JavaPackageImpl) o).shouldInclude();
            } finally {
                _unlock();
            }
        }

        public Iterator iterator() {
            _lock(false);
            try {
                return new Iterator() {
                    private JavaPackage next = null;
                    private Iterator iterator = subpackages.iterator();

                    public boolean hasNext() {
                        _lock(false);
                        try {
                            while (iterator.hasNext() && next == null) {
                                JavaPackageImpl temp = (JavaPackageImpl) iterator.next();
                                if (temp.subpackages.isReallyEmpty() && temp.resources.isReallyEmpty()) {
                                    iterator.remove();
                                    temp.refDelete();
                                } else {
                                    if (temp.shouldInclude()) {
                                        next = temp;
                                    }
                                }
                            }
                            return next != null;
                        } finally {
                            _unlock();
                        }
                    }

                    public Object next() {
                        _lock(false);
                        try {
                            if (!hasNext()) {
                                throw new NoSuchElementException();
                            }
                            JavaPackage result = next;
                            next = null;
                            return result;
                        } finally {
                            _unlock();
                        }
                    }

                    public void remove() {
                        throw new UnsupportedOperationException();
                    }
                };
            } finally {
                _unlock();
            }
        }

        void addSubPackage(JavaPackage pkg) {
            subpackages.add(pkg);
        }
        
        void removeSubPackage(JavaPackage pkg) {
            subpackages.remove(pkg);
        }

        boolean isReallyEmpty() {
            return subpackages.isEmpty();
        }
    }
    
    private void handleInvalid(InvalidObjectException e) {
        System.err.println("Invalid represented package for: " + getName() + " toString: " + toString());
        System.err.println("...number of represented packages: " + representedPackages.size());
        System.err.println("...resolvePackage(getName()) returns " + ((JavaPackageClassImpl) refClass()).resolvePackage(getName(), true));
        throw e;
    }

    private class ResourcesCollection extends AbstractCollection {
        private ResourcesCollection() {
        }

        public int size() {
            _lock(false);
            try {
                int size = 0;
                for (Iterator it = iterator(); it.hasNext(); size++, it.next());
                return size;
            } finally {
                _unlock();
            }
        }

        public boolean isEmpty() {
            _lock(false);
            try {
                return !iterator().hasNext();
            } finally {
                _unlock();
            }
        }

        public boolean contains(Object o) {
            _lock(false);
            try {
                Set roots = createRootsSet();
                Iterator it = representedPackages.iterator();
                boolean contains = false;
                while (it.hasNext() && !contains) {
                    JavaPackage pkg = (JavaPackage) it.next();
                    try {
                        if (roots.contains(pkg.refImmediatePackage())) {
                            contains = pkg.getResources().contains(o);
                        }
                    } catch (InvalidObjectException e) {
                        handleInvalid(e);
                    }
                }
                return contains;
            } finally {
                _unlock();
            }
        }

        private Set createRootsSet() {
            ClassPath cp = JavaMetamodel.getManager().getClassPath();
            FileObject[] fo = cp.getRoots();
            HashSet roots = new HashSet(fo.length);
            for (int i = 0; i < fo.length; i++) {
                roots.add(org.netbeans.modules.javacore.internalapi.JavaMetamodel.getManager().getJavaExtent(fo[i]));
            }
            return roots;
        }

        public Iterator iterator() {
            _lock(false);
            try {
                return new Iterator() {
                    private Iterator packages = representedPackages.iterator();
                    private Set roots = createRootsSet();
                    private Iterator resources = null;

                    public boolean hasNext() {
                        _lock(false);
                        try {
                            while ((resources == null || !resources.hasNext()) && packages.hasNext()) {
                                JavaPackage pkg = (JavaPackage) packages.next();
                                try {
                                    if (roots.contains(pkg.refImmediatePackage())) {
                                        resources = pkg.getResources().iterator();
                                    }
                                } catch (InvalidObjectException e) {
                                    handleInvalid(e);
                                }
                            }
                            return resources != null && resources.hasNext();
                        } finally {
                            _unlock();
                        }
                    }

                    public Object next() {
                        _lock(false);
                        try {
                            if (!hasNext()) {
                                throw new NoSuchElementException();
                            }
                            return resources.next();
                        } finally {
                            _unlock();
                        }
                    }

                    public void remove() {
                        throw new UnsupportedOperationException();
                    }
                };
            } finally {
                _unlock();
            }
        }

        boolean isReallyEmpty() {
            for (Iterator it = representedPackages.iterator(); it.hasNext();) {
                try {
                    if (!((JavaPackage) it.next()).getResources().isEmpty()) {
                        return false;
                    }
                } catch (InvalidObjectException e) {
                    handleInvalid(e);
                }
            }
            return true;
        }
    }
    
    public List getChildren() {
        ArrayList result = new ArrayList(getSubPackages());
        result.addAll(getResources());
        return result;
    }
    
    public void replaceChild(Element oldElement, Element newElement) {
        throw new UnsupportedOperationException();
    }
    
    public int getStartOffset() {
        throw new UnsupportedOperationException();
    }
    
    public int getEndOffset() {
        throw new UnsupportedOperationException();
    }
    
    public int getPartStartOffset(ElementPartKind part) {
        throw new UnsupportedOperationException();
    }

    public int getPartEndOffset(ElementPartKind part) {
        throw new UnsupportedOperationException();
    }
    
    public Element duplicate() {
        throw new UnsupportedOperationException("The operation is intentionally unsupported at this element."); // NOI18N 
    }
}
