/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.util.*;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.openide.ErrorManager;

import org.openide.util.WeakSet;

public class LightAttrList implements List {
    private List innerList;
    private int modCount = 0;
    
    private final MetadataElement parent;
    private final int changeMask;
    private List hardReferences;
    
    // weak set holding iterators - for debugging purposes to track down ConcurrentModificationException bug
    // (set to new WeakSet() if you want to debug)
    private final WeakSet iterators = new WeakSet();
    private RuntimeException exception = null;
    
    
    private void lock(boolean writeLock) {
        JavaMetamodel.getDefaultRepository().beginTrans(writeLock);
    }
    
    private void unlock() {
        unlock(false);
    }
    
    private void unlock(boolean rollback) {
        JavaMetamodel.getDefaultRepository().endTrans(rollback);
    }

    public LightAttrList(List innerList, MetadataElement parent, int changeMask) {
        this(innerList, parent, changeMask, false);
    }

    public LightAttrList(List innerList, MetadataElement parent, int changeMask, boolean hardReference) {
        this.parent = parent;
        this.changeMask = changeMask;
        setInnerList(innerList);
        if (hardReference) {
            updateHardReferences(true);
        }
    }

    private void updateHardReferences(boolean parentChanged) {
        Object[] elements=innerList.toArray();
        hardReferences = new ArrayList(elements.length);
        for (int i=0;i<elements.length;i++) {
            Object obj = elements[i];
            if (parentChanged) parentChanged(obj);
            hardReferences.add(obj);
        }
    }

    public void setInnerList(List innerList) {
        setInnerList(innerList, true);
    }
    
    public void setInnerList(List innerList, boolean parentChanged) {
        lock(false);
        try {
            modCount++;
            this.innerList = innerList;
            if (hardReferences != null) {
                updateHardReferences(parentChanged);
            }
            
            // --- for debugging purposes ------
            if (iterators != null) {
                Object[] iters = iterators.toArray();
                for (int i = 0; i < iters.length; i++) {
                    if (((LightAttrListIterator) iters[i]).modCount >= 0) {
                        exception = new RuntimeException();
                        break;
                    }
                }
            }
            // ---------------------------------
        } finally {
            unlock();
        }
    }
    
    public List getInnerList() {
        lock(false);
        try {
            return innerList;
        } finally {
            unlock();
        }
    }

    protected void objectChanged() {
        if (parent != null)
            parent.objectChanged(changeMask);
    }
    
    protected void parentChanged(Object obj) {
        if (obj instanceof MetadataElement) {
            ((MetadataElement) obj).parentChanged();
        }
    }
    
    // ..........................................................................
    
    public boolean remove(Object obj) {
        boolean fail = true;
        lock(true);
        try {
            objectChanged();
            boolean result = innerList.remove(obj);
            if (hardReferences != null && result) hardReferences.remove(obj);
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }
    
    public Object set(int param, Object obj) {
        boolean fail = true;
        lock(true);
        try {
            objectChanged();
            Object result = innerList.set(param, obj);
            if (hardReferences != null) hardReferences.set(param, obj);
            parentChanged(obj);
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }
    
    public Object remove(int param) {        
        boolean fail = true;
        lock(true);
        try {
            objectChanged();
            Object result = innerList.remove(param);
            if (hardReferences != null) hardReferences.remove(param);
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }
    
    public void add(int param, Object obj) {        
        boolean fail = true;
        lock(true);
        try {
            objectChanged();
            innerList.add(param, obj);
            if (hardReferences != null) hardReferences.add(param, obj);
            parentChanged(obj);
            fail = false;
        } finally {
            unlock(fail);
        }
    }
    
    public boolean add(Object obj) {        
        boolean fail = true;
        lock(true);
        try {
            objectChanged();
            boolean result = innerList.add(obj);
            if (hardReferences != null && result) hardReferences.add(obj);
            if (result) {
                parentChanged(obj);
            }
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }
    
    public ListIterator listIterator(int param) {
        lock(false);
        try {
            ListIterator result = new LightAttrListIterator(param);
            
            // -------------------------
            if (iterators != null) iterators.add(result);  // for debugging purposes
            // -------------------------
            
            return result;
        } finally {
            unlock();
        }
    }
    
    public Iterator iterator() {
        return listIterator();
    }
    
    public ListIterator listIterator() {
        return listIterator(0);
    }
    
    public List subList(int param, int param1) {
        lock(false);
        try {
            return new LightAttrList(innerList.subList(param,  param1), parent, changeMask);
        } finally {
            unlock();
        }
    }
    
    public boolean contains(Object obj) {
        return innerList.contains(obj);
    }
    
    public boolean containsAll(Collection collection) {
        return innerList.containsAll(collection);
    }
    
    public boolean addAll(Collection c) {
        boolean fail = true;
        lock(true);
        try {
            objectChanged();
            boolean result = innerList.addAll(c);
            if (hardReferences != null) hardReferences.addAll(c);
            if (result) {
                for (Iterator it = c.iterator(); it.hasNext();) {
                    parentChanged(it.next());
                }
            }
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }
    
    public void clear() {
        boolean fail = true;
        lock(true);
        try {
            objectChanged();
            innerList.clear();
            if (hardReferences != null) hardReferences.clear();
            fail = false;
        } finally {
            unlock(fail);
        }
    }
    
    public boolean isEmpty() {
        return innerList.isEmpty();
    }
    
    public boolean removeAll(Collection c) {
        boolean fail = true;
        lock(true);
        try {
            objectChanged();
            boolean result = innerList.removeAll(c);
            if (hardReferences != null) hardReferences.removeAll(c);
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }
    
    public boolean retainAll(Collection c) {
        boolean fail = true;
        lock(true);
        try {
            objectChanged();
            boolean result = innerList.retainAll(c);
            if (hardReferences != null) hardReferences.retainAll(c);
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }
    
    public int size() {
        return innerList.size();
    }
    
    public Object[] toArray() {
        return innerList.toArray();
    }
    
    public Object[] toArray(Object[] a) {
        return innerList.toArray(a);
    }
    
    public boolean addAll(int index, Collection c) {
        boolean fail = true;
        lock(true);
        try {
            objectChanged();
            boolean result = innerList.addAll(index, c);
            if (hardReferences != null) hardReferences.addAll(index, c);
            if (result) {
                for (Iterator it = c.iterator(); it.hasNext();) {
                    parentChanged(it.next());
                }
            }
            fail = false;
            return result;
        } finally {
            unlock(fail);
        }
    }
    
    public Object get(int index) {
        return innerList.get(index);
    }
    
    public int indexOf(Object o) {
        return innerList.indexOf(o);
    }
    
    public int lastIndexOf(Object o) {
        return innerList.lastIndexOf(o);
    }
    
    // ListIterator .................................................
    class LightAttrListIterator implements ListIterator {
        private ListIterator innerIterator;
        private ListIterator hrIterator;
        private int modCount = -1;
        private final int index;
        
        LightAttrListIterator(int param) {
            index = param;
        }
        
        private int getParentModCount() {
            return LightAttrList.this.modCount;
        }
        
        private void testModCount() throws ConcurrentModificationException {
            if (this.modCount == -1) {
                if (hardReferences != null) {
                    if (hardReferences.size() != innerList.size()) {
                        JMManager.getLog().notify(ErrorManager.INFORMATIONAL, new Exception("Inconsistency in LightAttrList: innerList.size() == " + innerList.size() + " while hardReferences.size() == " + hardReferences.size() + ". Fixing...")); // NOI18N
                        updateHardReferences(true);
                    }
                    hrIterator = hardReferences.listIterator(index);
                }
                innerIterator = innerList.listIterator(index);
                this.modCount = getParentModCount();
            } else
            if (getParentModCount() != modCount) {
                JMManager.getLog().notify(ErrorManager.INFORMATIONAL, exception);
                throw new ConcurrentModificationException();
            }
        }
        
        public void remove() {
            boolean fail = true;
            lock(true);
            try {
                testModCount();
                objectChanged();
                innerIterator.remove();
                if (hrIterator != null) hrIterator.remove();
                fail = false;
            } finally {
                unlock(fail);
            }
        }
        
        public void add(Object obj) {            
            boolean fail = true;
            lock(true);
            try {
                testModCount();
                objectChanged();
                innerIterator.add(obj);
                if (hrIterator != null) hrIterator.add(obj);
                parentChanged(obj);
                fail = false;
            } finally {
                unlock(fail);
            }
        }
        
        public void set(Object obj) {
            boolean fail = true;
            lock(true);
            try {
                testModCount();
                objectChanged();
                innerIterator.set(obj);
                if (hrIterator != null) hrIterator.set(obj);
                parentChanged(obj);
                fail = false;
            } finally {
                unlock(fail);
            }
        }
        
        public boolean hasNext() {
            lock(false);
            try {
                testModCount();
                return innerIterator.hasNext();
            } finally {
                unlock();
            }
        }
        
        public boolean hasPrevious() {
            lock(false);
            try {
                testModCount();
                return innerIterator.hasPrevious();
            } finally {
                unlock();
            }
        }
        
        public Object next() {
            lock(false);
            try {
                testModCount();
                if (hrIterator != null) hrIterator.next();
                return innerIterator.next();
            } finally {
                unlock();
            }
        }

        public int nextIndex() {
            lock(false);
            try {
                testModCount();
                return innerIterator.nextIndex();
            } finally {
                unlock();
            }
        }
        
        public Object previous() {
            lock(false);
            try {
                testModCount();
                if (hrIterator != null) hrIterator.previous();
                return innerIterator.previous();
            } finally {
                unlock();
            }
        }
        
        public int previousIndex() {
            lock(false);
            try {
                testModCount();
                return innerIterator.previousIndex();
            } finally {
                unlock();
            }
        }
    }
}
