/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.lib.java.parser.ASTreeTypes;
import org.netbeans.lib.java.parser.ParserTokens;
import org.netbeans.lib.java.parser.Token;
import org.netbeans.mdr.handlers.AttrListWrapper;
import org.netbeans.mdr.handlers.BaseObjectHandler;
import org.netbeans.mdr.handlers.InstanceHandler;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.storagemodel.StorableFeatured;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.parser.*;
import org.netbeans.modules.javacore.JMManager;
import org.openide.ErrorManager;
import org.openide.text.PositionBounds;
import javax.jmi.reflect.InvalidObjectException;
import javax.jmi.reflect.RefClass;
import javax.jmi.reflect.RefFeatured;
import javax.jmi.reflect.RefObject;
import java.lang.ref.Reference;
import java.util.*;

/** Defines method for children initialization and changes management.
 *
 * @author  Martin Matula
 */
public abstract class MetadataElement extends InstanceHandler implements Element {
    public static final int CHANGED_CHILDREN = 1;
    public static final int CHANGED_NAME = 1 << 1;
    public static final int CHANGED_MODIFIERS = 1 << 2;
    public static final int CHANGED_IS_STATIC = CHANGED_MODIFIERS;
    public static final int CHANGED_IS_FINAL = CHANGED_MODIFIERS;
    public static final int CHANGED_IS_ON_DEMAND = 1 << 3;
    public static final int CHANGED_THROWS = CHANGED_IS_ON_DEMAND;
    public static final int CHANGED_JAVADOC = 1 << 4;
    public static final int CHANGED_BODY = 1 << 5;
    public static final int CHANGED_TYPE = 1 << 6;
    public static final int CHANGED_EXTENDS = CHANGED_THROWS;
    public static final int CHANGED_IMPLEMENTS = 1 << 7;
    public static final int CHANGED_INITIAL_VALUE = CHANGED_BODY;
    public static final int CHANGED_PARAMETERS = 1 << 8;
    public static final int CHANGED_PACKAGE_NAME = 1 << 9;
    public static final int CHANGED_FEATURES = CHANGED_PARAMETERS;
    public static final int CHANGED_CLASSIFIERS = CHANGED_FEATURES;
    public static final int CHANGED_IMPORTS = CHANGED_BODY;
    public static final int CHANGED_STATEMENTS = CHANGED_FEATURES;
    public static final int CHANGED_LABEL = CHANGED_NAME;
    public static final int CHANGED_THEN_PART = CHANGED_BODY;
    public static final int CHANGED_ELSE_PART = 1 << 10;
    public static final int CHANGED_EXPRESSION = 1 << 11;
    public static final int CHANGED_STEPS = CHANGED_FEATURES;
    public static final int CHANGED_INIT = CHANGED_THROWS;
    public static final int CHANGED_CASES = CHANGED_STEPS;
    public static final int CHANGED_VALUE = CHANGED_EXPRESSION;
    public static final int CHANGED_LOCK = CHANGED_EXPRESSION;
    public static final int CHANGED_DETAIL = CHANGED_BODY;
    public static final int CHANGED_FINALIZER = CHANGED_EXPRESSION;
    public static final int CHANGED_CATCHES = CHANGED_FEATURES;
    public static final int CHANGED_PARAMETER = CHANGED_EXPRESSION;
    public static final int CHANGED_RIGHT_SIDE = CHANGED_ELSE_PART;
    public static final int CHANGED_LEFT_SIDE = CHANGED_THEN_PART;
    public static final int CHANGED_CONDITION = CHANGED_EXPRESSION;
    public static final int CHANGED_TRUE_PART = CHANGED_THEN_PART;
    public static final int CHANGED_FALSE_PART = CHANGED_ELSE_PART;
    public static final int CHANGED_ARRAY = CHANGED_LEFT_SIDE;
    public static final int CHANGED_INDEX = CHANGED_RIGHT_SIDE;
    public static final int CHANGED_PARENT_CLASS = CHANGED_EXPRESSION;
    public static final int CHANGED_ENCLOSING_CLASS = CHANGED_PARENT_CLASS;
    public static final int CHANGED_CLASS_DEFINITION = CHANGED_ELSE_PART;
    public static final int CHANGED_HAS_SUPER = CHANGED_MODIFIERS;
    public static final int CHANGED_ELEMENT_VALUES = CHANGED_FEATURES;
    public static final int CHANGED_DIMENSIONS = CHANGED_FEATURES;
    public static final int CHANGED_VARIABLES = CHANGED_FEATURES;
    public static final int CHANGED_TEXT = CHANGED_NAME;
    public static final int CHANGED_INITIALIZER = CHANGED_EXPRESSION;
    public static final int CHANGED_PARENT = CHANGED_EXPRESSION;
    public static final int CHANGED_CLASS_NAME = 1 << 12;
    public static final int CHANGED_DIM_COUNT = 1 << 13;
    public static final int CHANGED_ITERABLE = CHANGED_EXPRESSION;
    public static final int CHANGED_CONSTANTS = CHANGED_DIM_COUNT;
    public static final int CHANGED_IS_VARARG = CHANGED_BODY;
    public static final int CHANGED_TYPE_ARGUMENTS = 1 << 14;
    public static final int CHANGED_ANNOTATION = 1 << 15;
    public static final int CHANGED_BOUNDS = CHANGED_MODIFIERS;
    public static final int CHANGED_TYPE_PARAMETERS = 1 << 16;
    public static final int CHANGED_BOUND = CHANGED_BOUNDS;
    public static final int CHANGED_IS_LOWER = CHANGED_EXPRESSION;
    public static final int CHANGED_ARGUMENTS = CHANGED_STATEMENTS;
    public static final int CHANGED_ENUM_CONST_BODY = CHANGED_CLASS_DEFINITION;
    public static final int CHANGED_OPERATOR = 1 << 17;
    public static final int CHANGED_EXTERNAL = 1 << 18;
    
    // value of this field will be lazily initialized from editor setting
    protected static String INDENTATION = "    "; // NOI18N
    /**
     * Contains information about spaces per tab character. In addition to,
     * it holds also information if the INDENTATION field is initialized from
     * editor setting. -1 means it is not initialized yet.
     */
    protected static int spacesPerTab = -1;
    
    private int changes = 0;
    private boolean isValid = true;
    protected boolean childrenInited = false;
    public boolean disableChanges = false;

    public MetadataElement(StorableObject o) {
        super(o);
    }

    protected final void initCheck() {
        if (!isValid) {
            throwInvalidObject();
        }
        if (!isInitialized()) {
            if (this instanceof ResourceImpl) {
                ((ResourceImpl) this).initResource();
            } else {
                MetadataElement parent = (MetadataElement) refImmediateComposite();
                // the element must have a parent
                // with an exception of Resource
                if (parent != null)
                    parent.initChildren();
            }
        }
    }
    
    private void throwInvalidObject() {
        throw new InvalidObjectException(null, "Object was deleted."); // NOI18N
    }

    protected void _delete() {
        // --- delete from composite --------------------------------------
        MetadataElement parent = (MetadataElement) refImmediateComposite();
        if (parent != null) {
            parent.removeChild(this);
        }
        // --- delete children --------------------------------------------
//        for (Iterator it = getInitedChildren().iterator(); it.hasNext();) {
//            ((RefObject) it.next()).refDelete();
//        }
        // --- delete this instance ---------------------------------------
//        ((ExclusiveMutex) _getMdrStorage().getRepositoryMutex()).removeNew(this);
        
        isValid = false;
    }

    /** Part ot the rollback mechanism.
     *
     * In the case of any rollback, all classes should be rebuilt from ASTrees.
     * The layout for method is the next one:
     * protected void reset() {
     *   .. do your own job...
     *   super.reset()
     * }
     */
    protected void rollback() {
        if (isNew()) {
            isValid = false;
        } else {
            isValid = true;
            resetChange();
            boolean changes = disableChanges;
            disableChanges = true;
            try {
                //resetValues();
                for (Iterator it = getInitedChildren().iterator(); it.hasNext();) {
                    ((MetadataElement) it.next()).rollback();
                }
                resetChildren();
            } finally {
                disableChanges = changes;
            }
        }
    }

    protected final void commit() {
        //if (isNew()) {
            // the element is new => remove it from the registry
            //((ExclusiveMutex) _getMdrStorage().getRepositoryMutex()).removeNew(this);
        //}
        for (Iterator it = getInitedChildren().iterator(); it.hasNext();) {
            ((MetadataElement) it.next()).commit();
        }
        uninitialize();
    }

    protected abstract boolean isInitialized();
    protected abstract void uninitialize();

//    protected void resetValues() {
//    }

    public boolean isValid() {
        return isValid;
    }

    protected abstract Object getInternalForm();

    protected void initChildren() {
    }

    protected void resetChildren() {
    }

    protected List getInitedChildren() {
        if (childrenInited) {
            return getChildren();
        } else {
            return Collections.EMPTY_LIST;
        }
    }

    public List getChildren() {
        return Collections.EMPTY_LIST;
    }

    static void addIfNotNull(Collection collection, Object value) {
        if (value != null) {
            collection.add(value);
        }
    }

    protected void parentChanged() {
    }

    protected int getParentChangedMask(int mask) {
        return CHANGED_CHILDREN;
    }

    protected void objectChanged(int mask) {
        if (!isValid()) throwInvalidObject(); // NOI18N
        if (disableChanges) return;
        if (!isChanged()) {
            if (!childrenInited) {
                // todo: this is only necessary for elements that have persistent children
                initChildren();
            }
            Object parent = refImmediateComposite();
            if (parent != null && parent instanceof MetadataElement) {
                ((MetadataElement) parent).objectChanged(getParentChangedMask(mask));
            }
        }
        changes |= mask;
    }

    /**
     * Returns source code representation of this element in case of the element
     * does not contain any change and is not newly added. In such cases
     * (when element is new or changed) it returns null.
     *
     * @return  null in case of newly added or changed element, otherwise
     *          source-code representation.
     */
    protected final String checkChange() {
        // there is not any change, return original element in case of element
        // is not new element
        if (!isNew()) {
            if (!isChanged()) {
                ASTProvider p = getParser();
                MetadataElement indentElement = this;
                int b = IndentUtil.getElementStart(this);
                int e = IndentUtil.getElementEnd(this);
                if (this instanceof StatementBlock) {
                    // !!!! TODO: fix the following - this seems like it specially handles
                    // moving statement block in the finally clause
                    Object composite = refImmediateComposite();
                    if (composite instanceof TryStatement) {
                        TryStatement tryStmt = (TryStatement) composite;
                        if (this.equals(tryStmt.getFinalizer())) {
                            if (tryStmt.getCatches().isEmpty()) {
                                b = IndentUtil.getElementEnd(tryStmt.getBody());
                            } else {
                                CatchImpl catchBlock = null;
                                Iterator it = tryStmt.getCatches().iterator();
                                while (it.hasNext()) catchBlock = (CatchImpl) it.next();
                                b = IndentUtil.getElementEnd(catchBlock);
                                indentElement = (MetadataElement) refImmediateComposite();
                            }
                        }
                    }
                    // !!!!
                } else if (this instanceof Statement) {
                    Token startToken = p.getToken(getASTree().getFirstToken());
                    b = startToken.getStartOffset();
                    Token[] pad = startToken.getPadding();
                    if (pad.length > 0) {
                        startToken = pad[0];
                        for (int i = 0; i < pad.length; i++) {
                            if (pad[i].getType() == ParserTokens.EOL) {
                                b = pad[i].getStartOffset();
                                break;
                            }
                        }
                    }
                    int lastTokenIndex = getASTree().getLastToken();
                    Token endToken = p.getToken(lastTokenIndex+1);
                    if (endToken == null) {
                        endToken = p.getToken(lastTokenIndex);
                        e = endToken.getEndOffset();
                    } else {
                        pad = endToken.getPadding();
                        if (pad.length > 0) {
                            endToken = pad[0];
                            for (int i = 0; i < pad.length; i++) {
                                if (pad[i].getType() == ParserTokens.EOL) {
                                    e = pad[i].getStartOffset();
                                    break;
                                }
                            }
                        }
                    }
                } else if (this instanceof Catch) {
                    int tokenBefore = getASTree().getFirstToken();
                    b = IndentUtil.getElementEnd(((TryStatement) refImmediateComposite()).getBody());
                }
                String text = p.getSourceText().substring(b, e);
                int tokenId = getASTree().getFirstToken();
                // for the statement block, use closing curly as a base element
                // indentation will be computed from spaces before this curly
                if (this instanceof StatementBlock) {
                    tokenId = getASTree().getLastToken();
                }
                return IndentUtil.indentExistingElement(this, text, tokenId);
            } else {
                if (!childrenInited) {
                    initChildren();
                }
            }
        }
        return null; 
    }

    protected final void resetChange() {
        if (disableChanges) return;
        changes = 0;
    }

    protected final void setChanged() {
        if (disableChanges) return;
        changes = ~0;
    }

    protected final void resetChange(int mask) {
        if (disableChanges) return;
        changes &= ~mask;
    }

    protected final boolean isChanged() {
        return changes != 0;
    }

    protected final boolean isChanged(int mask) {
        return (changes & mask) != 0;
    }

    /** Method used to create children list from the AST
     * @param currentList Current list of children that the method should update with new ASTs or rebuild (in not applicable, pass null)
     * @param attrName Name of MOF attribute that holds the list returned by this method.
     * @param tree AS tree that represents a child or children to be put into the returned list.
     * @param groupNodeType In case there can be one child or a group node of several children
     *      represented by tree parameter, this attribute contains type of node that corresponds
     *      to the group node (to determine whether there is a single child or a group of children under a group node.
     * @param changeMask Bit that should be set in change flags of owner of the returned list if the list changes.
     * @param rebuild Indicates whether this method should completely rebuild the existing list (if passed in the first
     *      parameter) or whether it is expected that the list contains the same element, only their AS trees should be updated (in case of false value).
     * @return List of children.
     */
    protected final LightAttrList createChildrenList(LightAttrList currentList, String attrName, ASTree tree, int groupNodeType, int changeMask, boolean rebuild) {
        return createChildrenList(currentList, attrName, getChildNodes(tree, groupNodeType), changeMask, rebuild);
    }

    /** Method used to create children list from the AST */
    protected final LightAttrList createChildrenList(LightAttrList currentList, String attrName, ASTree[] trees, int changeMask, boolean rebuild) {
        return createChildrenList(currentList, attrName, trees, 0, changeMask, rebuild);
    }

    protected final LightAttrList createChildrenList(LightAttrList currentList, String attrName, ASTree[] trees, int index, int changeMask, boolean rebuild) {
        DeferredAttrList deferredList;
        try {
            if (currentList != null && trees != null) {
                deferredList = (DeferredAttrList) ((AttrListWrapper) currentList.getInnerList()).getInnerList();
                if (rebuild) {
                    for (Iterator it = deferredList.iterator(); it.hasNext();) {
                        MetadataElement temp = (MetadataElement) it.next();
                        it.remove();
                        temp.refDelete();
                    }
                } else {
                    int i = index;
                    for (ListIterator it = deferredList.listIterator(); it.hasNext(); i++) {
                        MetadataElement im = (MetadataElement) it.next();
                        if (isNew()) {
                            im.setNew();
                        } else {
                            if (i >= trees.length) {
                                it.remove();
                            } else {
                                Element newElement = initOrCreate(im, trees[i]);
                                if (newElement != im) {
                                    it.set(newElement);
                                }
                            }
                        }
                    }
                    for (; i < trees.length; i++) {
                        deferredList.add(createElement(trees[i]));
                    }
                    deferredList = null;
                }
            } else {
                StorableFeatured storable = (StorableFeatured) _getDelegate();
                deferredList = new DeferredAttrList(storable, storable.getClassProxy().getAttrDesc(attrName), new ArrayList());
                currentList = createWrapper(attrName, deferredList, changeMask);
            }
            if (deferredList != null && trees != null) {
                for (int i = index; i < trees.length; i++) {
                    MetadataElement s = createElement(trees[i]);
                    if (s != null) {
                        deferredList.add(s);
                    }
                }
            }
        } catch (StorageException e) {
            throw (GeneralException) ErrorManager.getDefault().annotate(new GeneralException(e.getMessage()), e);
        }
        return currentList;
    }

    /** Method used to create children list from a list provided into the statement constructor */
    protected final LightAttrList createChildrenList(String attrName, List elements, int changeMask) {
        DeferredAttrList deferredList;
        try {
            ArrayList innerList;
            if (elements != null) {
                innerList = new ArrayList(elements.size());
                for (Iterator it = elements.iterator(); it.hasNext();) {
                    MetadataElement element = (MetadataElement) it.next();
                    ((StorableObject) element._getDelegate()).setComposite(_getDelegate(), null, null);
                    innerList.add(element);
                }
            } else {
                innerList = new ArrayList();
            }
            StorableFeatured storable = (StorableFeatured) _getDelegate();
            deferredList = new DeferredAttrList(storable, storable.getClassProxy().getAttrDesc(attrName), innerList);
        } catch (StorageException e) {
            throw (GeneralException) ErrorManager.getDefault().annotate(new GeneralException(e.getMessage()), e);
        }
        return createWrapper(attrName, deferredList, changeMask);
    }

    protected final LightAttrList createWrapper(String attrName, List innerList, int changeMask) {
        AttrListWrapper list = new AttrListWrapper(this, innerList);
        list.setAttrName(attrName);
        return new LightAttrList(list, this, changeMask);
    }

    protected final void deleteChild(Element child) {
        if (child != null) {
            changeChild(child, null);
            child.refDelete();
        }
    }

    protected static void deleteChildren(Collection children) {
        for (Iterator it = children.iterator(); it.hasNext();) {
            RefObject element = (RefObject) it.next();
            it.remove();
            element.refDelete();
        }
    }

    protected final void changeChild(Element oldChild, Element newChild) {
        try {
            if (oldChild != null) {
                ((StorableObject) ((BaseObjectHandler) oldChild)._getDelegate()).clearComposite();
            }
            if (newChild != null) {
                ((StorableObject) ((BaseObjectHandler) newChild)._getDelegate()).setComposite(_getDelegate(), null, null);
            }
        } catch (StorageException e) {
            throw (GeneralException) ErrorManager.getDefault().annotate(new GeneralException(e.getMessage()), e);
        }
    }

    protected final TransientElement createElement(ASTree tree, RefClass proxy) {
        if (tree == null || proxy == null) return null;
        Object o = ((MEFactory) proxy).create(this);
        TransientElement result = (TransientElement) o;
        result.init(tree);
        return result;
    }

    public abstract MDRParser getParser();

    public ASTree getASTree() {
        return getASTree(getInternalForm());
    }

    public PositionBounds getPosition(boolean inclDoc) {
        testResourceChange();
        PositionBounds result = null;
        ASTree tree = getASTree();
        if (tree != null) {
            result = ((MDRParser) tree.getASTContext()).createBounds(tree, tree, inclDoc);
        }
        return result;
    }

    public final PositionBounds getPartPosition(ElementPartKind part, int position) {
        testResourceChange();
        return getParser().createBounds(getPartStartTree(part), getPartEndTree(part), false);
    }

    public Resource getResource() {
        RefObject parent = this;

        while (!(parent == null || parent instanceof Resource)) {
            parent = (RefObject) parent.refImmediateComposite();
        }
        return (Resource) parent;
    }

    protected MetadataElement createElement(ASTree tree) {
        if (tree == null) return null;
        switch (tree.getType()) {
            case ASTreeTypes.ENUM_DECLARATION:                
            case ASTreeTypes.ANNOTATION_TYPE_DECLARATION:
            case ASTreeTypes.INTERFACE_DECLARATION:
            case ASTreeTypes.CLASS_DECLARATION: {
                JavaClassImpl jcls=(JavaClassImpl)getParser().getSemanticInfo(tree, this);

                return jcls;
            }
            default:
                return createElement(tree, getElementProxy(tree));
        }
    }

    private RefClass getElementProxy(ASTree tree) {
        JavaModelPackage pkg = (JavaModelPackage) refImmediatePackage();
        int type = tree.getType();
        switch (type) {
            case ASTreeTypes.EXPLICIT_CONSTRUCTOR_INVOCATION:
                return pkg.getConstructorInvocation();
            case ASTreeTypes.ARRAY_INITIALIZER:
                return pkg.getArrayInitialization();
            case ASTreeTypes.ARRAY_CREATION_EXPRESSION:
                return pkg.getNewArrayExpression();
            case ASTreeTypes.LOCAL_VARIABLE_DECLARATION:
                return pkg.getLocalVarDeclaration();
            case ASTreeTypes.VARIABLE_DECLARATOR:
                return pkg.getLocalVariable();
            case ASTreeTypes.LABELED_STATEMENT:
                return pkg.getLabeledStatement();
            case ASTreeTypes.IF_STATEMENT:
                return pkg.getIfStatement();
            case ASTreeTypes.WHILE_STATEMENT:
                return pkg.getWhileStatement();
            case ASTreeTypes.FOR_STATEMENT:
                return pkg.getForStatement();
            case ASTreeTypes.FOR_EACH_STATEMENT:
                return pkg.getForEachStatement();
            case ASTreeTypes.BLOCK_STATEMENTS:
                return pkg.getStatementBlock();
            case ASTreeTypes.EXPRESSION_STATEMENT:
                return pkg.getExpressionStatement();
            case ASTreeTypes.EMPTY_STATEMENT:
                return pkg.getEmptyStatement();
            case ASTreeTypes.ASSIGNMENT:
                return pkg.getAssignment();
            case ASTreeTypes.INFIX_EXPRESSION:
                return pkg.getInfixExpression();
            case ASTreeTypes.PREFIX_EXPRESSION:
                return pkg.getPrefixExpression();
            case ASTreeTypes.POSTFIX_EXPRESSION:
                return pkg.getPostfixExpression();
            case ASTreeTypes.METHOD_INVOCATION:
                return pkg.getMethodInvocation();
            case ASTreeTypes.CLASS_INSTANCE_CREATION_EXPRESSION:
                return pkg.getNewClassExpression();
            case ASTreeTypes.COMPLEX_EXPRESSION:
                return pkg.getComplexExpression();
            case ASTreeTypes.CONDITIONAL_EXPRESSION:
                return pkg.getConditionalExpression();
            case ASTreeTypes.CAST_EXPRESSION:
                return pkg.getTypeCast();
            case ASTreeTypes.SWITCH_STATEMENT:
                return pkg.getSwitchStatement();
            case ASTreeTypes.SWITCH_LABEL:
            case ASTreeTypes.SWITCH_BLOCK_STATEMENT_GROUP:
                return pkg.getCase();
            case ASTreeTypes.DO_STATEMENT:
                return pkg.getDoStatement();
            case ASTreeTypes.BREAK_STATEMENT:
                return pkg.getBreakStatement();
            case ASTreeTypes.CONTINUE_STATEMENT:
                return pkg.getContinueStatement();
            case ASTreeTypes.RETURN_STATEMENT:
                return pkg.getReturnStatement();
            case ASTreeTypes.SYNCHRONIZE_STATEMENT:
                return pkg.getSynchronizedStatement();
            case ASTreeTypes.THROW_STATEMENT:
                return pkg.getThrowStatement();
            case ASTreeTypes.TRY_STATEMENT:
                return pkg.getTryStatement();
            case ASTreeTypes.CATCH_CLAUSE:
                return pkg.getCatch();
            case ASTreeTypes.ASSERT_STATEMENT:
                return pkg.getAssertStatement();
            case ASTreeTypes.REFERENCE_TYPE:
                return pkg.getArrayReference();
            case ASTreeTypes.MULTI_PART_ID:
            case ParserTokens.IDENTIFIER:
                if (getParser().isVariableAccess(tree)) {
                    return pkg.getVariableAccess();
                }
            case ASTreeTypes.PRIMITIVE_TYPE:
                return pkg.getMultipartId();
            case ASTreeTypes.FIELD_DECLARATION:
            case ASTreeTypes.METHOD_DECLARATION:
            case ASTreeTypes.CLASS_DECLARATION:
            case ASTreeTypes.INTERFACE_DECLARATION:
            case ASTreeTypes.INSTANCE_INITIALIZER:
            case ASTreeTypes.STATIC_INITIALIZER:
            case ASTreeTypes.CONSTRUCTOR_DECLARATION:
            case ASTreeTypes.CLASS_BODY_DECLARATIONS:
                //return pkg.getClassDefinition();
                // temporary -- ignore local classes
                return null;
            case ASTreeTypes.PRIMARY_CLASS:
                return pkg.getClassExpression();
            case ASTreeTypes.PRIMARY_THIS:
                return pkg.getThisExpression();
            case ASTreeTypes.FIELD_ACCESS:
                return pkg.getVariableAccess();
            case ASTreeTypes.ARRAY_ACCESS:
                return pkg.getArrayAccess();
            case ParserTokens.INT_LIT:
                if (((Token) tree).getValue() instanceof Long) {
                    return pkg.getLongLiteral();
                } else {
                    return pkg.getIntLiteral();
                }
            case ParserTokens.FLOAT_LIT:
                if (((Token) tree).getValue() instanceof Float) {
                    return pkg.getFloatLiteral();
                } else {
                    return pkg.getDoubleLiteral();
                }
            case ParserTokens.BOOL_LIT:
                return pkg.getBooleanLiteral();
            case ParserTokens.CHAR_LIT:
                return pkg.getCharLiteral();
            case ParserTokens.STRING_LIT:
                return pkg.getStringLiteral();
            case ParserTokens.NULL_LIT:
                return pkg.getNullLiteral();
            case ASTreeTypes.SUPER_:
                return pkg.getMultipartId();
            case ASTreeTypes.WILDCARD:
                return pkg.getWildCard();
            case ASTreeTypes.ARGUMENT_LIST:
                return pkg.getArgumentList();
            case ASTreeTypes.ELEMENT_VALUE_PAIR:
                return pkg.getAttributeValue();
            case ASTreeTypes.ERRONEOUS:
                return null;
            default:
                throw new RuntimeException("Unexpected type of ASTree: " + type); // NOI18N
        }
    }

    /**
     * Used for obtaining source-code representation of the element. This
     * method is called by parent when the element is newly added or we need
     * to regenerate whole element.
     *
     * @return  source-code representation of element
     */
    public String getSourceText() {
        String origElem;
        if ((origElem = checkChange()) != null)
            return origElem;
        else {
            if (isNew()) {
                return IndentUtil.indentNewElement(this);
            } else {
                // for all statements (except statement block) todo
                if (this instanceof Statement && !(this instanceof StatementBlock) &&
                                                 !(this instanceof JavaClass) &&
                                                 !(this instanceof LocalVarDeclaration && refImmediateComposite() instanceof ForStatement))
                {
                    StringBuffer buf = new StringBuffer(256);
                    IndentUtil.reformatHeadGarbage(this, buf);
                    buf.append(getRawText());
                    
                    // do not print tail for cycles as it was printed by
                    // their body
                    if (!(this instanceof WhileStatement ||
                         this instanceof ForStatement ||
                         this instanceof ForEachStatement ||
                         this instanceof IfStatement)
                       )
                    {
                        IndentUtil.printTailGarbage(this, buf);
                    }
                    return buf.toString();
                } else {
                    return getRawText();
                }
            }
        }
    }

    abstract String getRawText();
    
    /**
     * Used for obtaining changes and put them to the list.
     *
     * @param  diffList  list, where the changes are put by method. Contains
     *                   DiffElement instances.
     */
    public void getDiff(List diffList) {
    }

    protected abstract void setNew();
    protected abstract boolean isNew();

    public final void replaceNode(List diff, ASTProvider parser, ASTree node, String text, int start, String prefix) {
        int startOffset;
        int endOffset;
        if ((text == null && prefix != null) || node == null) {
            startOffset = start;
        } else {
            startOffset = getStartOffset(parser, node, false);
        }
        if (node == null) {
            endOffset = startOffset;
            if (text != null && prefix != null) {
                text = prefix + text;
            }
        } else {
            endOffset = parser.getToken(node.getLastToken()).getEndOffset();
        }
        if (text == null) {
            text = "";
        }
        diff.add(new DiffElement(startOffset, endOffset, text));
    }

    protected final void getChildDiff(List diff, ASTProvider parser, ASTree node, MetadataElement child, int changeMask) {
        getChildDiff(diff, parser, node, child, changeMask, 0, null);
    }

    protected final void getChildDiff(List diff, ASTProvider parser, ASTree node, MetadataElement child, int changeMask, int startOffset, String prefix) {
        if (isChanged(changeMask)) {
            replaceNode(diff, parser, node, child == null ? null : child.getSourceText(), startOffset, prefix);
        } else if (child != null && child.isChanged()) {
            child.getDiff(diff);
        }
    }

    public final void getElementsDiff(List diff, Collection newElements) {
        Object[] elements=newElements.toArray();
        for (int i=0;i<elements.length;i++) {
            MetadataElement me = (MetadataElement)elements[i];
            if (me.isChanged())
                me.getDiff(diff);
        }
    }
    public final void getCollectionDiff(List diff, ASTProvider parser, int changeMask, Object[] oldElements, Collection newElements, int endOffset, String separator) {
        getCollectionDiff(diff, parser, changeMask, oldElements, newElements, endOffset, separator, true);
    }

    public final void getCollectionDiff(List diff, ASTProvider parser, int changeMask, Object[] oldElements, Collection newElements, int endOffset, String separator, boolean includePadds) {
        if (isChanged(changeMask)) {
            getCollectionDiff(diff, parser, oldElements, newElements, endOffset, separator, null, 0, includePadds);
        } else if (isChanged(CHANGED_CHILDREN)) {
            getElementsDiff(diff, newElements);
        }
    }

    public final void getCollectionDiff(List diff, ASTProvider parser, int changeMask, ASTree oldParentNode, Collection newElements, int startOffset, String separator, String prefixIfNew) {
        getCollectionDiff(diff, parser, changeMask, oldParentNode, oldParentNode == null ? 0 : oldParentNode.getType(), newElements, startOffset, separator, prefixIfNew);
    }

    public final void getCollectionDiff(List diff, ASTProvider parser, int changeMask, ASTree oldParentNode, int groupNodeType, Collection newElements, int startOffset, String separator, String prefixIfNew) {
        if (isChanged(changeMask)) {
            ASTree[] oldASTs;
            int endOffset;
            if (oldParentNode == null) {
                endOffset = startOffset;
                oldASTs = new ASTree[0];
            } else {
                if (oldParentNode.getType() == groupNodeType) {
                    oldASTs = oldParentNode.getSubTrees();
                } else {
                    oldASTs = new ASTree[] {oldParentNode};
                }
                if (oldASTs.length == 0) {
                    endOffset = startOffset;
                } else {
                    endOffset = parser.getToken(oldASTs[oldASTs.length - 1].getLastToken()).getEndOffset();
                }
            }
            getCollectionDiff(diff, parser, oldASTs, newElements, endOffset, separator, prefixIfNew, startOffset, true);
        } else if (isChanged(CHANGED_CHILDREN)) {
            getElementsDiff(diff, newElements);
        }
    }

    private void getCollectionDiff(List diff, ASTProvider parser, Object[] oldElements, Collection newElements, int endOffset, String separator, String prefixIfNew, int startOffset, boolean includePadds) {
        //MethodInfo astInfo = (MethodInfo) getElementInfo();
        if (newElements.isEmpty()) {
            if (oldElements.length > 0 && prefixIfNew != null) {
                diff.add(new DiffElement(startOffset, endOffset, ""));
                return;
            }
        }
        Object[] newElementArr=newElements.toArray();
        CollectionMatcher matcher = new CollectionMatcher(oldElements, newElementArr);
        boolean isFirst = true;
        StringBuffer diffText = null;
        int positions[] = matcher.getPositions();
        Set deleted = matcher.getDeleted();
        int lastOffset = getOffset(0, parser, oldElements, endOffset, includePadds, false);
        ASTree lastElement = null;
        int newPos = 0;
        int oldPos = 0;
        for (;newPos<newElementArr.length; newPos++) {
            boolean shouldDelete = false;
            int newLastOffset = 0;
            while (oldPos < oldElements.length && deleted.remove(oldElements[oldPos])) {
                if (!shouldDelete) {
                    shouldDelete = true;
                    lastOffset = getOffset(oldPos, parser, oldElements, endOffset, oldPos == 0 ? includePadds : true, true);
                }
                oldPos++;
                newLastOffset = getOffset(oldPos, parser, oldElements, endOffset, true, false);
                // if we are going to delete all elements from a given element to the last old element,
                // we need to adjust the start offset of the deletion, since it should no longer
                // be the start offset of the first element to be deleted, but rather the start offset
                // of the token immediately following the last element that will not be deleted
                // (e.g. to eliminate tokens like colons in the list of parameters)
                if (lastElement != null && oldPos == oldElements.length && 
                   (separator.trim().length() != 0)) {
                    Token took = parser.getToken(lastElement.getLastToken() + 1);
                    lastOffset = took.getStartOffset();
                    Token[] pads = took.getPadding();
                    if (pads != null && pads.length > 0) {
                        for (int i = pads.length-1; i >= 0; i--) {
                            if (pads[i].getType() == ParserTokens.EOL) {
                                lastOffset = pads[i].getStartOffset();
                                break;
                            }
                        }
                    }
                }
            }
            if (shouldDelete) {
                diff.add(new DiffElement(lastOffset, newLastOffset, ""));
                lastOffset = newLastOffset;
            }
            MetadataElement e = (MetadataElement)newElementArr[newPos];
            if (positions[newPos] != oldPos) {
                if (diffText == null) {
                    if (prefixIfNew == null || oldElements.length > 0) {
                        if (oldPos >= oldElements.length && !isFirst) {
                            diffText = new StringBuffer(separator);
                        } else {
                            diffText = new StringBuffer();
                        }
                    } else {
                        diffText = new StringBuffer(prefixIfNew);
                    }
                } else {
                    diffText.append(separator);
                }
                diffText.append(e.getSourceText());
                if (positions[newPos] != -1) {
                    deleted.add(oldElements[positions[newPos]]);
                }
            } else {
                if (diffText != null) {
                    diffText.append(separator);
                    diff.add(new DiffElement(lastOffset, lastOffset, diffText.toString()));
                    diffText = null;
                }
                if (e.isChanged()) {
                    e.getDiff(diff);
                }
                lastElement = getLastElement(oldPos, oldElements);
                oldPos++;
                lastOffset = getOffset(oldPos, parser, oldElements, endOffset, true, false);
                isFirst = false;
            }
        }
        if (oldPos < oldElements.length) {
            int newLastOffset = getOffset(oldElements.length, parser, oldElements, endOffset, true, false);
            if (lastElement != null) {
                lastOffset = getStartOffset2(parser, parser.getToken(lastElement.getLastToken() + 1));
            }
            diff.add(new DiffElement(lastOffset, newLastOffset, diffText == null ? "" : diffText.toString()));
        } else if (diffText != null) {
            diff.add(new DiffElement(lastOffset, lastOffset, diffText.toString()));
        }
    }
    
    private int getImportsOffset(int index, ASTProvider parser, Object[] oldElements, int endOffset, boolean includePadds, boolean excludeComments) {
        if (oldElements.length <= index) {
            return endOffset;
        } else {
            ASTree tree = getChildASTree(oldElements[index]);
            if (index != 0) {
                return excludeComments ? getStartOffset2(parser, tree) : getStartOffset(parser, tree, includePadds);
            } else {
                Token startToken = parser.getToken(tree.getFirstToken());
                Token[] pad = startToken.getPadding();
                if (pad.length > 0) {
                    startToken = pad[0];
                    for (int i = 0; i < pad.length; i++) {
                        if (pad[i].getType() == ParserTokens.EOL) {
                            String value = parser.getText(pad[i]);
                            return pad[i].getEndOffset();
                        }
                    }
                }
                return startToken.getStartOffset();
            }
        }
    }
    
    void getImportsDiff(List diff, ASTProvider parser, Object[] oldElements, Collection newElements, int endOffset, int endOffset2, String separator, boolean includePadds, int changeMask) {
        if (!isChanged(changeMask) && isChanged(CHANGED_CHILDREN)) {
            getElementsDiff(diff, newElements);
            return;
        }
        Object[] newElementArr=newElements.toArray();
        CollectionMatcher matcher = new CollectionMatcher(oldElements, newElementArr);
        boolean isFirst = true;
        StringBuffer diffText = null;
        int positions[] = matcher.getPositions();
        Set deleted = matcher.getDeleted();
        int lastOffset = getOffset(0, parser, oldElements, endOffset, includePadds, false);
        ASTree lastElement = null;
        int newPos = 0;
        int oldPos = 0;
        for (;newPos<newElementArr.length; newPos++) {
            boolean shouldDelete = false;
            int newLastOffset = 0;
            while (oldPos < oldElements.length && deleted.remove(oldElements[oldPos])) {
                if (!shouldDelete) {
                    shouldDelete = true;
                    lastOffset = getImportsOffset(oldPos, parser, oldElements, endOffset, oldPos == 0 ? includePadds : true, true);
                }
                oldPos++;
                newLastOffset = getImportsOffset(oldPos, parser, oldElements, endOffset, true, false);
                // if we are going to delete all elements from a given element to the last old element,
                // we need to adjust the start offset of the deletion, since it should no longer
                // be the start offset of the first element to be deleted, but rather the start offset
                // of the token immediately following the last element that will not be deleted
                // (e.g. to eliminate tokens like colons in the list of parameters)
                if (lastElement != null && oldPos == oldElements.length) {
                    Token took = parser.getToken(lastElement.getLastToken() + 1);
                    lastOffset = took.getStartOffset();
                    Token[] pads = took.getPadding();
                    if (pads != null && pads.length > 0) {
                        for (int i = pads.length-1; i >= 0; i--) {
                            if (pads[i].getType() == ParserTokens.EOL) {
                                lastOffset = pads[i].getStartOffset();
                                break;
                            }
                        }
                    }
                }
            }
            if (shouldDelete) {
//                String s = oldPos == oldElements.length-1 ? "\n" : "";
                diff.add(new DiffElement(lastOffset, newLastOffset, ""));
                lastOffset = newLastOffset;
            }
            MetadataElement e = (MetadataElement)newElementArr[newPos];
            if (positions[newPos] != oldPos) {
                if (diffText == null) {
                    if ((oldElements.length > 0) && (oldPos >= oldElements.length && !isFirst)) {
                        diffText = new StringBuffer(separator);
                    } else {
                        diffText = new StringBuffer();
                    }
                } else {
                    diffText.append(separator);
                }
                diffText.append(e.getSourceText());
                if (positions[newPos] != -1) {
                    deleted.add(oldElements[positions[newPos]]);
                }
            } else {
                if (diffText != null) {
                    diffText.append(separator);
                    diff.add(new DiffElement(lastOffset, lastOffset, diffText.toString()));
                    diffText = null;
                }
                if (e.isChanged()) {
                    e.getDiff(diff);
                }
                lastElement = getLastElement(oldPos, oldElements);
                oldPos++;
                lastOffset = getImportsOffset(oldPos, parser, oldElements, endOffset, true, false);
                isFirst = false;
            }
        }
        if (oldPos < oldElements.length) {
            int newLastOffset = endOffset2 == -1 ? getImportsOffset(oldElements.length, parser, oldElements, endOffset, true, false) : endOffset2;
            if (lastElement != null) {
                lastOffset = getStartOffset2(parser, parser.getToken(lastElement.getLastToken() + 1));
            }
            diff.add(new DiffElement(lastOffset, newLastOffset, diffText == null ? "" : diffText.toString()));
        } else if (diffText != null) {
            diff.add(new DiffElement(lastOffset, lastOffset, diffText.toString()));
        }
    }

    private int getOffset(int index, ASTProvider parser, Object[] oldElements, int endOffset, boolean includePadds, boolean excludeComments) {
        if (oldElements.length <= index) {
            return endOffset;
        } else {
            ASTree tree = getChildASTree(oldElements[index]);
            return excludeComments ? getStartOffset2(parser, tree) : getStartOffset(parser, tree, includePadds);
        }
    }

    protected int getStartOffset(ASTProvider parser, ASTree tree, boolean includePaddings) {
        Token startToken = parser.getToken(tree.getFirstToken());
        if (includePaddings) {
            Token[] pad = startToken.getPadding();
            if (pad.length > 0) {
                startToken = pad[0];
                for (int i = 0; i < pad.length; i++) {
                    if (pad[i].getType() == ParserTokens.EOL) {
                        String value = parser.getText(pad[i]);
                        return pad[i].getStartOffset() + value.indexOf('\n') + 1;
                    }
                }
            }
        }
        return startToken.getStartOffset();
    }
    
    protected int getStartOffset2(ASTProvider parser, ASTree tree) {
        return getStartOffset(parser, tree, true);
    }
    
    public void getStatementsDiff(List diff, ASTProvider parser, Object[] oldElements, Collection newElements, int endOffset, String separator, String prefixIfNew, int startOffset, boolean includePadds) {
        //MethodInfo astInfo = (MethodInfo) getElementInfo();
        if (newElements.isEmpty()) {
            if (oldElements.length > 0 && prefixIfNew != null) {
                diff.add(new DiffElement(startOffset, endOffset, ""));
                return;
            }
        }
        Object[] newElementArr=newElements.toArray();
        CollectionMatcher matcher = new CollectionMatcher(oldElements, newElementArr);
        boolean isFirst = true;
        StringBuffer diffText = null;
        int positions[] = matcher.getPositions();
        Set deleted = matcher.getDeleted();
        int lastOffset = getOffset(0, parser, oldElements, endOffset, includePadds, false);
        if (oldElements.length > 0) {
            ASTree tree = getChildASTree(oldElements[0]);
            Token[] pad = getParser().getToken(tree.getFirstToken()).getPadding();
            if (pad.length > 0) {
                for (int i = 0; i < pad.length; i++) {
                    if (pad[i].getType() == ParserTokens.EOL) {
                        lastOffset = pad[i].getStartOffset();
                        break;
                    }
                }
            }
        }
        ASTree lastElement = null;
        int newPos = 0;
        int oldPos = 0;
        for (;newPos<newElementArr.length; newPos++) {
            boolean shouldDelete = false;
            int newLastOffset = 0;
            while (oldPos < oldElements.length && deleted.remove(oldElements[oldPos])) {
                if (!shouldDelete) {
                    shouldDelete = true;
                    if (oldPos != 0) {
                        lastOffset = getOffset(oldPos, parser, oldElements, endOffset, true, true);
                    }
                }
                oldPos++;
                newLastOffset = getOffset(oldPos, parser, oldElements, endOffset, true, false);
                // if we are going to delete all elements from a given element to the last old element,
                // we need to adjust the start offset of the deletion, since it should no longer
                // be the start offset of the first element to be deleted, but rather the start offset
                // of the token immediately following the last element that will not be deleted
                // (e.g. to eliminate tokens like colons in the list of parameters)
                if (lastElement != null && oldPos == oldElements.length && 
                   (separator.trim().length() != 0)) {
                    Token took = parser.getToken(lastElement.getLastToken() + 1);
                    lastOffset = took.getStartOffset();
                    Token[] pads = took.getPadding();
                    if (pads != null && pads.length > 0) {
                        for (int i = pads.length-1; i >= 0; i--) {
                            if (pads[i].getType() == ParserTokens.EOL) {
                                lastOffset = pads[i].getStartOffset();
                                break;
                            }
                        }
                    }
                }
            }
            if (shouldDelete) {
                diff.add(new DiffElement(lastOffset, newLastOffset, ""));
                lastOffset = newLastOffset;
            }
            MetadataElement e = (MetadataElement)newElementArr[newPos];
            if (positions[newPos] != oldPos) {
                if (diffText == null) {
                    if (prefixIfNew == null || oldElements.length > 0) {
                        if (oldPos >= oldElements.length && !isFirst) {
                            diffText = new StringBuffer(separator);
                        } else {
                            diffText = new StringBuffer();
                        }
                    } else {
                        diffText = new StringBuffer(prefixIfNew);
                    }
                } else {
                    diffText.append(separator);
                }
                diffText.append(e.getSourceText());
                if (positions[newPos] != -1) {
                    deleted.add(oldElements[positions[newPos]]);
                }
            } else {
                if (diffText != null) {
                    diffText.append(separator);
                    diff.add(new DiffElement(lastOffset, lastOffset, diffText.toString()));
                    diffText = null;
                }
                if (e.isChanged()) {
                    e.getDiff(diff);
                }
                lastElement = getLastElement(oldPos, oldElements);
                oldPos++;
                lastOffset = getOffset(oldPos, parser, oldElements, endOffset, true, false);
                isFirst = false;
            }
        }
        if (oldPos < oldElements.length) {
            int newLastOffset = getOffset(oldElements.length, parser, oldElements, endOffset, true, false);
            if (lastElement != null) {
                lastOffset = getStartOffset2(parser, parser.getToken(lastElement.getLastToken() + 1));
            }
            diff.add(new DiffElement(lastOffset, newLastOffset, diffText == null ? "" : diffText.toString()));
        } else if (diffText != null) {
            diff.add(new DiffElement(lastOffset, lastOffset, diffText.toString()));
        }
    }

    void testResourceChange() {
        if (((ResourceImpl) getResource()).isChanged()) {
            throw new IllegalStateException("The model was modified. Unable to return start offset."); // NOI18N
        }
    }

    /**
     * Returns the start offset of the element in the original source text.
     * 
     * @throws IllegalStateException  when element was changed
     * @return  start offset of the element
     */
    public int getStartOffset() {
        _lock();
        try {
        testResourceChange();
        return getStartOffset(getParser(), getASTree(), false);
        } finally {
            _unlock();
    }
    }

    /**
     * Start offset of part of the element.
     *
     * @param  part      identifies part of the element, see constants
     * @return   start offset of given part and position
     */
    public int getPartStartOffset(ElementPartKind part) {
        _lock();
        try {
        testResourceChange();
        return getParser().getToken(getPartStartTree(part).getFirstToken()).getStartOffset();
        } finally {
            _unlock();
    }
    }

    /**
     * Returns the end offset of the element in the original source text.
     * 
     * @return  end offset of the element
     */
    public int getEndOffset() {
        _lock();
        try {
        testResourceChange();
        return IndentUtil.getElementEnd(this);
        } finally {
            _unlock();
    }
    }

    /**
     * End offset of part of the element.
     *
     * @param  part      identifier part of the element, see constants
     * @return  end offset of given part and position
     */
    public int getPartEndOffset(ElementPartKind part) {
        _lock();
        try {
        testResourceChange();
        return getParser().getToken(getPartEndTree(part).getLastToken()).getEndOffset();
        } finally {
            _unlock();
    }
    }

    protected ASTree getPartStartTree(ElementPartKind part) {
        return getPartTree(part);
    }

    protected ASTree getPartEndTree(ElementPartKind part) {
        return getPartTree(part);
    }

    protected ASTree getPartTree(ElementPartKind part) {
        throw new IllegalArgumentException("Invalid part for this element: " + part.toString()); // NOI18N
    }

    protected static int getEndOffset(ASTProvider parser, ASTree tree) {
        Token endIndex = parser.getToken(tree.getLastToken());
        return endIndex.getEndOffset();
    }

    private ASTree getASTree(Object o) {
        if (o instanceof ElementInfo) {
            ElementInfo info = (ElementInfo) o;
            Reference sr = info.getASTree();
            if (sr == null) return null;

            ASTree result = (ASTree) sr.get();
            if (result == null) {
                try {
                    result = info.refreshASTree();
                } catch (IllegalArgumentException ex) {
                    StringBuffer report=new StringBuffer(100);
                    MetadataElement el=this;
                    
                    do {
                        report.append(el.getClass()).append(" ");
                        if (el instanceof NamedElement && el.isValid()) {
                            report.append(((NamedElement)el).getName()).append(" ");
                        }
                        report.append("valid:").append(el.isValid()).append(" | ");
                        Object obj = el.refImmediateComposite();
                        el = obj instanceof MetadataElement ? (MetadataElement)obj : null;
                    } while(el!=null);
                    throw new IllegalArgumentException(ex.getMessage()+" "+report.toString());
                }
            }
            return result;
        } else if (o instanceof ASTree) {
            return (ASTree) o;
        } else {
            return null;
        }
    }

    private ASTree getChildASTree(Object o) {
        return getASTree(o);
    }

    private ASTree getLastElement(int index, Object[] oldElements) {
        if (oldElements.length <= index) {
            return null;
        }
        return getChildASTree(oldElements[index]);
    }

    protected final Element initOrCreate(Element element, ASTree tree) {
        boolean changes = disableChanges;
        disableChanges = true;
        try {
            if (element != null) {
                if (element instanceof TransientElement) {
                    if (tree != null) {
                        RefClass proxy = getElementProxy(tree);
                        if (element.refClass().equals(proxy)) {
                            ((TransientElement) element).init(tree);
                            return element;
                        }
                    }
                    ((TransientElement) element).invalidate();
                } else {        // local class
                    element.refDelete();
                }
            }
            return createElement(tree);
        } finally {
            disableChanges = changes;
        }
    }

    public final boolean isTransient() {
        return _getDelegate() instanceof DeferredObject;
    }
    
    public final boolean checkElementType(ElementInfo astInfo, Element element) {
        switch (astInfo.infoType) {
            case FieldInfo.FIELD_TYPE:
                return (element instanceof Field) && !(element instanceof EnumConstant);
            case FieldGroupInfo.FIELDGROUP_TYPE:
                return element instanceof FieldGroup;
            case EnumInfo.ENUM_TYPE:
                return element instanceof JavaEnum;
            case ClassInfo.INTERFACE_TYPE:
            case ClassInfo.CLASS_TYPE:
                return (element instanceof JavaClass) && !(element instanceof AnnotationType) && !(element instanceof JavaEnum);
            case AnnotationTypeInfo.ANNOTATIONTYPE_TYPE:
                return element instanceof AnnotationType;
            case ClassInfo.ANON_CLASS_TYPE:
                return (element instanceof ClassDefinition) && !(element instanceof JavaClass);
            case MethodInfo.METHOD_TYPE:
                return element instanceof Method;
            case FeatureInfo.INSTANCE_INITIALIZER_TYPE:
            case FeatureInfo.STATIC_INITIALIZER_TYPE:
                return element instanceof Initializer;
            case MethodInfo.CONSTRUCTOR_TYPE:
                return element instanceof Constructor;
            case ElementInfo.IMPORT_ON_DEMAND_TYPE:
            case ElementInfo.SINGLE_IMPORT_TYPE:
                return element instanceof Import;
            case ParameterInfo.PARAMETER_TYPE:
                return element instanceof Parameter;
            case TypeParamInfo.TYPEPARAM_TYPE:
                return element instanceof TypeParameter;
            case FeatureInfo.ENUM_CONSTANT_TYPE:
                return element instanceof EnumConstant;
            case AttributeInfo.ATTRIBUTE_TYPE:
                return element instanceof Attribute;
            case AnnotationInfo.ANNOTATION_TYPE:
                return element instanceof Annotation;
            default:
                throw new IllegalArgumentException("Illegal type "+astInfo.infoType); // NOI18N
        }
    }
    
    public final SemiPersistentElement createElement(ElementInfo astInfo) {
        return createElement(astInfo, true);
    }

    protected final SemiPersistentElement createElement(ElementInfo astInfo, boolean initialize) {
        if (astInfo == null) return null;
        JavaModelPackage extent = (JavaModelPackage) refImmediatePackage();
        SemiPersistentElement element=null;
        boolean isTransient = isTransient();
        
        switch (astInfo.infoType) {
            case FieldInfo.FIELD_TYPE: {
                FieldInfo info = (FieldInfo) astInfo;
                element = ((FieldClassImpl) extent.getField()).create(info.name, info.modifiers, info.type, isTransient);
                break;
            } case FieldGroupInfo.FIELDGROUP_TYPE: {
                FieldGroupInfo info = (FieldGroupInfo) astInfo;
                element = ((FieldGroupClassImpl) extent.getFieldGroup()).create(info.modifiers, info.type, isTransient);
                break;
            } case EnumInfo.ENUM_TYPE:
                if (isTransient) {
                    ASTree tree=getASTree(astInfo);
                    MDRParser parser=(MDRParser) tree.getASTContext();
                    element = (JavaClassImpl)parser.getSemanticInfo(tree,this);
                } else {
                    ClassIndex index=ClassIndex.getIndex(extent);

                    Set classes = index.getClassesByFqn(astInfo.name);
                    for (Iterator it = classes.iterator(); it.hasNext();) {
                        JavaClassImpl temp = (JavaClassImpl) it.next();
                        if (temp.isValid()) {
                            Object composite = temp.refImmediateComposite();
                            if (composite == null || this.equals(composite)) {
                                element = temp;
                                break;
                            }
                        }
                    }
                    if (element == null) {
                        element = recoverFromCNFII(astInfo, extent.getJavaEnum());
                    }
                }
                break;
            case ClassInfo.INTERFACE_TYPE:
            case ClassInfo.CLASS_TYPE:
                if (isTransient) {
                    ASTree tree=getASTree(astInfo);
                    MDRParser parser=(MDRParser) tree.getASTContext();
                    element = (JavaClassImpl)parser.getSemanticInfo(tree,this);
                } else {
                    ClassIndex index=ClassIndex.getIndex(extent);

                    Set classes = index.getClassesByFqn(astInfo.name);
                    for (Iterator it = classes.iterator(); it.hasNext();) {
                        JavaClassImpl temp = (JavaClassImpl) it.next();
                        if (temp.isValid()) {
                            Object composite = temp.refImmediateComposite();
                            if (composite == null || this.equals(composite)) {
                                element = temp;
                                break;
                            }
                        }
                    }
                    if (element == null) {
                        element = recoverFromCNFII(astInfo, extent.getJavaClass());
                    }
                }
                break;
            case AnnotationTypeInfo.ANNOTATIONTYPE_TYPE:
                if (isTransient) {
                    ASTree tree=getASTree(astInfo);
                    MDRParser parser=(MDRParser) tree.getASTContext();
                    element = (JavaClassImpl)parser.getSemanticInfo(tree,this);
                } else {
                    ClassIndex index=ClassIndex.getIndex(extent);

                    Set classes = index.getClassesByFqn(astInfo.name);
                    for (Iterator it = classes.iterator(); it.hasNext();) {
                        JavaClassImpl temp = (JavaClassImpl) it.next();
                        if (temp.isValid()) {
                            Object composite = temp.refImmediateComposite();
                            if (composite == null || this.equals(composite)) {
                                element = temp;
                                break;
                            }
                        }
                    }
                    if (element == null) {
                        element = recoverFromCNFII(astInfo, extent.getAnnotationType());
                    }
                }
                break;            
            case ClassInfo.ANON_CLASS_TYPE:
                element = ((ClassDefinitionClassImpl) extent.getClassDefinition()).create();
                break;
            case MethodInfo.METHOD_TYPE: {
                MethodInfo info = (MethodInfo) astInfo;
                element = ((MethodClassImpl) extent.getMethod()).create(info.name, info.modifiers, info.type, info.exceptions, isTransient);
                break;
            } case FeatureInfo.INSTANCE_INITIALIZER_TYPE:
            case FeatureInfo.STATIC_INITIALIZER_TYPE:
                element = ((InitializerClassImpl) extent.getInitializer()).create(((FeatureInfo) astInfo).modifiers, isTransient);
                break;
            case MethodInfo.CONSTRUCTOR_TYPE: {
                MethodInfo info = (MethodInfo) astInfo;
                element = ((ConstructorClassImpl) extent.getConstructor()).create(info.modifiers, info.exceptions, isTransient);
                break;
            } case ElementInfo.IMPORT_ON_DEMAND_TYPE:
            case ElementInfo.SINGLE_IMPORT_TYPE:
                element = ((ImportClassImpl)extent.getImport()).create(astInfo.name, null, false, astInfo.infoType == ElementInfo.IMPORT_ON_DEMAND_TYPE, (ResourceImpl)this);
                break;
            case ParameterInfo.PARAMETER_TYPE: {
                ParameterInfo info = (ParameterInfo) astInfo;
                element = ((ParameterClassImpl) extent.getParameter()).create(info.name, info.isFinal, info.isVarArg, info.type, isTransient);
                break;
            } case TypeParamInfo.TYPEPARAM_TYPE: {
                TypeParamInfo info = (TypeParamInfo) astInfo;
                element = ((TypeParameterClassImpl) extent.getTypeParameter()).create(info.name, info.bounds, isTransient);
                break;
            } case FeatureInfo.ENUM_CONSTANT_TYPE:
                element = ((EnumConstantClassImpl) extent.getEnumConstant()).create(astInfo.name, isTransient);
                break;
            case AttributeInfo.ATTRIBUTE_TYPE:
                element = ((AttributeClassImpl) extent.getAttribute()).create(astInfo.name, ((AttributeInfo) astInfo).modifiers, ((AttributeInfo) astInfo).type, isTransient);
                break;
            case AnnotationInfo.ANNOTATION_TYPE:
                element = ((AnnotationClassImpl) extent.getAnnotation()).create(isTransient);
                break;
            case AnnotationValueInfo.ANNOTATIONVALUE_ANNOTATION:
            case AnnotationValueInfo.ANNOTATIONVALUE_ARRAY:
            case AnnotationValueInfo.ANNOTATIONVALUE_STRING:
                element = ((AttributeValueClassImpl) extent.getAttributeValue()).create(astInfo.name,isTransient);                
                break;
            default:
                throw new IllegalArgumentException("Illegal type "+astInfo.infoType); // NOI18N
        }
        element.updatePersistent(astInfo);
        if (initialize) {
            element.setElementInfo(astInfo);
        }
        return element;
    }
    
    private static SemiPersistentElement recoverFromCNFII(ElementInfo info, RefClass classProxy) {
        JMManager.getLog().notify(ErrorManager.INFORMATIONAL, new Exception("Class not found in index: " + info.name + ". Recovering...")); // NOI18N
        if (classProxy instanceof JavaEnumClassImpl) {
            return ((JavaEnumClassImpl) classProxy).create(info.name, ((EnumInfo) info).modifiers, false);
        } else if (classProxy instanceof AnnotationTypeClassImpl) {
            return ((AnnotationTypeClassImpl) classProxy).create(info.name, ((ClassInfo) info).modifiers, false);
        } else {
            return ((JavaClassClassImpl) classProxy).create(info.name, ((ClassInfo) info).modifiers, ((ClassInfo) info).superclass, ((ClassInfo) info).interfaces, false);
        }
    }

    public static ASTree[] getChildNodes(ASTree parent, int groupNodeType) {
        if (parent == null) {
            return new ASTree[0];
        } else if (parent.getType() == groupNodeType) {
            return parent.getSubTrees();
        } else {
            return new ASTree[] {parent};
        }
    }
    
    public Type resolveType(TypeRef typeRef) {
        if (typeRef == null) return null;
        
        JavaModelPackage extent = (JavaModelPackage) refImmediatePackage();
        if (typeRef instanceof LocalClassNameRef) {
            LocalClassNameRef locRef=(LocalClassNameRef)typeRef;
            
            return (Type)JavaMetamodel.getDefaultRepository().getByMofId(locRef.mofId);
        } else if (typeRef instanceof NameRef) {
            NameRef nRef = (NameRef) typeRef;
            JavaClass parent = (JavaClass) resolveType(nRef.parent);
            ArrayList args = new ArrayList(nRef.args.length);
            boolean hasWildCard = false;
            for (int i = 0; i < nRef.args.length; i++) {
                args.add(resolveType(nRef.args[i]));
                if (nRef.args[i] instanceof WildCardRef) {
                    hasWildCard = true;
                }
            }
            JavaClass definition;
            if (parent != null) {
                definition = parent.getInnerClass(nRef.name, true);
                if (definition == null) {
                    definition = ((UnresolvedClassClassImpl) extent.getUnresolvedClass()).resolveUnresolved(nRef.name);
                }
            } else {
                definition = ((JavaClassClassImpl) extent.getJavaClass()).resolveClass(nRef.name, false);
            }
            if ((definition instanceof AnnotationType) && (parent == null) && (args.size() == 0)) {
                return definition;
            }
            if (!(parent instanceof ParameterizedType)) {
                parent = null;
            }
            ParameterizedTypeClass parTypeClass = ((JavaModelPackage)definition.refImmediatePackage()).getParameterizedType();
            ParameterizedTypeImpl result = (ParameterizedTypeImpl) parTypeClass.resolveParameterizedType(definition, args, (ParameterizedType)parent);
            if (hasWildCard) {
                for (int i = 0; i < nRef.args.length; i++) {
                    if (nRef.args[i] instanceof WildCardRef) {
                        WildCardRef wcRef = (WildCardRef) nRef.args[i];
                        result.setWildCardStatus(i, wcRef.bound == null ? 3 : (wcRef.isLower ? 1 : 2));
                    }
                }
            }
            return result;
        } else if (typeRef instanceof ArrayRef) {
            ArrayRef aRef = (ArrayRef) typeRef;
            Type result = resolveType(aRef.parent);
            ArrayClass arrClass = ((JavaModelPackage)result.refImmediatePackage()).getArray();
            for (int i = 0; i < aRef.dimCount; i++) {
                result = arrClass.resolveArray(result);
            }
            return result;
        } else if (typeRef instanceof TypeParamRef) {
            TypeParamRef tpRef = (TypeParamRef) typeRef;
            Element element=this;
            while (!((element instanceof Resource) || element == null)) {
                if (element instanceof GenericElement) {
                    GenericElement gel=(GenericElement)element;
                    Iterator typeParsIt=gel.getTypeParameters().iterator();
                    
                    while(typeParsIt.hasNext()) {
                        TypeParameter typePar=(TypeParameter)typeParsIt.next();
                        
                        if (typePar.getName().equals(tpRef.name))
                            return typePar;

                    }
                }
                element=(Element)element.refImmediateComposite();
            }
            // type parameter not found, create unresolved class - this is OK - this case happens when parsing the file and parser
            // is resolving the identifiers
            return ((JavaClassClassImpl) extent.getJavaClass()).resolveClass(tpRef.name, false);
        } else if (typeRef instanceof PrimitiveTypeRef) {
            PrimitiveTypeRef ptRef = (PrimitiveTypeRef) typeRef;
            return ((PrimitiveTypeClassImpl) extent.getPrimitiveType()).resolveType(ptRef.name);
        } else if (typeRef instanceof WildCardRef) {
            WildCardRef wcRef = (WildCardRef) typeRef;
            if (wcRef.bound != null) {
                return resolveType(wcRef.bound);
            }
            return resolveType(NameRef.java_lang_Object);
        }
        throw new IllegalStateException();
    }

    /**
     * Returns indentation of the element. Every element, which should be
     * indented has to override this method.
     *
     * @return indentation for element.
     */
    protected String getIndentation() {
        return "";
    }

    // temporary part - should be read from options
    static String[][] data = null;
    static String[] EMPTY = { null, null, null };
    static {
        //
        data = new String[][] {
            EMPTY,                     //  0 EMPTY
            { "s", "{", "n" },         //  1 BODY_OPEN_CURLY // NOI18N
            { "i", "}", "" },          //  2 BODY_CLOSE_CURLY // NOI18N
            { "", "(", "" },           //  3 PAR_OPEN_BRACKET // NOI18N
            { "", ")", "" },           //  4 PAR_CLOSE_BRACKET // NOI18N
            { "", ",", "s" },          //  5 COMMA // NOI18N
            { null, null, null },      //  6 CALLABLE_IDENTIFIER
            { "s" , "throws", "s" },   //  7 THROWS_KEYWORD // NOI18N
            { "s", "=", "s" },         //  8 FIELD_EQUALS // NOI18N
            { "s", "extends", "s" },   //  9 EXTENDS_KEYWORD // NOI18N
            { "s", "implements", "s" },// 10 IMPLEMENTS_KEYWORD // NOI18N
            { "s", "{", "n" },         // 11 CLASS_OPEN_CURLY // NOI18N
            { "i", "}", "n" },         // 12 CLASS_CLOSE_CURLY // NOI18N
            { "", "[", "" },           // 13 ARRAY_OPEN_BRACKET // NOI18N
            { "", "]", "" },           // 14 ARRAY_CLOSE_BRACKET // NOI18N
            { "", "{", "" },           // 15 ARRAY_OPEN_CURLY // NOI18N
            { "", "}", "" },           // 16 ARRAY_CLOSE_CURLY // NOI18N
            { "i", "assert", "s" },    // 17 ASSERT_KEYWORD // NOI18N
            { "s", "catch", "s" },     // 18 CATCH_KEYWORD // NOI18N
            { "", "(", "" },           // 19 STMT_OPEN_BRACKET // NOI18N
            { "", ")", "" },           // 20 STMT_CLOSE_BRACKET // NOI18N
            { "i", "case", "s" },      // 21 CASE_KEYWORD // NOI18N
            { "i", "default", ":" },   // 22 DEFAULT_KEYWORD // NOI18N
            { "s", "?", "s" },         // 23 COND_EXPR_QUESTION // NOI18N
            { "s", ":", "s" },         // 24 COND_EXPR_COLON // NOI18N
            { "i", "do", "s" },        // 25 DO_KEYWORD // NOI18N
            { "s", "while", "s" },     // 26 DO_WHILE_KEYWORD // NOI18N
            { "", "for", "s" },        // 27 FOR_KEYWORD // NOI18N
            { "", ";", "s" },          // 28 FOR_SEMICOLON // NOI18N
            { "", "if", "s" },         // 29 IF_KEYWORD // NOI18N
            { "s", "else", "" },       // 30 ELSE_KEYWORD // NOI18N
            { "", "", "" },            // 31 --- not used ---
            { "s", "{", "" },          // 32 BLOCK_OPEN_CURLY // NOI18N
            { "ni", "}", "" },         // 33 BLOCK_CLOSE_CURLY // NOI18N
            { "i", "switch", "s" },    // 34 SWITCH_KEYWORD // NOI18N
            { "i", "synchronized", "s" },// 35 SYNCHRONIZED_KEYWORD // NOI18N
            { "", "throw", "s" },      // 36 THROW_KEYWORD // NOI18N
            { "", "try", "" },         // 37 TRY_KEYWORD // NOI18N
            { "s", "finally", "" },    // 38 FINALLY_KEYWORD // NOI18N
            { "", "while", "s" },      // 39 WHILE_KEYWORD // NOI18N
            { "i", "}", "" }           // 40 ANON_CLASS_CLOSE_CURLY // NOI18N
        };
    }
    // end temporary

    // formatting constants
    public static final int BODY_OPEN_CURLY = 1;
    public static final int BODY_CLOSE_CURLY = 2;
    public static final int PAR_OPEN_BRACKET = 3;
    public static final int PAR_CLOSE_BRACKET = 4;
    public static final int COMMA = 5;
    public static final int CALLABLE_IDENTIFIER = 6;
    public static final int THROWS_KEYWORD = 7;
    public static final int FIELD_EQUALS = 8;
    public static final int EXTENDS_KEYWORD = 9;
    public static final int IMPLEMENTS_KEYWORD = 10;
    public static final int CLASS_OPEN_CURLY = 11;
    public static final int CLASS_CLOSE_CURLY = 12;
    public static final int ARRAY_OPEN_BRACKET = 13;
    public static final int ARRAY_CLOSE_BRACKET = 14;
    public static final int ARRAY_OPEN_CURLY = 15;
    public static final int ARRAY_CLOSE_CURLY = 16;
    public static final int ASSERT_KEYWORD = 17;
    public static final int CATCH_KEYWORD = 18;
    public static final int STMT_OPEN_BRACKET = 19;
    public static final int STMT_CLOSE_BRACKET = 20;
    public static final int CASE_KEYWORD = 21;
    public static final int DEFAULT_KEYWORD = 22;
    public static final int COND_EXPR_QUESTION = 23;
    public static final int COND_EXPR_COLON = 24;
    public static final int DO_KEYWORD = 25;
    public static final int DO_WHILE_KEYWORD = 26;
    public static final int FOR_KEYWORD = 27;
    public static final int FOR_SEMICOLON = 28;
    public static final int IF_KEYWORD = 29;
    public static final int ELSE_KEYWORD = 30;
    public static final int BLOCK_OPEN_CURLY = 32;
    public static final int BLOCK_CLOSE_CURLY = 33;
    public static final int SWITCH_KEYWORD = 34;
    public static final int SYNCHRONIZED_KEYWORD = 35;
    public static final int THROW_KEYWORD = 36;
    public static final int TRY_KEYWORD = 37;
    public static final int FINALLY_KEYWORD = 38;
    public static final int WHILE_KEYWORD = 39;
    public static final int ANON_CLASS_CLOSE_CURLY = 40;
    
    /**
     * Looks for the formatting used for defined element (type). Puts
     * source representation to the parameter buffer. If the data are not
     * available for the specified formatting type, it throws exception.
     *
     * @param  type  represents type of element which should be formatted
     * @throws IllegalArgumentException  if the formatting for the type
     *         is not available or the formatting data are not accessible
     */
    protected final void formatElementPart(int type, StringBuffer buf) throws IllegalArgumentException {
        if (data == null || type >= data.length) {
            throw new IllegalArgumentException("Format data for the argument " + // NOI18N
                "type #" + type + " not available or formatting data are not " + // NOI18N
                "accessible."); // NOI18N
        }
        // prints leading formatting
        format(data[type][0], buf);
        // prints token representation
        if (data[type][1] != null)
            buf.append(data[type][1]);
        // prints trailing formatting
        format(data[type][2], buf);
    }

    /**
     * @see ...
     * @return  formatted element part
     */
    protected final String formatElementPart(int type) {
        StringBuffer buf = new StringBuffer();
        formatElementPart(type, buf);
        return buf.toString();
    }
    
    protected void formatElementPart(int type, StringBuffer buf, boolean isNew) {
        if (isNew)
            formatElementPart(type, buf);
        else
            buf.append(data[type][1]);
    }
    
    /**
     *
     * @return 
     */
    String getElementPrefix(int type) {
        StringBuffer buf = new StringBuffer(64);
        format(data[type][0], buf);
        return buf.toString();
    }

    /**
     * Sets the text used before the element and after the
     * element. See ... how strings are interpreted.
     */
    public static final void setFormattingFor(int type, String pre, String post) {
        data[type][0] = pre;
        data[type][2] = post;
    }

    /**
     * Returns the array of the three strings:
     * Characters before the text,
     * formatted text,
     * characters after the text.
     * 
     * @return  array of strings
     */
    public static final String[] getFormattingFor(int type) {
        return data[type];
    }

    /**
     * Creates whitespaces from the formatting mask.
     * Format characters representation:
     * n ... new line ('\n') character
     * i ... indentation
     * s ... space
     *
     * @param   mask represents formatting mask
     * @param   buf  buffer to append whitespaces to
     *
     * @return  if anything was added to the buffer
     */
    private boolean format(String mask, StringBuffer buf) {
        if (mask == null || mask.length() == 0)
            return false;

        char[] c = mask.toCharArray();
        for (int i = 0; i < c.length; i++) {
            switch (c[i]) {
                case 'n':
                    buf.append('\n');
                    break;
                case 's':
                    buf.append(' ');
                    break;
                case 'i':
                    buf.append(getIndentation());
                    break;
            }
        }
        return true;
    }

    protected StringBuffer appendDims(StringBuffer buf, int dimCount) {
        for (int i = 0; i < dimCount; i++) {
            formatElementPart(ARRAY_OPEN_BRACKET, buf);
            formatElementPart(ARRAY_CLOSE_BRACKET, buf);
        }
        return buf;
    }

    /**
     * Sets the default indentation of the element.
     *
     * @param  spaces      number of spaces used for indentation
     * @param  expandTabs  if this parameter is true, tab character is used
     *                     instead of spaces
     */
    public static void setIndentSpace(int spaces, boolean expandTabs) {
        if (!expandTabs) {
            // user want to use \t character instead of spaces
            INDENTATION = "\t"; // NOI18N
        } else {
            if (spaces >= 0) {
                spacesPerTab = spaces;
                char[] buf = new char[spaces];
                Arrays.fill(buf, ' ');
                INDENTATION = new String(buf);
            }
        }
    }
    
    public static boolean isExpandTab() {
        return !INDENTATION.equals("\t"); // NOI18N
    }
    
    public static int getIndentSpace() {
        return spacesPerTab;
    }
    
    protected RefFeatured _immediateComposite() {
        return getRealComposite(super._immediateComposite());
    }
    
    protected RefFeatured _realImmediateComposite() {
        return super._immediateComposite();
    }

    protected RefFeatured _outermostComposite() {
        return getRealComposite(super._outermostComposite());
    }

    private RefFeatured getRealComposite(RefFeatured refFeatured) {
        if (refFeatured instanceof JavaPackage) {
            JavaPackage pkg = (JavaPackage) refFeatured;
            return ((JavaPackageClass) pkg.refClass()).resolvePackage(pkg.getName());
        } else {
            return refFeatured;
        }
    }

    void childChanged(MetadataElement element) {
    }

    protected final void removeChild(MetadataElement element) {
        replaceChild(element, null);
    }

    public void replaceChild(Element oldElement, Element newElement) {
    }

    protected final boolean replaceObject(List list, Element oldElement, Element newElement) {
        int index=list.indexOf(oldElement);

        if (index!=-1) {
            if (newElement==null)
                list.remove(index);
            else
                list.set(index,newElement);
            return true;
        }
        return false;
    }
    
    static void fixImportsInClassList(Element scope,List newEl,List origEls) {
        List oels=new ArrayList(origEls);
        Iterator fIt=oels.iterator();

        if (fIt.hasNext()) {
            newEl.clear();
            while(fIt.hasNext()) {
                newEl.add(JavaModelUtil.resolveImportsForClass(scope,(JavaClass)fIt.next()));
            }
        }        
    }
    
    static void fixImports(Element scope, List newEl, List origEls) {
        List nels=new ArrayList(newEl);
        List oels=new ArrayList(origEls);
        int size = nels.size();
        
        for (int i=0;i<size;i++) {
            Element oe=(Element)oels.get(i);
            MetadataElement ne=(MetadataElement)nels.get(i);
            
            ne.fixImports(scope,oe);
        }        
    }
    
    public void fixImports(Element scope,Element original) {
        fixImports(scope, getChildren(), original.getChildren());
    }
    
    /**
     * Duplicates elements in list provided in parameter and returns 
     * ArrayList with the elements' duplicates.
     * 
     * @param  list  original list with elements, which should be duplicated
     * @return  new list with duplicated elements.
     */
    static List duplicateList(List list, JavaModelPackage targetExtent) {
        if (list == null) {
            return Collections.EMPTY_LIST;
        }
        Object[] elements=list.toArray();
        int size = elements.length;
        if (size == 0) {
            return Collections.EMPTY_LIST;
        }
        List result = new ArrayList(size);
        for (int i=0;i<elements.length;i++) {
            Object o = elements[i];
            result.add(((MetadataElement) o).duplicate(targetExtent));
        }
        return result;
    }
    
    /**
     * Duplicates provided element by calling duplicate() method on them.
     * Method is provided only for null value handling.
     *
     * @param   element which has to be duplicated
     * @return  duplicated element or null
     */
    static Element duplicateElement(Element e, JavaModelPackage targetExtent) {
        return e == null ? null : ((MetadataElement) e).duplicate(targetExtent);
    }
    
    public Element duplicate() {
        return duplicate((JavaModelPackage) refImmediatePackage());
    }

    public abstract Element duplicate(JavaModelPackage targetExtent);
}
