/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.lang.reflect.Modifier;
import java.util.*;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.parser.ASTProvider;
import org.netbeans.modules.javacore.parser.ElementInfo;
import org.netbeans.modules.javacore.parser.MethodInfo;
import org.netbeans.modules.javacore.parser.TypeRef;

/**
 * Implementation of Method object instance interface.
 *
 * @author  Martin Matula
 * @author  Pavel Flaska
 */
public abstract class MethodImpl extends CallableFeatureImpl implements Method {
    private static final ElementInfo DEFAULT_INFO = new MethodInfo(null, MethodInfo.METHOD_TYPE, null, 0, null, null, null, null, null);

    private TypeReference typeName;

    /** Creates a new instance of MethodImpl */
    public MethodImpl(StorableObject s) {
        super(s);
    }

    public String toString() {
        return "method " + getName(); // NOI18N
    }

    protected ElementInfo getDefaultInfo() {
        return DEFAULT_INFO;
    }

    public TypeReference getTypeName() {
        checkUpToDate();
        if (!elementsInited) {
            initASTElements();
        }
        return typeName;
    }

    public void setTypeName(TypeReference typeName) {
        _setTypeName(typeName, typeReferenceToTypeRef(typeName, 0));
    }
    
    private void _setTypeName(TypeReference typeName, TypeRef typeRef) {
        if (!disableChanges) {
            objectChanged(CHANGED_TYPE);
            changeChild(getTypeName(), typeName);
            this.typeName = typeName;
        }
        setTypeRef(typeRef);
    }
    
    private void fireTypeNameChange(TypeReference typeReference) {
        Object oldValue = null;
        Object newValue = null;
        if (elementsInited && !disableChanges) {
            oldValue = getTypeName();
            newValue = typeReference;
        }
        fireAttrChange("typeName", oldValue, newValue); // NOI18N
    }
    
    /**
     * Returns the value of reference type.
     * @return Value of reference type.
     */
    public Type getType() {
        checkUpToDate();
        return resolveType(getTypeRef());
    }

    /**
     * Sets the value of reference type. See {@link #getType} for description
     * on the reference.
     * @param newValue New value to be set.
     */
    public void setType(Type newValue) {
        TypeRef tr = typeToTypeRef(newValue);
        TypeReference typeReference = (TypeReference) typeRefToTypeReference(tr, 0);
        fireTypeNameChange(typeReference);
        _setTypeName(typeReference, tr);
    }

    protected void initASTElements() {
        elementsInited = false;
        super.initASTElements();
        ElementInfo info = getElementInfo();
        ASTree tree = info.getTypeAST(this);
        typeName = (TypeReference) initOrCreate(typeName, tree);
        elementsInited = true;
    }

    protected void setData(List annotations, java.lang.String javadocText, JavaDoc javadoc, StatementBlock body, java.lang.String bodyText, List typeArguments, List parameters, List exceptionNames, TypeReference typeName, int dimCount) {
        super.setData(annotations, javadocText, javadoc, body, bodyText, typeArguments, parameters, exceptionNames);
        changeChild(null, typeName);
        this.typeName = typeName;
        setTypeRef(typeReferenceToTypeRef(typeName, 0));
    }

    protected void resetASTElements() {
        if (elementsInited) {
            if (typeName != null) {
                TypeReference temp = typeName;
                changeChild(typeName, null);
                typeName = null;
                temp.refDelete();
            }
            super.resetASTElements();
        }
    }

    public void replaceChild(Element oldElement, Element newElement) {
        if (elementsInited) {
            if (oldElement.equals(typeName)) {
                setTypeName((TypeReference) newElement);
                return;
            }
        }
        super.replaceChild(oldElement, newElement);
    }

    public List getChildren() {
        List list = new ArrayList();
        list.addAll(getAnnotations());
        addIfNotNull(list, getTypeName());
        list.addAll(super.getChildren());
        return list;
    }

    public void fixImports(Element scope, Element original) {
        setTypeName(JavaModelUtil.resolveImportsForType(scope,((Method)original).getType()));
        setDimCount(0);
        super.fixImports(scope,original);
    }

    protected List getInitedChildren() {
        List list = super.getInitedChildren();
        if (elementsInited) {
            addIfNotNull(list, typeName);
        }
        return list;
    }

    // .........................................................................
    // printing and formatting fuctionality
    // .........................................................................

    /**
     * Implementation of abstract method from CallableFeatureImpl. Appends
     * to buffer return type and method name separated by space.
     *
     * @param  buf  buffer to append to
     */
    void generateTypeAndName(StringBuffer buf) {
        MetadataElement melem = (MetadataElement)getTypeName();
        if (melem != null) {
            buf.append(melem.getSourceText());
        } else {
            buf.append(getType().getName());
        }
        buf.append(' ');
        buf.append(getName());
        formatElementPart(CALLABLE_IDENTIFIER, buf);
    }

    /**
     * Returns source code representation of this object.
     *
     * @return  source code representation of the object.
     */
    String getRawText() {
        if (!elementsInited) {
            initASTElements();
        }
        StringBuffer buf = new StringBuffer();
        generateHeader(buf);
        int modifiers = getSourceModifiers();
        boolean parentAbstract = false;
        if (refImmediateComposite() instanceof JavaClass) {
            JavaClass jc = (JavaClass) refImmediateComposite();
            parentAbstract = jc.isInterface();
        }
        // for abstract, native methods and methods in interfaces put semicolon only
        if ((!parentAbstract && (modifiers & Modifier.ABSTRACT) != Modifier.ABSTRACT) &&
            ((modifiers & Modifier.NATIVE) != Modifier.NATIVE))
        {
            generateBody(buf);
        }
        else
            buf.append(';');
        return buf.toString();
    }
    
    protected void _delete() {
        if (elementsInited) {
            deleteChild(typeName);
        }
        super._delete();
    }
    
    public void getDiff(List diffList) {
        MethodInfo astInfo = (MethodInfo) getElementInfo();
        ASTProvider parser = getParser();
        ASTree tree = getASTree();
        ASTree[] children = tree.getSubTrees();

        // javadoc print
        replaceJavaDoc(diffList);
        // modifier print
        ASTree type = children[2];
        if (isChanged(CHANGED_MODIFIERS) || isChanged(CHANGED_ANNOTATION)) {
            diffModifiers(diffList, type == null ? children[3] : type, parser);
        } else if (children[0] != null) {
            getCollectionDiff(diffList, parser, CHANGED_ANNOTATION, astInfo.annotations, getAnnotations(), parser.getToken(children[0].getLastToken()).getEndOffset(), "\n"); // NOI18N
        }
        getTypeParamsDiff(diffList);
        // type
        getChildDiff(diffList, parser, type, (MetadataElement) getTypeName(), CHANGED_TYPE);
        // name
        ASTree declarator = tree.getSubTrees()[3];
        if (isChanged(CHANGED_NAME)) {
            String newName = getName();
            replaceNode(diffList, parser, declarator.getSubTrees()[0], newName, 0, null);
        }
        // parameters
        String comma = formatElementPart(COMMA);
        int endOffset = parser.getToken(declarator.getLastToken()).getStartOffset();
        getCollectionDiff(diffList, parser, CHANGED_PARAMETERS,
            astInfo.parameters, getParameters(), endOffset, comma);
        // exceptions
        int startOffset = parser.getToken(declarator.getLastToken()).getEndOffset();
        getCollectionDiff(diffList, parser, CHANGED_THROWS, tree.getSubTrees()[4],
                getExceptionNames(), startOffset, comma, formatElementPart(THROWS_KEYWORD));
        // body print
        createBodyDiffs(diffList);
    }

    /**
     */
    protected ASTree getPartTree(ElementPartKind part) {
        ASTree[] children = getASTree().getSubTrees();
        if (ElementPartKindEnum.NAME.equals(part))
            return children[3].getSubTrees()[0];

        throw new IllegalArgumentException("Invalid part for this element: " + part); // NOI18N
    }

    /**
     */
    protected ASTree getPartStartTree(ElementPartKind part) {
        if (ElementPartKindEnum.HEADER.equals(part)) {
            return getASTree();
        }
        return super.getPartStartTree(part);
    }

    public Collection getReferences() {
        return findDependencies(true, false, false);
    }
    
    public boolean signatureEquals(Method m) {
        return signatureEquals(this, m);
    }
    
    static boolean signatureEquals(Method m1, Method m2) {
        if (!m1.getName().equals(m2.getName()))
            return false;
        
        Parameter[] params1 = (Parameter[]) m1.getParameters().toArray(new Parameter[0]);
        Parameter[] params2 = (Parameter[]) m2.getParameters().toArray(new Parameter[0]);
        if (params1.length != params2.length)
            return false;
        
        for (int i = 0; i < params2.length; i++) {
            Type t1 = TypeClassImpl.getRawType(params1[i].getType());
            Type t2 = TypeClassImpl.getRawType(params2[i].getType());
            if (!t1.getName().equals(t2.getName())) {
                return false;
            }
        }
        return true;
    }
    
    public Collection getOverriddenMethods() {
        Collection result = new LinkedHashSet(2);
        
        ClassDefinition declaringClass = this.getDeclaringClass();
        
        List params = new ArrayList();
        for (Iterator i = this.getParameters().iterator(); i.hasNext(); params.add(((Parameter)i.next()).getType()));
        
        Iterator i = declaringClass.getInterfaces().iterator();
        while (i.hasNext()) {
            JavaClass jc = (JavaClass) i.next();
            Method m = jc.getMethod(this.getName(), params, true);
            if (m!=null) {
                if (m instanceof ParameterizedTypeImpl.Wrapper) {
                    m = (Method) ((ParameterizedTypeImpl.Wrapper) m).getWrappedObject();
                }
                result.addAll(((MethodImpl) m).getOverriddenMethods());
                result.add(m);
            }
        }

        ClassDefinition parent = declaringClass.getSuperClass();
        if (parent != null) {
            Method m = parent.getMethod(this.getName(), params, true);
            if (m!=null) {
                if (m instanceof ParameterizedTypeImpl.Wrapper) {
                    m = (Method) ((ParameterizedTypeImpl.Wrapper) m).getWrappedObject();
                }
                result.addAll(((MethodImpl) m).getOverriddenMethods());
                result.add(m);
            }
        }
        return result;
    }
    
    void childChanged(MetadataElement mpi) {
        super.childChanged(mpi);
        if (elementsInited) {
            if (mpi == typeName) {
                setTypeName((TypeReference) mpi);
            }
        }
    }

    public Element duplicate(JavaModelPackage targetExtent) {
        StatementBlock body;
        String bodyText;
        
        if (isChanged(CHANGED_BODY) && this.bodyText != null) {
            body = null;
            bodyText = this.bodyText;
        } else {
            body = (StatementBlock) duplicateElement(getBody(), targetExtent);
            bodyText = null;
        }
        
        return targetExtent.getMethod().createMethod(
                getName(),
                duplicateList(getAnnotations(), targetExtent),
                getModifiers(),
                null,
                (JavaDoc) duplicateElement(getJavadoc(), targetExtent),
                body,
                bodyText,
                duplicateList(getTypeParameters(), targetExtent),
                duplicateList(getParameters(), targetExtent),
                duplicateList(getExceptionNames(), targetExtent),
                (TypeReference) duplicateElement(getTypeName(), targetExtent),
                getDimCount()
            );
    }

}
