/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

/*
 * UsageFinder.java
 *
 * Created on 06 January 2004, 11:44
 */

package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.util.*;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.mdr.NBMDRepositoryImpl;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.ProgressSupport;
import org.openide.ErrorManager;
import org.openide.util.Cancellable;

/**
 *
 * @author  Jan Becicka
 */
public class UsageFinder {
    
    private NamedElement what;
    private ElementNavigator navigator;
    private Collection usages;
    private static final int MAX_COUNT = 30;
    private Set declaringClasses;
    private boolean overriders = false;
    private boolean findUsages = true;
    
    /** Creates a new instance of UsageFinder */
    public UsageFinder(NamedElement what) {
        if (what instanceof ParameterizedType) {
            this.what = convert(getRealClassDefinition((ParameterizedType) what));
        } else {
            this.what = convert(what);
        }
    }
    
    private static List getParameters(CallableFeature f) {
        List params = new ArrayList();
        for (Iterator i = f.getParameters().iterator(); i.hasNext(); params.add(convert(TypeClassImpl.getRawType(((Parameter)i.next()).getType()))));
        return params;
    }
    
    private static NamedElement convert(NamedElement what) {
        if (what instanceof TypeParameter) {
            return what;
        }
        if (what instanceof JavaClass) {
            return convert((JavaClass) what);
        }

        NamedElement convertedElement = null;
        
        if (what instanceof ClassMember) {
            ClassDefinition cls = ((ClassMember) what).getDeclaringClass();
            if (cls instanceof JavaClass) {
                JavaClass clazz = (JavaClass) convert(cls);
                if (clazz.equals(cls)) {
                    return what;
                }

                if (what instanceof Field) {
                    convertedElement = clazz.getField(what.getName(), false);
                } else {
                    if (what instanceof Method) {
                        convertedElement = clazz.getMethod(what.getName(), getParameters((CallableFeature) what), false);
                    } else if (what instanceof Constructor) {
                        convertedElement = clazz.getConstructor(getParameters((CallableFeature) what), false);
                    }
                }
            } 
        }
        return convertedElement == null ? what:convertedElement ;
    }
    
    private static Type convert(Type clazz) {
        if (clazz instanceof JavaClassImpl && ((JavaClassImpl)clazz).isTransient())  // local class
            return clazz;
        return JavaMetamodel.getManager().getDefaultExtent().getType().resolve(clazz.getName());
    }
    
    /**
     * specific contructor for method
     * @param method method for search
     * @param findUsages find usages of methods
     * @param fromBaseClass find all occurrences including superclass
     * @param overriders get Overriding Methods
     */
    public UsageFinder(CallableFeature method, boolean findUsages, boolean fromBaseClass, boolean overriders) {
        this(method);
        this.overriders = overriders;
        this.findUsages = findUsages;
        declaringClasses = new HashSet();
        ClassDefinition clazz = method.getDeclaringClass();
        declaringClasses.add(getRealClassDefinition(clazz));
        List argTypes = new ArrayList();
        for (Iterator i = method.getParameters().iterator(); i.hasNext(); argTypes.add(TypeClassImpl.getRawType(((Parameter)i.next()).getType())));
        
        if (fromBaseClass) {
            LinkedList q = new LinkedList();
            q.add(clazz);
            HashSet visited = new HashSet();
            while (!q.isEmpty()) {
                clazz = (ClassDefinition) q.removeFirst();
                if (visited.add(clazz)) {
                    CallableFeature callFeature = null;
                    if (method instanceof Method) 
                        callFeature = clazz.getMethod(method.getName(), argTypes, false);
                    else
                        callFeature = clazz.getConstructor(argTypes, false);
                    if (callFeature != null) {
                        declaringClasses.add(getRealClassDefinition(clazz));
                    }
                    q.addAll(clazz.getInterfaces());
                    clazz = clazz.getSuperClass();
                    if (clazz!=null) {
                        q.add(clazz);
                    }
                }
            }
        }
    }
    
    private ClassDefinition getRealClassDefinition(ClassDefinition cls) {
        return (ClassDefinition) TypeClassImpl.getRawType(cls);
    }
    
    private Collection getUsers(MetadataElement where) {
        usages = new ArrayList();
        List l = new ArrayList(1);
        l.add(where);
        getUsers(l);
        return usages;
    }
    
    public Collection getUsers(Resource res[]) {
        return new LazyCollection(res);
    }
    
    private void getUsers(Collection elements) {
        if (elements == null || elements.isEmpty())
            return;
        
        for (Iterator iter = elements.iterator(); iter.hasNext();) {
            MetadataElement el = (MetadataElement) iter.next();
            if ((el instanceof Resource) || navigator.containsIdentifierIn(el)) {
                if (isMatch(what, el)) {
                    usages.add(el);
                }
                getUsers(el.getChildren());
            }
        }
    }
    
    private boolean isMatch(Element what, Element r) {
        //I'm not usage of myself
        if (what.equals(r))
            return false;

        Element ref = r;
        if (findUsages) {
            if (ref instanceof ElementReference) {
                ref = ((ElementReference) r).getElement();
            } else {
                if (!overriders) {
                    return false; 
                }
            }
        }

        if (!(what instanceof Method)) {
            if (ref instanceof ParameterizedType) {
                ref = getRealClassDefinition((ParameterizedType) ref);
            }
            return what.equals(ref);
        }

//        if (!overriders) {
//            if (ref instanceof Method)
//                return false;
//        }

        if (!(ref instanceof Method)) {
            return false;
        }
        
        if (!overriders && what.equals(ref)) {
            return true;
        }
        
        Method m = (Method) ref;
        if (m.signatureEquals((Method) what)) {
            ClassDefinition collectedClass, declaringClass = getRealClassDefinition(m.getDeclaringClass());
            for (Iterator i = declaringClasses.iterator(); i.hasNext();) {
                collectedClass = (ClassDefinition) i.next();
                if (declaringClass.isSubTypeOf(collectedClass)) {
                    return true;
                }
            }
        }
        return false;
    }

    private void lock() {
        JavaMetamodel.getDefaultRepository().beginTrans(false);
    }

    private void unlock() {
        JavaMetamodel.getDefaultRepository().endTrans(false);
    }
	
    private class LazyCollection extends AbstractCollection {
        MOFID res[];
        
        public LazyCollection(Resource res[]) {
            this.res = new MOFID[res.length];
            for (int i = 0; i < res.length; i++) {
                this.res[i] = ((MetadataElement) res[i])._getMofId();
            }
        }
        
        public Iterator iterator() {
            return new LazyIterator(res);
        }

        public int size() {
            lock();
            try {
                int size = 0;
                for (Iterator i = iterator(); i.hasNext(); i.next(), size++ );
                return size;
            } finally {
                unlock();
            }
        }

        public boolean isEmpty() {
            lock();
            try {
                return !iterator().hasNext();
            } finally {
                unlock();
            }
        }
    }
    
    private class LazyIterator implements Iterator, Cancellable {
        
        MOFID res[];
        Collection currentUsages = Collections.EMPTY_LIST;
        Iterator inner;
        int currentIndex;
        Object next;
        boolean hasNext;
        float step;
        int last;
        ProgressSupport progressSupport;
        private boolean cancelRequest;
        
        LazyIterator(MOFID[] res) {
            this.res = res;
            inner = currentUsages.iterator();
            currentIndex = -1;
            hasNext = true;
            
            step = 1;
            if (res.length > MAX_COUNT) {
                step = (float) MAX_COUNT / res.length;
            }
            last = 0;
        
            progressSupport = JavaMetamodel.getManager().getProgressSupport();
        }
                
        public void remove() {
            throw new UnsupportedOperationException();
        }
        
        public Object next() {
            lock();
            try {
                if (next==null) {
                    findNext();
                }
                Object result = next;
                next = null;
                return result;
            } finally {
                unlock();
            }
        }
        
        private void findNext() {
            if (!hasNext) {
                throw new NoSuchElementException();
            }
            while (next==null && hasNext == true) {
                if (!inner.hasNext()) {
                    if (cancelRequest)
                        return;                    
                    currentIndex++;
                    if (currentIndex == 0) {
                        progressSupport.fireProgressListenerStart(0, Math.min(res.length,MAX_COUNT));
                    }
                    if (currentIndex < res.length) {
                        MOFID mofId = res[currentIndex];
                        if (mofId != null) {
                            ResourceImpl resource = (ResourceImpl) ((NBMDRepositoryImpl) JMManager.getDefaultRepository()).getByMofId(mofId);
                            try {
                                if (resource != null && !resource.getName().endsWith("class")) { // NOI18N
                                    navigator = new ElementNavigator(resource, what);
                                    currentUsages = getUsers(resource);
                                    inner = currentUsages.iterator();
                                    if (!inner.hasNext()) {
                                        res[currentIndex] = null;
                                    }
                                } else {
                                    res[currentIndex] = null;
                                }
                            } catch (IllegalArgumentException e) {
                                JMManager.getLog().notify(ErrorManager.INFORMATIONAL, new Exception("Resource " + resource == null ? null : resource.getName() + " is not parsable.", e)); // NOI18N
                            }
                        }

                        if (currentIndex*step >= last) {
                            progressSupport.fireProgressListenerStep();
                            last++;
                        }

                    } else {
                        hasNext = false;
                        progressSupport.fireProgressListenerStop();
                    }
                }
                if (inner.hasNext()) {
                    next = inner.next(); 
                }
            }
        }
        
        public boolean hasNext() {
            lock();
            try {
                findNext();
                if (cancelRequest)
                    return false;
                return hasNext;
            } finally {
                unlock();
            }
        }
        
        public boolean cancel() {
            cancelRequest = true;
            return true;
        }
    }
}
